// Register block for I2C
// Inputs:
// SDA      - Data input bus for writing to registers
// SCL      - clocking signal that is driven by I2C initiator
// 
// Internal nets (connecting with I2C target):
// reg_en   - signal that enables the write/read operation in the register
// RW       - The current RW value in I2C target SDA
// addr     - The register address that the target is currently reading from or writing to.
// data     - Data that will be sent or received (according to RW signal) from I2C target
// 
// Outputs:
// pw_adj   (8) - Pulse width adjust for fx2
// en_mux       - Mux enable
// sel_mux  (2) - Mux select
// ampl_adj (2) - Clock driver amplitude adjust
// sr_adj   (2) - Clock driver slew rate adjust
//
// Registers:
// 00h - FX2_REG (8) - | pw_adj (8) |
// 01h - MUX_REG (8) - | 5'bx | en_mux (1) | sel_mux (2) |
// 02h - DRV_REG (8) - | 4'bx | ampl_adj (2) | sr_adj (2) | 

`timescale 1ns/1ps

module I2C_registers (
  inout  wire     SCL, // forced to EEnet through coercion
  inout  wire     SDA,
  output wire [7:0] pw_adj,
  output wire       en_mux,
  output wire [1:0] sel_mux,
  output wire [1:0] ampl_adj,
  output wire [1:0] sr_adj
  );
  
  parameter real Vsup_p = 1.8;
  
  wire [7:0] addr;
  wire [7:0] data;
  logic [7:0] data_int;
  wire RW, reg_en; // test
  //Internal registers
  reg [7:0] INT_REG [2:0];
  
  //Internal nets
  wire [7:0] FX2_REG, MUX_REG, DRV_REG;
  

  I2C_target #(.Vsup_p(Vsup_p)) target (.SCL(SCL), .SDA(SDA), .RW(RW), .reg_en(reg_en), .addr(addr), .data(data));
  
  always @(posedge reg_en) begin
    if(!RW && reg_en) begin  // target is reading from SDA
      if(addr < 3)
        INT_REG[addr] <= data;
    end
    else if (RW && reg_en) begin // writing to SDA 
      if(addr < 3)
        data_int <= INT_REG[addr];
      else // when the addr is out of range
        data_int <= 8'h00;
    end
    else
      data_int <= 8'h00;
  end

  assign data = (RW && reg_en) ? data_int : 8'bz; 
  assign FX2_REG = INT_REG[0];
  assign MUX_REG = INT_REG[1];
  assign DRV_REG = INT_REG[2];
  
  assign pw_adj = FX2_REG;
  assign en_mux = MUX_REG[2];
  assign sel_mux = MUX_REG[1:0];
  assign ampl_adj = DRV_REG[3:2];
  assign sr_adj = DRV_REG[1:0];
  
endmodule
