// 
//  (c) Copyright OCP-IP 2003, 2004
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//
//  Description : Layer Adapter, TL0-TL1 example, Asynchronous TL0 OCP Master (OCP2.0)
//  $Id: ocp_tl0_master_async_burst.cpp,v 1.1.1.1 2004/09/24 09:26:13 sguntz Exp $
//  
//    Features:
//    - Asynchronous TL0 master implementing OCP2.0 Basic and Burst Signals
//    - Propagation delays emulation for output signals
//    - Performs 5 WRITE operations followed by 5 READ operations
//    - For WRITE cells:
//        * MByteEn = 0x7
//        * MBurstLength = 5
//        * MBurstPrecise = true
//        * MBurstSingleReq = false
//        * MBurstSeq = INCR
//    - For READ cells:
//        * MByteEn = 0xf
//        * MBurstLength = 1
//        * MBurstPrecise = true
//        * MBurstSingleReq = false
//        * MBurstSeq = WRAP
//    - Checks received responses
//    
//  Parameter    :
//    Template arguments.
//     - TdataCl: Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//     - Td: Data type
//     - Ta: Address type
//                
//   Constructor arguments.
//     - sc_module_name: Name of the module instance
//     - sc_time RequestDelay:
//        delay between clock rising edge and 'MCmd' assertion 
//     - sc_time ResponseDelay:
//        delay between valid response reception ('SResp'==DVA)
//            and 'MRespAccept' signal assertion
// ============================================================================

// Used for debug purposes
#define DEBUG_TL0_MASTER_ASYNC_BURST

#include "ocp_tl0_master_async_burst.h"
#include "ocp_tl1_data_cl.h"

// ----------------------------------------------------------------------------
// Process : OCP_TL0_Master_async_burst::OCP_TL0_Master_async_burst
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_Master_async_burst<TdataCl>::OCP_TL0_Master_async_burst
       (
          sc_module_name name_
          , sc_time RequestDelay
          , sc_time ResponseDelay
        )
          : sc_module(name_)
          , m_RequestDelay(RequestDelay) 
          , m_ResponseDelay(ResponseDelay)
{
  // Test Constructor parameters
  if(ResponseDelay < RequestDelay) {
    cout << "Error: TL0 OCP Master Constructor" << endl
      << "       ResponseDelay' parameter must be >= 'RequestDelay' parameter" << endl
      << "       ResponseDelay=" << ResponseDelay
      << "       RequestDelay=" << ResponseDelay
      << endl;
    assert(0);
  }
  
  // Output ports initializing 
  MCmd.initialize(sc_bv<3>(OCP_MCMD_IDLE));
  MData.initialize(Td(0));
  MAddr.initialize(Ta(0));
  MByteEn.initialize(0);
  MBurstSeq.initialize(OCP_MBURSTSEQ_INCR);
  MBurstSingleReq.initialize(false);
  MBurstPrecise.initialize(true);
  MBurstLength.initialize(0);
  MRespAccept.initialize(false);

  // Response Processes
  SC_THREAD(GetResponse);
  sensitive << SResp;
  
  // Request Processes
  SC_THREAD(SendRequest);
  sensitive_pos << Clk;
   
} // end of constructor


// ----------------------------------------------------------------------------
// Process : OCP_TL0_Master_async_burst::~OCP_TL0_Master_async_burst
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_Master_async_burst<TdataCl>::~OCP_TL0_Master_async_burst()
{
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_Master_async_burst::GetResponse()
//
//  - Catch 'SResp' variations
//  - Compare Response with expected value
//  - Issue a Response acknowledge after 'ResponseDelay' ns 
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_Master_async_burst<TdataCl>::GetResponse()
{
  Td CorrectData = 0;
  Td Rdata;
  int i = 0;

  while(true)
  {
    do {
      wait(); // for an event on SResp
    }
    while(SResp.read() != sc_bv<2>(OCP_SRESP_DVA));

    Rdata = SData.read();

#ifdef DEBUG_TL0_MASTER_ASYNC_BURST
    cout << "TL0 Master got response "
      << " delta " << simcontext()->delta_count()
      << " time " << sc_time_stamp().to_seconds()
      << " received data " << Rdata ;
    if(Rdata != CorrectData++) 
      cout << " is BAD" << endl;
    else
      cout << " is good" << endl;
#endif

    wait(m_ResponseDelay);

    MRespAccept.write(true);

#ifdef DEBUG_TL0_MASTER_ASYNC_BURST
    cout << "TL0 Master released response "
      << " delta " << simcontext()->delta_count()
      << " time " << sc_time_stamp().to_seconds()
      << " data " << Rdata << endl;
#endif

    wait(Clk.posedge_event());
    MRespAccept.write(false);

    i++;
  } // end of while(true) loop

} // end of SC_THREAD


// ----------------------------------------------------------------------------
//  Method : OCP_TL0_Master_async_burst::SendRequest()
//  
//  Send OCP requests: 5 writes followed by 5 reads
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_Master_async_burst<TdataCl>::SendRequest()
{
  int i = 0;
  int j;
  int Pause[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0}; // # of cycles between requests;
  //int Pause[] = {1, 1, 1, 1, 1, 1, 1, 1, 1, 1}; // # of cycles between requests;
  //int Pause[] = {2, 2, 2, 2, 2, 2, 2, 2, 2, 2}; // # of cycles between requests;

  Ta Addr[] =     {0,4,8,12,16,0,4,8,12,16};    // MAddr field for each request
  Td SentData = 0;
  int cnt = 0;

  while(true)
  {
    if (cnt == 0)
    {
      // get new Pause value
      j=i;
      i++;
      cnt = Pause[i%10];

      // Delay Request signals
      // This must work with or without the next line
      wait(m_RequestDelay);

      // Set Request signals
      MAddr.write(Addr[j%10]);
      if(j%10<5) {
        MCmd.write(sc_bv<3>(OCP_MCMD_WR));
        MData.write(SentData++);
        MBurstSeq.write(OCP_MBURSTSEQ_INCR);
        MBurstSingleReq.write(false);
        MBurstPrecise.write(true);
        MBurstLength.write(5);
        MByteEn.write(0x7);
      }
      else {
        MCmd.write(sc_bv<3>(OCP_MCMD_RD));
        MBurstSeq.write(OCP_MBURSTSEQ_WRAP);
        MBurstSingleReq.write(false);
        MBurstPrecise.write(true);
        MBurstLength.write(1);
        MByteEn.write(0xf);
      }

#ifdef DEBUG_TL0_MASTER_ASYNC_BURST
      cout << "TL0 Master sent request "
        << " delta " << simcontext()->delta_count()
        << " time " << sc_time_stamp().to_seconds();
      if(j%10<5) 
        cout << " written data " << SentData-1 << endl;
      else 
        cout << endl;
#endif
      // Wait for slave acknowledge
      do {
        wait();
      } while (!SCmdAccept.read());

      MCmd.write(sc_bv<3>(OCP_MCMD_IDLE));

    }
    else
    {
      cnt--;
      wait();
    }
  } // end of while loop
} // end of SC_THREAD


// ----------------------------------------------------------------------------
//
//  Instantiation of the Master
//
// ----------------------------------------------------------------------------

template class OCP_TL0_Master_async_burst<OCP_TL1_DataCl<int,int>  >; // see ocp_tl1_globals.h
