// 
//  (c) Copyright OCP-IP 2003, 2004
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level Layer-1
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//                Anssi Haverinen, Nokia Inc., anssi.haverinen@nokia.com
//                Yann Bajot, Prosilog, bajot@prosilog.com
//  
//  Description : Transaction Level - Layer-1 example Slave
//  $Id: ocp_tl1_slave_sync_hs.cpp,v 1.1.1.1 2004/09/24 09:26:16 sguntz Exp $
//
//    Features:
//    - Synchronous slave using OCP 2.0 data class
//    - Non-blocking methods are used
//    - Pipelined Request and Response threads
//    - Datahandshake support
//                
// Parameter    :
//   Template arguments.
//     - TdataCl: Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//     - Td: Data type
//     - Ta: Address type
//                
//   Constructor arguments.
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Slave.
//           ID must be unique among all Slaves attached to the same Bus.
//     - StartAddress: First address of the Slaves memory.
//     - EndAddress: Last address of the Slaves memory.
//                   The address range of all Slaves in a system
//                   must be mutually distinct.
//     - RequestDelay: Non-negative number specifying the delay in percent
//         of a clock cycle, the requests are acknowledged after request
//         arrival.
//     - DataDelay: Non-negative number specifying the delay in percent
//         of a clock cycle, the data are acknowledged after data request
//         arrival.
//     - ResponseDelay: Non-negative number specifying the delay in percent
//         of a clock cycle, the response is sent after clock rising edge
//         following the request arrival.
//
//
// ============================================================================


// Used for debug purposes
#define DEBUG_TL1_SLAVE_SYNC_HS


#include "ocp_tl1_slave_sync_hs.h"
#include "ocp_tl1_data_cl.h"

// ----------------------------------------------------------------------------
// Process : OCP_TL1_Slave_sync_hs::OCP_TL1_Slave_sync_hs
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL1_Slave_sync_hs<TdataCl>::OCP_TL1_Slave_sync_hs
       (
          sc_module_name name_
        , int ID
        , Ta StartAddress
        , Ta EndAddress
        , double RequestDelay
        , double DataDelay
        , double ResponseDelay
        )
          : sc_module          (name_)
          , SlaveP             ("SlavePort")
          , m_ID               (ID)
          , m_StartAddress     (StartAddress)
          , m_EndAddress       (EndAddress)
          , m_RequestDelay     (RequestDelay)
          , m_DataDelay        (DataDelay)
          , m_ResponseDelay    (ResponseDelay)
{
  int i;

  // InitParameters();

  m_NumWordsPerBeat = sizeof(Td);
  m_SmemoryLength = (m_EndAddress - m_StartAddress + 1) / m_NumWordsPerBeat;

  // allcoate memory
  if (m_SmemoryLength > 0)
  {
    try
    {
      m_SlaveMemory = new Td[m_SmemoryLength];
    }
    catch(std::bad_alloc ex)
    {
      cerr << "ERROR: Memory allocation for Slave memory failed "
            << name() << ">" << endl;
      cerr << "       Size  is: " << m_SmemoryLength << endl;
      cerr << "       Error is: " << ex.what() << endl;
      sc_stop();
      return;
    }
  }

  for (i = 0; i < m_SmemoryLength; i++)
    m_SlaveMemory[i] = 0;

  for (i = 0; i < TL_SLAVE_FIFO_DEPTH; i++)
    m_RespEventS[i] = false;

  SC_THREAD(Request);
  sensitive << SlaveP.RequestStartEvent();
  SC_THREAD(DataRequest);
  sensitive << SlaveP.DataHSStartEvent();
  SC_THREAD(Response);
  sensitive_pos << clk;
}


// ----------------------------------------------------------------------------
// Process : OCP_TL1_Slave_sync_hs::~OCP_TL1_Slave_sync_hs
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL1_Slave_sync_hs<TdataCl>::~OCP_TL1_Slave_sync_hs()
{
  delete [] m_SlaveMemory;
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Slave_sync_hs::Request()
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Slave_sync_hs<TdataCl>::Request()
{
  int i = 0;
  int ReqIdx = 0;
  int Pause[] = {0, 0, 0, 0, 0, 1, 1, 1, 2, 3};
//  int Pause[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
  int cnt = 0;

  while(true)
  {
    wait(); // for request start event

    if (SlaveP->getOCPRequest(TL1_request,false))
    {
      cnt = Pause[i%10];
      i++;
      // check which request was posted
      if (TL1_request.MCmd != OCP_MCMD_IDLE)
      {
        if (TL1_request.MCmd == OCP_MCMD_RD)
        {
          // send delayed response 
          m_RespEventS[ReqIdx] = true;

          // get new index (only for those requests, where the
          // response thread is called)
          ReqIdx++;
          if (ReqIdx >= TL_SLAVE_FIFO_DEPTH)
            ReqIdx = 0;
        }

#ifdef DEBUG_TL1_SLAVE_SYNC_HS
      cout << "Slave got request "
           << "  delta = " << simcontext()->delta_count()
           << "  time  = " << sc_time_stamp().to_seconds()
           << "  addr  = " << TL1_request.MAddr << endl;
#endif

        // Delay data signal
        // This must work with or without the next line
        if (m_RequestDelay > 0)
          wait(TL_CLK_PERIOD * (m_RequestDelay/100.0), TL_TIME_UNIT);

        SlaveP->putSCmdAccept();

#ifdef DEBUG_TL1_SLAVE_SYNC_HS
      cout << "Slave released request "
           << "  delta = " << simcontext()->delta_count()
           << "  time  = " << sc_time_stamp().to_seconds()
           << "  addr  = " << TL1_request.MAddr
           << endl;
#endif

      }
    }
  } // end of while(true) loop
} // end of method

// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Slave_sync_hs::DataRequest()
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Slave_sync_hs<TdataCl>::DataRequest()
{
  int i = 0;
  int Pause[] = {0, 0, 0, 0, 0, 1, 1, 1, 2, 3};
  //  int Pause[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
  int cnt = 0;

  while(true)
  {
    wait(); // for dataHS start event

    if (SlaveP->getOCPDataHS(TL1_dataHS,false))
    {
      cnt = Pause[i%10];
      i++;

#ifdef DEBUG_TL1_SLAVE_SYNC_HS
      cout << "Slave got data "
        << "  delta = " << simcontext()->delta_count()
        << "  time  = " << sc_time_stamp().to_seconds()
        << "  data  = " << TL1_dataHS.MData
        << endl;
#endif

      // Delay data signal
      // This must work with or without the next line
      if (m_DataDelay > 0)
        wait(TL_CLK_PERIOD * (m_DataDelay/100.0), TL_TIME_UNIT);

      SlaveP->putSDataAccept();

#ifdef DEBUG_TL1_SLAVE_SYNC_HS
      cout << "Slave released data "
        << "  delta = " << simcontext()->delta_count()
        << "  time  = " << sc_time_stamp().to_seconds()
        << "  data  = " << TL1_dataHS.MData << endl;
#endif

    }
  } // end of while(true) loop
} // end of method



// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Slave_sync_hs::Response()
//
//  Top level response method (synchronous)
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Slave_sync_hs<TdataCl>::Response()
{
  Td Rdata = 0;
  int ResIdx = 0;

  while(true)
  {
    while(!m_RespEventS[ResIdx])
      wait(); // wait for a response to send

    if(m_ResponseDelay > 0)
      wait(TL_CLK_PERIOD * (m_ResponseDelay/100.0), TL_TIME_UNIT);

    TL1_response.SResp = OCP_SRESP_DVA;
    TL1_response.SData = Rdata++;
    

    // reset + get new index
    m_RespEventS[ResIdx] = false;
    ResIdx++;
    if (ResIdx >= TL_SLAVE_FIFO_DEPTH)
      ResIdx = 0;

#ifdef DEBUG_TL1_SLAVE_SYNC_HS
      cout << "Slave sent response "
           << "  delta = " << simcontext()->delta_count()
           << "  time  = " << sc_time_stamp().to_seconds()
           << "  SResp= " << TL1_response.SResp
           << "  data  = " << TL1_response.SData << endl;
#endif

    // send response to Master
   bool ret = SlaveP->startOCPResponse(TL1_response);
   // For Debug purposes only
   assert(ret);

    do {
      wait();
    } while (!SlaveP->getMRespAccept());
  }
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Slave_sync_hs::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Slave_sync_hs<TdataCl>::end_of_elaboration()
{
  sc_module::end_of_elaboration();

  // Initialize debug interface
  SlaveP->SregisterDirectIF(this);

  // Put parameter into Channel
  // No parameters needed
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Slave_sync_hs::MPutDirect()
//
//  Debug interface method.
//  Read/Write data, Master to Slave direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl> bool OCP_TL1_Slave_sync_hs<TdataCl>::MputDirect(
         int MasterID, bool IsWrite, Td *Data, Ta Address, int NumWords)
{
  int i;
  int NumBeats = NumWords / m_NumWordsPerBeat;
  int Offset = (Address - m_StartAddress) / m_NumWordsPerBeat;

  if ((Offset >= 0) && ((Offset + NumBeats) <= m_SmemoryLength))
  {
    if (IsWrite)
    {
      // write request
      for (i = 0; i < NumBeats; i++)
        m_SlaveMemory[Offset + i] = Data[i];
    }
    else
    {
      // read request
      for (i = 0; i < NumBeats; i++)
        Data[i] = m_SlaveMemory[Offset + i];
    }
    return(true);
  }
  else
  {
    cout << "Error: Address out of range at Slave " << m_ID << endl
         << "       Address=" << Address << "  StartAddr=" << m_StartAddress
         << "  _EndAddr=" << m_EndAddress
         << "  NumWords=" << NumWords
         << endl;
    return(false);
  }
}


// ----------------------------------------------------------------------------
//
//  Instantiation of the Slave
//
// ----------------------------------------------------------------------------
template class OCP_TL1_Slave_sync_hs<OCP_TL1_DataCl<int,int>  >; // see ocp_tl1_globals.h
