// 
//  (c) Copyright OCP-IP 2003, 2004
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//                Stephane Guntz, PROSILOG, guntz@prosilog.com
//
//  Description : Layer Adapter, TL0-TL1 example
//             Top level, TL0 synchronous Master <---> TL1 synchronous Slave
//             with datahandshake
//  $Id: top_master_sync_hs.cpp,v 1.1.1.1 2004/09/24 09:26:18 sguntz Exp $
//
//  This example shows an OCP connexion between a synchronous TL0 Master and
//    a TL1 Slave.
//    
//  The master issues 5 writes followed by 5 reads, and checks received data.
//    
//  Operation timings are controlled by the following parameters: T1, T2, T3,
//  T4 ,T5 ,T6, T7 and T8
//    - T1, T3, T6, T7 and T8 are expressed as 'sc_time' values
//    - T2, T4 and T5 are expressed as a percentage (double) of the clock cycle 
//    - Clock cycle duration in this example is 10 ns
//  
//  OCP transfers occurs such as follows: 
//    
//    * 'T1' ns after the clock rising edge, TL0 Master sends a request 
//    * 'T7' ns after the clock rising edge, TL0 Master Adapter samples
//        request signals group, and issues a request call to the TL1 channel
//        depending on the 'MCmd' value.
//    * After request reception, TL1 slave waits for:
//                  - 'T2*clock_cycle/100' ns and releases the request
//                  - 'one cycle + T5*clock_cycle/100' ns and issues the
//                    response (for READ operations)
//    WRITE OPERATION:
//    * One cycle later following request sending, TL0 Master sends a data
//      request with a delay of T3
//    * 'T7' ns after the clock rising edge, TL0 Master Adapter samples
//        datahandshake group signals , and issues a data request call to the
//        TL1 channel depending on the 'MDataValid' value.
//    * After data request reception, TL1 slave waits for 'T4*clock_cycle/100'
//      ns and releases the data
//
//    READ OPERATION:
//    * One cycle later following request sending, TL1 Slave sends a response
//      with a delay of T5
//    * TL0 Master Adapter catches the TL1 response and drives corresponding
//      TL0 signals
//    * 'T6' ns after the response assertion, TL0 Master set the 'MRespAccept'
//      signal.
//    * 'T8' ns after the clock rising edge, TL0 Master Adapater samples
//      'MRespAccept' and conditionally releases the TL1 response.
//
//  T7 & T8 specifies the sample delays for the TL0 Master Adapter.  To get a
//  correct behaviour, T7 must be larger than 'max(T1,T3)' (Request and
//  Datahandshake signals) and T8 must be larger than 'T5 + T6' ('MRespAccept'
//  signal).
//
//  TL0 signals can be viewed using the VCD file 'top_master_sync_hs.vcd'
//    
// ============================================================================

#include <map>
#include <string>

#include "ocp_tl0_master_sync_hs.h"
#include "ocp_tl1_slave_sync_hs.h"
#include "ocp_tl0_tl1_master_adapter_hs.h"
#include "ocp_tl1_data_cl.h"
#include "ocp_tl1_channel.h"  


// Timings 
const sc_time_unit TL_TIME_UNIT = SC_NS; 
const double TL_CLK_PERIOD = 10 ;     // clock period

void readMapFromFile(const string &myFileName, MapStringType &myParamMap) 
{
    // read pairs of data from the passed file
    string leftside;
    string rightside;
    
    // (1) open the file
    ifstream inputfile(myFileName.c_str());
    assert( inputfile );

    // set the formatting
    inputfile.setf(std::ios::skipws);

    // Now read through all the pairs of values and add them to the passed map
    while ( inputfile ) {
        inputfile >> leftside;
        inputfile >> rightside;
        myParamMap.insert(std::make_pair(leftside,rightside));
    }

    // All done, close up
    inputfile.close();
}



int sc_main(int, char*[])
{
    MapStringType   ocpParamMap;
   string  ocpParamFileName = "ocpParams_master_adapter_hs_tl0_tl1";
   
   if ( ! ocpParamFileName.empty() ) {
        readMapFromFile(ocpParamFileName, ocpParamMap);
    }
    
    // TL1 channel
    //
    // 
    bool sync = true; // mandatory for OCP data classes
    bool defaultevent = true; // mandatory for the tl1 master used in this example 
    // 'SyncEvent' must be set for the adapter to work correctly
    bool syncevent = true;
    OCP_TL1_Channel<OCP_TL1_DataCl<int,int> > ch1("ch0", sync, syncevent, defaultevent, NULL, TL_CLK_PERIOD, TL_TIME_UNIT,"ch0.ocp",true);
    ch1.setConfiguration(ocpParamMap);
    

    // TL0 synchronous Master with Datahandshake 
    //
    // 
    // Request group is valid after 'T1' ns
    sc_time T1 = sc_time(1,SC_NS);     
    // Datahandshake signals are valid after 'T3' ns
    sc_time T3 = sc_time(2,SC_NS);     
    // Response acknowledge is valid after 'T6' ns, following the response arrival 
    sc_time T6 = sc_time(2,SC_NS);     
 
    OCP_TL0_Master_sync_hs<OCP_TL1_DataCl<int,int>  > ms0("ms0", T1, T3, T6 );

    // TL1 Synchronous Slave with Datahandshake 
    //
    // 
    double T2 = 10;     // Request acknowledge is valid after 'T2'% of the clock
                        // cycle following the request detection
    double T4 = 10;     // Data Request acknowledge is valid after 'T4'% of the clock
                        // cycle following the data request detection
    double T5 = 20;    // Response is sent after 'T5'% of the clock cycle
    OCP_TL1_Slave_sync_hs<OCP_TL1_DataCl<int,int>  > sl1("sl1", 1, 0x0, 0x3ff, T2, T4, T5);


    // TL0 Master Adapter with Datahandshake
    //
    // Sample delay T7 is used to sample Request group and Datahandshake group signals: it must be > max(T1,T3) in this case (asynchronous request/datahandshake acknowledge)
    // Sample delay T8 is used to sample MRespAccept signal: it must be > T5*clock_cycle/100 + T6 in this case (asynchronous response acknowledge)
    // check_setup_time parameter is actived to check TL0 setup time violations
    
    sc_time T7 = sc_time(3,SC_NS);
    sc_time T8 = sc_time(5,SC_NS);

    bool check_setup_time = true;
    OCP_TL0_TL1_Master_Adapter_Hs<OCP_TL1_DataCl<int,int>  > ms_adapter("ms_adapter", 1, 1, T7, T8, check_setup_time );


    // Testbench building
    //
    //
    sc_clock clk("clock", TL_CLK_PERIOD, TL_TIME_UNIT);

    sc_signal<sc_bv<3> > 	s_MCmd;
    sc_signal<int>       	s_MAddr;
    sc_signal<bool> 	    s_SCmdAccept;
    sc_signal<bool> 	    s_MDataValid;
    sc_signal<int>       	s_MData;
    sc_signal<bool> 	    s_SDataAccept;
    sc_signal<int>       	s_SData;
    sc_signal<sc_bv<2> >  s_SResp;
    sc_signal<bool>  	    s_MRespAccept;



    sl1.clk(clk);
    sl1.SlaveP(ch1);

    ms_adapter.MasterP(ch1);

    ms_adapter.Clk(clk);
    ms_adapter.MCmd(s_MCmd);
    ms_adapter.MAddr(s_MAddr);
    ms_adapter.SCmdAccept(s_SCmdAccept);
    ms_adapter.MDataValid(s_MDataValid);
    ms_adapter.MData(s_MData);
    ms_adapter.SDataAccept(s_SDataAccept);
    ms_adapter.SData(s_SData);
    ms_adapter.SResp(s_SResp);
    ms_adapter.MRespAccept(s_MRespAccept);

    ms0.Clk(clk);
    ms0.MCmd(s_MCmd);
    ms0.MAddr(s_MAddr);
    ms0.SCmdAccept(s_SCmdAccept);
    ms0.MDataValid(s_MDataValid);
    ms0.MData(s_MData);
    ms0.SDataAccept(s_SDataAccept);
    ms0.SData(s_SData);
    ms0.SResp(s_SResp);
    ms0.MRespAccept(s_MRespAccept);

    // VCD Trace file for TL0 signals
    //
    //
    sc_trace_file *vcd_file = sc_create_vcd_trace_file("top_master_sync_hs"); 

    sc_trace(vcd_file,clk,"Clk");
    sc_trace(vcd_file,s_MCmd,"MCmd");
    sc_trace(vcd_file,s_MAddr,"MAddr");
    sc_trace(vcd_file,s_SCmdAccept,"SCmdAccept");
    sc_trace(vcd_file,s_MDataValid,"MDataValid");
    sc_trace(vcd_file,s_MData,"MData");
    sc_trace(vcd_file,s_SDataAccept,"SDataAccept");
    sc_trace(vcd_file,s_SData,"SData");
    sc_trace(vcd_file,s_SResp,"SResp");
    sc_trace(vcd_file,s_MRespAccept,"MRespAccept");


    sc_start(500, SC_NS);

    sc_close_vcd_trace_file(vcd_file);

    return(0);
}
