// 
//  (c) Copyright OCP-IP 2003, 2004
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//                Stephane Guntz, PROSILOG, guntz@prosilog.com
//
//  Description : Layer Adapter, TL0-TL1 example
//              Top level, TL1 Synchronous Master <---> TL0 Synchronous Slave (OCP 2.0)
//             (with Datahandshake support)
//  $Id: top_slave_sync_hs.cpp,v 1.1.1.1 2004/09/24 09:26:18 sguntz Exp $
//
//  This example shows an OCP connexion between a synchronous TL0 Slave and
//  a TL1 Master supporting Datahandshake.
//    
//  The master issues 5 writes followed by 5 reads, and checks received data
//    assuming that OCP Target acts as a RAM.
//    
//  Operation timings are controlled by the following parameters: T1, T2, T3,
//  T4 ,T5, T6, T7 and T8
//    - T2, T4, T5, T7 and T8 are expressed as 'sc_time' values
//    - T1, T3 and T6 are expressed as a percentage (double) of the clock cycle 
//    - Clock cycle duration in this example is 10 ns
//  
//  OCP transfers occurs such as follows:
//    * 'T1*clock_cycle/100' ns after the clock rising edge, TL1 Master sends a
//      request 
//    * TL0 Slave adapter catchs the request and set the TL0 Request Signal
//      group
//    * After request reception, TL0 slave waits for:
//                  - 'T2' ns and releases the request
//                  - Next cycle + 'T5' ns and issues the response (for READ
//                    operations)
//    * As soon as 'SCmdAccept' is set, TL0 adapter releases the TL1 request

//    WRITE OPERATION:
//    * One cycle later following request sending, TL1 Master sends data with a
//      propagation delay of T3
//    * After data reception, TL0 slave waits for'T4' ns and releases the data
//    
//    READ OPERATION:
//    * As soon as 'SResp' is set, TL0 adapter sends the TL1 response
//    * After Response reception, TL1 Master waits for T6 and releases
//      the TL1 response
//    * As soon as TL1 Response is released, TL0 Adapter set 'MRespAccept'
//      signal
//
//  T7 & T8 specifies the sample delays for the TL0 Slave Adapter.
//  To get a correct behaviour, T7 must be larger than 'max(T1+T2,T3+T4)'
//  (Accept signals) and T8 must be larger than 'T5' (response signals).
//
//  TL0 signals can be viewed using the VCD file 'top_slave_sync_hs.vcd'
//    
// ============================================================================

#include <map>
#include <string>

#include "ocp_tl1_master_async_hs.h"
#include "ocp_tl0_slave_sync_hs.h"
#include "ocp_tl0_tl1_slave_adapter_hs.h"
#include "ocp_tl1_data_cl.h"
#include "ocp_tl1_channel.h"  

// Timings 
const sc_time_unit TL_TIME_UNIT = SC_NS; 
const double TL_CLK_PERIOD = 10 ;     // clock period

void readMapFromFile(const string &myFileName, MapStringType &myParamMap) 
{
    // read pairs of data from the passed file
    string leftside;
    string rightside;
    
    // (1) open the file
    ifstream inputfile(myFileName.c_str());
    assert( inputfile );

    // set the formatting
    inputfile.setf(std::ios::skipws);

    // Now read through all the pairs of values and add them to the passed map
    while ( inputfile ) {
        inputfile >> leftside;
        inputfile >> rightside;
        myParamMap.insert(std::make_pair(leftside,rightside));
    }

    // All done, close up
    inputfile.close();
}


int sc_main(int, char*[])
{
   MapStringType   ocpParamMap;
   string  ocpParamFileName = "ocpParams_slave_adapter_hs_tl0_tl1";
   
	if ( ! ocpParamFileName.empty() ) {
        readMapFromFile(ocpParamFileName, ocpParamMap);
    }
	
    // TL1 channel
    //
    // 
    bool sync = true; // mandatory for OCP data classes
    bool defaultevent = true; // mandatory for the tl1 master used in this example 
    // 'SyncEvent' must be set fot the adapter to work correctly
    bool syncevent = true;
    OCP_TL1_Channel<OCP_TL1_DataCl<int,int> > ch1("ch0", sync, syncevent, defaultevent, NULL, TL_CLK_PERIOD, TL_TIME_UNIT,"ch0.ocp",true);
    ch1.setConfiguration(ocpParamMap);
        
    // TL1 Master
    //
    // 
    // Request group is valid after 'T1'% of the clock cycle
    double T1 = 10;     
    // Datahandshake signals are valid after 'T3'% of the clock cycle 
    double T3 = 10;     
    // Response acknowledge is valid after 'T6'% of the clock cycle, following the response arrival 
    // T6 value > 100% is used to emulate a sequential path for 'MRespAccept' signal
    double T6 = 10;     

    OCP_TL1_Master_Async_hs<OCP_TL1_DataCl<int,int>  > ms1("ms1", 1, 1, T1, T3, T6);
    
    
    // TL0 Synchronous Slave with Datahandshake 
    //
    // 
    sc_time T2 = sc_time(2, SC_NS);   // Request delay
    sc_time T4 = sc_time(1, SC_NS);   // Data delay
    sc_time T5 = sc_time(1, SC_NS);   // Response delay
    OCP_TL0_Slave_sync_hs<OCP_TL1_DataCl<int,int> > sl0("sl0", T2, T4, T5);

    // TL0 Slave Adapter 
    //
    // Sample delay T7 is used to sample Accept signals (SCmdAccept and SDataAccept):
    //      it must be > max(T1+T2,T3+T4) in this case 
    // Sample delay T8 is used to sample Response signals: it must be > T5 in this case
    // check_setup_time parameter is actived to check TL0 setup time violations
    sc_time T7 = sc_time(4, SC_NS);
    sc_time T8 = sc_time(2, SC_NS);
    bool check_setup_time = true;
    OCP_TL0_TL1_Slave_Adapter_Hs<OCP_TL1_DataCl<int,int> > sl_adapter("slave_adapter", 1, T7, T8, check_setup_time );

    // Default Constructor
    //OCP_TL0_TL1_Slave_Adapter_Hs<TL1_TEMPL_DATA_CL > sl_adapter("slave_adapter", 1, true, sc_time(TL_CLK_PERIOD,TL_TIME_UNIT), check_setup_time );


    // Testbench building
    //
    //
   
    sc_clock clk("clock", TL_CLK_PERIOD, TL_TIME_UNIT);
    sc_signal<sc_bv<3> > 	s_MCmd;
    sc_signal<int>       	s_MAddr;
    sc_signal<bool> 	    s_SCmdAccept;
    sc_signal<bool> 	    s_MDataValid;
    sc_signal<int>       	s_MData;
    sc_signal<bool> 	    s_SDataAccept;
    sc_signal<int>       	s_SData;
    sc_signal<sc_bv<2> >  s_SResp;
    sc_signal<bool>  	    s_MRespAccept;



    sl_adapter.Clk(clk);
    sl_adapter.MCmd(s_MCmd);
    sl_adapter.MAddr(s_MAddr);
    sl_adapter.SCmdAccept(s_SCmdAccept);
    sl_adapter.MDataValid(s_MDataValid);
    sl_adapter.MData(s_MData);
    sl_adapter.SDataAccept(s_SDataAccept);
    sl_adapter.SData(s_SData);
    sl_adapter.SResp(s_SResp);
    sl_adapter.MRespAccept(s_MRespAccept);

    sl0.Clk(clk);
    sl0.MCmd(s_MCmd);
    sl0.MAddr(s_MAddr);
    sl0.SCmdAccept(s_SCmdAccept);
    sl0.MDataValid(s_MDataValid);
    sl0.MData(s_MData);
    sl0.SDataAccept(s_SDataAccept);
    sl0.SData(s_SData);
    sl0.SResp(s_SResp);
    sl0.MRespAccept(s_MRespAccept);
    
    sl_adapter.SlaveP(ch1);
    ms1.MasterP(ch1);
    ms1.clk(clk);
    
    // VCD Trace file for TL0 signals
    //
    //
    sc_trace_file *vcd_file = sc_create_vcd_trace_file("top_slave_sync_hs"); 

    sc_trace(vcd_file,clk,"Clk");
    sc_trace(vcd_file,s_MCmd,"MCmd");
    sc_trace(vcd_file,s_MAddr,"MAddr");
    sc_trace(vcd_file,s_SCmdAccept,"SCmdAccept");
    sc_trace(vcd_file,s_MDataValid,"MDataValid");
    sc_trace(vcd_file,s_MData,"MData");
    sc_trace(vcd_file,s_SDataAccept,"SDataAccept");
    sc_trace(vcd_file,s_SData,"SData");
    sc_trace(vcd_file,s_SResp,"SResp");
    sc_trace(vcd_file,s_MRespAccept,"MRespAccept");


    sc_start(500, SC_NS);

    sc_close_vcd_trace_file(vcd_file);

    return(0);
}
