// 
//  (c) Copyright OCP-IP 2003, 2004
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//				  Anssi Haverinen, Nokia Inc., anssi.haverinen@nokia.com
//				  modified by by Joe Chou, Sonics Inc., joechou@sonicsinc.com,
//				  Yann Bajot, PROSILOG, bajot@prosilog.com 
//                Stephane Guntz, PROSILOG, guntz@prosilog.com)
//
//  Description : Transaction Level - Layer-2 example Slave
//  $Id: ocp_tl2_slave.cpp,v 1.1.1.1 2004/09/24 09:26:20 sguntz Exp $
// 
//  Features: 
//     - Blocking methods are used.
//     - Slave puts data into his memory (write request)
//       or sends data from his memory to the Master (read request).
//     - Slave can be pipelined or non-pipelined.
//     - In the pipelined case the Slave uses FIFO to not drop
//       requests.
//                
// Parameter    : 
//   Template arguments.
//   - TdataCl: Data class containing data members and
//              access methods for the data exchanged between
//              Masters and Slaves
//   - Td: Data type
//   - Ta: Address type
//                
//   Constructor arguments.
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//     - StartAddress: First address of the Slaves memory.
//     - EndAddress: Last address of the Slaves memory.
//                   The address range of all Slaves in a system
//                   must be mutually distinct.
//     - Pipelined: Switch to change between non-pipelined and
//                  pipelined operation mode of the Master.
//     - WriteRequest: Switch to enable/disable the sending of a
//                     response packet to a Master write request.
//                     Note that all modules involved in a system
//                     must use the same value.
//     - ReadAcceptCycles: Number of cycles per beat the Slave waits
//                         before accepting a read request.
//     - WriteAcceptCycles: Number of cycles per beat the Slave waits
//                          before accepting a write request.
//     - ReadResponseCycles: Number of cycles per beat the Slave waits
//                           before starting a read response.
//     - WriteResponseCycles: Number of cycles per beat the Slave waits
//                            before starting a write response.
//
// ============================================================================

// Pipelined versus non-pipelined operation mode
// ---------------------------------------------
// The Slave can be pipelined or non-pipelined.
// In the non-pipelined case the requests and responses
// are handled sequentially in a single thread. This is pretty secure
// against deadlocks.
// In the piplined case the request and reponse methods are called from
// two different threads. The response thread starts after the request
// thread has triggered an event.
//
// Timing
// ------
// The Slave has the following timing:
// Non-pipelined case:
// - Write request transfer: 
//         Accept  : wait NumWords * WriteAcceptCycles cycles
//         Response: wait WriteResponseCycles cycles
// - Read  request  transfer:
//         Accept  : wait ReadAcceptCycles cycles
//         Response: wait NumWords * ReadResponseCycles cycles
// Pipelined case:
// - Write request transfer: 
//         Accept  : wait NumWords * WriteAcceptCycles cycles
//         Response: wait WriteResponseCycles cycles
// - Read  request  transfer:
//         Accept  : wait ReadAcceptCycles cycles
//         Response: wait NumWords * ReadResponseCycles cycles
// Note that in the non-pipelined case the response is started after
// the request, while in the pipelined case request and response are started
// in parallel.
// ============================================================================

#include "ocp_tl2_slave.h"


//#define DEBUG_G1

// ----------------------------------------------------------------------------
// Process : OCP_TL2_Slave::OCP_TL2_Slave
// 
// Constructor
// ----------------------------------------------------------------------------
OCP_TL2_Slave::OCP_TL2_Slave(sc_module_name name_, int ID,
  Ta StartAddress, Ta EndAddress, bool Pipelined, bool WriteResponse,
  int ReadAcceptCycles, int WriteAcceptCycles, int ReadResponseCycles,
  int WriteResponseCycles)
   : sc_module(name_),
     SlaveP("SlavePort"),
     m_ID(ID),
     m_StartAddress(StartAddress),
     m_EndAddress(EndAddress),
     m_Pipelined(Pipelined),
     m_WriteResponse(WriteResponse),
     m_ReadAcceptCycles(ReadAcceptCycles),
     m_WriteAcceptCycles(WriteAcceptCycles),
     m_ReadResponseCycles(ReadResponseCycles),
     m_WriteResponseCycles(WriteResponseCycles),
     m_ReqTaNum(0),
     m_ResTaNum(0)
{
  int i;

  // InitParameters();

  //
  m_NumBytesPerWord = sizeof(Td);
  m_SmemoryLength = (m_EndAddress - m_StartAddress + 1) / m_NumBytesPerWord;

  // allocate memory
  if (m_SmemoryLength > 0) {
    try {
      m_SlaveMemory = new Td[m_SmemoryLength];
    }
    catch(std::bad_alloc ex) {
      cerr << "ERROR: Memory allocation for Slave memory failed "
            << name() << ">" << endl;
      cerr << "       Size  is: " << m_SmemoryLength << endl;
      cerr << "       Error is: " << ex.what() << endl;
      sc_stop();
      return;
    }
  }

  //change internal memory for debugging purposes
  for (i = 0; i < m_SmemoryLength; i++)
    m_SlaveMemory[i]=i;


  for (i = 0; i < TL_SLAVE_TL2_FIFO_DEPTH; i++)  {
    m_SlaveRespWait[i] = true;
	last_chunk[i]=false;
  }

  //
  if (m_Pipelined) {
    SC_THREAD(SlaveRequest);

    SC_THREAD(SlaveResponse);
    sensitive << m_SlaveRespEvent;
  } else {
    SC_THREAD(Slave);
  }
}

// ----------------------------------------------------------------------------
// Process : OCP_TL2_Slave::~OCP_TL2_Slave
// 
// Destructor
// ----------------------------------------------------------------------------
OCP_TL2_Slave::~OCP_TL2_Slave()
{
  delete [] m_SlaveMemory;
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Slave::Slave()
//
//  Top level method of the Slave (non-pipelined case)
//
// ----------------------------------------------------------------------------

void OCP_TL2_Slave::Slave()
{
  Ta            Address;
  OCPSRespType  Sresp;
  int           Sconnid;
  OCPRequestGrp<Td, Ta> req;
  OCPResponseGrp<Td> resp; 
  unsigned int chunk_length;
  bool last_chunk_of_burst;

  // assuming the master side will pass in valid data pointers
  Td*           Mdata;
  Td*           Sdata;

  while (true) {
    // Get a new OCP request; use blocking methods
    if ((SlaveP->getOCPRequestBlocking(req, false, chunk_length, last_chunk_of_burst))) {
      // no release

      // retrieve the request
      Address=req.MAddr;

      //
      if (req.MCmd==OCP_MCMD_WR) {
        // write request
 
        // retrieve the write data and OCP word length
        Mdata=req.MDataPtr;
		Sconnid = req.MConnID;

        // Acceptance delay
        wait(chunk_length*m_WriteAcceptCycles,TL_TIME_SCALE);

#ifdef DEBUG_S
        PrintTrans(m_ResTaNum,Address,chunk_length," Blocking Response ");
#endif
#ifdef DEBUG_G1
        PrintTime("Accept write request in Slave ",m_ID,Address);
#endif

        // accept the write request

        // write data into the memory (the direct method call is just
        // a reuse of an existing method, and there is nothing to do with
        // the debug interface).
        Sresp = (MputDirect(m_BusID,true,Mdata,Address,chunk_length))?
                OCP_SRESP_DVA : OCP_SRESP_ERR;

        // Release the OCP request channel
       	SlaveP->putSCmdAccept() ;

        // generate a write response, if needed
        if (m_WriteResponse) {
          // Response delay
          wait(m_WriteResponseCycles,TL_TIME_SCALE);

          // response phase

          // setup a write response
 		  resp.SResp=Sresp;
			
#ifdef DEBUG_G1
          PrintTime("Start write response in Slave ",m_ID,Address);
#endif
 
          // Send write response
		  if (!SlaveP->sendOCPResponseBlocking(resp, chunk_length, last_chunk_of_burst)) {
            // Deep ship: Should never happen
            cout << "Error: Master (" << m_BusID << ") dropped Slave ("
                 << m_ID << ") response <" << name() << ">" << endl;
          }
          m_ResTaNum++;
        }
      } else if(req.MCmd==OCP_MCMD_RD){
        // read request
 
        // retrieve the read request OCP word length and a return data pointer
   	    Sdata=req.MDataPtr;
	    Sconnid=req.MConnID;

        // Acceptance delay
        wait(m_ReadAcceptCycles,TL_TIME_SCALE);

#ifdef DEBUG_G1
        PrintTime("Accept read request in Slave ",m_ID,Address);
#endif
#ifdef DEBUG_S
        PrintTrans(m_ResTaNum,Address,chunk_length," Blocking Response ");
#endif

        // accept the read request

        // read the Data (the direct method call is just a reuse of an
        // existing method, and there is nothing to do with the debug
        // interface).
        Sresp = (MputDirect(m_BusID,false,Sdata,Address,chunk_length))?
                OCP_SRESP_DVA : OCP_SRESP_ERR;

        // Release the OCP request Channel
        SlaveP->Srelease();

        // Response delay
        wait(chunk_length*m_ReadResponseCycles,TL_TIME_SCALE);

        // Response phase

#ifdef DEBUG_G1
        PrintTime("Start read response in Slave ",m_ID,Address);
#endif

        // setup a read response
  		resp.SResp=Sresp;
		resp.SDataPtr=Sdata;

        // Send response to Master/Bus
		if (!SlaveP->sendOCPResponseBlocking(resp, chunk_length, last_chunk_of_burst)) {

          // Deep ship: Should never happen
          cout << "Error: Master (" << m_BusID << ") dropped Slave ("
               << m_ID << ") response <" << name() << ">" << endl;
        }

        m_ResTaNum++;
      } // end of read branch
    } // end of SgetRequest() branch
  } // end of while (true) loop
} // end of Slave method

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Slave::SlaveRequest()
//
//  Top level request method of the Slave (pipelined case)
//
// ----------------------------------------------------------------------------
void OCP_TL2_Slave::SlaveRequest()
{
  //TO DO: CHECK parameter usefulness
  int ReqIdx = 0;
  OCPRequestGrp<Td, Ta> req;
  unsigned int chunk_length;
  bool last_chunk_of_burst;

  while (true) {
    // make sure there is a space left for another outstanding request
    while (m_ReqTaNum - m_ResTaNum >= TL_SLAVE_TL2_FIFO_DEPTH) {
      // FIFO full - need to work on the response backlog

#ifdef DEBUG_S
      cout << "Slave (" << m_ID << ") waits to process responses <"
           << name() << ">, m_ReqTaNum: " << m_ReqTaNum
           << " m_ResTaNum:" << m_ResTaNum << endl;
#endif

      // skip a few cycles
      wait(TL_SLAVE_TL2_FIFO_DEPTH*(1+m_ReadResponseCycles),TL_TIME_SCALE);
    }

    // use the blocking method to get a new request, do not release the channel
 	if ((SlaveP->getOCPRequestBlocking(req, false, chunk_length, last_chunk_of_burst))) {
      // get the new request and store it in the FIFO
      if(req.MCmd==OCP_MCMD_WR)
		m_Write[ReqIdx] = true;
	  else if(req.MCmd==OCP_MCMD_RD)
		m_Write[ReqIdx] = false;
	  m_Address[ReqIdx] = req.MAddr;
	  m_MconnID[ReqIdx] = req.MConnID;
	  m_MthreadID[ReqIdx]= req.MThreadID;
	 
      m_NumWords[ReqIdx]=chunk_length;

		#ifdef DEBUG_S
			  PrintTrans(m_ReqTaNum,m_Address[ReqIdx],m_NumWords[ReqIdx],
						 " Blocking Request ");
		#endif

      

      //
      if (m_Write[ReqIdx]) {
        // write request
		  m_Mdata[ReqIdx] = req.MDataPtr;

		  // Put data into the memory (write) or into the Data field (read).
		  // The direct method call is just a reuse of an existing method.
		  // This has nothing to do with the debug interface.
		  m_Sresp[ReqIdx] = (MputDirect(m_BusID,m_Write[ReqIdx],
							m_Mdata[ReqIdx],m_Address[ReqIdx],m_NumWords[ReqIdx]))?
							OCP_SRESP_DVA: OCP_SRESP_ERR;

        // Acceptance delay
		//TO DO: CHECK PARAM NumWords not initialized
        int AcceptDelay = m_NumWords[ReqIdx]* m_WriteAcceptCycles;

        // schedule a release of the OCP request channel (i.e., assert the
        // SCmdAccept signal) at a delta time later
        wait(sc_time(AcceptDelay,TL_TIME_SCALE));
	    SlaveP->putSCmdAccept();

		  #ifdef DEBUG_G1
			PrintFuTime("Accept write request in Slave ",m_ID,m_Address[ReqIdx],
								(double)AcceptDelay*TL_TIME_SCALE);
		#endif

        // check whether need to generate a write response
        if (m_WriteResponse) {
          // setup a response and notify the response thread

          // compute pipelined response delay and trigger response thread
          sc_time ResponseDelay = sc_time(m_WriteResponseCycles,
                                          TL_TIME_SCALE);
          m_RespStartTime[ReqIdx] = sc_time_stamp() + ResponseDelay;

          // schedule a future notification to the response thread
          m_SlaveRespWait[ReqIdx] = false;
          m_SlaveRespEvent.notify(ResponseDelay);
		  last_chunk[ReqIdx]=last_chunk_of_burst;

	      #ifdef DEBUG_S1
				  cout << "Slave (" << m_ID << ") starts response process at "
					   << m_RespStartTime[ReqIdx].to_double() << "  <"
					   << name() << ">" << endl;
		  #endif

          // advance the outstanding request's FIFO index
          ReqIdx++;
          if (ReqIdx >= TL_SLAVE_TL2_FIFO_DEPTH)
            ReqIdx = 0;

          // advance the outstanding request counter
          m_ReqTaNum++;
        }
      } else {  // read request
        // Acceptance delay
        int AcceptDelay = m_ReadAcceptCycles;
		m_Mdata[ReqIdx]=new Td[chunk_length];

		// Put data into the memory (write) or into the Data field (read).
		// The direct method call is just a reuse of an existing method.
		// This has nothing to do with the debug interface.
		m_Sresp[ReqIdx] = (MputDirect(m_BusID,m_Write[ReqIdx],
							m_Mdata[ReqIdx],m_Address[ReqIdx],m_NumWords[ReqIdx]))?
							OCP_SRESP_DVA: OCP_SRESP_ERR;


        // schedule a release of the OCP request channel (i.e., assert the
        // SCmdAccept signal) at a delta time later
       	wait(sc_time(AcceptDelay,TL_TIME_SCALE));
		SlaveP->putSCmdAccept();

		#ifdef DEBUG_G1
				PrintFuTime("Accept read request in Slave ",m_ID,m_Address[ReqIdx],
							(double)AcceptDelay*TL_TIME_SCALE_SEC);
		#endif
        // setup a response and notify the response thread

        // compute pipelined response delay and trigger response thread
        //TO DO: CHECK NumWords parameter
		sc_time ResponseDelay = sc_time(m_NumWords[ReqIdx]*m_ReadResponseCycles,TL_TIME_SCALE);
                                        
        m_RespStartTime[ReqIdx] = sc_time_stamp() + ResponseDelay;

        // schedule a future notification to the response thread
        m_SlaveRespWait[ReqIdx] = false;
		last_chunk[ReqIdx]=last_chunk_of_burst;
        m_SlaveRespEvent.notify(ResponseDelay);

		#ifdef DEBUG_S1
				cout << "Slave (" << m_ID << ") starts response process at "
					 << m_RespStartTime[ReqIdx].to_double() << "  <"
					 << name() << ">" << endl;
		#endif

        // advance the outstanding request's FIFO index
        ReqIdx++;
        if (ReqIdx >= TL_SLAVE_TL2_FIFO_DEPTH)
          ReqIdx = 0;

        // advance the outstanding request counter
        m_ReqTaNum++;
      }
    } // end of Sget  = true branch
  } // end of while (true) loop
} // end of method

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Slave::SlaveResponse()
//
//  Top level response method of the Slave (pipelined case)
//
// ----------------------------------------------------------------------------
void OCP_TL2_Slave::SlaveResponse()
{
  int ResIdx = 0;
  OCPResponseGrp<Td> resp; 

  while (true) {
    // use blocking methods

	#ifdef DEBUG_S1
		cout << "Slave (" << m_ID << ", " << ResIdx << ") starts waiting at "
			 << sc_time_stamp().to_double() << "  <"
			   << name() << ">" << endl;
	#endif

    // wait for a notification event
    if (m_SlaveRespWait[ResIdx]) wait();

    // check if we still have to wait
    sc_time CurrentTime = sc_time_stamp();

	#ifdef DEBUG_S1
		sc_time ti1 = (m_RespStartTime[ResIdx] > CurrentTime)?
					  m_RespStartTime[ResIdx] - CurrentTime : SC_ZERO_TIME;
		cout << "Slave (" << m_ID << ", " << ResIdx << ") waits "
			 << ti1.to_double() << " in response process at "
			 << CurrentTime.to_double() << "  <" << name() << ">" << endl;
	#endif

    if (m_RespStartTime[ResIdx] > CurrentTime)
      wait(m_RespStartTime[ResIdx] - CurrentTime);

	#ifdef DEBUG_S
		PrintTrans(m_ResTaNum,m_Address[ResIdx],m_NumWords[ResIdx],
				   " Blocking Response ");
	#endif

    // ----------------
    // setup a response
    // ----------------

    // set the response and address filed
   	resp.SResp=m_Sresp[ResIdx];
	resp.SThreadID=m_MthreadID[ResIdx];
	resp.SDataPtr=m_Mdata[ResIdx];
   
    // set the command type (and the returned data, if needed)
    if (m_Write[ResIdx]) {
		// a write response
		#ifdef DEBUG_G1
		  PrintTime("Start write response in Slave ",m_ID,m_Address[ResIdx]);
		#endif

		// Send a response to the Master
		if (!SlaveP->sendOCPResponseBlocking(resp, 0, last_chunk[ResIdx])) {
		  // Deep ship: Should never happen
		  cout << "Error: Master (" << m_BusID << ") dropped Slave (" 
			   << m_ID << ") response <" << name() << ">" << endl;
		}
    } else {
      // a read response and its data
	  #ifdef DEBUG_G1
		  PrintTime("Start read response in Slave ",m_ID,m_Address[ResIdx]);
	  #endif

	  // Send a response to the Master
	  if (!SlaveP->sendOCPResponseBlocking(resp, m_NumWords[ResIdx], last_chunk[ResIdx])) {
		 // Deep ship: Should never happen
		 cout << "Error: Master (" << m_BusID << ") dropped Slave (" 
			   << m_ID << ") response <" << name() << ">" << endl;
	  } 
    }

    // reset the FIFO entry
    m_SlaveRespWait[ResIdx] = true;

    // get to the next waiting response
    ResIdx++;
    if (ResIdx >= TL_SLAVE_TL2_FIFO_DEPTH)
      ResIdx = 0;

    // advance the committed response counter
    m_ResTaNum++;
  }
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Slave::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
void OCP_TL2_Slave::end_of_elaboration()
{
  //
  sc_module::end_of_elaboration();


  // Get system parameter structure
  m_ParamCl = SlaveP->GetParamCl();
  
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Slave::MPutDirect()
//
//  Read/Write data, Master to Slave direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
bool OCP_TL2_Slave::MputDirect(int MasterID, bool IsWrite, Td* Data,
  Ta Address, int NumWords)
{
  int i;
  int Offset = (Address - m_StartAddress) / m_NumBytesPerWord;

  if ((Offset >= 0) && ((Offset + NumWords) <= m_SmemoryLength)) {
    if (IsWrite) {
      // write request
      for (i = 0; i < NumWords; i++)
        m_SlaveMemory[Offset + i] = Data[i];
    } else {
      // read request
      for (i = 0; i < NumWords; i++)
        Data[i] = m_SlaveMemory[Offset + i];
    }
    return(true);
  } else {
    cout << "Error: Address out of range at Slave " << m_ID << endl
         << "       Address=" << Address << "  StartAddr=" << m_StartAddress
         << "  _EndAddr=" << m_EndAddress
         << "  NumWords=" << NumWords
         << endl;
    return(false);
  }
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Slave::PrintTrans()
//
//  Print transaction
//
// ----------------------------------------------------------------------------
void OCP_TL2_Slave::PrintTrans(int TaNum, Ta Addr, int NumWords,
  const char* Method)
{
  double ttime = sc_time_stamp().to_double();
  cout << "Slave " << m_ID << " : " << "Req = " << TaNum
       << "  Address = " << Addr 
       << "  NumWords = " << NumWords << Method 
       << " at "<< ttime << " sec" << endl;
}

// ----------------------------------------------------
//  Method : PrintTime()
//
//  Print time stamp
// ----------------------------------------------------
void OCP_TL2_Slave::PrintTime(const char* Pstring, int ID, Ta Addr)
{
  cout << Pstring << ID << " : Address = " << Addr << " at "
       << sc_time_stamp().to_double()
       << endl;
}

// ----------------------------------------------------
//  Method : PrintFuTime()
//
//  Print future time stamp
// ----------------------------------------------------
void OCP_TL2_Slave::PrintFuTime(const char* Pstring, int ID, Ta Addr,
  double AddTime)
{
  cout << Pstring << ID << " : Address = " << Addr << " at "
       << (sc_time_stamp().to_double() + AddTime)
       << endl;
}

