// (c) Copyright OCP-IP 2004-2006
// OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG
//       Original Authors :
//                Anssi Haverinen, Nokia Inc.
//                Alan Kamas, for Sonics Inc.
//                Joe Chou, Sonics Inc.
//                James Aldis, Texas Instruments Inc.
//		  Tim Kogel, CoWare Inc.
//           $Id: ocp_tl1_trace_monitor_clocked.h,v 1.1 2006/09/04 14:01:07 tkogel Exp $
//
//  Description : Monitor component for OCP TL1 transaction API
//                Prints out a CoreCreator-compatible trace file.
//                Note: Instantiate after the channel monitored has been configured.
//   Old History:
//          04/09/2004 Original revision, based on Dec 2003 OCP TL1 channel
//          07/02/2004 Changed time to simulation time.
//          05/08/2004 Added a destructor so that
//                     the m_ocpMonPtr destructor can be called.
//	    02/03/2006 New unified monitor interface, renamed file
//	    09/06/2006 separate trace monitors for clocked and untimed channel
//
// ============================================================================


#ifndef _OCP_TL1_TRACE_MONITOR_CLOCKED_H
#define _OCP_TL1_TRACE_MONITOR_CLOCKED_H

#include "ocp_tl1_monitor_port.h"
#include "ocp_tl1_ocpmongen_cl.h"

template<typename TdataCl>
class OCP_TL1_Trace_Monitor_Clocked : public sc_module
{
 public:
  // forward type declarations in monitor interface
  typedef typename OCP_TL1_MonitorIF<TdataCl>::Td		Td;
  typedef typename OCP_TL1_MonitorIF<TdataCl>::Ta		Ta;
  typedef typename OCP_TL1_MonitorIF<TdataCl>::request_type	request_type;
  typedef typename OCP_TL1_MonitorIF<TdataCl>::datahs_type	datahs_type;
  typedef typename OCP_TL1_MonitorIF<TdataCl>::response_type	response_type;
  typedef typename OCP_TL1_MonitorIF<TdataCl>::paramcl_type	paramcl_type;

  OCP_TL1_MonitorPort<TdataCl> p_mon;
  sc_in<bool> p_clk;

  SC_HAS_PROCESS(OCP_TL1_Trace_Monitor_Clocked);
  OCP_TL1_Trace_Monitor_Clocked(sc_module_name mn,
				std::string monFileName = "",
				bool print_params = true)
    : sc_module(mn), 
      p_mon("p_mon"),
      p_clk("p_clk"),
      m_print_params(print_params)
  {
    std::string n = (std::string)mn+"gen";
    std::string m = (std::string)monFileName;
    if ( m.empty() ) {
      m = (std::string)mn+".ocp";
    }
    m_ocpMonPtr = new OCP_TL1_OCPMonGenCl<TdataCl>((std::string)n, m);
    m_clk_count = 0;
    SC_METHOD(OCPClockTick);
    sensitive << p_clk.pos();
    dont_initialize();

    mcmd = OCP_MCMD_IDLE;
    scmdaccept = true;
    mdatavalid = false;
    sdataaccept = true;
    sresp = OCP_SRESP_NULL;
    mrespaccept = true;
  }

  // Destructor
  ~OCP_TL1_Trace_Monitor_Clocked()
  {
      // delete the OCP_TL1_OCPMonGenCl which contains a file to flush
      delete m_ocpMonPtr;
  }



  // This function is called every OCP clock tick
  void
  OCPClockTick(void)
  {
 
    // NOTE: would be more efficient to simply change the current MCmd/SResp/MDataValid
    //       and then change it back after the call.
    //       However, this is cleaner.

    bool monReqAccept =  ((p_mon->peekRequestEnd()) || 
			  (p_mon->peekRequestStart() && p_mon->peekRequestEarlyEnd()));
    bool monRespAccept = ((p_mon->peekResponseEnd())  || 
			  (p_mon->peekResponseStart() && p_mon->peekResponseEarlyEnd()));
    bool monDataAccept = ((p_mon->peekDataRequestEnd())  || 
			  (p_mon->peekDataRequestStart() &&p_mon->peekDataRequestEarlyEnd()));


    m_monRequest = p_mon->peekOCPRequest();
    m_monDataHS = p_mon->peekDataHS();
    m_monResponse = p_mon->peekOCPResponse();

    OCPMCmdType  a;
    bool         b;
    OCPSRespType c;

    if (scmdaccept) {
      a = p_mon->getMCmdTrace();
      if (a != OCP_MCMD_IDLE) {
	mcmd = a;
	scmdaccept = false;
	m_monRequest.MCmd = a;
      }
    }
    else {
      m_monRequest.MCmd = mcmd;
    }
    if (mcmd != OCP_MCMD_IDLE)
      if (monReqAccept) {
	mcmd = OCP_MCMD_IDLE;
	scmdaccept = true;
      }

    if (sdataaccept) {
      b = p_mon->getMDataValidTrace();
      if (b) {
	mdatavalid = b;
	sdataaccept = false;
	m_monDataHS.MDataValid = b;
      }
    }
    else {
      m_monDataHS.MDataValid = mdatavalid;
    }
    if (mdatavalid)
      if (monDataAccept) {
	mdatavalid = false;
	sdataaccept = true;
      }

    if (mrespaccept) {
      c = p_mon->getSRespTrace();
      if (c != OCP_SRESP_NULL) {
	sresp = c;
	mrespaccept = false;
	m_monResponse.SResp = c;
      }
    }
    else {
      m_monResponse.SResp = sresp;
    }
    if (sresp != OCP_SRESP_NULL)
      if (monRespAccept) {
	sresp = OCP_SRESP_NULL;
	mrespaccept = true;
      }

    // Print a line out
    m_ocpMonPtr-> monitorOut( p_mon->GetParamCl(),
			      //m_clk_count,
			      sc_time_stamp(),
			      m_monRequest,
			      monReqAccept,
			      m_monDataHS,
			      monDataAccept,
			      m_monResponse,
			      monRespAccept,
			      p_mon->peekSThreadBusy(),
			      p_mon->peekSDataThreadBusy(),
			      p_mon->peekMThreadBusy(),
			      p_mon->peekMReset_n(),
			      p_mon->peekSReset_n(),
			      p_mon->peekMError(),
			      p_mon->peekMFlag(),
			      p_mon->peekSError(),
			      p_mon->peekSFlag(),
			      p_mon->peekSInterrupt(),
			      p_mon->peekControl(),
			      p_mon->peekControlWr(),
			      p_mon->peekControlBusy(),
			      p_mon->peekStatus(),
			      p_mon->peekStatusRd(),
			      p_mon->peekStatusBusy());

    // Now that the trace as been dumped, exit if there were any
    // fatal errors during this OCP cycle
    if (p_mon->peekExitAfterOCPMon())
    {
      cout << name() << ": Exiting due to earlier errors." << endl;
      exit(1000);
    }

  }

  void startup(void)
  {
    m_ocpMonPtr->monitorHeader(p_mon->GetParamCl());
  }

  void runcheck(void)
  {
    // for future
  }

  void start_of_simulation(void)
  {
    cout << "start_of_simulation\n";
    startup();
  }
 protected:

  OCP_TL1_OCPMonGenCl<TdataCl> *m_ocpMonPtr;

  bool m_print_params;

  double m_clk_count;

  request_type  m_monRequest;
  datahs_type   m_monDataHS;
  response_type m_monResponse;

  OCPMCmdType mcmd;
  bool scmdaccept;
  bool mdatavalid;
  bool sdataaccept;
  OCPSRespType sresp;
  bool mrespaccept;
};

#endif // _OCP_TL1_TRACE_MONITOR_CLOCKED_H
