// 
//  Copyright 2003 OCP-IP
//  OCP-IP Confidential & Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level
//       Author : Norman Weyrich, Synopsys Inc.
//                (modified by Joe Chou, Sonics Inc.)
//         $Id: ocp_tl2_master.cpp,v 1.2 2004/09/05 17:57:56 halexan Exp $
//
//  Description : Transaction Level - Layer-2 example Master
//                Features: 
//                - Blocking methods are used
//                - Read/write selection depends on random number
//                - Address depends on Master ID
//                - Master can be pipelined or non-pipelined.
//                
// Parameter    : 
//   Template arguments.
//   - TdataCl: Data class containing data members and
//              access methods for the data exchanged between
//              Masters and Slaves
//   - Td: Data type
//   - Ta: Address type
//                
//   Constructor arguments.
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//     - Priority: Positive number specifying the priority relative to the
//                 other Masters for Bus access. Higher numbers means higher
//                 priority. Masters can have the same priority.
//     - Pipelined: Switch to change between non-pipelined and
//                  pipelined operation mode of the Master.
//     - WriteResponse: Switch to enable/disable the sending of a
//                      response packet to a Master write request.
//                      Note that all modules involved in a system
//                      must use the same value.
//     - ReadRequestCycles: Number of cycles per word the Master waits
//                          before starting a read request.
//     - WriteRequestCycles: Number of cycles per word the Master waits
//                           before starting a write request.
//     - ReadAcceptCycles: Number of cycles per word the Master waits
//                         before accepting a read response.
//     - WriteAcceptCycles: Number of cycles per word the Master waits
//                          before accepting a write response.
//
// ============================================================================
//
// Pipelined versus non-pipelined operation mode
// ---------------------------------------------
// The Master can be pipelined or non-pipelined.
// In the non-pipelined case the requests and responses
// are handled sequentially in a single thread. This is pretty secure
// against deadlocks.
// In the piplined case the request and reponse methods are called from
// two different threads. They run completly in parallel with no dependency
// between both threads.
//
// Timing
// ------
// The Master has the following timing:
// - Write request  transfer: wait NumWords * WriteRequestCycles cycles
// - Read  request  transfer: wait    1     * ReadRequestCycles  cycles
// - Write response transfer: wait    1     * WriteAcceptCycles cycles
// - Read  response transfer: wait NumWords * ReadAcceptCycles  cycles
// ============================================================================

#include "ocp_tl2_master.h"

// ----------------------------------------------------------------------------
// Process : OCP_TL2_Master::OCP_TL2_Master
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl>
OCP_TL2_Master<TdataCl>::OCP_TL2_Master(sc_module_name name_, int ID,
  int Priority, bool Pipelined, bool WriteResponse, int ReadRequestCycles,
  int WriteRequestCycles, int ReadAcceptCycles, int WriteAcceptCycles)
  : sc_module(name_),
    MasterP("MasterPort"),
    m_ID(ID),
    m_Priority(Priority),
    m_Pipelined(Pipelined),
    m_WriteResponse(WriteResponse),
    m_ReadRequestCycles(ReadRequestCycles),
    m_WriteRequestCycles(WriteRequestCycles),
    m_ReadAcceptCycles(ReadAcceptCycles),
    m_WriteAcceptCycles(WriteAcceptCycles),
    m_Write(1),
    m_Read(0),
    m_TaNum(0),
    m_TaNumS(0)
{
  // InitParameters();

  //
  m_NumBytesPerWord = sizeof(Td);

  //
  if (m_Pipelined) {
    SC_THREAD(MasterRequest);
    SC_THREAD(MasterResponse);
  } else {
    SC_THREAD(Master);

    // to try the MasterD thread: uncomment the following line and
    // comment out the previous SC_THREAD() line
    //SC_THREAD(MasterD);
  }
}

// ----------------------------------------------------------------------------
// Process : OCP_TL2_Master::~OCP_TL2_Master
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl>
OCP_TL2_Master<TdataCl>::~OCP_TL2_Master()
{
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Master::MasterD()
//
//  Top level method of the Master, non-pipelined case,
//  using the direct interface. This method is not used in the constructor
//  and serves as example, on how to use direct access method calls.
//
// ----------------------------------------------------------------------------
template<class TdataCl>
void OCP_TL2_Master<TdataCl>::MasterD()
{
  unsigned int  i;
  bool rt;
  Ta   Addr;
  bool chunk_last ;

  // test the debug interface
  SetNumWords(10);
  for (i = 0; i < m_NumWords; i++)
    m_DirectDataW[i] = i+1;

  // the address should always align to an OCP word
  Addr = 0x40;
  rt = MasterP->MputDirect(m_ID,m_Write,m_DirectDataW,Addr,m_NumWords);
  rt = MasterP->MputDirect(m_ID,m_Read,m_DirectDataR,Addr,m_NumWords);

  // compare the results
  CompareData(m_DirectDataW,m_DirectDataR,m_NumWords);

  //
  m_TaNum++;

  // read the direct data (blocking request)
#ifdef DEBUG_M
  PrintTrans(m_TaNum,Addr," Blocking Read");
#endif
  
  // set address and return data (pointer passing) for the read request
  m_DataCl->MputMAddr(Addr);
  m_DataCl->MputMData(m_DataR,m_NumWords);

  if (rt = MasterP->MputReadRequestBlocking()) {
    if (rt = MasterP->MgetResponseBlocking(true)) {
      // release immediately

      // get Slave return response and data from the channel
      m_Sresp = m_DataCl->MgetSResp();
      m_Sdata = m_DataCl->MgetSData(m_SNumWords,chunk_last);

      //
      if (m_Sresp == OCP_SRESP_DVA) {
        CompareData(m_DirectDataW,m_Sdata,m_SNumWords);
      } else {
#ifdef DEBUG_M
        PrintSresp(m_Sresp,m_TaNum);
#endif
      }
    } else {
      PrintError();
    }
  } else {
    PrintError();
  }
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Master::Master()
//
//  Top level method of the Master (non-pipelined case)
//
// ----------------------------------------------------------------------------
template<class TdataCl>
void OCP_TL2_Master<TdataCl>::Master()
{
  int  i;
  bool Write;

  //
  int  Words;
  unsigned int  NumWords;

  //
  Ta   BaseAddr;
  Ta   Addr = 0;
  int  AddrInc = 0;

  // 
  bool chunk_last;

  // --------------
  // initialization
  // --------------

  // initialize the rendom seed (we only need simple mechanism)
  int jran = 0xDEAD + m_ID;

  // initialize Data array
  for (i = 0; i < TL2_MAX_BURST_LENGTH; i++)
    m_DataW[i] = 2*i+1;

  // setup different burst sizes and base addresses for different Masters
  switch (m_ID) {
  case 1:
    Words = 1;
    BaseAddr = 0;
    break;
  case 2:
    Words = 2;
    BaseAddr = 0x1000;
    break;
  case 3:
    Words = 3;
    BaseAddr = 0x1400;
    break;
  default:
    Words = 1;
    BaseAddr = 0;
    break;
  }

  //
  SetNumWords(Words);

  // ---------------------------------
  // main loop: send blocking requests
  // ---------------------------------
  while (true) {
    // -----------------
    // prepare a request
    // -----------------

    // set address in the channel
    Addr = BaseAddr + AddrInc;

    // randomly decide whether to send a read request or a write request
    jran = (jran * 7141 + 54773) % 259200;
    Write = (((float)jran / 259200.0) < 0.5)? 0 : 1;

    //
    if (Write) {
      // send a write request

      // emulate send delay
      wait(m_NumWords*m_WriteRequestCycles,TL_TIME_SCALE);

      // request phase

#ifdef DEBUG_G1
      PrintTime("Start write request in Master ",m_ID,Addr);
#endif
#ifdef DEBUG_M
      PrintTrans(m_TaNum,Addr," Blocking Write");
#endif

      // set data (pointer passing) and address in the request channel
      m_DataCl->MputMAddr(Addr);
      m_DataCl->MputMData(m_DataW,m_NumWords);

      // Send a write request to the OCP channel
      if ((MasterP->MputWriteRequestBlocking())) {
        // calculate the byte address of the next request
        AddrInc += m_NumWords * m_NumBytesPerWord;
        if (AddrInc > 0x3FF) AddrInc = 0;

        // update the transaction counter
        m_TaNum++;

        // wait for write response, if configured
        if (m_WriteResponse) {
          // Get response

          if ((MasterP->MgetResponseBlocking(false))) {
            // release later

            // get the response from the OCP channel
            m_Sresp = m_DataCl->MgetSResp();

            // emulate a response acceptance delay (only for good transmission)
            if (m_Sresp == OCP_SRESP_DVA)
              wait(m_WriteAcceptCycles,TL_TIME_SCALE);

#ifdef DEBUG_M
            PrintSresp(m_Sresp,m_TaNum);
#endif
#ifdef DEBUG_G1
            PrintTime("End write response in Master ",m_ID);
#endif

            // Release the OCP channel (end of response transaction)
       	    MasterP->Mrelease();
          } else {
            PrintError();
          }
        }
      } else {
        PrintError();
      }
    } else {
      // read request

      // emulate send delay
      wait(m_ReadRequestCycles,TL_TIME_SCALE);

      // request phase

#ifdef DEBUG_G1
      PrintTime("Start read request in Master ",m_ID,Addr);
#endif
#ifdef DEBUG_M
      PrintTrans(m_TaNum,Addr," Blocking Read");
#endif

      // set address and return data (pointer passing) in the request channel
      m_DataCl->MputMAddr(Addr);
      m_DataCl->MputMData(m_DataR,m_NumWords);

      if ((MasterP->MputReadRequestBlocking())) {
        AddrInc += m_NumWords * m_NumBytesPerWord;
        if (AddrInc > 0x3FF) AddrInc = 0;

        m_TaNum++;

        // Send a read request to the OCP channel
        if ((MasterP->MgetResponseBlocking(false))) {
          // release later

          // get Slave return response and data from the OCP channel
          m_Sresp = m_DataCl->MgetSResp();
          m_Sdata = m_DataCl->MgetSData(NumWords,chunk_last);

          // emulate a response acceptance delay (only for good transmission)
          if (m_Sresp == OCP_SRESP_DVA)
            wait(NumWords*m_ReadAcceptCycles,TL_TIME_SCALE);

#ifdef DEBUG_M
          PrintSresp(m_Sresp,m_TaNum);
#endif
#ifdef DEBUG_G1
          PrintTime("Accept read response in Master ",m_ID);
#endif

          // Release the OCP channel (end of response transaction)
	  MasterP->Mrelease();
        } else {
          PrintError();
        }
      } else {
        PrintError();
      }
    } // end read branch
  } // end while(true) loop
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Master::MasterRequest()
//
//  Top level request method of the Master (pipelined case)
//
// ----------------------------------------------------------------------------
template<class TdataCl>
void OCP_TL2_Master<TdataCl>::MasterRequest()
{
  int  i;
  int  Words;
  bool Write;

  Ta   BaseAddr;
  Ta   Addr = 0;
  int  AddrInc = 0;

  // we only need simple mechanism
  int jran = 0xDEAD + m_ID;

  // initialize Data array
  for (i = 0; i < TL2_MAX_BURST_LENGTH; i++)
    m_DataW[i] = 2*i+1;

  // setup different burst sizes and base addresses for different Masters
  switch (m_ID) {
  case 1:
    Words = 1;
    BaseAddr = 0;
    break;
  case 2:
    Words = 2;
    BaseAddr = 0x1000;
    break;
  case 3:
    Words = 3;
    BaseAddr = 0x1400;
    break;
  default:
    Words = 1;
    BaseAddr = 0;
    break;
  }

  //
  SetNumWords(Words);

  // ---------------------------------
  // main loop: send blocking requests
  // ---------------------------------
  while (true) {

    // -----------------
    // prepare a request
    // -----------------

    // set address in the channel
    Addr = BaseAddr + AddrInc;

    // randomly decide whether to send a read request or a write request
    jran = (jran * 7141 + 54773) % 259200;
    Write = (((float)jran / 259200.0) < 0.5)? 0 : 1;

    //
    if (Write) {
      // send a write request

      // emulate send delay
      wait(m_NumWords*m_WriteRequestCycles,TL_TIME_SCALE);

      // request phase

#ifdef DEBUG_G1
      PrintTime("Start write request in Master ",m_ID,Addr);
#endif
#ifdef DEBUG_M
      PrintTrans(m_TaNum,Addr," Blocking Write");
#endif

      // set data (pointer passing) and address in the request channel
      m_DataCl->MputMAddr(Addr);
      m_DataCl->MputMData(m_DataW,m_NumWords);

      // Send a write request to the OCP channel
      if ((MasterP->MputWriteRequestBlocking())) {
        // calculate the byte address of the next request
        AddrInc += m_NumWords * m_NumBytesPerWord;
        if (AddrInc > 0x3FF) AddrInc = 0;

        // update the transaction counter
        m_TaNum++;
      } else {
        PrintError();
      }
    } else {
      // read request

      // emulate send delay
      wait(m_ReadRequestCycles,TL_TIME_SCALE);

      // request phase

#ifdef DEBUG_G1
      PrintTime("Start read request in Master ",m_ID,Addr);
#endif
#ifdef DEBUG_M
      PrintTrans(m_TaNum,Addr," Blocking Read");
#endif

      // set address and return data (pointer passing) in the request channel
      m_DataCl->MputMAddr(Addr);
      m_DataCl->MputMData(m_DataR,m_NumWords);

      if ((MasterP->MputReadRequestBlocking())) {
        AddrInc += m_NumWords * m_NumBytesPerWord;
        if (AddrInc > 0x3FF) AddrInc = 0;

        m_TaNum++;
      } else {
        PrintError();
      }
    } // end read branch
  } // end while(true) loop
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Master::MasterResponse()
//
//  Top level reponse method of the Master (pipelined case)
//
// ----------------------------------------------------------------------------
template<class TdataCl>
void OCP_TL2_Master<TdataCl>::MasterResponse()
{
  unsigned int  NumWords;
  bool chunk_last;

  // get response (blocking)
  while (true) {
    // Get response
    if ((MasterP->MgetResponseBlocking(false))) {
      // release later

      // get the response from the OCP channel
      m_Sresp = m_DataCl->MgetSResp();
    
      // Is it a write response (SDataInfo = 1) ?
      if (MasterP->GetDataCl()->MgetSDataInfo()==1) {
        // write response

        // Sanity check
        if (!m_WriteResponse) {
          cerr << "ERROR: Bus/Slave responded to a write request with "
               << "WriteResponse = false >" << name() << ">" << endl;
               sc_stop();
               return;
        } else {
          // emulate a response acceptance delay (only for
          // good transmission)
          if (m_Sresp == OCP_SRESP_DVA)
            wait(m_WriteAcceptCycles,TL_TIME_SCALE);
        }
      } else {
        // read response

        // get the return data pointer and word length
        m_Sdata = m_DataCl->MgetSData(NumWords,chunk_last);

        // Response accept delay
#ifdef DEBUG_M1
        cout << "Master " << m_ID << " : Waits in read response for "
             << sc_time(NumWords*m_ReadAcceptCycles,TL_TIME_SCALE) << endl;
#endif

        // emulate a response acceptance delay (only for good transmission)
        if (m_Sresp == OCP_SRESP_DVA)
          wait(NumWords*m_ReadAcceptCycles,TL_TIME_SCALE);
      }

#ifdef DEBUG_M
      PrintSresp(m_Sresp,m_TaNum);
#endif
#ifdef DEBUG_G1
      PrintTime("Accept response in Master ",m_ID);
#endif

      // Release the OCP response channel (end of response transaction)
      MasterP->Mrelease();

      // update the transaction counter
      m_TaNumS++;
    } else {
      PrintError();
    }
  } // end while(true) loop
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Master::SetNumWords()
//
//  Set the burst length
//
// ----------------------------------------------------------------------------
template<class TdataCl>
bool OCP_TL2_Master<TdataCl>::SetNumWords(int NumWords)
{
  if (NumWords > TL2_MAX_BURST_LENGTH) {
    cerr << "ERROR: Master " << m_ID << " request exceeds max burst length. <"
         << name() << ">" << endl;
    sc_stop();
    return(false);
  } else {
    m_NumWords = NumWords;
    return(true);
  }
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Master::CompareData()
//
//  Read/Write Data comparision
//
// ----------------------------------------------------------------------------
template<class TdataCl>
bool OCP_TL2_Master<TdataCl>::CompareData(Td* DataW, Td* DataR, int NumWords)
{
  int i;

  for (i = 0; i < NumWords; i++) {
    if (DataW[i] != DataR[i]) {
      cout << "Error: Master " << m_ID << " TaNum " << m_TaNum
           << " Read/Write data comparision failed"
           << endl;
      return(false);
    }
  }

#ifdef DEBUG_M
  cout << "Master " << m_ID << " : " << "Req = " << m_TaNum
       << " Read/Write data comparision successful"
       << endl;
#endif

  return(true);
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Master::SputDirect()
//
//  Debug interface method.
//  Read/Write data, Slave to Master direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl>
bool OCP_TL2_Master<TdataCl>::SputDirect(int SlaveID, bool IsWrite,
  Td* Data, Ta Address, int NumWords)
{
  // not implemented
  return(false);
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Master::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl>
void OCP_TL2_Master<TdataCl>::end_of_elaboration()
{
  sc_module::end_of_elaboration();

  // Initialize debug interface
  MasterP->MregisterDirectIF(this);

  // Get data structure
  m_DataCl = MasterP->GetDataCl();

  // Get communication structure
  m_CommCl = MasterP->GetCommCl();

  // Get system parameter structure
  m_ParamCl = MasterP->GetParamCl();

  // Put parameter into channel
  m_ParamCl->MasterID = m_ID;
  m_ParamCl->Priority = m_Priority;
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Master::PrintTrans()
//
//  Print transaction
//
// ----------------------------------------------------------------------------
template<class TdataCl>
void OCP_TL2_Master<TdataCl>::PrintTrans(int TaNum, Ta Addr,
  const char* Method)
{
  double ttime = sc_time_stamp().to_double();
  cout << "Master " << m_ID << " : " << "Req = " << TaNum
       << "  Address = " << Addr 
       << "  NumWords = " << m_NumWords << Method 
       << " at "<< ttime << " sec" << endl;
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Master::PrintError()
//
//  
//
// ----------------------------------------------------------------------------
template<class TdataCl>
void OCP_TL2_Master<TdataCl>::PrintError()
{
  cout << "Error: Master " << m_ID << " : " << "Req = " << m_TaNum << " failed"
       << endl;
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Master::PrintSresp()
//
//  
//
// ----------------------------------------------------------------------------
template<class TdataCl>
void OCP_TL2_Master<TdataCl>::PrintSresp(OCPSRespType Sresp, int TaNum)
{
  cout << "Master " << m_ID << " : " << "Req = " << TaNum << " Response = "
       << Sresp << " at "
       << sc_time_stamp().to_double() << endl;
}

// ----------------------------------------------------
//  Method : PrintTime()
//
//  Print time stamp
// ----------------------------------------------------
template<class TdataCl>
void OCP_TL2_Master<TdataCl>::PrintTime(const char* Pstring, int ID, Ta Addr)
{
  cout << Pstring << ID << " : Address = " << Addr << " at "
       << sc_time_stamp().to_double()
       << endl;
}

// ----------------------------------------------------------------------------
//
//  Instantiation of the Master
//
// ----------------------------------------------------------------------------

#include "ocp_tl2_data_cl.h"
template class OCP_TL2_Master< OCP_TL2_TEMPL_DATA_CL >;
