///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2003, 2004 OCP-IP
// OCP-IP Confidential & Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : Yann Bajot, Prosilog
//          $Id: ocp_tl2_channel.h,v 1.2 2004/09/05 17:57:56 halexan Exp $
//
//  Description :  OCP TL 2 SystemC Channel Model
//        This version of Channel is derived from the original 
//        OCP SLD WG, OCP Transaction Level Layer-2 model by
//        Norman Weyrich, Anssi Haverinen, and Joe Chou.
//        Please see that model for details.
//
// This version adds support for OCP2.0, full channel access through the ports,
// single function calls to send and receive requests and reponses.
//
// Change History:
//
// 12/14/2003 Original revision
// 05/17/2004 Added timed accept methods and TL1-style reset
//            Reformating, TL1-like get/put*Accept() methods 
//            Changes in sendOCP*() methods: TL2 Data class members are not
//            updated if the call fails.
//            Bug fixed in startOCP*Blocking() methods: now acts differently
//            than sendOCP*Blocking() methods.
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#ifndef _OCP_TL2_CHANNEL_H
#define _OCP_TL2_CHANNEL_H

#include <string>
#include <map>

#include "systemc.h"

#include "tl_channel.h"
#include "ocp_tl_param_cl.h"
#include "ocp_tl2_data_cl.h"
#include "ocp_tl2_master_if.h"
#include "ocp_tl2_slave_if.h"

// -----------------------------------------------------------------
// -----------------------------------------------------------------

template <class Tdata, class Taddr>
class OCP_TL2_Channel :
  public TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >,
  public OCP_TL2_MasterIF<Tdata,Taddr>,
  public OCP_TL2_SlaveIF<Tdata,Taddr>
{

  public:

    //---------------------------------------------------------------
    // constructor
    //---------------------------------------------------------------
    // For the TL2 channel, Generic Channel constructor parameters are always
    // true: sync, use_event and use_default_event
    OCP_TL2_Channel(std::string name)
      : TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >(name.c_str(),true,true,true)
      {}

    //---------------------------------------------------------------
    // destructor
    //---------------------------------------------------------------
    virtual ~OCP_TL2_Channel()
    { }

  public:

    //////////////////////////////////////////////////////////////
    // OCP TL2 Master Interface methods
    //////////////////////////////////////////////////////////////


    // 'sendOCPRequest()'
    //
    bool sendOCPRequest(OCPRequestGrp<Tdata,Taddr>& req, unsigned int
        ReqChunkLen = 1, bool last_chunk_of_a_burst = true) {
      // is the channel in reset?
      if (get_reset()) {
        // In reset. Give up.
        return false;
      }
      // Is the thread free?
      if (m_ParamCl->sthreadbusy_exact == 1){
        // SThreadBusy is part of the channel and we are required to check it.
        // Now we need to test to see if the thread of the request is busy.
        if ( m_DataCl->isThisThreadBusy(req.MThreadID, getSThreadBusy()) ) {
          // The thread is busy. Fail.
#ifdef DEBUG_C
          cout << name() << ": sendOCPRequest failed. It was called for a busy"
          << " thread on a channel with sthreadbusy_exact." << endl;
#endif
          return false;
        }
      }


      // Send request
      if(!MputRequest())
          return false;

      // Set request fields of the data class
      SetRequestFields(req, ReqChunkLen , last_chunk_of_a_burst);

      return true;

    }

    // 'startOCPRequest()'
    //
    // This method is an alias for 'sendOCPRequest()', to ensure compatibility
    // with the TL1 API
    //
    bool startOCPRequest(OCPRequestGrp<Tdata,Taddr>& req, unsigned int
        ReqChunkLen = 1, bool last_chunk_of_a_burst = true) {

      return sendOCPRequest(req,ReqChunkLen,last_chunk_of_a_burst);
    }


    // 'sendOCPRequestBlocking()'
    //
    bool sendOCPRequestBlocking(OCPRequestGrp<Tdata,Taddr>& req, unsigned int
        ReqChunkLen = 1, bool last_chunk_of_a_burst = true)
    {
      // is the channel in reset?
      if (get_reset()) {
        // In reset. Give up.
        return false;
      }
      // Is the thread free?
      if (m_ParamCl->sthreadbusy_exact == 1)  {
        // SThreadBusy is part of the channel and we are required to check it.
        // Now we need to test to see if the thread of the request is busy.
        if ( m_DataCl->isThisThreadBusy(req.MThreadID, getSThreadBusy()) ){
          // The thread is busy. Fail.
#ifdef DEBUG_C
          cout << name() << ": sendOCPRequest failed. It was called for a busy"
           << " thread on a channel with sthreadbusy_exact." << endl;
#endif
          return false;
        }
      }


      // Place the request onto the channel (possibly wait until the channel is
      // free)
      if(!MputRequestSemiBlocking()) {
        return false;
      }

      // Set request fields of the data class
      SetRequestFields(req, ReqChunkLen, last_chunk_of_a_burst);

      // Wait until the slave has acknowledged the request
      waitSCmdAccept();

      if (get_reset()) {
        // In reset. Give up.
        return false;
      }


      return true;

    }

    // 'startOCPRequestBlocking()'
    //
    bool startOCPRequestBlocking(OCPRequestGrp<Tdata,Taddr>& req, unsigned int
        ReqChunkLen = 1, bool last_chunk_of_a_burst = true)
    {
      // is the channel in reset?
      if (get_reset()) {
        // In reset. Give up.
        return false;
      }
      // Is the thread free?
      if (m_ParamCl->sthreadbusy_exact == 1) {
        // SThreadBusy is part of the channel and we are required to check it.
        // Now we need to test to see if the thread of the request is busy.
        if ( m_DataCl->isThisThreadBusy(req.MThreadID, getSThreadBusy()) ){
          // The thread is busy. Fail.
#ifdef DEBUG_C
          cout << name() << ": sendOCPRequest failed. It was called for a busy"
          << " thread on a channel with sthreadbusy_exact." << endl;
#endif
          return false;
        }
      }

      // Wait until the request channel is free, place the request onto the
      // channel but do not wait for acknowledgment
      if(!MputRequestSemiBlocking()) {
        return false;
      }

      // Set request fields of the data class
      SetRequestFields(req, ReqChunkLen, last_chunk_of_a_burst);

      return true;

    }


    // 'getOCPResponse()'
    //
    bool getOCPResponse(OCPResponseGrp<Tdata>& resp, bool accept, unsigned int&
        RespChunkLen, bool& last_chunk_of_a_burst)
    {
      // is the channel in reset?
      if (get_reset()) {
        // In reset. Give up.
        return false;
      }

      // Is there a response on the channel ?
      bool ret_val=MgetResponse(accept);

      if (ret_val) {

        // Get response fields of the data class
        GetResponseFields(resp, false,  RespChunkLen, last_chunk_of_a_burst);
        return true;
      }

#ifdef DEBUG_C
      cout << name() << ": WARNING - getOCPResponse returns false because"
      << " MgetResponse() call failed"  << endl;
#endif

      return false;
    }

    // 'getOCPResponseBlocking()'
    //
    bool getOCPResponseBlocking(OCPResponseGrp<Tdata>& resp, bool accept,
        unsigned int& RespChunkLen, bool& last_chunk_of_a_burst)
    {
      // is the channel in reset?
      if (get_reset()) {
        // In reset. Give up.
        return false;
      }

      // Waits for a response on the channel...
      bool ret_val=MgetResponseBlocking(accept);

      if(ret_val) {
        // A response is present on the channel
        // Get response fields of the data class ('SData' array is not copied here)
        GetResponseFields(resp, false,  RespChunkLen, last_chunk_of_a_burst);
        return true;
      }

      // Blocking call failed
#ifdef DEBUG_C
      cout << name() << ": WARNING - getOCPResponseBlocking returns false"
      << " because MgetResponseBlocking() call failed"  << endl;
#endif
      return false;

    }

    //---------------------------------------------------------------------
    // Serialized methods
    //---------------------------------------------------------------------

    // 'OCPReadTransfer()'
    //
    bool OCPReadTransfer(OCPRequestGrp<Tdata,Taddr>& req,
        OCPResponseGrp<Tdata>& resp, unsigned int TransferLen = 1)
    {
      // is the channel in reset?
      if (get_reset()) {
        // In reset. Give up.
        return false;
      }

      // Test if MCmd is really 'READ'
      if(req.MCmd != OCP_MCMD_RD) 
        fatalError("OCPReadTransfer()","'MCmd' is not equal to 'OCP_MCMD_RD'");

      // Send the request (blocking call)
      if(!sendOCPRequestBlocking(req, TransferLen, true)) {
        return false;
      }

      unsigned int chunklength;
      bool chunklast;

      // Get the response, and immediatly release
      // the response
      bool ret_val = getOCPResponseBlocking(resp, true , chunklength,
          chunklast); 

      // Check that the response chunk is atomic and has the same length than
      // the request chunk
      if( !(chunklast && ( chunklength == TransferLen) ) ) {
#ifdef DEBUG_C
        cout << name() << ": WARNING - response chunk is not atomic ." << endl;
#endif
        return false;
      }


      return ret_val;
    }

    // 'OCPWriteTransfer()'
    //
    bool OCPWriteTransfer(OCPRequestGrp<Tdata,Taddr>& req, unsigned int
        TransferLen = 1) 
    {
      // is the channel in reset?
      if (get_reset()) {
        // In reset. Give up.
        return false;
      }

      // Test if MCmd is really 'WRITE'
      if(req.MCmd != OCP_MCMD_WR) {
        fatalError("OCPWriteTransfer()","'MCmd' is not equal to 'OCP_MCMD_WR'");
      }

      // Send the request (blocking call)
      return sendOCPRequestBlocking(req, TransferLen, true);
    }


    //---------------------------------------------------------------------
    // Synchronization methods 
    //---------------------------------------------------------------------

    bool getSBusy(void) const {
      return  MgetSbusy();
    }

    bool getSCmdAccept(void) const {
      if (!m_ParamCl->cmdaccept)
        return(true);
      else
        return (!getSBusy());
    }

    void waitSCmdAccept(void) {
      // is the channel in reset?
      if (!get_reset()) {
        if (!getSBusy() || !m_ParamCl->cmdaccept){
          // The command has already been accepted
          return;
        } 
        wait(m_CommCl->RequestEndEvent);
      }

    }

    bool putMRespAccept() {

      // If 'respaccept' is set to 0, ignore
      if(m_ParamCl->respaccept == 0) {
#ifdef DEBUG_C
        cout << name() << ": WARNING - putMRespAccept called when 'respaccept'"
        << " parameter set to 0 " << endl;
#endif
        return true;
      }

      // Was this command called at the correct time?
      if (! m_CommCl->ResponseStart )
      {
#ifdef DEBUG_C
        cout << name() << ": WARNING - putMRespAccept called when there was no"
         << " response on the channel." << endl;
#endif
        return false;
      }
      Mrelease();
      return true;
    }

    bool putMRespAccept(sc_time after) {

      // If 'respaccept' is set to 0, ignore
      if(m_ParamCl->respaccept == 0) {
#ifdef DEBUG_C
        cout << name() << ": WARNING - putMRespAccept called when 'respaccept'"
        << " parameter set to 0 " << endl;
#endif
        return true;
      }

      // Was this command called at the correct time?
      if (! m_CommCl->ResponseStart )
      {
#ifdef DEBUG_C
        cout << name() << ": WARNING - putMRespAccept called when there was no"
        << " response on the channel." << endl;
#endif
        return false;
      }
      Mrelease(after);
      return true;
    }

    unsigned int getSThreadBusy(void) const
    { 
      // is the channel in reset?
      if (m_CommCl->Reset) {
        // In reset. Give up.
        return false;
      }
      if (m_ParamCl->sthreadbusy == 0 )  {
#ifdef DEBUG_C
        cout << name() << ": WARNING - getSThreadBusy called on a channel with"
          << "no SThreadBusy signal." << endl;
#endif
        // No SThreadBusy signal so no threads are busy
        return 0;
      } 

      return m_DataCl->MgetSThreadBusy();
    }

    bool getSThreadBusyBit(unsigned int ThreadID = 0) const 
    { 
      // is the channel in reset?
      if (m_CommCl->Reset) {
        // In reset. Give up.
        return false;
      }
      if ( m_ParamCl->sthreadbusy == 0 ){
#ifdef DEBUG_C
        cout << name() << ": WARNING - getSThreadBusyBit called on a channel"
        << " with no SThreadBusy signal." << endl;
#endif
        // No SThreadBusy signal so no threads are busy
        return false;
      } 

      return m_DataCl->isThisThreadBusy(ThreadID, m_DataCl->MgetSThreadBusy());
    }


    void putMThreadBusy(unsigned int mthreadbusy) 
    { 
      // is the channel in reset?
      if (!get_reset())
        m_DataCl->MputMThreadBusy( mthreadbusy );
    }

    void putMThreadBusyBit(bool nextBitValue, unsigned int ThreadID = 0) 
    {
      // is the channel in reset?
      if (!get_reset()) {
        if(nextBitValue) {
          // Set bit
          unsigned int mask = ( ((unsigned int) 1) << ThreadID );
          m_DataCl->MputMThreadBusy(m_DataCl->SgetMThreadBusy() | mask );
        } else {
          // Reset bit
          unsigned int mask = ~( ((unsigned int) 1) << ThreadID );
          m_DataCl->MputMThreadBusy(m_DataCl->SgetMThreadBusy() & mask );
        }
      }
    }


    //---------------------------------------------------------------------
    // Request Group Access methods
    //---------------------------------------------------------------------
    void MputMAddr(Taddr a) {m_DataCl->MputMAddr(a);}
    void MputMAddrSpace(unsigned int a) {m_DataCl->MputMAddrSpace(a);}
    void MputMAtomicLength(unsigned int a) {m_DataCl->MputMAtomicLength(a);}
    void MputMBurstSeq(OCPMBurstSeqType a) {m_DataCl->MputMBurstSeq(a);}
    void MputMByteEn(unsigned int a) {m_DataCl->MputMByteEn(a);}
    void MputMCmd(OCPMCmdType a) {m_DataCl->MputMCmd(a);}
    void MputMConnID(unsigned int a) {m_DataCl->MputMConnID(a);}
    void MputMData(Tdata* d, unsigned int w = 1, bool last_of_a_burst = true)
    {m_DataCl->MputMData(d,w,last_of_a_burst);}
    void MputMThreadID(unsigned int a) {m_DataCl->MputMThreadID(a);}
    void MputMBurstLength(unsigned int a) {m_DataCl->MputMBurstLength(a);}
    void MputMBurstPrecise(bool a) {m_DataCl->MputMBurstPrecise(a);}
    void MputMBurstSingleReq(bool a) {m_DataCl->MputMBurstSingleReq(a);}
    void MputMReqInfo(unsigned int a) {m_DataCl->MputMReqInfo(a);}
    void MputMReqLast(bool a) {m_DataCl->MputMReqLast(a);}
    void SetRequestFields(OCPRequestGrp<Tdata,Taddr>& req, unsigned int
        ReqChunkLen = 1, bool last_chunk_of_a_burst = true)
    {m_DataCl->SetRequestFields(req,ReqChunkLen,last_chunk_of_a_burst);}
    void GetRequestFields(OCPRequestGrp<Tdata,Taddr>& req, bool mdata_copy,
        unsigned int& ReqChunkLen, bool& last_chunk_of_a_burst)
    {m_DataCl->GetRequestFields(req,mdata_copy,ReqChunkLen,last_chunk_of_a_burst);}

    //---------------------------------------------------------------------
    // Response Group Access methods
    //---------------------------------------------------------------------
    Tdata* MgetSData(unsigned int& w, bool& last_of_a_burst){return
      m_DataCl->MgetSData(w,last_of_a_burst);}
    Tdata* MgetSData(unsigned int& w){return m_DataCl->MgetSData(w);}
    Tdata* MgetSData(){return m_DataCl->MgetSData();}
    unsigned int MgetSDataInfo(){return m_DataCl->MgetSDataInfo();}
    OCPSRespType MgetSResp(){return m_DataCl->MgetSResp();}
    unsigned int MgetSRespInfo(){return m_DataCl->MgetSRespInfo();}
    bool MgetSRespLast(){return m_DataCl->MgetSRespLast();}
    unsigned int MgetSThreadID(){return m_DataCl->MgetSThreadID();}
    void SetResponseFields(OCPResponseGrp<Tdata>& resp, unsigned int
        RespChunkLen = 1, bool last_chunk_of_a_burst = true)
    {m_DataCl->SetResponseFields(resp,RespChunkLen,last_chunk_of_a_burst);}
    void GetResponseFields(OCPResponseGrp<Tdata>& resp, bool sdata_copy,
        unsigned int& RespChunkLen, bool& last_chunk_of_a_burst)
    {m_DataCl->GetResponseFields(resp,sdata_copy,RespChunkLen,last_chunk_of_a_burst);}

    //---------------------------------------------------------------------
    // DataHS Group Access methods
    //---------------------------------------------------------------------

    void MputMDataInfo(unsigned int a){m_DataCl->MputMDataInfo(a);}

    //---------------------------------------------------------------------
    // ThreadBusy Group Access methods
    //---------------------------------------------------------------------
    unsigned int MgetSThreadBusy(void) const{return
      m_DataCl->MgetSThreadBusy();}
    void MputMThreadBusy(unsigned int
        mthreadbusy){m_DataCl->MputMThreadBusy(mthreadbusy);}

    //---------------------------------------------------------------------
    // SideBand Group Access methods
    //---------------------------------------------------------------------
    void MputMError(bool nextValue){m_DataCl->MputMError(nextValue);}
    void MputMFlag(unsigned int nextValue){m_DataCl->MputMFlag(nextValue);}

    // Sideband reset replaced with reset assert/deassert calls
    void MputMReset_n(bool nextValue){
      cout << "MputMReset_n is deprecated " << endl;
      //	  m_DataCl->MputMReset_n(nextValue);
      //m_DataCl->SBResetEvent.notify();
    }
    bool MgetSReset_n(void) const{
      cout << "MgetSReset_n is deprecated " << endl;
      return false;
    }
    bool MgetSError(void) const{return m_DataCl->MgetSError();}
    unsigned int MgetSFlag(void) const{return m_DataCl->MgetSFlag();}
    bool MgetSInterrupt(void) const{return m_DataCl->MgetSInterrupt();}

    //---------------------------------------------------------------------
    // TL2-specific burst chunks methods
    //---------------------------------------------------------------------

    void MputMReqChunkLen(unsigned int w){m_DataCl->MputMReqChunkLen(w);}
    void MputMReqChunkLast(bool w){m_DataCl->MputMReqChunkLast(w);}
    unsigned int MgetSRespChunkLen() const{return
      m_DataCl->MgetSRespChunkLen();}
    bool MgetSRespChunkLast() const{return m_DataCl->MgetSRespChunkLast();}


    //////////////////////////////////////////////////////////////
    // OCP TL2 Slave Interface methods
    //////////////////////////////////////////////////////////////

    // 'getOCPRequest()'
    //
    bool getOCPRequest(OCPRequestGrp<Tdata,Taddr>& req, bool accept, unsigned
        int& ReqChunkLen, bool& last_chunk_of_a_burst){

      // is the channel in reset?
      if (get_reset()) {
        // In reset. Give up.
        return false;
      }

      // Is it a request on the channel ?
      bool ret_val=SgetRequest(accept);

      if(ret_val) {
        // A request is present on the channel

        // Get request fields of the data class ('MData' array is not copied
        // here)
        GetRequestFields(req, false,  ReqChunkLen, last_chunk_of_a_burst);
        return true;
      }

      // No request on the channel
#ifdef DEBUG_C
      cout << name() << ": WARNING - getOCPRequest returns false because"
      << " SgetRequest() call failed"  << endl;
#endif

      return false;
    };

    // 'getOCPRequestBlocking()'
    //
    bool getOCPRequestBlocking(OCPRequestGrp<Tdata,Taddr>& req, bool accept,
        unsigned int& ReqChunkLen, bool& last_chunk_of_a_burst){

      // Waits for a request on the channel.
      bool ret_val=SgetRequestBlocking(accept);

      if(ret_val) {
        // A request is present on the channel

        // Get request fields of the data class ('MData' array is not copied
        // here)
        GetRequestFields(req, false,  ReqChunkLen, last_chunk_of_a_burst);
        return true;
      }

      // Blocking call failed
#ifdef DEBUG_C
      cout << name() << ": WARNING - getOCPRequestBlocking returns false"
      << " because SgetRequestBlocking() call failed"  << endl;
#endif

      return false;
    };

    // 'sendOCPResponse()'
    //
    bool sendOCPResponse(OCPResponseGrp<Tdata>& resp, unsigned int RespChunkLen
        = 1, bool last_chunk_of_a_burst = true){
      // is the channel in reset?
      if (get_reset()) {
        // In reset. Give up.
        return false;
      }
      // Is the thread free?
      if (m_ParamCl->mthreadbusy_exact == 1) {
        // MThreadBusy is part of the channel and we are required to check it.
        // Now we need to test to see if the thread of the response is busy.
        if ( m_DataCl->isThisThreadBusy(resp.SThreadID, getMThreadBusy()) ) {
          // The thread is busy. Fail.
#ifdef DEBUG_C
          cout << name() << ": sendOCPResponse failed. It was called for a busy"
          << " thread on a channel with mthreadbusy_exact." << endl;
#endif
          return false;
        }
      }

      if(!SputResponse())
          return false;
        
      // Set response fields of the data class
      SetResponseFields(resp, RespChunkLen , last_chunk_of_a_burst);

      return true;
    }

    // 'startOCPResponse()'
    //
    // This method is an alias for 'startOCPRequest()', to ensure compatibility
    // with the TL1 API
    //
    bool startOCPResponse(OCPResponseGrp<Tdata>& resp, unsigned int
        RespChunkLen = 1, bool last_chunk_of_a_burst = true){
      return sendOCPResponse(resp,RespChunkLen,last_chunk_of_a_burst);
    }


    // 'sendOCPResponseBlocking()'
    //
    bool sendOCPResponseBlocking(OCPResponseGrp<Tdata>& resp, unsigned int
        RespChunkLen = 1, bool last_chunk_of_a_burst = true){
      // is the channel in reset?
      if (get_reset()) {
        // In reset. Give up.
        return false;
      }
      // Is the thread free?
      if (m_ParamCl->mthreadbusy_exact == 1){
        // MThreadBusy is part of the channel and we are required to check it.
        // Now we need to test to see if the thread of the response is busy.
        if ( m_DataCl->isThisThreadBusy(resp.SThreadID, getMThreadBusy()) ) {
          // The thread is busy. Fail.
#ifdef DEBUG_C
          cout << name() << ": sendOCPResponseBlocking failed. It was called"
          << " for a busy thread on a channel with mthreadbusy_exact." << endl;
#endif
          return false;
        }
      }

      // Place the response onto the channel (possibly wait until the channel is
      // free)
      if(!SputResponseSemiBlocking()) {
        return false;
      }

      // Set response fields of the data class
      SetResponseFields(resp, RespChunkLen , last_chunk_of_a_burst);

      // Wait until the master has acknowledged the response
      waitMRespAccept();

      if (get_reset()) {
        // In reset. Give up.
        return false;
      }

      return true;
    }

    // 'startOCPResponseBlocking()'

    bool startOCPResponseBlocking(OCPResponseGrp<Tdata>& resp, unsigned int
        RespChunkLen = 1, bool last_chunk_of_a_burst = true){
      // is the channel in reset?
      if (get_reset()) {
        // In reset. Give up.
        return false;
      }
      // Is the thread free?
      if (m_ParamCl->mthreadbusy_exact == 1) {
        // MThreadBusy is part of the channel and we are required to check it.
        // Now we need to test to see if the thread of the response is busy.
        if ( m_DataCl->isThisThreadBusy(resp.SThreadID, getMThreadBusy()) ) {
          // The thread is busy. Fail.
#ifdef DEBUG_C
          cout << name() << ": sendOCPResponseBlocking failed. It was called"
          << " for a busy thread on a channel with mthreadbusy_exact." << endl;
#endif
          return false;
        }
      }

      // Wait until the response channel is free, place the response onto the
      // channel but do not wait for acknowledgment
      if(!SputResponseSemiBlocking()) {
        return false;
      }

      // Set response fields of the data class
      SetResponseFields(resp, RespChunkLen , last_chunk_of_a_burst);

      return true;
    }


    //---------------------------------------------------------------------
    // Synchronization methods 
    //---------------------------------------------------------------------

    bool getMBusy(void) const {
      return SgetMbusy();
    }

    bool putSCmdAccept() {

      // If 'cmdaccept' is set to 0, ignore
      if(m_ParamCl->cmdaccept == 0) {
#ifdef DEBUG_C
        cout << name() << ": WARNING - putSCmdAccept called when 'cmdaccept'"
        << " parameter set to 0 " << endl;
#endif
        return true;
      }

      // Was this command called at the correct time?
      if (! m_CommCl->RequestStart ) {
#ifdef DEBUG_C
        cout << name() << ": WARNING - putSCmdAccept called when there was no"
        << " request on the channel." << endl;
#endif
        return false;
      }
      Srelease();
      return true;
    }

    bool putSCmdAccept(sc_time after) {

      // If 'cmdaccept' is set to 0, ignore
      if(m_ParamCl->cmdaccept == 0) {
#ifdef DEBUG_C
        cout << name() << ": WARNING - putSCmdAccept called when 'cmdaccept'"
        << " parameter set to 0 " << endl;
#endif
        return true;
      }

      // Was this command called at the correct time?
      if (! m_CommCl->RequestStart ) {
#ifdef DEBUG_C
        cout << name() << ": WARNING - putSCmdAccept called when there was no"
        << " request on the channel." << endl;
#endif
        return false;
      }
      Srelease(after);
      return true;
    }

    void putSThreadBusy(unsigned int sthreadbusy) 
    { 
      if(!get_reset())
        m_DataCl->SputSThreadBusy( sthreadbusy );
    }

    void putSThreadBusyBit(bool nextBitValue, unsigned int ThreadID = 0) 
    {
      // is the channel in reset?
      if (!get_reset()) {
        if(nextBitValue) {
          // Set bit
          unsigned int mask = ( ((unsigned int) 1) << ThreadID );
          m_DataCl->SputSThreadBusy(m_DataCl->MgetSThreadBusy() | mask );
        } else {
          // Reset bit
          unsigned int mask = ~( ((unsigned int) 1) << ThreadID );
          m_DataCl->SputSThreadBusy(m_DataCl->MgetSThreadBusy() & mask );
        }
      }
    }


    bool getMRespAccept(void) const {
      if (!m_ParamCl->respaccept) {
        return(true);
      }
      else {
        return(!getMBusy());
      }
    }

    unsigned int getMThreadBusy(void) const
    {
      // is the channel in reset?
      if (m_CommCl->Reset) {
        // In reset. Give up.
        return false;
      } 
      if (m_ParamCl->mthreadbusy == 0 ){
#ifdef DEBUG_C
        cout << name() << ": WARNING - getMThreadBusy called on a channel with"
        << " no MThreadBusy signal." << endl;
#endif
        // No MThreadBusy signal so no threads are busy
        return 0;
      }
      return m_DataCl->SgetMThreadBusy();
    }

    bool getMThreadBusyBit(unsigned int ThreadID = 0) const 
    {
      // is the channel in reset?
      if (m_CommCl->Reset) {
        // In reset. Give up.
        return false;
      } 
      if (m_ParamCl->mthreadbusy == 0){
#ifdef DEBUG_C
        cout << name() << ": WARNING - getMThreadBusyBit called on a channel"
        << " with no MThreadBusy signal." << endl;
#endif
        // No MThreadBusy signal so no threads are busy
        return false;
      } 

      return m_DataCl->isThisThreadBusy(ThreadID, m_DataCl->SgetMThreadBusy());
    }

    void waitMRespAccept(void) {
      // is the channel in reset?
      if (!get_reset()) {
        if (!getMBusy() || !m_ParamCl->respaccept) {
          // The command has already been accepted
          return;
        } 
        wait(m_CommCl->ResponseEndEvent);
      }
    }


    //---------------------------------------------------------------------
    // Request Group Access methods
    //---------------------------------------------------------------------
    Taddr SgetMAddr(){return m_DataCl->SgetMAddr();}
    unsigned int SgetMAddrSpace(){return m_DataCl->SgetMAddrSpace();}
    unsigned int SgetMAtomicLength(){return m_DataCl->SgetMAtomicLength();}
    OCPMBurstSeqType SgetMBurstSeq(){return m_DataCl->SgetMBurstSeq();}
    unsigned int SgetMByteEn(){return m_DataCl->SgetMByteEn();}
    OCPMCmdType SgetMCmd(){return m_DataCl->SgetMCmd();}
    unsigned int SgetMConnID(){return m_DataCl->SgetMConnID();}
    Tdata* SgetMData(unsigned int& w, bool& last_of_a_burst){return
      m_DataCl->SgetMData(w, last_of_a_burst);}
    Tdata* SgetMData(unsigned int& w){return m_DataCl->SgetMData(w);}
    Tdata* SgetMData(){return m_DataCl->SgetMData();}
    unsigned int SgetMThreadID(){return m_DataCl->SgetMThreadID();}
    unsigned int SgetMBurstLength(){return m_DataCl->SgetMBurstLength();}
    unsigned int SgetMBurstPrecise(){return m_DataCl->SgetMBurstPrecise();}
    unsigned int SgetMBurstSingleReq(){return m_DataCl->SgetMBurstSingleReq();}
    unsigned int SgetMReqInfo(){return m_DataCl->SgetMReqInfo();}
    unsigned int SgetMReqLast(){return m_DataCl->SgetMReqLast();}

    //---------------------------------------------------------------------
    // Response Group Access methods
    //---------------------------------------------------------------------
    void SputSData(Tdata* d, unsigned int w = 1, bool last_of_a_burst =
        true){m_DataCl->SputSData(d,w,last_of_a_burst);}
    void SputSDataInfo(unsigned int a){m_DataCl->SputSDataInfo(a);}
    void SputSResp(OCPSRespType a){m_DataCl->SputSResp(a);}
    void SputSRespInfo(unsigned int a){m_DataCl->SputSRespInfo(a);}
    void SputSRespLast(bool a){m_DataCl->SputSRespLast(a);}
    void SputSThreadID(unsigned int a){m_DataCl->SputSThreadID(a);}

    //---------------------------------------------------------------------
    // DataHS Group Access methods
    //---------------------------------------------------------------------
    unsigned int SgetMDataInfo(){return m_DataCl->SgetMDataInfo();}

    //---------------------------------------------------------------------
    // ThreadBusy Group Access methods
    //---------------------------------------------------------------------
    unsigned int SgetMThreadBusy(void) const{return m_DataCl->SgetMThreadBusy();}
    void SputSThreadBusy(unsigned int
        sthreadbusy){m_DataCl->SputSThreadBusy(sthreadbusy);}

    //---------------------------------------------------------------------
    // SideBand Group Access methods
    //---------------------------------------------------------------------
    bool SgetMError(void) const               { return m_DataCl->SgetMError();}
    unsigned int SgetMFlag(void) const        { return m_DataCl->SgetMFlag();}

    // Sideband reset replaced with reset assert/deassert calls
    void SputSReset_n(bool nextValue)         {
      cout << "SputSReset_n is deprecated " << endl;
    }
    bool SgetMReset_n(void) const             { 
      cout << "SgetMReset_n is deprecated " << endl;
      return false;
    }

    void SputSError(bool nextValue){ m_DataCl->SputSError(nextValue);}
    void SputSFlag(unsigned int nextValue){ m_DataCl->SputSFlag(nextValue);}
    void SputSInterrupt(bool nextValue){ m_DataCl->SputSInterrupt(nextValue);}

    //---------------------------------------------------------------------
    // TL2-specific burst chunks methods
    //---------------------------------------------------------------------
    void SputSRespChunkLen(unsigned int w){m_DataCl->SputSRespChunkLen(w);}
    void SputSRespChunkLast(bool w){m_DataCl->SputSRespChunkLast(w);}
    unsigned int SgetMReqChunkLen() const{return m_DataCl->SgetMReqChunkLen();}
    bool SgetMReqChunkLast() const{return m_DataCl->SgetMReqChunkLast();}


    //////////////////////////////////////////////////////////////
    // OCP TL2 Slave/Master Interfaces common methods
    //////////////////////////////////////////////////////////////

    //---------------------------------------------------------------------
    // System/Core Sideband signals 
    //---------------------------------------------------------------------
    void SysputControl(int nextValue){m_DataCl->SysputControl(nextValue);}
    bool SysgetControlBusy(void) const{return m_DataCl->SysgetControlBusy();}
    void SysputControlWr(bool nextValue){m_DataCl->SysputControlWr(nextValue);}
    unsigned int SysgetStatus(void) const{return m_DataCl->SysgetStatus();}
    bool SysgetStatusBusy(void) const{return m_DataCl->SysgetStatusBusy();}
    void SysputStatusRd(bool nextValue){m_DataCl->SysputStatusRd(nextValue);}
    unsigned int CgetControl(void) const{return m_DataCl->CgetControl();}
    void CputControlBusy(bool nextValue){m_DataCl->CputControlBusy(nextValue);}
    bool CgetControlWr(void) const{return m_DataCl->CgetControlWr();}
    void CputStatus(int nextValue){m_DataCl->CputStatus(nextValue);}
    void CputStatusBusy(bool nextValue){m_DataCl->CputStatusBusy(nextValue);}
    bool CgetStatusRd(void) const{return m_DataCl->CgetStatusRd();}


    //---------------------------------------------------------------------
    // Event accesses
    //---------------------------------------------------------------------

    const sc_event& RequestStartEvent(void) const{return
      m_CommCl->RequestStartEvent;}
    const sc_event& RequestEndEvent(void) const{return
      m_CommCl->RequestEndEvent;}
    const sc_event& DataRequestStartEvent(void) const{return
      m_CommCl->DataRequestStartEvent;}
    const sc_event& DataRequestEndEvent(void) const{return
      m_CommCl->DataRequestEndEvent;}
    const sc_event& ResponseStartEvent(void) const{return
      m_CommCl->ResponseStartEvent;}
    const sc_event& ResponseEndEvent(void) const{return
      m_CommCl->ResponseEndEvent;}
    const sc_event& SThreadBusyEvent(void) const{return
      m_DataCl->SBSThreadBusyEvent;}
    const sc_event& MThreadBusyEvent(void) const{return
      m_DataCl->SBMThreadBusyEvent;}
    const sc_event& SidebandMasterEvent(void) const{return
      m_DataCl->SBMasterEvent;} 
    const sc_event& SidebandSlaveEvent(void) const{return
      m_DataCl->SBSlaveEvent;} 
    // Sideband reset is removed
    //        const sc_event& SidebandResetEvent(void) const{return
    //        m_DataCl->SBResetEvent;} 
    const sc_event& SidebandControlEvent(void) const{return
      m_DataCl->SBControlEvent;}
    const sc_event& SidebandStatusEvent(void) const{return m_DataCl->SBStatusEvent;}
    const sc_event& ResetStartEvent(void) const{return
      m_CommCl->ResetStartEvent;} 
    const sc_event& ResetEndEvent(void) const{return m_CommCl->ResetEndEvent;} 

    //---------------------------------------------------------------------
    // Configuration Functions
    //---------------------------------------------------------------------

    void setConfiguration( MapStringType& passedMap )
    {
      // Use the passed data base to set up the configuration class
      m_ParamCl->setOCPConfiguration( name(), passedMap );

    } 

    //////////////////////////////////////////////////////////////
    // Transparent access to the generic channel's functions
    //////////////////////////////////////////////////////////////

    // Generic Methods for requests/responses synchronization
    // 
    void Mrelease(sc_time time)
        { TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::Mrelease(time); }
    void Srelease(sc_time time)
        { TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::Srelease(time); }
    void Mrelease(void)
        { TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::Mrelease(); }
    void Srelease(void)
        { TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::Srelease(); }
    void SreleaseData(sc_time time)
        { TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::SreleaseData(time); }
    void SreleaseData(void)
        { TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::SreleaseData(); }
    void MreleasePE(void)
        { TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::MreleasePE(); }
    void MunreleasePE(void)
        { TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::MunreleasePE(); }
    void SreleasePE(void)
        { TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::SreleasePE(); }
    void SunreleasePE(void)
        { TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::SunreleasePE(); }
    void SreleaseDataPE(void)
        { TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::SreleaseDataPE(); }
    void SunreleaseDataPE(void)
        { TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::SunreleaseDataPE(); }
    bool MgetSbusy(void) const
        { return (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::MgetSbusy()); }
    bool MgetSbusyData(void) const
        { return (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::MgetSbusyData()); }
    bool SgetMbusy(void) const
        { return (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::SgetMbusy()); }
    bool IsWrite(void)
        { return (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::IsWrite()); }

    // Generic Methods to send/get requests/responses
    // 
    bool MputWriteRequest(void)
    { return (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::MputWriteRequest()); }
    bool MputReadRequest(void)
    { return (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::MputReadRequest()); }
    bool MputWriteRequestBlocking(void) { return
      (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::MputWriteRequestBlocking()); }
    bool MputReadRequestBlocking(void) { return
      (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::MputReadRequestBlocking()); }
    bool MputRequest(void) 
        { return (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::MputRequest()); }
    bool MputRequestBlocking(void) { return
        (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::MputRequestBlocking()); }
    bool SgetRequest(bool Release) { return
      (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::SgetRequest(Release)); }
    bool SgetRequestBlocking(bool Release) { return
      (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::SgetRequestBlocking(Release));
    }
    bool SgetRequestPE(void)
        { return (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::SgetRequestPE()); }
    bool SputResponse(void) 
        { return (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::SputResponse()); }
    bool SputResponseBlocking(void) { return
      (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::SputResponseBlocking()); }
    bool MgetResponse(bool Release) { return
      (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::MgetResponse(Release)); }
    bool MgetResponseBlocking(bool Release) { return
     (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::MgetResponseBlocking(Release));
    }
    bool MgetResponsePE(void) { return
      (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::MgetResponsePE()); }
    bool MputDataRequest(void) { return
      (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::MputDataRequest()); }
    bool MputDataRequestBlocking(void) { return
      (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::MputDataRequestBlocking()); }
    bool SgetDataRequest(bool Release) { return
      (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::SgetDataRequest(Release)); }
    bool SgetDataRequestBlocking(bool Release) { return
     (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::SgetDataRequestBlocking(Release)
      );
    }
    bool SgetDataRequestPE(void) { return
      (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::SgetDataRequestPE()); }

    // Generic Methods for Reset
    void reset()
    { 
      // Start reset now
      TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::reset(); 
    }
    void remove_reset()
    { 
      // End reset at the end of delta cycle
      TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::remove_reset();
    }
    bool get_reset()
    { 
      return(TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::get_reset());
    }
    void Reset() {
      reset();
      remove_reset();
    }


    // OCP Reset methods, use these
    bool getReset() { return (get_reset()); }
    void MResetAssert(void)
    {
      // reset the channel
      reset();
      m_DataCl->MputMReset_n(1);
    }
    void MResetDeassert(void) 
    {
      // Now pull out of reset mode
      remove_reset();
      m_DataCl->MputMReset_n(0);
    }
    void SResetAssert(void)
    { 
      reset();
      m_DataCl->SputSReset_n(1);
    }
    void SResetDeassert(void)
    {
      remove_reset();
      m_DataCl->SputSReset_n(0);
    }

    // Generic Methods for direct access
    // 
    virtual bool MputDirect(int MasterID, bool IsWrite, Tdata *Data, Taddr
        Address, int NumWords)
    {
      return (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr>
          >::MputDirect(MasterID,IsWrite,Data, Address,NumWords));
    }
    virtual bool SputDirect(int SlaveID, bool IsWrite, Tdata *Data, Taddr Address,
        int NumWords)
    {
      return (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr>
          >::SputDirect(SlaveID,IsWrite,Data, Address,NumWords));
    }

    // Register port method 
    // 
    virtual void register_port(sc_port_base& port, const char* if_typename)
    {
      TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::register_port(port,if_typename);
    }
    void SregisterDirectIF(SdirectIF<OCP_TL2_DataCl<Tdata,Taddr> >* A)
    {
      TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::SregisterDirectIF(A);
    }
    void MregisterDirectIF(MdirectIF<OCP_TL2_DataCl<Tdata,Taddr> >* A)
    {
      TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::MregisterDirectIF(A);
    }


    // 'Param', 'Data' and 'Comm' class accesses 
    // 
    OCP_TL2_DataCl<Tdata,Taddr> * GetDataCl(void)
    { return (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::GetDataCl()); }

    ParamCl<OCP_TL2_DataCl<Tdata,Taddr> >* GetParamCl(void)
    { return (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::GetParamCl()); }

    CommCl* GetCommCl(void) 
    { return (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::GetCommCl()); }

    // 'Default event' 
    // 
    const sc_event& default_event() const
    { return (TL_Channel<OCP_TL2_DataCl<Tdata,Taddr> >::default_event()); }



  protected:
    //---------------------------------------------------------------
    // protected methods
    //---------------------------------------------------------------
    // display a fatal error message and quick the program
    void fatalError(char* sub_title, char* msg)
    {
      cout << endl << "ERROR: " << sub_title << ":" << msg << endl;
      assert(0);
    }



  protected:
    //----------------------------------------------------------------------
    // member data
    //----------------------------------------------------------------------

};

#endif  // _OCP_TL2_CHANNEL_H
