// Copyright 2003, 2004 OCP-IP
// OCP-IP Confidential & Proprietary
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Norman Weyrich, Synopsys Inc.
//                Anssi Haverinen, Nokia Inc.
//                Joe Chou, Sonics Inc.
//                Yann Bajot, Prosilog
//         $Id: ocp_tl2_data_cl.h,v 1.3 2004/09/10 21:50:21 akamas Exp $
//
//  Description : Transaction Level - Layer-2 data class to model
//                the Open Core Protocol (OCP) 2.0
//                The data class can be used in conjunction with OCP
//                transaction channel (ocp_tl2_channel.h) to transport data
//                between Initiator and Target modules.  Contains member
//                variables and access methods to set or query member
//                variables.
//                The member variables are split in to three groups:
//                - Request-Channel members (set by Master read by Slave)
//                - Response-Channel members (set by Slave, read by Master)
//                - Sideband members
//
//                See definition of these groups in OCP 2.0 specification
// ============================================================================

// This file contains the definition of the classes:
// template <class Td, class Ta> class OCP_TL2_DataCl

#ifndef _OCP_TL2_DATA_CL
#define _OCP_TL2_DATA_CL

#include "systemc.h"
#include "ocp_globals.h"


// -------------------------------------------------
// OCP TL2 Data class
// -------------------------------------------------
template <class Td, class Ta> class OCP_TL2_DataCl : public sc_prim_channel
{
    public:

        typedef Td DataType;
        typedef Ta AddrType;


        // Constructor
        OCP_TL2_DataCl()
        {
            Reset();
        }

        // Destructor
        ~OCP_TL2_DataCl()
        {
        }

        void Reset()
        {
            // Request Signal Group
            m_ReqGrp.reset();

            // Response Signal Group
            m_RespGrp.reset();

            // DataHandshake Signal Group
            m_DataHSGrp.reset();

            // Sideband signals
            m_SidebandGrp.reset();

            // ThreadBusy signals
            m_ThreadBusyGrp.reset();

            // TL2-specific fields
            m_SRespChunkLast = true;
            m_SRespChunkLen = 1;
            m_MReqChunkLast  = true;
            m_MReqChunkLen = 1;

            // 
            m_MCmdSaved = OCP_MCMD_IDLE;
            m_SRespSaved = OCP_SRESP_NULL;

        }

        //---------------------------------------------------------------------
        // -------------- Start of access methods -------------------------------

        // These three methods are used in the Channel,
        // and hence they must always be present!

        bool IsWriteRequest()
        {
            return((m_ReqGrp.MCmd == OCP_MCMD_WR) ? true : false);
        }
        void SetWriteRequest()
        {
            m_ReqGrp.MCmd = OCP_MCMD_WR;
        }
        void SetReadRequest() 
        {
            m_ReqGrp.MCmd = OCP_MCMD_RD;
        }

        // ---------- Request phase update 
        void update_Fw(int eventSelect) {
        }

        // ---------- Data Request phase update 
        void update_FwD(int) {}

        // ---------- Response phase update
        void update_Bw(int eventSelect) {
        }


        // Time stamp methods: These methods could e.g. be used by
        // Masters and Slaves to indicate a duration of a packet.
        void MputEndTime(sc_time tt)
        {
            ReqEndTime = tt;
        }
        sc_time SgetEndTime()
        {
            return ReqEndTime;
        }
        void SputEndTime(sc_time tt)
        {
            ResEndTime = tt;
        }
        sc_time MgetEndTime()
        {
            return ResEndTime;
        }

        //-----------------------------------------------------------
        // The following methods are needed to run the examples
        // We recommend to use them in user applications as well
        //
        // These are access methods for data fields to
        //   - make code more readable and
        //   - make Master and Slave code independent of the
        //     data structure. If a different data structure is used
        //     only the data class must be changed but not Master and Slave code.


        // ---------- Request Signal Group ----------

        // Access methods for MAddr
        void MputMAddr(Ta a)
        {
            m_ReqGrp.MAddr = a;
        }
        Ta SgetMAddr()
        {
            return m_ReqGrp.MAddr;
        }

        // Access methods for MAddrSpace
        void MputMAddrSpace(unsigned int a)
        {
            m_ReqGrp.MAddrSpace = a;
        }
        unsigned int SgetMAddrSpace()
        {
            return m_ReqGrp.MAddrSpace;
        }

        // Access methods for MAtomicLength
        void MputMAtomicLength(unsigned int a)
        {
            m_ReqGrp.MAtomicLength = a;
        }
        unsigned int SgetMAtomicLength()
        {
            return m_ReqGrp.MAtomicLength;
        }


        // Access methods for MBurstSeq
        void MputMBurstSeq(OCPMBurstSeqType a)
        {
            m_ReqGrp.MBurstSeq = a;
        }
        OCPMBurstSeqType SgetMBurstSeq()
        {
            return m_ReqGrp.MBurstSeq;
        }

        // Access methods for MByteEn
        void MputMByteEn(unsigned int a)
        {
            m_ReqGrp.MByteEn = a;
        }
        unsigned int SgetMByteEn()
        {
            return m_ReqGrp.MByteEn;
        }

        // Access methods for MCmd
        void MputMCmd(OCPMCmdType a)
        {
            m_ReqGrp.MCmd = a;
            m_MCmdSaved = a;
        }
        OCPMCmdType SgetMCmd()
        {
            OCPMCmdType a = m_ReqGrp.MCmd;
            m_ReqGrp.MCmd = OCP_MCMD_IDLE; // reset
            return a;
        }

        // Special function that can be used by non-intrusive channel readers (e.g.
        // transaction recorders)
        OCPMCmdType SgetMCmdSaved()
        {
            return m_MCmdSaved;
        }

        // Access methods for MConnID
        void MputMConnID(unsigned int a)
        {
            m_ReqGrp.MConnID = a;
        }
        unsigned int SgetMConnID()
        {
            return m_ReqGrp.MConnID;
        }

        // Access methods for MData 
        //
        // Note: this is a pointer-passing method
        void MputMData(Td* d, unsigned int w = 1, bool last_of_a_burst = true)
        {
            m_ReqGrp.MDataPtr = d;
            m_MReqChunkLen = w;
            m_MReqChunkLast = last_of_a_burst;
        }
        Td* SgetMData(unsigned int& w, bool& last_of_a_burst)
        {
            w = m_MReqChunkLen;
            last_of_a_burst = m_MReqChunkLast;

            return (m_ReqGrp.MDataPtr);
        }
        Td* SgetMData(unsigned int& w)
        {
            w = m_MReqChunkLen;
            return (m_ReqGrp.MDataPtr);
        }
        Td* SgetMData()
        {
            return (m_ReqGrp.MDataPtr);
        }

        // Access methods for MDataInfo
        void MputMDataInfo(unsigned int a)  
        {
            m_DataHSGrp.MDataInfo = a;
        }
        unsigned int SgetMDataInfo()
        {
            return m_DataHSGrp.MDataInfo;
        }


        // Access methods for MThreadID
        void MputMThreadID(unsigned int a)
        {
            m_ReqGrp.MThreadID = a;
        }
        unsigned int SgetMThreadID()
        {
            return m_ReqGrp.MThreadID;
        }

        // Access methods for MBurstLength
        void MputMBurstLength(unsigned int a)
        {
            m_ReqGrp.MBurstLength = a;
        }
        unsigned int SgetMBurstLength()
        {
            return m_ReqGrp.MBurstLength;
        }

        // Access methods for MBurstPrecise 
        void MputMBurstPrecise(bool a)
        {
            m_ReqGrp.MBurstPrecise = a;
        }
        unsigned int SgetMBurstPrecise()
        {
            return m_ReqGrp.MBurstPrecise;
        }

        // Access methods for MBurstSingleReq 
        void MputMBurstSingleReq(bool a)
        {
            m_ReqGrp.MBurstSingleReq = a;
        }
        unsigned int SgetMBurstSingleReq()
        {
            return m_ReqGrp.MBurstSingleReq;
        }

        // Access methods for MReqInfo 
        void MputMReqInfo(unsigned int a)
        {
            m_ReqGrp.MReqInfo = a;
        }
        unsigned int SgetMReqInfo()
        {
            return m_ReqGrp.MReqInfo;
        }

        // Access methods for MReqLast 
        void MputMReqLast(bool a)
        {
            m_ReqGrp.MReqLast = a;
        }
        unsigned int SgetMReqLast()
        {
            return m_ReqGrp.MReqLast;
        }

        // Not part of OCP 2.0, but necessary for TL2 to work
        void MputMReqChunkLen(unsigned int w) {
            m_MReqChunkLen = w;
        }
        unsigned int SgetMReqChunkLen() {
            return m_MReqChunkLen;
        }

        void MputMReqChunkLast(bool w) {
            m_MReqChunkLast = w;
        }
        unsigned int SgetMReqChunkLast() {
            return m_MReqChunkLast;
        }

        // ---------- Response Signal Group ----------

        // Access methods for SData
        void SputSData(Td* d, unsigned int w = 1, bool last_of_a_burst = true)
        {
            m_RespGrp.SDataPtr = d;
            m_SRespChunkLen = w;
            m_SRespChunkLast = last_of_a_burst;
        }

        Td* MgetSData(unsigned int& w, bool& last_of_a_burst)
        {
            w = m_SRespChunkLen;
            last_of_a_burst = m_SRespChunkLast;

            return (m_RespGrp.SDataPtr);
        }
        Td* MgetSData(unsigned int& w)
        {
            w = m_SRespChunkLen;

            return (m_RespGrp.SDataPtr);
        }
        Td* MgetSData()
        {
            return (m_RespGrp.SDataPtr);
        }

        // Access methods for SDataInfo
        void SputSDataInfo(unsigned int a)
        {
            m_RespGrp.SDataInfo = a;
        }
        unsigned int MgetSDataInfo()
        {
            return m_RespGrp.SDataInfo;
        }

        // Access methods for SResp
        void SputSResp(OCPSRespType a)
        {
            m_RespGrp.SResp = a;
            m_SRespSaved = a;
        }

        OCPSRespType MgetSResp()
        {
            OCPSRespType a = m_RespGrp.SResp;
            m_RespGrp.SResp = OCP_SRESP_NULL; // reset
            return a;
        }

        // Special function that can be used by non-intrusive channel readers (e.g.
        // transaction recorders)
        OCPSRespType MgetSRespSaved()
        {
            return m_SRespSaved;
        }

        // Access methods for SRespInfo
        void SputSRespInfo(unsigned int a)
        {
            m_RespGrp.SRespInfo = a;
        }
        unsigned int MgetSRespInfo()
        {
            return m_RespGrp.SRespInfo;
        }

        // Access methods for SRespLast
        void SputSRespLast(bool a)
        {
            m_RespGrp.SRespLast = a;
        }
        bool MgetSRespLast()
        {
            return m_RespGrp.SRespLast;
        }

        // Access methods for SThreadID
        void SputSThreadID(unsigned int a)
        {
            m_RespGrp.SThreadID = a;
        }
        unsigned int MgetSThreadID()
        {
            return m_RespGrp.SThreadID;
        }

        // Not part of OCP 2.0, but necessary for TL2 to work
        void SputSRespChunkLen(unsigned int w) {
            m_SRespChunkLen = w;
        }
        unsigned int MgetSRespChunkLen() {
            return m_SRespChunkLen;
        }
        void SputSRespChunkLast(bool w) {
            m_SRespChunkLast = w;
        }
        bool MgetSRespChunkLast() {
            return m_SRespChunkLast;
        }



        //---------------------------------------------------------------------
        // Sideband signals
        //---------------------------------------------------------------------

        // Reset signals
        bool SgetMReset_n()
        { return (m_SidebandGrp.MReset_n); }

        void MputMReset_n(bool mreset_n)
        { 
            m_SidebandGrp.MReset_n = mreset_n; 
        }

        bool MgetSReset_n()
        { return (m_SidebandGrp.SReset_n); }

        void SputSReset_n(bool sreset_n)
        { 
            m_SidebandGrp.SReset_n = sreset_n; 
        }

        // System/Core signals
        unsigned int  CgetControl()
        { return (m_SidebandGrp.Control); }

        void SysputControl(unsigned int control)
        { 
            m_SidebandGrp.Control = control; 
            SBControlEvent.notify(SC_ZERO_TIME);
        }

        bool  CgetControlWr()
        { return (m_SidebandGrp.ControlWr); }

        void SysputControlWr(bool controlwr)
        { 
            m_SidebandGrp.ControlWr = controlwr; 
            SBControlEvent.notify(SC_ZERO_TIME);
        }

        bool  SysgetControlBusy()
        { return (m_SidebandGrp.ControlBusy); }

        void CputControlBusy(bool controlbusy)
        { 
            m_SidebandGrp.ControlBusy = controlbusy; 
            SBControlEvent.notify(SC_ZERO_TIME);
        }

        unsigned int  SysgetStatus()
        { return (m_SidebandGrp.Status); }

        void CputStatus(unsigned int status)
        { 
            m_SidebandGrp.Status = status; 
            SBStatusEvent.notify(SC_ZERO_TIME);
        }

        bool  SysgetStatusBusy()
        { return (m_SidebandGrp.StatusBusy); }

        void CputStatusBusy(bool statusbusy)
        { 
            m_SidebandGrp.StatusBusy = statusbusy; 
            SBStatusEvent.notify(SC_ZERO_TIME);
        }

        bool  CgetStatusRd()
        { return (m_SidebandGrp.StatusRd); }

        void SysputStatusRd(bool statusrd)
        {
            m_SidebandGrp.StatusRd = statusrd;
            SBStatusEvent.notify(SC_ZERO_TIME);

        }

        // Threadbusy signals
        unsigned int MgetSThreadBusy() 
        {
            return m_ThreadBusyGrp.SThreadBusy;
        }

        void SputSThreadBusy(unsigned int sthreadbusy)
        {
            m_ThreadBusyGrp.SThreadBusy = sthreadbusy;
            SBSThreadBusyEvent.notify(SC_ZERO_TIME);
        }

        unsigned int SgetMThreadBusy()
        {
            return m_ThreadBusyGrp.MThreadBusy;
        }

        void MputMThreadBusy(unsigned int mthreadbusy)
        {
            m_ThreadBusyGrp.MThreadBusy = mthreadbusy;
            SBMThreadBusyEvent.notify(SC_ZERO_TIME);
        }

        // Sideband signals

        bool MgetSError()
        { return (m_SidebandGrp.SError); }

        void SputSError(bool serror)
        { 
            m_SidebandGrp.SError = serror; 
            SBSlaveEvent.notify(SC_ZERO_TIME);
        }

        bool MgetSInterrupt()
        { return (m_SidebandGrp.SInterrupt); }

        void SputSInterrupt(bool sinterrupt)
        { 
            m_SidebandGrp.SInterrupt = sinterrupt; 
            SBSlaveEvent.notify(SC_ZERO_TIME);
        }

        unsigned int  MgetSFlag()
        { return (m_SidebandGrp.SFlag); }

        void SputSFlag(unsigned int sflag)
        { 
            m_SidebandGrp.SFlag = sflag; 
            SBSlaveEvent.notify(SC_ZERO_TIME);
        }

        bool SgetMError()
        { return (m_SidebandGrp.MError); }

        void MputMError(bool merror)
        { 
            m_SidebandGrp.MError = merror; 
            SBMasterEvent.notify(SC_ZERO_TIME);
        }

        unsigned int  SgetMFlag()
        { return (m_SidebandGrp.MFlag); }

        void MputMFlag(unsigned int mflag)
        { 
            m_SidebandGrp.MFlag = mflag; 
            SBMasterEvent.notify(SC_ZERO_TIME);
        }

        //---------------------------------------------------------------------
        // Complete Request Group setting method
        //---------------------------------------------------------------------

        // Set OCP TL2 dataclass fields according to the passed 
        //  'OCPRequestGrp' structure and 'chunk' parameters
        // 
        void SetRequestFields(OCPRequestGrp<Td,Ta>& req, int ReqChunkLen = 1, bool last_chunk_of_a_burst = true)
        {
            m_ReqGrp        = req;
            m_MReqChunkLen  = ReqChunkLen;
            m_MReqChunkLast = last_chunk_of_a_burst;
        }

        //---------------------------------------------------------------------
        // Complete Response Group access method
        //---------------------------------------------------------------------
        // Store OCP TL2 dataclass response fields into the passed 
        //  'OCPResponseGrp' structure, and return 'chunk' parameters
        // 
        // 'sdata_copy' parameter:
        //    * If 'sdata_copy' is set, SData array is copied from the slave (using
        //    the 'SData*' TL2 dataclass pointer) into the location pointed by
        //    'OCPResponseGrp.SDataPtr'. It assumes that memory allocation (e.g.
        //    malloc() or new) has been performed for this location. This method DOES
        //    NOT ALLOCATE MEMORY itself.
        //  
        //    * If 'sdata_copy' is not set, only a pointer copy is performed.
        //
        void GetResponseFields(OCPResponseGrp<Td>& resp, bool sdata_copy, unsigned int& RespChunkLen , bool& last_chunk_of_a_burst)
        {
            resp.SResp    = m_RespGrp.SResp;
            resp.SDataInfo= m_RespGrp.SDataInfo;
            resp.SThreadID= m_RespGrp.SThreadID;
            resp.SRespInfo= m_RespGrp.SRespInfo;
            resp.SRespLast= m_RespGrp.SRespLast;

            // Get 'chunk' parameters
            RespChunkLen = m_SRespChunkLen;
            last_chunk_of_a_burst = m_SRespChunkLast;


            if(sdata_copy) {

                // Test NULL pointer
                if(resp.SDataPtr == NULL) {
                    fatalError("GetResponseFields()","'SDataPtr' is NULL");
                }

                // Data copy
                for (unsigned int i=0; i<RespChunkLen; i+=1 ) {
                    resp.SDataPtr[i]= m_RespGrp.SDataPtr[i];
                }

            } else {

                // Pointer passing
                resp.SDataPtr= m_RespGrp.SDataPtr;
            }

        }

        //---------------------------------------------------------------------
        // Complete Response Group setting method
        //---------------------------------------------------------------------

        // Set OCP TL2 dataclass fields according to the passed
        //  'OCPResponseGrp' structure and 'chunk' parameters
        // 
        void SetResponseFields(OCPResponseGrp<Td>& resp, int RespChunkLen = 1, bool last_chunk_of_a_burst = true)
        {
            m_RespGrp = resp;
            m_SRespChunkLen  = RespChunkLen;
            m_SRespChunkLast = last_chunk_of_a_burst;
        }


        // Complete Request Group access method
        //
        // 'mdata_copy' parameter:
        //    * If 'mdata_copy' is set, MData array is copied from the master (using
        //    the 'MData*' TL2 dataclass pointer) into the location pointed by
        //    'OCPRequestGrp.MDataPtr'. It assumes that memory allocation (e.g.
        //    malloc() or new) has been performed for this location. This method DOES
        //    NOT ALLOCATE MEMORY itself.
        //  
        //    * If 'mdata_copy' is not set, only a pointer copy is performed.
        //

        void GetRequestFields(OCPRequestGrp<Td,Ta>& req, bool mdata_copy, unsigned int& ReqChunkLen, bool& last_chunk_of_a_burst)
        {
            req.MCmd            = m_ReqGrp.MCmd            ; 
            req.MAddr           = m_ReqGrp.MAddr           ; 
            req.MAddrSpace      = m_ReqGrp.MAddrSpace      ; 
            req.MAtomicLength   = m_ReqGrp.MAtomicLength   ; 
            req.MBurstSeq       = m_ReqGrp.MBurstSeq       ; 
            req.MByteEn         = m_ReqGrp.MByteEn         ; 
            req.MConnID         = m_ReqGrp.MConnID         ; 
            req.MThreadID       = m_ReqGrp.MThreadID       ; 
            req.MBurstLength    = m_ReqGrp.MBurstLength    ; 
            req.MBurstPrecise   = m_ReqGrp.MBurstPrecise   ; 
            req.MBurstSingleReq = m_ReqGrp.MBurstSingleReq ; 
            req.MReqInfo        = m_ReqGrp.MReqInfo        ; 
            req.MReqLast        = m_ReqGrp.MReqLast        ;
            req.MDataInfo       = m_ReqGrp.MDataInfo       ;

            // Get 'chunk' parameters
            ReqChunkLen = m_MReqChunkLen;
            last_chunk_of_a_burst = m_MReqChunkLast;

            if(mdata_copy) {

                // Test NULL pointer
                if(req.MDataPtr == NULL) {
                    fatalError("GetRequestFields()","'MDataPtr' is NULL");
                }

                // Data copy
                for (unsigned int i=0; i<ReqChunkLen; i+=1 ) {
                    req.MDataPtr[i]= m_ReqGrp.MDataPtr[i];
                }
            } else {
                req.MDataPtr= m_ReqGrp.MDataPtr;
            }

        }




        //---------------------------------------------------------------------
        // Not used in TL2
        //---------------------------------------------------------------------
        void update()
        {
        }

        sc_event SBControlEvent;
        sc_event SBStatusEvent;
        sc_event SBMasterEvent;
        sc_event SBSlaveEvent;
        sc_event SBResetEvent;

        sc_event SBSThreadBusyEvent;
        sc_event SBMThreadBusyEvent;


        // -------------- End of access methods -------------------------------
        
        // is the passed thread ID number busy?
        bool isThisThreadBusy( int threadNum, unsigned int threadField )
        {
            return ( 1 & (threadField >> threadNum) );
        }

    private:

        // --------------- Data members --------------- 

        // Request Signal Group
        OCPRequestGrp<Td, Ta> m_ReqGrp;

        // Response Signal Group
        OCPResponseGrp<Td> m_RespGrp;

        // DataHandshake Signal Group
        //    Provided to allow access to 'MDataInfo' fields
        //    Other fields should not be used in TL2 
        OCPDataHSGrp<Td> m_DataHSGrp;

        // Threadbusy Signal Group
        OCPThreadBusyGrp m_ThreadBusyGrp;

        // Sideband Signal Group
        OCPSidebandGrp m_SidebandGrp;

        // -------------------------------------------------------------
        // for TL2 burst transactions (atomic and/or non-atomic)
        // -------------------------------------------------------------

        // for non-atomic burst transactions
        bool  m_MReqChunkLast;
        bool  m_SRespChunkLast;

        //
        unsigned int   m_MReqChunkLen;
        unsigned int   m_SRespChunkLen;

        // ---------------------------------------------------------
        // Helper Functions
        // ---------------------------------------------------------

        // The ReqEndTime time stamps can be used to pass on information
        // about the durations of request packets to other modules.
        sc_time ReqEndTime;

        // The ResEndTime time stamps can be used to pass on information
        // about the durations of response packets to other modules.
        sc_time ResEndTime;

        // Copies of 'MCmd' and 'SResp' OCP signals
        // These signals are automatically reset by the channel when they are read.
        // The following special members keep track of their latest values, and can
        // be used by non-intrusive Masters/Slaves (e.g. transaction recorders).
        OCPMCmdType   m_MCmdSaved;
        OCPSRespType  m_SRespSaved;

        // display a fatal error message and quit the program
        void fatalError(char* sub_title, char* msg)
        {
            cout << endl << "ERROR: " << sub_title << ":" << msg << endl;
            assert(0);
        }



};

#endif // _OCP_TL2_DATA_CL
