// 
//  Copyright 2004 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Stephane Guntz, PROSILOG, guntz@prosilog.com
//				  Yann Bajot, PROSILOG, bajot@prosilog.com               
//         Date : 01/05/2004
//
//  Description : Transaction Level - Layer-1 to Layer-2 Master Adapter (OCP 2.0 supported)  
//  Version: 1.0
//    
//  Parameters    :
//    Template arguments.
//     - TdataCl_tl1: TL1 Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//	   - TdataCl_tl2: TL2 Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//     - Td: Data type 
//     - Ta: Address type
//                
//	Constructor arguments:
//      - sc_module_name: Name of the module instance
//	    - max_chunk_length: maximum size of a chunk
//		- adapter_depth: Number of missed events and responses to be cached.
//						 The adapter stores new request/response infos which have arrived during the processing of the current request/response
//						 and sends them when the current request/response is finished
//		- timeout: a warning is sent if during a timeout period, no TL1 request or
//		TL2 response has been received by the master adapter 
//	 
//	 
//  The TL1 master adapter has a TL2 master interface on one side, and a TL1 slave interface on the other side.
//  It must be connected to:
//      - a TL1 master through a TL1 channel
//      - a TL2 slave through a TL2 channel
//
//	The adapter retrieves TL1 requests and stores them in its internal buffer. When a new request arrived, the adapter
//	compares it with former requests to decide whether it is part of the current burst or the beginning of a new burst
//  (it can be a single request too).
//  When a burst is finished, a TL2 request is sent to the slave.
//  
//  On the response side, the TL1 master adapter retrieves the TL2 response from the slave, and sends corresponding
//  TL1 responses to the master. 
// 
//	The following is unsupported:
//	- Single request/multiple data
//	- Data handshake
//	- Multiple threads
//	- SThreadBusy/MThreadBusy compliances (m_threadbusy_exact=0, s_threadbusy_exact=0)
// 
// ============================================================================

#include "ocp_tl1_tl2_master_adapter.h"

#define DEBUG_MASTER_ADAPTER_TL1


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter constructor
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::OCP_TL1_TL2_Master_Adapter(sc_module_name name_, int max_chunk_length_, int adapter_depth_, sc_time timeout): 
	sc_module(name_),
	SlaveP("SlaveP"),
	MasterP("MasterP"),
    max_chunk_length(max_chunk_length_),
	adapter_depth(adapter_depth_),
	m_timeout(timeout)
{

	buffer_size=max_chunk_length*adapter_depth;

	m_request_buffer=new OCPRequestGrp<Td, Ta>[buffer_size];
	m_response_buffer= new OCPResponseGrp<Td>[adapter_depth];
	m_data=new Td[buffer_size];
	m_response_data= new Td[buffer_size];

	//arrays for storing characteristics of TL2 requests
	burst_request_index=new int[adapter_depth];
	burst_request_length=new int[adapter_depth];

	//arrays for storing characteristics of TL1 responses
	response_index=new int[adapter_depth];
	response_length=new unsigned int[adapter_depth];

	//initialize internal request, data and response buffer
	for(int i=0; i<buffer_size; i++)  {
		reset_request_parameters(i);
		m_data[i]=0;
		m_response_data[i]=0;
	}

	beginning_burst_index= current_index=0;

	for(int i=0; i<adapter_depth; i++)   {
		burst_request_index[i]=0;
		burst_request_length[i]=0;
		response_index[i]=0;
		response_length[i]=0;
	}
	new_burst_transaction=true;
	new_chunk_transaction=true;

	last_of_a_burst=false;
	release_request_thread=false;
	release_response_thread=false;
	m_pending_request=0;
	m_request_number=0;
	m_pending_response=0;
	m_response_number=0;
	current_processed_request=current_processed_response=0;

	m_NumBytesPerWord = sizeof(Td);

	//SystemC processes 
	SC_THREAD(SlaveRequest);
	sensitive<<SlaveP;

	SC_THREAD(MasterRequest);
	sensitive<<send_TL2_request_event;

	SC_THREAD(SlaveResponse);

	SC_THREAD(MasterResponse);
	sensitive<<send_TL1_response_event;

	SC_METHOD(timeout_activated);
	dont_initialize();
	sensitive<<timeout_event;


}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter destructor
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::~OCP_TL1_TL2_Master_Adapter()    {
	
	delete [] m_request_buffer;
	delete [] m_response_buffer;
	delete [] m_response_data;
	delete [] m_data;
	delete [] burst_request_index;
	delete [] burst_request_length;
	delete [] response_index;
	delete [] response_length;
	if(burst_precise)
		delete [] precise_burst_length;
}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::SlaveRequest 
//get the TL1 requests, store them in the internal buffer, prepare TL2 request to be sent to the TL2 slave
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::SlaveRequest()   {
	OCPRequestGrp<Td,Ta> req;

	while(true)   {
		wait();			//wait for default event of the channel
		
        if(SlaveP->getOCPRequest(req, false))   {  //get a pending request

			//a request has arrived: cancel the previous timeout notification 
			timeout_event.cancel();
			//notify the timeout event 
			timeout_event.notify(m_timeout);

			//store the request fields in the internal buffer
			m_request_buffer[current_index].copy(req);

			#ifdef DEBUG_MASTER_ADAPTER_TL1   
				cout<<"---\nSlaveRequest(): master adapter gets a TL1 request at time "<<sc_time_stamp()<<endl;
				cout<<"MAddr: "<<m_request_buffer[current_index].MAddr<<endl;
				cout<<"MBurstSeq: "<<m_request_buffer[current_index].MBurstSeq<<endl;
				cout<<"MThreadID: "<<m_request_buffer[current_index].MThreadID<<endl;
				cout<<"MCmd: "<<m_request_buffer[current_index].MCmd<<"\n---"<<endl;
			#endif

			//store the data in the internal buffer
		    m_data[current_index]=req.MData;

			switch(define_adapter_task())   {

			case SEND_CHUNK_WITH_REQUEST:
				//specify burst length and beginning
				burst_request_index[m_request_number]=beginning_burst_index;
				burst_request_length[m_request_number]=abs(current_index-beginning_burst_index+1); //use the abs function because of circular buffer

				//change the internal index
				beginning_burst_index=current_index+1;

				if(new_burst_transaction)
					last_of_a_burst=true;
				else last_of_a_burst=false;
				
				//increment number of pending requests that the adapter has to send
				m_pending_request++;
				//increment number of the next request that is going to be stored
				//check if the request buffer is full: next request is currently being processed
				if((m_request_number+1)%adapter_depth==current_processed_request)
					release_request_thread=true;
				else {
					release_request_thread=false;
					m_request_number=(m_request_number+1)%adapter_depth;
					SlaveP->putSCmdAccept();
				}
				
				//notify request event
				send_TL2_request_event.notify();	
				break;

			case SEND_CHUNK_WITHOUT_REQUEST:
				//specify burst length and beginning
				burst_request_index[m_request_number]=beginning_burst_index;
				burst_request_length[m_request_number]=abs(current_index-beginning_burst_index); //use the abs function because of circular buffer

				//change the internal index
				beginning_burst_index=current_index;

				if(new_burst_transaction)
					last_of_a_burst=true;
				else last_of_a_burst=false;

				//increment number of pending requests that the adapter has to send
				m_pending_request++;
				//increment number of the next request that is going to be stored
				//check if the request buffer is full: next request is not currently being processed
				if((m_request_number+1)%adapter_depth==current_processed_request)
					release_request_thread=true;
				else {
					release_request_thread=false;
					m_request_number=(m_request_number+1)%adapter_depth;
					SlaveP->putSCmdAccept();
				}
				
				//notify request event
				send_TL2_request_event.notify();
				break;

			case STORE_NEW_REQUEST:
				SlaveP->putSCmdAccept();
				release_request_thread=false;
				break;

			case SEND_CHUNK_AND_SINGLE_REQUEST:
				//specify the characteristics of the current burst
				//specify burst length and beginning
				burst_request_index[m_request_number]=beginning_burst_index;
				burst_request_length[m_request_number]=abs(current_index-beginning_burst_index); //use the abs function because of circular buffer

				//change the internal index
				beginning_burst_index=current_index;
				//change the internal index
				if(current_index!=buffer_size-1)
					current_index++;
				else current_index=0;

				//increment number of pending requests that the adapter has to send
				m_pending_request++;

				//increment number of the next request that is going to be stored
				//check if the request buffer is full: next request is not currently being processed
				if((m_request_number+1)%adapter_depth==current_processed_request)    {
					release_request_thread=true;
					send_TL2_request_event.notify();
					wait(thread_released_event);  //wait for the current request to be finished to store the characteristics of the next request
				}
				else {
					release_request_thread=false;
					m_request_number=(m_request_number+1)%adapter_depth;
				}

				//continue with single request
				burst_request_index[m_request_number]=beginning_burst_index;
				burst_request_length[m_request_number]=1; //use the abs function because of circular buffer

				//change the internal index
				beginning_burst_index=current_index+1;

				//last datum of the burst is part of the burst
				last_of_a_burst=true;

				//increment number of pending requests that the adapter has to send
				m_pending_request++;

				//increment number of the next request that is going to be stored
				//check if the request buffer is full: next request is not currently being processed
				if((m_request_number+1)%adapter_depth==current_processed_request)
					release_request_thread=true;
				else {
					release_request_thread=false;
					m_request_number=(m_request_number+1)%adapter_depth;
					SlaveP->putSCmdAccept();
				}
				
				//notify request event
				send_TL2_request_event.notify();
				break;

			default:
				break;
			}		

			//change the internal index
			if(current_index!=buffer_size-1)
				current_index++;
			else current_index=0;

		}  //get a pending request
	} //end of infinite loop
}  //end of SlaveRequest


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::MasterRequest
//send TL2 request made of atomic TL1 requests
//
//---------------------------------------------------------------------------
template< class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::MasterRequest()   {
	OCPRequestGrp<Td,Ta> req;

	while(true)   {
		
		if(m_pending_request==0)
			wait();  //wait for notification of send_TL2_request_event

		//put pointer to data and chunk length in the TL2 channel
		//put request fields into the TL2 channel
		req.copy(m_request_buffer[burst_request_index[current_processed_request]]);
		req.MDataPtr=&m_data[burst_request_index[current_processed_request]];
		
		#ifdef DEBUG_MASTER_ADAPTER_TL1
			switch(req.MCmd)   {
				case OCP_MCMD_WR:
					cout<<"\n*****\nMasterRequest: send a TL2 write request at time t="<<sc_time_stamp()<<endl;
					break;
				case OCP_MCMD_RD:
					cout<<"\n*****\nMasterRequest: send a TL2 read request at time t="<<sc_time_stamp()<<endl;
					break;
				case OCP_MCMD_RDEX:
					cout<<"\n*****\nMasterRequest: send an exclusive TL2 read request at time t="<<sc_time_stamp()<<endl;
					break;
				default:
					cout<<"++++\nMasterRequest: default\n++++"<<endl;
					break;
			} //end of switch
			cout<<"Length: "<< burst_request_length[current_processed_request]<<endl;
			if(last_of_a_burst)
				cout<<"Last chunk of the burst\n*****"<<endl;
			else cout<<"Not the last chunk of the burst\n*****\n"<<endl;
		#endif
			

		//send TL2 request to slave
		MasterP->sendOCPRequestBlocking(req, burst_request_length[current_processed_request], last_of_a_burst);
		
		current_processed_request=(current_processed_request+1)%adapter_depth;

		//release the request thread, if it was not released before
		if(release_request_thread)  {
			m_request_number=(m_request_number+1)%adapter_depth;
			SlaveP->putSCmdAccept();
			release_request_thread=false;
			thread_released_event.notify();
		}
				
		if(m_pending_request!=0)
			m_pending_request--;

	} //end of infinite loop

} //end of MasterRequest method



//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::SlaveResponse
//get the TL2 response from the slave
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::SlaveResponse()   {

	//Td* temp;
	unsigned int i=0;
	OCPResponseGrp<Td> resp;
	bool last_chunk=false;

	while(true)    {
		//get TL2 response from the slave 
        if(MasterP->getOCPResponseBlocking(resp, false, response_length[m_response_number], last_chunk))   {
			
			//a response has arrived: cancel the previous timeout notification 
			timeout_event.cancel();
			//notify the timeout event 
			timeout_event.notify(m_timeout);

			#ifdef DEBUG_MASTER_ADAPTER_TL1
				cout<<"\n---Slave has sent a TL2 response at time t="<<sc_time_stamp()<<endl;
			#endif

			//copy response fields from TL2 data class to adapter
            m_response_buffer[m_response_number].copy(resp);

			//response
			#ifdef DEBUG_MASTER_ADAPTER_TL1
				cout<<"TL2 response:"<<endl;
				cout<<"SResp= "<<m_response_buffer[m_response_number].SResp<<endl;
				cout<<"SThreadID= "<<m_response_buffer[m_response_number].SThreadID<<endl;
				cout<<"Read data= ";
			#endif
			
			//copy the response data from the TL2 data class to the adapter
			if(m_response_number==0)
				response_index[m_response_number]=0;
			else response_index[m_response_number]=response_index[m_response_number-1]+response_length[m_response_number-1];

            for(i=0; i<response_length[m_response_number]; i++)   {
				m_response_data[i+response_index[m_response_number]] =resp.SDataPtr[i];
				#ifdef DEBUG_MASTER_ADAPTER_TL1
					cout<<m_response_data[i+response_index[m_response_number]] <<" ";
				#endif
				
			}
			#ifdef DEBUG_MASTER_ADAPTER_TL1
				cout<<"\n***"<<endl;
			#endif

			send_TL1_response_event.notify();

			//increment number of pending responses that the adapter has to send
			m_pending_response++;

			//increment next response number
			if((m_response_number+1)%adapter_depth==current_processed_response)
				release_response_thread=true;
			else {
				release_response_thread=false;
				m_response_number=(m_response_number+1)%adapter_depth;
				//release response channel
				MasterP->putMRespAccept(); 
			}
			
		}  //end of get Response
	} //end of infinite loop
} //end of SlaveResponse method


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::MasterResponse
//send the TL1 response to the master, made from the TL2 response from the slave 
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::MasterResponse()   {

	unsigned int i=0;
	OCPResponseGrp<Td> resp;

	while(true)    {
		if(m_pending_response==0)
			wait();  //wait for notification of send_TL1_response_event

		//copies response fields into TL1 data class
		resp.copy(m_response_buffer[current_processed_response]);
        
		//send atomic TL1 responses
		for(i=response_index[current_processed_response]; i<response_index[current_processed_response]+response_length[current_processed_response]; i++)  {
            
            resp.SData=m_response_data[i];

			#ifdef DEBUG_MASTER_ADAPTER_TL1
				cout<<"***TL1 response at time= "<<sc_time_stamp()<<endl;
				cout<<"SData= "<<m_response_data[i]<<"\n***"<<endl;
			#endif
            //send response
			SlaveP->startOCPResponse(resp);

            //wait for the release of the response channel
            SlaveP->waitMRespAccept();            
                
			wait(clk->posedge_event());  //wait for next clock edge to send next TL1 response;
		}
		#ifdef DEBUG_MASTER_ADAPTER_TL1
			cout<<"\n-----------------------------------------\nProcessing next TL1 responses at t= "<<sc_time_stamp();
			cout<<"\n-----------------------------------------"<<endl;
		#endif

		current_processed_response=(current_processed_response+1)%adapter_depth;

		//release response_thread if it not was done before
		if(release_response_thread)
			MasterP->putMRespAccept();

		if(m_pending_response!=0)
			m_pending_response--;

	} //end of infinite loop
} //end of MasterResponse method



//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::define_adapter_task
//
//---------------------------------------------------------------------------
template< class TdataCl_tl1, class TdataCl_tl2>
MasterAdapterTask OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::define_adapter_task()   {
	bool same_fields=same_request_fields(m_request_buffer[current_index], m_request_buffer[beginning_burst_index]);
	bool same_other_fields=same_other_request_fields(m_request_buffer[current_index], m_request_buffer[beginning_burst_index]);
	bool single=false;

	//completed chunk has maximum length
	if(abs(current_index-beginning_burst_index+1)==max_chunk_length)  {
		//the request fields are the same, and address increment is right		
		if( same_fields && is_address_increment_right(m_request_buffer[current_index], m_request_buffer[beginning_burst_index]) )  {
			new_chunk_transaction=true;
			
			if(max_chunk_length==1)
				single=true;
			else single=false;

			//check if the burst is finished
			if(is_last_of_burst(m_request_buffer[current_index], single)) 
				new_burst_transaction=true;
			else new_burst_transaction=false;

			return SEND_CHUNK_WITH_REQUEST;
		}
		else {
			//the fields have changed, or the address increment is wrong
			//check if there is a SINGLE request
			if(is_last_of_burst(m_request_buffer[current_index], false))  {
				new_burst_transaction=new_chunk_transaction=true;
				return SEND_CHUNK_AND_SINGLE_REQUEST;
			}
			//or not: send current burst, store new request
			else {
				new_burst_transaction=new_chunk_transaction=false;
				return SEND_CHUNK_WITHOUT_REQUEST;
			}
		}		
	}
	else  {
		//this is the beginning of a new transaction
		if(new_burst_transaction)  {
			//for precise burst, set the remaining number of requests in the burst
			if(burst_precise)
                precise_burst_length[current_index]=m_request_buffer[current_index].MBurstLength-1;
		
			//check if there is a SINGLE request
			if(is_last_of_burst(m_request_buffer[current_index], true))   {
				new_chunk_transaction=new_burst_transaction=true;
				return SEND_CHUNK_WITH_REQUEST;
			}
			//or not: store new request along with current burst
			else {
				new_chunk_transaction=new_burst_transaction=false;
				return STORE_NEW_REQUEST;
			}
		}
		else {
			//the request fields are the same, and address increment is right		
			if( same_fields && is_address_increment_right(m_request_buffer[current_index], m_request_buffer[beginning_burst_index]) )  {
				//the fields are the same:
				if(same_other_fields)  {
					//check if this is the last of the burst
					if(is_last_of_burst(m_request_buffer[current_index], false))   {
						new_burst_transaction=new_chunk_transaction=true;
						return SEND_CHUNK_WITH_REQUEST;
					}
					//or not: store new request along with current burst
					else {
						new_burst_transaction=new_chunk_transaction=false;
						return STORE_NEW_REQUEST;	
					}
				}	
				//the fields have changed, new chunk:
				else   {
					new_chunk_transaction=true;
					//check if this is the last of the burst: sends the previous chunk with the chunk composed of one request 
					if(is_last_of_burst(m_request_buffer[current_index], false))   {
						new_burst_transaction=true;
						return SEND_CHUNK_AND_SINGLE_REQUEST;
					}
					//or not: store new request along with current burst
					else {
						new_burst_transaction=false;
						return SEND_CHUNK_WITHOUT_REQUEST;	
					}
				}
			}
			else {
				//the fields have changed, or the address increment is wrong : current request is the beginning of a new burst
				//this is the beginning of a new burst: for precise burst, get the new length of the burst
				if(burst_precise)
                    precise_burst_length[current_index]=precise_burst_length[current_index-1]-1;

				//check if there is a SINGLE request
				if(is_last_of_burst(m_request_buffer[current_index], false))  	{
					new_burst_transaction=new_chunk_transaction=true;
					return SEND_CHUNK_AND_SINGLE_REQUEST;
				}
				//or not: send current burst, store new request
				else {
					new_burst_transaction=new_chunk_transaction=false;
					return SEND_CHUNK_WITHOUT_REQUEST;	
				}
			}
		}
	}
}


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::same_request_fields
//
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
bool OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::same_request_fields(OCPRequestGrp<Td, Ta>& new_req_param, OCPRequestGrp<Td, Ta>& current_req_param )  {

	//check if the fields have changed 
	//some fields can change during the burst: MAddr, MByteEn, MDataInfo, MReqLast
	//Multiple threads are not supported (hence MThreadID variation ends simulation)
	
	if(new_req_param.MThreadID != current_req_param.MThreadID) {
		cerr<<"\n****************************\nError:\n";
		cerr<<"Multiple threads are not supported\n****************************"<<endl;
		sc_stop();
	}
	else if(new_req_param.MCmd != current_req_param.MCmd) 
		return false;
	else if(new_req_param.MAddrSpace != current_req_param.MAddrSpace)
		return false;
	else if(new_req_param.MConnID != current_req_param.MConnID)
		return false;
	else if(new_req_param.MReqInfo != current_req_param.MReqInfo)
		return false;
	else if(new_req_param.MAtomicLength != current_req_param.MAtomicLength)
		return false;
	//tests the other fields relative to burst support, if necessary 
	else if(burst_support) {
		if(new_req_param.MBurstPrecise != current_req_param.MBurstPrecise)
			return false;
		else if(new_req_param.MBurstSeq != current_req_param.MBurstSeq)
			return false;
		else if(burst_precise)  {
			if(new_req_param.MBurstLength != current_req_param.MBurstLength)
				return false;
		}
	}
		
	//fields have not changed:
	return true;
}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::same_other_request_fields
//fields MByteEn or MDataInfo have changed: a new chunk begins
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
bool OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::same_other_request_fields(OCPRequestGrp<Td, Ta>& new_req_param, OCPRequestGrp<Td, Ta>& current_req_param )  {
	if(new_req_param.MByteEn != current_req_param.MByteEn) 
		return false;
	else if(new_req_param.MDataInfo != current_req_param.MDataInfo)
		return false;

	//fields have not changed:
	return true;
}


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::is_address_increment_right
//
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
bool OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::is_address_increment_right(OCPRequestGrp<Td, Ta>& new_req_param, OCPRequestGrp<Td, Ta>& current_req_param )  {

	if(burst_support)  {
		//if burst is a streaming burst, verify if the address has not changed
		if(new_req_param.MBurstSeq==OCP_MBURSTSEQ_STRM)  {	
			if( (current_req_param.MBurstSeq==OCP_MBURSTSEQ_STRM) && (new_req_param.MAddr ==current_req_param.MAddr))
				return true;
			else return false;
		}
		//if burst is a incremental burst, verify if the address increment has changed
		else if(new_req_param.MBurstSeq==OCP_MBURSTSEQ_INCR) {
			if(new_req_param.MAddr == current_req_param.MAddr+(current_index-beginning_burst_index)*m_NumBytesPerWord)
				return true;
			else return false;
		}
	}
	else  {
		//no burst support: verify if the address increment is right
		if(new_req_param.MAddr == current_req_param.MAddr+(current_index-beginning_burst_index)*m_NumBytesPerWord)
				return true;
			else return false;
	}


	//in the other cases (custom bursts), send as single TL2 requests 
	return false;
}


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::reset_request_parameters
//reset request parameters
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::reset_request_parameters(int i)   {
	//calls the reset function of OCPRequestGrp class
	m_request_buffer[i].reset(false);
}


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::is_last_of_burst
//indicates if it is the last TL1 request of the burst
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
bool OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::is_last_of_burst(OCPRequestGrp<Td, Ta>& new_req_param, bool new_burst)   {
	//no burst support: test if MReqLast is true 
	if(!burst_support)   {
		if(new_req_param.MReqLast==true)
			return true;
		else return false;
	}
	else {
		//imprecise burst
		if(!burst_precise)   {
			//check if MBurstLength=1, which indicates the end of the burst
			if(new_req_param.MBurstLength==1)  
				return true;
			else return false;
		}
		//precise burst: check if burst is finished 
		else {  
			if(!new_burst)  {
				if(current_index>0)
					precise_burst_length[current_index]=precise_burst_length[current_index-1]-1;
				else precise_burst_length[0]=precise_burst_length[buffer_size-1]-1;	
				
				if(precise_burst_length[current_index]==0)  
					return true;		
				else return false;
			}
			else {
				if(m_request_buffer[current_index].MBurstLength==1)
					return true;
				else return false;
			}
        }
	}	
}


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::timeout_activated 
//no request or response has arrived during the m_timeout time
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::timeout_activated()   {
	cout<<"\n\n***************************\nWarning (t="<<sc_time_stamp()<<") :\n";
	cout<<"Timeout for TL1-TL2 Master adapter \""<<name()<<"\"\n***************************\n"<<endl;
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_TL2_Master_Adapter::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------

template<class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::end_of_elaboration()
{

  sc_module::end_of_elaboration();

  // Initialize debug interface
  MasterP->MregisterDirectIF((MdirectIF<TdataCl_tl2>*)(this));
  SlaveP->SregisterDirectIF((SdirectIF<TdataCl_tl1>*)(this));


  // Get data structure
  m_DataCl_tl1 = SlaveP->GetDataCl();
  m_DataCl_tl2 = MasterP->GetDataCl();


  // Get communication structure
  m_CommCl = SlaveP->GetCommCl();


  // Get system parameter structure
  m_ParamCl = SlaveP->GetParamCl();

  //multiple threads are not supported
  if(m_ParamCl->threads>1)  {
	  cerr<<"\n****************************\nError:\n";
	  cerr<<"Multiple threads are not supported\n****************************"<<endl;
	  sc_stop();
  }

  //check for burst support
  if((m_ParamCl->burstseq)&&(m_ParamCl->burstlength))   {
	  burst_support=true;
	  //check if bursts are precise
	  if(m_ParamCl->burstprecise)
		  burst_precise=true;
	  else burst_precise=false;
  }
  else burst_support=false;

  if(burst_precise)  {
	precise_burst_length=new unsigned int[buffer_size];
	for(int i=0; i<buffer_size; i++)   
		precise_burst_length[i]=0;

  }

  //check for threadbusy compliances
  if((m_ParamCl->mthreadbusy_exact) || (m_ParamCl->sthreadbusy_exact))  {
	  cerr<<"\n****************************\nError:\n";
	  cerr<<"Threadbusy compliance is not supported\n****************************"<<endl;
	  sc_stop();
  }

  //check for single request/multiple data
  if((m_ParamCl->burstsinglereq))  {
	  cerr<<"\n****************************\nError:\n";
	  cerr<<"Single Request/Multiple Data compliance is not supported\n****************************"<<endl;
	  sc_stop();
  }

  //check for datahandshake 
  if((m_ParamCl->datahandshake))  {
	  cerr<<"\n****************************\nError:\n";
	  cerr<<"Single Request/Multiple Data compliance is not supported\n****************************"<<endl;
	  sc_stop();
  }

  //check for cmdaccept 
  if(!m_ParamCl->cmdaccept)  {
	  cerr<<"\n****************************\nError:\n";
	  cerr<<"cmdaccept parameters must be set\n****************************"<<endl;
	  sc_stop();
  }

  //check for srespaccept 
  if(!m_ParamCl->respaccept)  {
	  cerr<<"\n****************************\nError:\n";
	  cerr<<"respaccept parameters must be set\n****************************"<<endl;
	  sc_stop();
  }

  // Put parameter into Channel
  // No parameters needed

}

// ----------------------------------------------------------------------------
//  Method : OCP_TL1_TL2_Master_Adapter::MPutDirect()
//  Debug interface method.
//  Read/Write data, Master to Slave direction
//  Returns true for success and false for failure
//
//	calls Slave's MputDirect function
//
// ----------------------------------------------------------------------------

template<class TdataCl_tl1, class TdataCl_tl2>
bool OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::MputDirect(
    int MasterID, bool IsWrite, Td *Data, Ta Address, int NumWords)

{
  return(SlaveP->MputDirect(MasterID, IsWrite, Data, Address, NumWords));

}


//----------------------------------------------------------------------------
//  Method : OCP_TL1_TL2_Master_Adapter::SPutDirect()
//  Debug interface method.
//  Read/Write data, Slave to Master direction
//
//	calls Master's SputDirect function
//
// ----------------------------------------------------------------------------

template<class TdataCl_tl1, class TdataCl_tl2>
bool OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::SputDirect(
         int SlaveID, bool IsWrite, Td *Data, Ta Address, int NumWords)
{
	return(MasterP->SputDirect(SlaveID, IsWrite, Data, Address, NumWords));	
}


// ----------------------------------------------------------------------------
//
//  Instantiation of the adapter
//
// ----------------------------------------------------------------------------

template class OCP_TL1_TL2_Master_Adapter<OCP_TL1_DataCl<int , int>, OCP_TL2_DataCl<int , int> >; 

