// 
//  (c) Copyright OCP-IP 2003, 2004
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Alan Kamas, for Sonics Inc.
//                Anssi Haverinen, Nokia Inc.
//                Joe Chou, Sonics Inc.
//                
//         Date: 07/25/2004
//
//  Description : OCP Transaction Level Channel
//                - generate an OCP monitor file
//
//  Change History
//             12/14/2003 Original revision for OCP 2.0
//             07/25/2004 Minor bug fixes for printouts
// ============================================================================


#ifndef _OCP_TL1_OCPMONGEN_CL_h
#define _OCP_TL1_OCPMONGEN_CL_h

#include <iostream>
#include <iomanip>
#include <string>
#include <bitset>
#include "ocp_globals.h"
#include "ocp_tl_param_cl.h"

template <typename TdataCl>
class OCP_TL1_OCPMonGenCl {
  public:

    typedef typename TdataCl::DataType Td;
    typedef typename TdataCl::AddrType Ta;

    // Constructor
    OCP_TL1_OCPMonGenCl(string my_ocpmon_prefix, string file_name) 
    : monFile(file_name.c_str()), ocpmon_prefix(my_ocpmon_prefix)
    {
        // Did the initialization work?
        // Is it open for writing?
        if (!monFile) {
            cout << "Error: OCP Monitor for OCP named " << ocpmon_prefix << " - cannot open OCP monitor file: " << file_name
                 << endl;
            // give up
            exit(1100);
        }

        // clean up the prefix (in case there are leading and trailing '_' chars)
        if ((my_ocpmon_prefix[0]=='_') && (my_ocpmon_prefix[my_ocpmon_prefix.length()-1]=='_')) { 
            ocpmon_prefix = my_ocpmon_prefix.substr(1,my_ocpmon_prefix.length()-2);
        }
    }

    // Destructor 
    ~OCP_TL1_OCPMonGenCl() 
    {
        // close out the file
        monFile.close();
    }

    void
    monitorHeader( ParamCl<TdataCl> *mParamCl )
    {
        // Send out the header information
        monFile << "# ocp20version=1.0" << std::endl;
        monFile << "# name=" << ocpmon_prefix << "_ocpmon" << std::endl;

        monFile << "# broadcast_enable=" << mParamCl->broadcast_enable << std::endl;
        monFile << "# burst_aligned=" << mParamCl->burst_aligned << std::endl;
        monFile << "# burstseq_dflt1_enable=" << mParamCl->burstseq_dflt1_enable << std::endl;
        monFile << "# burstseq_dflt2_enable=" << mParamCl->burstseq_dflt2_enable << std::endl;
        monFile << "# burstseq_incr_enable=" << mParamCl->burstseq_incr_enable << std::endl;
        monFile << "# burstseq_strm_enable=" << mParamCl->burstseq_strm_enable << std::endl;
        monFile << "# burstseq_unkn_enable=" << mParamCl->burstseq_unkn_enable << std::endl;
        monFile << "# burstseq_wrap_enable=" << mParamCl->burstseq_wrap_enable << std::endl;
        monFile << "# burstseq_xor_enable=" << mParamCl->burstseq_xor_enable << std::endl;
        monFile << "# endian=" << mParamCl->endian << std::endl;
        monFile << "# force_aligned=" << mParamCl->force_aligned << std::endl;
        monFile << "# mthreadbusy_exact=" << mParamCl->mthreadbusy_exact << std::endl;
        monFile << "# rdlwrc_enable=" << mParamCl->rdlwrc_enable << std::endl;
        monFile << "# read_enable=" << mParamCl->read_enable << std::endl;
        monFile << "# readex_enable=" << mParamCl->readex_enable << std::endl;
        monFile << "# sdatathreadbusy_exact=" << mParamCl->sdatathreadbusy_exact << std::endl;
        monFile << "# sthreadbusy_exact=" << mParamCl->sthreadbusy_exact << std::endl;
        monFile << "# write_enable=" << mParamCl->write_enable << std::endl;
        monFile << "# writenonpost_enable=" << mParamCl->writenonpost_enable << std::endl;
        monFile << "# datahandshake=" << mParamCl->datahandshake << std::endl;
        monFile << "# reqdata_together=" << mParamCl->reqdata_together << std::endl;
        monFile << "# writeresp_enable=" << mParamCl->writeresp_enable << std::endl;
        monFile << "# addr=" << mParamCl->addr << std::endl;
        monFile << "# addr_wdth=" << mParamCl->addr_wdth << std::endl;
        monFile << "# addrspace=" << mParamCl->addrspace << std::endl;
        monFile << "# addrspace_wdth=" << mParamCl->addrspace_wdth << std::endl;
        monFile << "# atomiclength=" << mParamCl->atomiclength << std::endl;
        monFile << "# atomiclength_wdth=" << mParamCl->atomiclength_wdth << std::endl;
        monFile << "# burstlength=" << mParamCl->burstlength << std::endl;
        monFile << "# burstlength_wdth=" << mParamCl->burstlength_wdth << std::endl;
        monFile << "# burstprecise=" << mParamCl->burstprecise << std::endl;
        monFile << "# burstseq=" << mParamCl->burstseq << std::endl;
        monFile << "# burstsinglereq=" << mParamCl->burstsinglereq << std::endl;
        monFile << "# byteen=" << mParamCl->byteen << std::endl;
        monFile << "# cmdaccept=" << mParamCl->cmdaccept << std::endl;
        monFile << "# connid=" << mParamCl->connid << std::endl;
        monFile << "# connid_wdth=" << mParamCl->connid_wdth << std::endl;
        monFile << "# dataaccept=" << mParamCl->dataaccept << std::endl;
        monFile << "# datalast=" << mParamCl->datalast << std::endl;
        monFile << "# data_wdth=" << mParamCl->data_wdth << std::endl;
        monFile << "# mdata=" << mParamCl->mdata << std::endl;
        monFile << "# mdatabyteen=" << mParamCl->mdatabyteen << std::endl;
        monFile << "# mdatainfo=" << mParamCl->mdatainfo << std::endl;
        monFile << "# mdatainfo_wdth=" << mParamCl->mdatainfo_wdth << std::endl;
        monFile << "# mdatainfobyte_wdth=" << mParamCl->mdatainfobyte_wdth << std::endl;
        monFile << "# mthreadbusy=" << mParamCl->mthreadbusy << std::endl;
        monFile << "# reqinfo=" << mParamCl->reqinfo << std::endl;
        monFile << "# reqinfo_wdth=" << mParamCl->reqinfo_wdth << std::endl;
        monFile << "# reqlast=" << mParamCl->reqlast << std::endl;
        monFile << "# resp=" << mParamCl->resp << std::endl;
        monFile << "# respaccept=" << mParamCl->respaccept << std::endl;
        monFile << "# respinfo=" << mParamCl->respinfo << std::endl;
        monFile << "# respinfo_wdth=" << mParamCl->respinfo_wdth << std::endl;
        monFile << "# resplast=" << mParamCl->resplast << std::endl;
        monFile << "# sdata=" << mParamCl->sdata << std::endl;
        monFile << "# sdatainfo=" << mParamCl->sdatainfo << std::endl;
        monFile << "# sdatainfo_wdth=" << mParamCl->sdatainfo_wdth << std::endl;
        monFile << "# sdatainfobyte_wdth=" << mParamCl->sdatainfobyte_wdth << std::endl;
        monFile << "# sdatathreadbusy=" << mParamCl->sdatathreadbusy << std::endl;
        monFile << "# sthreadbusy=" << mParamCl->sthreadbusy << std::endl;
        monFile << "# threads=" << mParamCl->threads << std::endl;
        monFile << "# control=" << mParamCl->control << std::endl;
        monFile << "# controlbusy=" << mParamCl->controlbusy << std::endl;
        monFile << "# control_wdth=" << mParamCl->control_wdth << std::endl;
        monFile << "# controlwr=" << mParamCl->controlwr << std::endl;
        monFile << "# interrupt=" << mParamCl->interrupt << std::endl;
        monFile << "# merror=" << mParamCl->merror << std::endl;
        monFile << "# mflag=" << mParamCl->mflag << std::endl;
        monFile << "# mflag_wdth=" << mParamCl->mflag_wdth << std::endl;
        monFile << "# mreset=" << mParamCl->mreset << std::endl;
        monFile << "# serror=" << mParamCl->serror << std::endl;
        monFile << "# sflag=" << mParamCl->sflag << std::endl;
        monFile << "# sflag_wdth=" << mParamCl->sflag_wdth << std::endl;
        monFile << "# sreset=" << mParamCl->sreset << std::endl;
        monFile << "# status=" << mParamCl->status << std::endl;
        monFile << "# statusbusy=" << mParamCl->statusbusy << std::endl;
        monFile << "# statusrd=" << mParamCl->statusrd << std::endl;
        monFile << "# status_wdth=" << mParamCl->status_wdth << std::endl;
        monFile << "##" << std::endl;
    }

    // A helper function to print out monitor dump
    void
    printBinIf( bool my_condition, unsigned int my_value, int my_bit_width = 1 ) 
    {
        monFile << " " << std::setw(1);
        if (my_condition) {
            // print out binary with MSB on left and bit zero on right
            unsigned int my_one = 1 << (my_bit_width-1);
            for (int i=0;i<my_bit_width;i++) {
                if (my_value & my_one) {
                    monFile << '1';
                } else {
                    monFile << '0';
                }
                my_one = my_one>>1;
            }
        } else {
            // condition not met. Print "don't cares" 
            for (int i=0;i<my_bit_width;i++) {
                monFile << "x";
            }
        }
    }

    // A helper function to print out monitor dump
    void
    printIf( bool my_condition, int my_value, int my_bit_width = 4 ) 
    {
        int my_hex_size = 1 + (my_bit_width-1)/4; 
        if (my_condition) {
            monFile << " " << std::setw(my_hex_size) << std::setfill('0') << my_value; 
        } else {
            // condition not met. Print "don't cares" 
            monFile << " ";
            for (int i=0;i<my_hex_size;i++) {
                monFile << "x";
            }
        }
    }

    // Prints out one line of ocp state information in ocpMon format
    //  - should be called every ocp clock tick
    //
    void
    monitorOut( ParamCl<TdataCl> *mParamCl,
            double dtime, 
            const OCPRequestGrp<Td,Ta>& curRequest,
            const bool& commandAccepted,
            const OCPDataHSGrp<Td>& curDataHS, 
            const bool& dataAccepted,
            const OCPResponseGrp<Td>& curResponse,
            const bool& responseAccepted,
            const unsigned int& mySThreadBusy,
            const unsigned int& mySDataThreadBusy,
            const unsigned int& myMThreadBusy,
            const bool& myMReset_n,
            const bool& mySReset_n,
            const bool& myMError,
            const unsigned int& myMFlag,
            const bool& mySError,
            const unsigned int& mySFlag,
            const bool& mySInterrupt,
            const unsigned int& myControl,
            const bool& myControlWr,
            const bool& myControlBusy,
            const unsigned int& myStatus,
            const bool& myStatusRd,
            const bool& myStatusBusy)
    {
      if (dtime>0) {
        bool requestInProgress = (curRequest.MCmd != OCP_MCMD_IDLE);
        bool writerequestInProgress = (curRequest.MCmd == OCP_MCMD_WR);
        bool dataInProgress = curDataHS.MDataValid;
        bool responseInProgress = (curResponse.SResp != OCP_SRESP_NULL);

        // Field Description : Required OCP Parameter : Width(bits) : Format

        // Simulation Time	None		float in (ns)
        // the time is decimal
        monFile << dec;

        // Clk
        monFile << std::setw(12) << std::setfill(' ') << dtime;
        monFile << " ";

        // most other values are in hex
        monFile << hex;

        // MReset_n	mreset parameter is 1	Always 1	hexadecimal
        if (mParamCl->mreset) {
            monFile << " " << std::setw(1) << myMReset_n; 
        }

        // SReset_n	sreset parameter is 1	Always 1	hexadecimal
        if (mParamCl->sreset) {
            monFile << " " << std::setw(1) << mySReset_n; 
        }

        // MCmd	None	Always 3	hexadecimal
        if (requestInProgress) {
            monFile << " " << curRequest.MCmd;
        } else {
            // No request in progress so this value must be Idle 
            monFile << " " << OCP_MCMD_IDLE;
        }

        // MAddr	addr is 1	addr_wdth	hexadecimal
        if (mParamCl->addr) {
	  int addr_size = 1 + ((mParamCl->addr_wdth)-1)/4; 
	  if (requestInProgress) {
            monFile << " " << std::setw(addr_size) << std::setfill('0')
		    << curRequest.MAddr;
	  } else {
            // No request in progress so this value is not defined right now 
            monFile << " ";
            for (int i=0;i<addr_size;i++) {
	      monFile << "x";
            }
	  }
	}
        // MAddrSpace	addrspace is 1	addrspace_wdth	hexadecimal
        if (mParamCl->addrspace) {
            printIf(requestInProgress, curRequest.MAddrSpace, mParamCl->addrspace_wdth);
        }

        // MByteEn	byteen	data_wdth / 8	hexadecimal
        if (mParamCl->byteen) {
            printIf(requestInProgress, curRequest.MByteEn, (mParamCl->data_wdth)/8 );
        }

        // MConnID	connid is 1	connid_wdth	hexadecimal
        if (mParamCl->connid) {
            printIf(requestInProgress, curRequest.MConnID, mParamCl->connid_wdth);
        }

        // MReqInfo	reqinfo is 1	reqinfo_wdth	hexadecimal
        if (mParamCl->reqinfo) {
            printIf(requestInProgress, curRequest.MReqInfo, mParamCl->reqinfo_wdth);
        }

        // NOTE: A hack to compute thread width
        int my_thread_wdth = 4;  // one hex number
        if (mParamCl->threads > 256 ) {
            my_thread_wdth = 12;  // three hex numbers
        } else if (mParamCl->threads > 16) {
            my_thread_wdth = 8;  // two hex numbers
        }

        // MThreadID	threads > 1	threadid_wdth	hexadecimal
        if (mParamCl->threads > 1) {
            printIf(requestInProgress, curRequest.MThreadID, my_thread_wdth);
        }

        // MAtomicLength	atomiclength is 1	atomiclength_wdth	hexadecimal
        if (mParamCl->atomiclength) {
            printIf(requestInProgress, curRequest.MAtomicLength, mParamCl->atomiclength_wdth);
        }

        // MBurstLength	burstlength is 1	burstlength_wdth	hexadecimal
        if (mParamCl->burstlength) {
            printIf(requestInProgress, curRequest.MBurstLength, mParamCl->burstlength_wdth);
        }

        // MBurstPrecise	burstprecise is 1	Always 1	hexadecimal
        if (mParamCl->burstprecise) {
            printIf(requestInProgress, curRequest.MBurstPrecise, 1);
        }

        // MBurstSeq	burstseq is 1	Always 3	hexadecimal
        if (mParamCl->burstseq) {
            printIf(requestInProgress, curRequest.MBurstSeq, 3);
        }

        // MBurstSingleReq	burstsinglereq is 1	Always 1	hexadecimal
        if (mParamCl->burstsinglereq) {
            printIf(requestInProgress, curRequest.MBurstSingleReq, 1);
        }

        // MReqLast	reqlast is 1	Always 1	hexadecimal
        if (mParamCl->reqlast) {
            printIf(requestInProgress, curRequest.MReqLast, 1);
        }

        // SCmdAccept	cmdaccept is 1	Always 1	hexadecimal
        if (mParamCl->cmdaccept) {
	    monFile << " " << commandAccepted;
        }

        // SThreadBusy	sthreadbusy is 1	threads	hexadecimal
        if (mParamCl->sthreadbusy) {
            printIf(true , mySThreadBusy, mParamCl->threads);
        }

        // MData	mdata is 1	data_wdth	hexadecimal
        if (mParamCl->mdata) {
            int my_data_size = 1 + ((mParamCl->data_wdth)-1)/4; 
            if ( mParamCl->datahandshake && dataInProgress ) {
                monFile << " " << std::setw(my_data_size) << std::setfill('0') << curDataHS.MData;
            } else if (!(mParamCl->datahandshake) && writerequestInProgress) {
                monFile << " " << std::setw(my_data_size) << std::setfill('0') << curRequest.MData;
            } else {
                // no data - just don't cares
                monFile << " ";
                for (int i=0;i<my_data_size;i++) {
                    monFile << "x";
                }
            }
        }

        // MDataInfo	mdatainfo is 1	mdatainfo_wdth	hexadecimal
        if (mParamCl->datahandshake && mParamCl->mdatainfo) {
            printIf(dataInProgress, curDataHS.MDataInfo, mParamCl->mdatainfo_wdth);
        }

        // MDataValid	datahandshake is 1	Always 1	hexadecimal
        if (mParamCl->datahandshake) {
            monFile << " " << dataInProgress;
        }

        // MDataByteEn	mdatabyteen is 1	data_wdth / 8	hexadecimal
        if ( mParamCl->datahandshake && mParamCl->mdatabyteen) {
            printIf(dataInProgress, curDataHS.MDataByteEn, mParamCl->data_wdth / 8);
        }

        // MDataThreadID	threads > 1 and datahandshake is 1	threadid_wdth	hexadecimal
        if ( (mParamCl->datahandshake) && (mParamCl->threads > 1)) {
            printIf(dataInProgress, curDataHS.MDataThreadID, my_thread_wdth );
        }

        // MDataLast	datalast is 1	Always 1	hexadecimal
        if ( mParamCl->datahandshake && mParamCl->datalast) {
            printIf(dataInProgress, curDataHS.MDataLast, 1);
        }

        // SDataAccept	dataaccept is 1	Always 1	hexadecimal
        if ((mParamCl->datahandshake) && (mParamCl->dataaccept) ) {
            monFile << " " << dataAccepted;
        }


        // SDataThreadBusy	sdatathreadbusy is 1	threads	hexadecimal
        if ( (mParamCl->datahandshake) && (mParamCl->sdatathreadbusy) ) {
            printIf(true , mySDataThreadBusy, mParamCl->threads);
        }

        // SResp	resp is 1	Always 2	hexadecimal
        if (mParamCl->resp) {
            if (responseInProgress) {
                monFile << " " << curResponse.SResp;
            } else {
                // No request in progress so this value must be Idle 
                monFile << " " << OCP_SRESP_NULL;
            }
        }

        // SRespInfo	respinfo is 1	respinfo_wdth	hexadecimal
        if ((mParamCl->resp) && (mParamCl->respinfo)) {
            printIf(responseInProgress , curResponse.SRespInfo, mParamCl->respinfo_wdth);
        }

        // SThreadID	threads > 1 and resp is 1	threadid_wdth	hexadecimal
        if ((mParamCl->resp) && (mParamCl->threads > 1) ) {
            printIf(responseInProgress, curResponse.SThreadID, my_thread_wdth);
        }

        // SData	sdata is 1	data_wdth	hexadecimal
        if ((mParamCl->sdata) && (mParamCl->resp) ) {
            int my_data_size = 1 + ((mParamCl->data_wdth)-1)/4; 
            if (responseInProgress) {
                monFile << " " << std::setw(my_data_size) << std::setfill('0') << curResponse.SData;
            } else {
                // no data - just don't cares
                monFile << " ";
                for (int i=0;i<my_data_size;i++) {
                    monFile << "x";
                }
            }
        }

        // SDataInfo	sdatainfo is 1	sdatainfo_wdth	hexadecimal
        if ((mParamCl->sdatainfo) && (mParamCl->resp) ) {
            printIf(responseInProgress , curResponse.SDataInfo, mParamCl->sdatainfo_wdth);
        }

        // SRespLast	resplast is 1	Always 1	hexadecimal
        if ( (mParamCl->resp) && (mParamCl->resplast)) {
            printIf(responseInProgress , curResponse.SRespLast, 1);
        }

        // MRespAccept	respaccept is 1	Always 1	hexadecimal
        if (mParamCl->respaccept) {
            monFile << " " << responseAccepted;
        }

        // MThreadBusy	mthreadbusy is 1	threads	hexadecimal
        if (mParamCl->mthreadbusy) {
            printIf(true , myMThreadBusy, mParamCl->threads);
        }

        // MFlag	mflag is 1	mflag_wdth	binary
        if (mParamCl->mflag) {
            printIf(true, myMFlag, mParamCl->mflag_wdth);
        }

        // MError	merror is 1	Always 1	binary
        if (mParamCl->merror) {
            printBinIf(true, myMError, 1);
        }

        // SFlag	sflag is 1	sflag_wdth	binary
        if (mParamCl->sflag) {
            printIf(true, mySFlag, mParamCl->sflag_wdth);
        }

        // Serror	serror is 1	Always 1	binary
        if (mParamCl->serror) {
            printBinIf(true, mySError, 1);
        }

        // SInterrupt	interrupt is 1	Always 1	binary
        if (mParamCl->interrupt) {
            printBinIf(true, mySInterrupt, 1);
        }

        // Control	control is 1	control_wdth	hexadecimal
        if (mParamCl->control) {
            printIf(true , myControl, mParamCl->control_wdth);
        }

        // ControlWr	controlwr is 1	Always 1	binary
        if (mParamCl->controlwr) {
            printBinIf(true, myControlWr, 1);
        }

        // ControlBusy	controlbusy is 1	Always 1	binary
        if (mParamCl->controlbusy) {
            printBinIf(true, myControlBusy, 1);
        }

        // Status	status is 1	status_wdth	hexadecimal
        if (mParamCl->status) {
            printIf(true , myStatus, mParamCl->status_wdth);
        }

        // StatusRd	statusrd is 1	Always 1	binary
        if (mParamCl->statusrd) {
            printBinIf(true, myStatusRd, 1);
        }

        // StatusBusy	statusbusy is 1	Always 1	binary
        if (mParamCl->statusbusy) {
            printBinIf(true, myStatusBusy, 1);
        }

        // finish off the line 
        monFile << endl;
      }
    }

    protected:
        ofstream monFile;       // starts out with no file.
        string ocpmon_prefix;   // name of the OCP channel
  };

#endif // _OCP_TL1_OCPMONGEN_CL_h
