/*
  James Aldis
  TI France

  OCP TL1 Channel Timing Distribution
  29 June 2005

  C++ Header File for:
    multi-threaded thread-busy-exact test slave.  can produce
    responses combinatorially
*/


// multiple inclusion protection
#ifndef TL1_TIMING_SLAVE_H
#define TL1_TIMING_SLAVE_H


#include "tl1_timing_common.h"

class tl1_timing_slave : 
  public sc_module,
  public OCP_TL1_Master_TimingIF
{
  typedef OCP_RESP* OCP_RESP_PTR;
  SC_HAS_PROCESS(tl1_timing_slave);
  public:
    tl1_timing_slave(sc_module_name name, int nr_threads) :
      sc_module(name), ocp("ocp") {

      threads = nr_threads;

      // array of FIFOs, one per thread, for responses
      rd_ptr = new int[threads];
      wr_ptr = new int[threads];
      full = new bool[threads];
      responses_waiting = new OCP_RESP_PTR[threads];
      for(int i=0; i<threads; i++) {
        rd_ptr[i] = 0;
        wr_ptr[i] = 0;
        full[i] = false;
        responses_waiting[i] = new OCP_RESP[max_responses_waiting];
      }

      requests_got = 0;
      responses_sent = 0;

      SC_THREAD(main_thread);
      sensitive(clk.pos());
      dont_initialize();

      time_quantum = sc_get_time_resolution();
      request_sample_time = time_quantum;  // initial guess
      mthreadbusy_sample_time = time_quantum;  // initial guess
    };

    ~tl1_timing_slave() {
      cout << "Deleting slave:   " << name() << endl;
      cout << "  Total requests got:   " << requests_got << endl;
      cout << "  Total responses sent: " << responses_sent << endl;
      cout << "  Per thread outstanding:  ";
      for(int i=0; i<threads; i++) {
        cout << " " << (full[i] ? max_responses_waiting :
          (wr_ptr[i] - rd_ptr[i] + max_responses_waiting) % max_responses_waiting);
      }
      cout << endl;
      cout << "  MThreadBusy sample time:  " << mthreadbusy_sample_time << endl;
      cout << "  Request sample time:       " << request_sample_time << endl;
    }

    // ports
    sc_in<bool> clk;
    OCP_SLAVE_PORT ocp;

    // processes
    void main_thread() {

      while(true) {
        wait();  // clock rising edge

        // generate sthreadbusy
        int mask = 1;
        int stb_out = 0;
        for(int i=0; i<threads; i++) {
          if(full[i]) {
            stb_out |= mask;
          }
          mask <<= 1;
        }
        ocp->putSThreadBusy(stb_out);

        // wait for requests and capture them
        wait(request_sample_time);
        OCP_REQ tmp;
        if(ocp->getOCPRequest(tmp)) {
          requests_got++;
          int t = tmp.MThreadID;
          OCP_RESP *ptr = &((responses_waiting[t])[wr_ptr[t]]);
          ptr->SResp = OCP_SRESP_DVA;
          ptr->SThreadID = t;
          ptr->SData = tmp.MAddr;
          wr_ptr[t] = (wr_ptr[t] + 1) % max_responses_waiting;
          if(wr_ptr[t] == rd_ptr[t]) {
            full[t] = true;
          }
        }

        // wait for mthreadbusy if necessary
        if(request_sample_time < mthreadbusy_sample_time) {
          wait(mthreadbusy_sample_time - request_sample_time);
        }
        int mtb = ocp->getMThreadBusy();

        // generate responses
        if(unit_rand() < pr_response) {
          // priority goes to lowest thread
          for(int i=0; i<threads; i++) {
            if((full[i] || (wr_ptr[i] != rd_ptr[i])) && !(mtb & 1)) {
              responses_sent++;
              ocp->startOCPResponse((responses_waiting[i])[rd_ptr[i]]);
              rd_ptr[i] = (rd_ptr[i] + 1) % max_responses_waiting;
              full[i] = false;
              break;
            }
            mtb >>= 1;
          }
        }
      }
    }

    // module is both timing-sensitive and non-default-timing, because
    // the response output time depends on the mthreadbusy and request
    // input times
    void end_of_elaboration() {
      cout << "<<<< E-O-E >>>> " << name() << endl;

      // non-default timing
      OCP_TL1_Slave_TimingCl mytiming;
      mytiming.ResponseGrpStartTime = sc_max(request_sample_time,
        mthreadbusy_sample_time);
      ocp->setOCPTL1SlaveTiming(mytiming);

      // timing-sensitive
      ocp->registerTimingSensitiveOCPTL1Slave(this);
    }

    // callback from the channel to indicate master timing
    // need to capture mthreadbusy and response sample times,
    // and inform channel if this changes the response timing
    void setOCPTL1MasterTiming(OCP_TL1_Master_TimingCl master_timing) {
      cout << "  << S-M-T >>   " << name() << endl;

      sc_time current_resp_out_t = sc_max(request_sample_time,
        mthreadbusy_sample_time);

      if(master_timing.MThreadBusyStartTime + time_quantum >
            mthreadbusy_sample_time) {
        mthreadbusy_sample_time =
            master_timing.MThreadBusyStartTime + time_quantum;
      }

      if(master_timing.RequestGrpStartTime + time_quantum >
            request_sample_time) {
        request_sample_time =
            master_timing.RequestGrpStartTime + time_quantum;
      }

      sc_time new_resp_out_t = sc_max(request_sample_time,
        mthreadbusy_sample_time);

      if(new_resp_out_t > current_resp_out_t) {
        OCP_TL1_Slave_TimingCl sto;
        sto.ResponseGrpStartTime = new_resp_out_t;
        ocp->setOCPTL1SlaveTiming(sto);
      }
    }

  private:
    sc_time time_quantum;
    sc_time request_sample_time;
    sc_time mthreadbusy_sample_time;

    static const double pr_response;
    static const int max_responses_waiting = 4;
    int *rd_ptr, *wr_ptr;
    bool *full;
    OCP_RESP **responses_waiting;
    int threads;

    int requests_got, responses_sent;
};

const double tl1_timing_slave::pr_response=0.25;

// end of multiple inclusion protection
#endif

