//////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2005 Sonics, Inc.
//
// Confidential and Proprietary Information of Sonics, Inc.
// Use, disclosure, or reproduction is prohibited without
// written permission from Sonics, Inc.
//
// $Id: ThreadArbiter.cpp,v 1.1 2007/01/25 21:51:22 halexan Exp $
//
//////////////////////////////////////////////////////////////////////

#include "ThreadArbiter.h"
#include <algorithm>

using namespace std;
using namespace Sonics;

//////////////////////////////////////////////////////////////////
// FUNCTION: ThreadArbiter::ThreadArbiter
// DESCRIPTION: constructor
// ARGUMENTS: module name, number of threads (default -1, meaning to
// be set later by setNumThreads) and threadbusy proxy if applicable
// RETURNS: Nothing
//////////////////////////////////////////////////////////////////
ThreadArbiter::ThreadArbiter( sc_module_name name,
                              ThreadBusyProxy* pThreadBusyProxy,
                              int numThreads ) :
    sc_module          ( name ),    
    m_numThreads       ( numThreads ),
    m_lastWinner       ( 0 ),
    m_pThreadBusyProxy ( pThreadBusyProxy )
{
    SC_METHOD( arbitrateMethod );
    sensitive << m_arbitrationTrigger;
    if ( m_pThreadBusyProxy != NULL )
        sensitive << m_pThreadBusyProxy->threadBusyEvent();
    dont_initialize();
}

//////////////////////////////////////////////////////////////////
// FUNCTION: ThreadArbiter::setNumThreads
// DESCRIPTION: Sets number of threads. If the number is not known
// at construction time, then this function must be called at end of
// elaboration
// ARGUMENTS: number of threads
// RETURNS: Nothing
//////////////////////////////////////////////////////////////////
void
ThreadArbiter::setNumThreads( int numThreads )
{
    m_numThreads = numThreads;
    assert( m_numThreads > 0 );
}
//////////////////////////////////////////////////////////////////
// FUNCTION: ThreadArbiter::threadReady
// DESCRIPTION: notifies the arbiter that a thread is candidate
// for arbitration
// ARGUMENTS: thread id
//////////////////////////////////////////////////////////////////
void
ThreadArbiter::threadReady( int threadId )
{
    assert( static_cast<unsigned int>( threadId ) < m_threadActive.size() );
    m_threadActive[threadId] = 1;
    // notify with delay, so multiple threads can declare readiness within
    // delta cycle
    m_arbitrationTrigger.notify( SC_ZERO_TIME );
}

//////////////////////////////////////////////////////////////////
// FUNCTION: ThreadArbiter::threadWinner
// DESCRIPTION: the event indicating the given thread wins arbitration
// ARGUMENTS: thread id
//////////////////////////////////////////////////////////////////
sc_event&
ThreadArbiter::threadWinner( int threadId ) const
{
    assert( static_cast<unsigned int>( threadId ) < m_threadActive.size() );
    assert( m_threadWinner[threadId] != NULL );
    return *( m_threadWinner[threadId] );
}

//////////////////////////////////////////////////////////////////
// FUNCTION: ThreadArbiter::end_of_elaboration()
// DESCRIPTION: the thread structure setup
// ARGUMENTS: None
// RETURNS: Nothing
//////////////////////////////////////////////////////////////////
void
ThreadArbiter::end_of_elaboration()
{
    assert( m_numThreads > 0 );
    m_threadActive.resize( m_numThreads, 0 );
    m_threadWinner.resize( m_numThreads, NULL );
    for ( int t=0; t < m_numThreads; ++t )
        m_threadWinner[t] = new sc_event();
    m_lastWinner = m_numThreads - 1;
}

//////////////////////////////////////////////////////////////////
// FUNCTION: ThreadArbiter::arbitrateMethod
// DESCRIPTION: The core of the arbiter.
// ARGUMENTS: None
// PRECONDITIONS:
//   sensitive <<  m_arbitrationTrigger
//   sensitive <<  m_pThreadBusyProxy->ThreadBusyEvent()
// SIDE EFFECTS: notifies a winner
//////////////////////////////////////////////////////////////////
void
ThreadArbiter::arbitrateMethod()
{
    unsigned int threadBusyWord = 0;
    if ( m_pThreadBusyProxy )
        threadBusyWord = m_pThreadBusyProxy->getThreadBusy();

    int winner = -1;

    if ( m_numThreads == 1 ) {
        if ( m_threadActive[0] && !threadBusyWord ) {
            winner = 0;
        }
    } else {
        // round robin arbitration loop, first eligible goes
        int threadsTried = 0;
        for ( int candidate = ( m_lastWinner + 1 ) % m_numThreads;
              threadsTried != m_numThreads;
              candidate = ( candidate + 1 ) % m_numThreads, ++threadsTried ) {
            if ( m_threadActive[candidate] <= 0 )
                continue;
            if ( ( threadBusyWord & ( 1 << candidate ) ) == 0 ) {
                winner = candidate;
                break;
            }
        }
    }

    if ( winner != -1 ) {
        m_threadWinner[winner]->notify();
        m_threadActive[winner] = 0;
        m_lastWinner = winner;
        vector<int>::iterator foundActive = find(
            m_threadActive.begin(), m_threadActive.end(), 1 );
        if ( foundActive != m_threadActive.end() ) {
            // there are active threads still. Make sure they get their turn
            m_arbitrationTrigger.notify( SC_ZERO_TIME );
        }            
    }
}
