
// 
//  Copyright 2003 OCP-IP
//  OCP-IP Confidential & Proprietary
//
//  Simple TL2 sideband example

#define DEBUG

#include <systemc.h>
#include "ocp_tl2_globals.h"
#include "ocp_tl2_data_cl.h"
#include "tl_channel.h"  

#include "tl_master_port.h"
#include "tl_master_if.h"

SC_MODULE(master) {
  
  // Port definitions
  sc_port<TLmasterIF<OCP_TL2_DataCl<int, int> > > ocp;

  void get_sb_thread() {
    int serror, sflag;  
    while(1) {
      wait(ocp->GetDataCl()->SBSlaveEvent);
      serror = ocp->GetDataCl()->MgetSError();
#ifdef DEBUG
      cout << "Master gets SError = " << serror;
      cout << " at time = " << sc_time_stamp().to_seconds() << endl;
      sflag = ocp->GetDataCl()->MgetSFlag();
      cout << "Master gets SFlag = " << sflag;
      cout << " at time = " << sc_time_stamp().to_seconds() << endl;
#endif
    }
  }

  void request_thread () {
    bool read_cycle = 0;
    int shift = 0;

    while (1) {
      if (!read_cycle) {
	read_cycle = 1; // Toggle write and read cycles
	ocp->GetDataCl()->MputMData(&wdata[shift], 10); 
	if (ocp->MputWriteRequestBlocking()) {
#ifdef DEBUG
	  cout << "Master write request finished";
	  cout << ", data = ";
	  for (int i=shift;i<shift+10;i++)
	    cout << wdata[i] <<",";
	  cout << "  at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
	  if (shift<90)
	    shift += 10;
	  else
	    shift = 0;
	}
      }
      else {
	if (ocp->MputReadRequestBlocking()) {
	  read_cycle = 0; // Toggle write and read cycles
#ifdef DEBUG
	  cout << "Master read request finished";
	  cout << " at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
	}
      }
    }
  }

  void response_thread () {
    unsigned int len;
    bool last_chunk;

    while (1) {
      if (ocp->MgetResponseBlocking(0)) {
	ocp->Mrelease(sc_time(1000, SC_MS)); 
	rdata = ocp->GetDataCl()->MgetSData(len,last_chunk); 
#ifdef DEBUG
	cout << "Master gets read response";
	cout << ", data = ";
	for (unsigned int i=0;i<len;i++)
	  cout << rdata[i] <<",";
	cout << " at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
      }
    }
  }
  
  SC_CTOR(master) {
    for (int i=0;i<100;i++)
      wdata[i] = i;
    SC_THREAD(get_sb_thread);
    SC_THREAD(request_thread);
    SC_THREAD(response_thread);
  }

  int wdata[100], *rdata;

};



// Copyright (c 2003 OCP-IP
// Simple pipelined TL2 slave

#include "tl_slave_if.h"
#include "tl_slave_port.h"

SC_MODULE(slave) {

  // Port definitions
  sc_port<TLslaveIF<OCP_TL2_DataCl<int, int> > > ocp;

  // Globals
  int * data;
  unsigned int len;

  sc_event response_event;

  void put_sb_thread() {
    bool serror = 0;
    int sflag = 0;

    while(1) {
      wait(sc_time(100, SC_MS));
      ocp->GetDataCl()->SputSError(serror);
#ifdef DEBUG
      cout << "Slave puts SError = " << serror;
      cout << " at time = " << sc_time_stamp().to_seconds() << endl;
      ocp->GetDataCl()->SputSFlag(sflag);
      cout << "Slave puts SFlag = " << sflag;
      cout << " at time = " << sc_time_stamp().to_seconds() << endl;
#endif
      serror = !serror;
      if (sflag<100) {
	sflag++;
      }
      else
	sflag = 0;
    }
  }
  void request_thread () {
    bool wr;
    bool tmp;
    bool last_chunk;

    while (true) {

      tmp = ocp->SgetRequestBlocking(0);
      if (tmp) {
	ocp->Srelease(sc_time(1000, SC_MS)); 
	wr = ocp->IsWrite(); // Test read or write
	if (wr) {
	  data = ocp->GetDataCl()->SgetMData(len,last_chunk); 
#ifdef DEBUG
	  cout << "Slave gets write request";
	  cout << ", data = ";
	  for (unsigned int i=0;i<len;i++)
	    cout << data[i] <<",";
	  cout << "  at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
	}
	else {
#ifdef DEBUG
	  cout << "Slave gets read request";
	  cout << "  at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
	  response_event.notify(sc_time(100, SC_MS));
	}
      }
    }
  }

  void response_thread () {
    bool tmp;

    while (true) {
      wait (response_event);
      ocp->GetDataCl()->SputSData(data, len);
      tmp = ocp->SputResponseBlocking();
      if (tmp) {
#ifdef DEBUG
	cout << "Slave read response finished";
	cout << ", data = ";
	for (unsigned int i=0;i<len;i++)
	  cout << data[i] <<",";
	cout << "  at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
      }
    }
  }

  SC_CTOR(slave) {
    SC_THREAD(put_sb_thread);
    SC_THREAD(request_thread);
    SC_THREAD(response_thread);
  }
};




// Copyright (c 2003 OCP-IP
// Top level netlist for simple TL2 example

int sc_main(int, char*[])
{

  bool sync = true;
  TL_Channel<OCP_TL2_DataCl<int, int> >  ch0("ch0", sync);

  ch0.GetParamCl()->respaccept=true;

  slave sl1("sl1");
  master ms1("ms1");

  ms1.ocp(ch0);

  sl1.ocp(ch0);

  sc_start(20000, SC_MS);

  return(0);
}

