// 
// Copyright 2003 OCP-IP
// OCP-IP Confidential & Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//         Date : 10/16/2003
//
//  Description:  Regresion test 2 for the OCP-specific TL2 API.
//                Tests the behaviour of the master/slave interface methods
//                described in the OCP TL2 specific channel, in a single-thread
//                context
// ============================================================================

//#define DEBUG_C
//#define TRACE_C

#include <string>

using namespace std;

#include <systemc.h>
#include "ocp_tl2_channel.h"  
#include "ocp_tl2_master_port.h"  
#include "ocp_tl2_slave_port.h"  

// ----------------------------------------------------------------------------
// SIMPLE MASTER MODULE:
//
//  * non-pipelined module: uses one thread for request and response
//  * uses an OCP-specific TL2 master port
//  * uses a'OCPRequestGrp' structure to pass all the request signals to the
//      channel, and a 'OCPResponseGrp' structure to store the response
//      signals
//  * models a single thread master: threads = 1 
//  * performs the following actions:
//      - sends a 10-length READ burst request to the slave using two
//        sendOCPRequest() non-blocking calls with different chunk lengths.
//      - waits and get the corresponding response using two successive
//        getOCPResponse() calls catching two chunks.
//      - sends a 10-length READ burst request to the slave using two
//        sendOCPRequestBlocking() calls with different chunk lengths.
//      - waits and get the corresponding response using two successive
//        getOCPResponseBlocking() calls catching two chunks.
// ----------------------------------------------------------------------------


SC_MODULE(master) {

    // OCP-specific TL2 specialized master port 
    OCP_TL2_MasterPort<unsigned int, unsigned int> ocp;

    // this module has SC processes
    SC_HAS_PROCESS(master);

    // ----------------------------------------------------------------------------
    // Request/Response thread  
    // ----------------------------------------------------------------------------

    void request_response_thread () {

            //
            //
            // Sends the first request (10-length READ burst): 
            //
            //
            
            // Set the members of the request group stucture
            m_req.MCmd            =     OCP_MCMD_RD;    // this is a READ transaction
            m_req.MDataPtr        =     wdata;          // pass a pointer to the master data array


            while(ocp->getSBusy()) {
                wait(ocp->RequestEndEvent());
            }

#ifdef DEBUG_C
            cout << "Master starts sending request 1 chunk 1 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif
            assert(ocp->sendOCPRequest(m_req,6,false));
            ocp->waitSCmdAccept();

#ifdef DEBUG_C
            cout << "Master ends sending request 1 chunk 1 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif

            m_req.MDataPtr        =     &wdata[6];          // pass a pointer to the master data array (second chunk)


            while(ocp->getSBusy()) {
                wait(ocp->RequestEndEvent());
            }

#ifdef DEBUG_C
            cout << "Master starts sending request 1 chunk 2 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif

            assert(ocp->sendOCPRequest(m_req,4,true));
            while(!ocp->getSCmdAccept()) {
                wait(ocp->RequestEndEvent());
            }

#ifdef DEBUG_C
            cout << "Master ends sending request 1 chunk 2 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif


            //
            //
            // Waits and gets the first response (2 chunks)
            //
            //
            
            unsigned int RespChunkLen;
            bool RespChunkLast;

            // get the first response chunk and tests

#ifdef DEBUG_C
            cout << "Master starts getting response 1 chunk 1 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif
            while(!ocp->getOCPResponse(m_resp,false,RespChunkLen,RespChunkLast)) {
                wait(ocp->ResponseStartEvent());
            }

            // Test the received data
            assert(RespChunkLen==4);
            assert(!RespChunkLast);
            for (unsigned int  i=0; i< RespChunkLen; i+=1 ) {
                assert(m_resp.SDataPtr[i] == i);
            }

            ocp->putMRespAccept();

#ifdef DEBUG_C
            cout << "Master ends getting response 1 chunk 1 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif

            // get the second response chunk and tests

#ifdef DEBUG_C
            cout << "Master starts getting response 1 chunk 2 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif
            while(!ocp->getOCPResponse(m_resp,true,RespChunkLen,RespChunkLast)) {
                wait(ocp->ResponseStartEvent());
            }

            // Test the received data
            assert(RespChunkLen==6);
            assert(RespChunkLast);
            for (unsigned int  i=0; i< RespChunkLen; i+=1 ) {
                assert(m_resp.SDataPtr[i] == i+4);
            }

#ifdef DEBUG_C
            cout << "Master ends getting response 1 chunk 2 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif


            //
            //
            // Sends the second request (10-length READ burst): 
            //
            //
            
            // Set the members of the request group stucture
            m_req.MCmd            =     OCP_MCMD_RD;    // this is a READ transaction
            m_req.MDataPtr        =     wdata;          // pass a pointer to the master data array

#ifdef DEBUG_C
            cout << "Master starts sending request 2 chunk 1 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif

            assert(ocp->sendOCPRequestBlocking(m_req,7,false));

#ifdef DEBUG_C
            cout << "Master ends sending request 2 chunk 1 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif

            m_req.MDataPtr        =     &wdata[7];          // pass a pointer to the master data array

#ifdef DEBUG_C
            cout << "Master starts sending request 2 chunk 2 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif

            assert(ocp->sendOCPRequestBlocking(m_req,3,true));

#ifdef DEBUG_C
            cout << "Master ends sending request 2 chunk 2 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif


            //
            //
            // Waits and gets the second response (2 chunks)
            //
            //
            
            // get the first response chunk and tests

#ifdef DEBUG_C
            cout << "Master starts getting response 2 chunk 1 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif
            assert(ocp->getOCPResponseBlocking(m_resp,false,RespChunkLen,RespChunkLast));

            // Test the received data
            assert(RespChunkLen==3);
            assert(!RespChunkLast);
            for (unsigned int  i=0; i< RespChunkLen; i+=1 ) {
                assert(m_resp.SDataPtr[i] == i);
            }

            ocp->putMRespAccept();

#ifdef DEBUG_C
            cout << "Master ends getting response 2 chunk 1 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif

            // get the second response chunk and tests


#ifdef DEBUG_C
            cout << "Master starts getting response 2 chunk 2 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif
            assert(ocp->getOCPResponseBlocking(m_resp,true,RespChunkLen,RespChunkLast));
            // Test the received data
            assert(RespChunkLen==7);
            assert(RespChunkLast);
            for (unsigned int  i=0; i< RespChunkLen; i+=1 ) {
                assert(m_resp.SDataPtr[i] == i+3);
            }

#ifdef DEBUG_C
            cout << "Master ends getting response 2 chunk 2 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif


#ifdef DEBUG_C
            cout << "MASTER THREAD finished -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif

            master_finished = true;
    }

    // ----------------------------------------------------------------------------
    // Constuctor
    // ----------------------------------------------------------------------------
    master(sc_module_name mod):
        sc_module(mod),
    ocp("Master_Port")
    {

        master_finished = false;

        // Init Master Data array
        for (unsigned int i=0;i<100;i++)
            wdata[i] = i;
        SC_THREAD(request_response_thread);
    }

    // ----------------------------------------------------------------------------
    // Internal class members 
    // ----------------------------------------------------------------------------
    bool master_finished;

    private:
    
    // Array storing data to be sent
    unsigned int wdata[100];

    // OCP Request/Response structures
    OCPRequestGrp<unsigned int, unsigned int> m_req;
    OCPResponseGrp<unsigned int> m_resp;

};

// ----------------------------------------------------------------------------
// SIMPLE SLAVE MODULE:
// 
//  * non-pipelined module: uses one thread for request and response
//  * uses an OCP-specific TL2 slave port
//  * uses an'OCPRequestGrp' structure to store all the request signals from the
//      channel, and an 'OCPResponseGrp' structure to send response signals
//  * performs the following actions:
//      - receives a 10-length READ burst from the master using getOCPRequest()
//        non-blocking calls (2 chunks of length 6 and 4), and sends a response
//        using sendOCPResponse()(two response chunks of length 4 and 6)
///      - receives a 10-length READ burst from the master using getOCPRequestBlocking()
//        calls (2 chunks of length 7 and 3), and sends a response
//        using sendOCPResponseBlocking()(two response chunks of length 3 and 7)
// ----------------------------------------------------------------------------

SC_MODULE(slave) {

    // OCP-specific TL2 specialized master port 
    OCP_TL2_SlavePort<unsigned int,unsigned int> ocp;

    // this module has SC processes
    SC_HAS_PROCESS(slave);


    // ----------------------------------------------------------------------------
    // Request/Response thread
    // ----------------------------------------------------------------------------
    void request_response_thread () {
            
            //
            //
            // Waits and gets the first request (10-length READ)
            //
            //
            
            unsigned int ReqChunkLen;
            bool ReqChunkLast;

            // get the first request chunk
#ifdef DEBUG_C
            cout << "Slave starts getting request 1 chunk 1-- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif
            while(!ocp->getOCPRequest(m_req,false,ReqChunkLen,ReqChunkLast))
            {
                wait(ocp->RequestStartEvent());
            }

            // Test the received data
            assert(ReqChunkLen == 6);      
            assert(ReqChunkLast == false);
            for (unsigned int  i=0; i< ReqChunkLen; i+=1 ) {
                assert(m_req.MDataPtr[i] == i);
            }

            ocp->putSCmdAccept();

            // Copy the received data into the internal buffer
            for (unsigned int i=0; i< ReqChunkLen; i+=1 ) {
                data[i] = m_req.MDataPtr[i];
            }

#ifdef DEBUG_C
            cout << "Slave ends getting request 1 chunk 1-- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif

            // get the second request chunk
#ifdef DEBUG_C
            cout << "Slave starts getting request 1 chunk 2-- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif
            while(!ocp->getOCPRequest(m_req,true,ReqChunkLen,ReqChunkLast))
            {
                wait(ocp->RequestStartEvent());
            }

            // Test the received data
            assert(ReqChunkLen == 4);      
            assert(ReqChunkLast == true);
            for (unsigned int  i=0; i< ReqChunkLen; i+=1 ) {
                assert(m_req.MDataPtr[i] == i+6);
            }


            // Copy the received data into the internal buffer
            for (unsigned int i=0; i< ReqChunkLen; i+=1 ) {
                data[i+6] = m_req.MDataPtr[i];
            }

#ifdef DEBUG_C
            cout << "Slave ends getting request 1 chunk 2-- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif




            //
            //
            // Sends the response using 2 chunks
            //
            //
            
            // Set the members of the response group stucture
            m_resp.SResp          =     OCP_SRESP_DVA;      // Response is VALID
            m_resp.SDataPtr       =     data;               // pass a pointer to the slave data

            // Sends the first chunk 
            while(ocp->getMBusy()) {
                wait(ocp->ResponseEndEvent());
            }

#ifdef DEBUG_C
            cout << "Slave starts sending Response 1 chunk 1 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif

            assert(ocp->sendOCPResponse(m_resp,4,false));
            ocp->waitMRespAccept();

            // Sends the second chunk 
            m_resp.SDataPtr       =     &data[4];               // pass a pointer to the slave data

            while(ocp->getMBusy()) {
                wait(ocp->ResponseEndEvent());
            }

#ifdef DEBUG_C
            cout << "Slave starts sending Response 1 chunk 2 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif
            assert(ocp->sendOCPResponse(m_resp,6,true));
            while(!ocp->getMRespAccept()) {
                wait(ocp->ResponseEndEvent());
            }

            //
            //
            // Waits and gets the second request (10-length READ)
            //
            //
            
            // get the first request chunk
#ifdef DEBUG_C
            cout << "Slave starts getting request 2 chunk 1-- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif
            assert(ocp->getOCPRequestBlocking(m_req,false,ReqChunkLen,ReqChunkLast));

            // Test the received data
            assert(ReqChunkLen == 7);      
            assert(ReqChunkLast == false);
            for (unsigned int  i=0; i< ReqChunkLen; i+=1 ) {
                assert(m_req.MDataPtr[i] == i);
            }

            ocp->putSCmdAccept();

            // Copy the received data into the internal buffer
            for (unsigned int i=0; i< ReqChunkLen; i+=1 ) {
                data[i] = m_req.MDataPtr[i];
            }

#ifdef DEBUG_C
            cout << "Slave ends getting request 2 chunk 1-- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif

            // get the second request chunk
#ifdef DEBUG_C
            cout << "Slave starts getting request 2 chunk 2-- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif
            assert(ocp->getOCPRequestBlocking(m_req,true,ReqChunkLen,ReqChunkLast));

            // Test the received data
            assert(ReqChunkLen == 3);      
            assert(ReqChunkLast == true);
            for (unsigned int  i=0; i< ReqChunkLen; i+=1 ) {
                assert(m_req.MDataPtr[i] == i+7);
            }


            // Copy the received data into the internal buffer
            for (unsigned int i=0; i< ReqChunkLen; i+=1 ) {
                data[i+7] = m_req.MDataPtr[i];
            }

#ifdef DEBUG_C
            cout << "Slave ends getting request 2 chunk 2-- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif


            //
            //
            // Sends the response using 2 chunks
            //
            //
            
            // Set the members of the response group stucture
            m_resp.SResp          =     OCP_SRESP_DVA;      // Response is VALID
            m_resp.SDataPtr       =     data;               // pass a pointer to the slave data

            // Sends the first chunk 
#ifdef DEBUG_C
            cout << "Slave starts sending Response 2 chunk 1 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif
            assert(ocp->sendOCPResponseBlocking(m_resp,3,false));

            // Sends the second chunk 
            m_resp.SDataPtr       =     &data[3];               // pass a pointer to the slave data
#ifdef DEBUG_C
            cout << "Slave starts sending Response 2 chunk 2 -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif
            assert(ocp->sendOCPResponseBlocking(m_resp,7,true));

            

#ifdef DEBUG_C
            cout << "SLAVE THREAD finished -- " << "\t" << simcontext()->delta_count() << "\t" << sc_time_stamp() << endl;
#endif

            slave_finished = true;
    }

    // ----------------------------------------------------------------------------
    // constuctor
    // ----------------------------------------------------------------------------
    slave(sc_module_name mod):
        sc_module(mod),
    ocp("Slave_Port")
    {
        slave_finished = false;
        SC_THREAD(request_response_thread);
    }

    // ----------------------------------------------------------------------------
    // Internal class members 
    // ----------------------------------------------------------------------------
    bool slave_finished;
    
    private:
    // slave memory 
    unsigned int data[100];

    // OCP Request/Response structures
    OCPRequestGrp<unsigned int, unsigned int> m_req;
    OCPResponseGrp<unsigned int> m_resp;
};

// ----------------------------------------------------------------------------
// MAIN PROGRAM
// ----------------------------------------------------------------------------

void readMapFromFile(const string &myFileName, MapStringType &myParamMap) 
{
    // read pairs of data from the passed file
    string leftside;
    string rightside;
    
    // (1) open the file
    ifstream inputfile(myFileName.c_str());
    assert( inputfile );

    // set the formatting
    inputfile.setf(std::ios::skipws);

    // Now read through all the pairs of values and add them to the passed map
    while ( inputfile ) {
        inputfile >> leftside;
        inputfile >> rightside;
        myParamMap.insert(std::make_pair(leftside,rightside));
    }

    // All done, close up
    inputfile.close();
}

  // Performs some basic checks on the parameter values
  // To be completed....
  template <class TdataCl> 
  bool ConfigChecking(ParamCl<TdataCl>* pcl) {
     bool configError = false;
      
     // 'Exact ThreadBusy' conditions (OCP 2.0 spec. page 49)
     if (pcl->sthreadbusy_exact==1 && !(pcl->cmdaccept == 0) )
     {
         cout <<  " OCP parameter error: " << endl;
         cout << "      sthreadbusy_exact==1 and cmdaccept parameter is not 0" ;
         configError = true;
     }
     if (pcl->sthreadbusy_exact==1 && pcl->datahandshake==1 && pcl->sdatathreadbusy==0 && !(pcl->dataaccept == 0) )
     {
         cout <<  " OCP parameter error: " << endl;
         cout << "      sthreadbusy_exact==1,datahandshake==1,sdatathreadbusy==0 and dataaccept parameter is not 0" ;
         configError = true;
     }
     if (pcl->sdatathreadbusy_exact==1 && !(pcl->dataaccept == 0) )
     {
         cout <<  " OCP parameter error: " << endl;
         cout << "      sdatathreadbusy_exact==1 and dataaccept parameter is not 0" ;
         configError = true;
     }
     if (pcl->mthreadbusy_exact==1 && !(pcl->respaccept == 0) )
     {
         cout <<  " OCP parameter error: " << endl;
         cout << "      mthreadbusy_exact==1 and respaccept parameter is not 0" ;
         configError = true;
     }
     if (pcl->sthreadbusy_exact==0 && pcl->cmdaccept== 0 && pcl->sthreadbusy==1 )
     {
         cout <<  " OCP parameter error: " << endl;
         cout << "      sthreadbusy_exact is set to 0, cmdaccept is set to 0, but sthreadbusy is set to 1" ;
         configError = true;
     }
     if (pcl->mthreadbusy_exact==0 && pcl->respaccept== 0 && pcl->mthreadbusy==1 )
     {
         cout <<  " OCP parameter error: " << endl;
         cout << "      mthreadbusy_exact is set to 0, respaccept is set to 0, but mthreadbusy is set to 1" ;
         configError = true;
     }

     return !configError;
      
  };



int sc_main(int argc, char* argv[])
{
    // Get the ocp parameters file name
    string ocp_params_file_name = "";
    if (argc > 1) {
        string file_name(argv[1]);
        ocp_params_file_name = file_name;
    } else {
        cout    << "enter an ocp parameter file as first argument...\n";
        exit(-1);
    }

    // Creates the OCP TL2 channel
    OCP_TL2_Channel<unsigned int, unsigned int>  ch0("ch0");

    // Set the OCP parameters for this channel
    MapStringType  ocpParamMap;
    readMapFromFile(ocp_params_file_name, ocpParamMap);
    ch0.setConfiguration(ocpParamMap);
    // performs some basic checks on the OCP parameter values
    ParamCl<OCP_TL2_DataCl<unsigned int, unsigned int> > *pcl = ch0.GetParamCl();
    assert(ConfigChecking(pcl));


    // Creates masters and slaves 
    slave sl1("sl1");
    master ms1("ms1");

    // Connect masters and slaves using channels
    ms1.ocp(ch0);
    sl1.ocp(ch0);

    // Starts simulation 
    sc_start(20000, SC_MS);

    if(sl1.slave_finished && ms1.master_finished) 
        cout << endl << "OCP TL2 API Regression test 2 PASSED...." << endl;
    else
        cout << endl << "OCP TL2 API Regression test 2 FAILED...." << endl;

    return(0);
}



