////////////////////////////////////////////////////////////////////////
//                                                                    //
//                                                                    //
// (c) Copyright OCP-IP 2003-2005                                     //
// OCP-IP Confidential and Proprietary                                //
// $Id:
// Joe Chou, Sonics Inc.
// Alan Kamas, for Sonics Inc.
//                                                                    //
//                                                                    //
// 
// MasterSysC.cc
// simple reference master 
//
////////////////////////////////////////////////////////////////////////


#include "MasterSysC.h"
#include "ocp_globals.h"
#include "ocp_tl1_data_cl.h"


#define NUM_TESTS         9
#define TL1_FIFO_SIZE     4
#define OPTION1_NUM_TESTS 2

// ----------------------------------------------------------------------------
// constructor
// ----------------------------------------------------------------------------
template<typename TdataCl>
MasterSysC<TdataCl>::MasterSysC(
    sc_module_name name,
    double         ocp_clock_period,
    sc_time_unit   ocp_clock_time_unit,
    int            id,
    ostream*       debug_os_ptr
) : sc_module(name),
    ipP("ipPort"),
    m_ID(id),
    m_ocpClkPeriod(ocp_clock_period),
    m_ocpClkTimeUnit(ocp_clock_time_unit),
    m_curMThreadBusy(0),
    m_debug_os_ptr(debug_os_ptr),
    m_OCPParamP(NULL),
    m_threads(1),
    m_addrspace(false),
    m_sthreadbusy(false),
    m_must_use_sthreadbusy(false),
    m_sdatathreadbusy(false),
    m_sdatathreadbusy_exact(false),
    m_mthreadbusy(false),
    m_must_use_mthreadbusy(false),
    m_respaccept(true),
    m_datahandshake(false),
    m_writeresp_enable(false),
    m_readex_enable(false),
    m_writenonpost_enable(false),
    m_max_burst_len(1),
    m_burstseq(false),
    m_burstsinglereq(false),
    m_address_incr(1),
    m_respaccept_fixeddelay(true),
    m_respaccept_delay(0),
    m_command_cycles(0)
{
    // setup a SystemC thread process, which uses dynamic sensitive
    SC_THREAD(requestThreadProcess);

    // setup a SystemC thread process, which uses dynamic sensitive
    SC_THREAD(responseThreadProcess);

    // setup a SysetmC thread process to handle data handshakes
    SC_THREAD(dataHSThreadProcess);

    // setup a SystemC thread process to drive any connected sideband signals
    SC_THREAD(exerciseSidebandThreadProcess);
}

// ----------------------------------------------------------------------------
// destructor
// ----------------------------------------------------------------------------
template<typename TdataCl>
MasterSysC<TdataCl>::~MasterSysC()
{
    // do nothing
}

// ----------------------------------------------------------------------------
// SystemC Method MasterSysC::end_of_elaboration()
// ----------------------------------------------------------------------------
// 
//  At this point, everything has been built and connected.
//  We are now free to get our OCP parameters and to set up our
//  own variables that depend on them.
//
template<typename TdataCl>
void MasterSysC<TdataCl>::end_of_elaboration()
{
    // Call the System C version of this function first
    sc_module::end_of_elaboration();

    //-----------------------------------------
    //  OCP Parameters
    //-----------------------------------------

    // This Master adjusts to the OCP it is connected to.

    // Now get my OCP parameters from the port.
    m_OCPParamP = ipP->GetParamCl();

    // Get the number of threads
    m_threads = m_OCPParamP->threads;
    
    // is the MAddrSpace field part of the OCP channel?
    m_addrspace = m_OCPParamP->addrspace;

    // is SThreadBusy part of the channel?
    m_sthreadbusy = m_OCPParamP->sthreadbusy;

    // Is SThreadBusy compliance required?
    if (m_OCPParamP->sthreadbusy_exact) {
        m_must_use_sthreadbusy = true;
    } else if ( (m_OCPParamP->sthreadbusy) && (! m_OCPParamP->cmdaccept) ) {
        m_must_use_sthreadbusy = true;
    } else {
        m_must_use_sthreadbusy = false;
    }

    // is SDataThreadBusy part of the channel?
    m_sdatathreadbusy = m_OCPParamP->sdatathreadbusy;

    // Is SDataThreadBusy compliance required?
    m_sdatathreadbusy_exact = m_OCPParamP->sdatathreadbusy_exact;

    // is MThreadBusy part of the channel?
    m_mthreadbusy = m_OCPParamP->mthreadbusy;

    // is MRespAccept part of the channel?
    m_respaccept = m_OCPParamP->respaccept;

    // Is MThreadBusy compliance required?
    if (m_OCPParamP->mthreadbusy_exact) {
        m_must_use_mthreadbusy = true;
    } else if ( (m_OCPParamP->mthreadbusy) && (! m_OCPParamP->respaccept) ) {
        m_must_use_mthreadbusy = true;
    } else {
        m_must_use_mthreadbusy = false;
    }

    // is Data Handshake part of the channel?
    m_datahandshake = m_OCPParamP->datahandshake;

    // is write response part of the channel?
    m_writeresp_enable = m_OCPParamP->writeresp_enable;

    // is READ-EX part of the channel?
    m_readex_enable = m_OCPParamP->readex_enable;

    // Are non-posted writes (write commands that receive responses) part of the channel?
    m_writenonpost_enable = m_OCPParamP->writenonpost_enable;

    // Burst Parameters
    // The Slave only supports precise bursts with a maximum length of at least 4.
    m_max_burst_len = (1 << (m_OCPParamP->burstlength_wdth));
    m_burstseq = m_OCPParamP->burstseq;
    if (m_burstseq) {
        assert(m_OCPParamP->burstprecise);
        assert(m_OCPParamP->burstlength);
        assert(m_max_burst_len >= 4);
        assert(m_OCPParamP->burstseq_incr_enable);
        // data width must be a byte length
        assert( (m_OCPParamP->data_wdth)%8 == 0 );
    }
    // amount to increment the address by each burst transaction
    m_address_incr = (m_OCPParamP->data_wdth) / 8;
    // Support for single request / multiple data
    m_burstsinglereq = m_OCPParamP->burstsinglereq;
    if (m_burstsinglereq) {
        assert(m_burstseq && m_datahandshake);
    }
    // burstdatalast, burstreqlast, and burstresplast are not supported
    assert(!m_OCPParamP->datalast);
    assert(!m_OCPParamP->reqlast);
    assert(!m_OCPParamP->resplast);

    
    //-----------------------------------------
    //  Master Specific Parameters
    //-----------------------------------------

    // Retrieve any configuration parameters that were passed to this block
    // in the setConfiguration command.

// #ifndef NDEBUG
//     cout << "I am configuring a Master!" << endl;
//     cout << "Here is my configuration map for Master >" << name() << "< that was passed to me." << endl;
//     MapStringType::iterator map_it;
//     for (map_it = m_ParamMap.begin(); map_it != m_ParamMap.end(); ++map_it) {
//         cout << "map[" << map_it->first << "] = " << map_it->second << endl;
//     }
//     cout << endl;
// #endif

    string myPrefix = "";
    string paramName = "undefined";

    // MRespAccept delay in OCP cycles 
    paramName = "mrespaccept_delay";
    if (!(ParamCl<TdataCl>::getIntOCPConfigValue(myPrefix, paramName, m_respaccept_delay, m_ParamMap,name())) ) {
        // Could not find the parameter so we must set it to a default
#ifndef NDEBUG 
        cout << "Warning: master paramter \"" << paramName << "\" for Master \"" << name() << "\" was not found in the parameter map." << endl;
        cout << "         setting missing parameter to 1." << endl;
#endif
        m_respaccept_delay = 1;
    }

    // MRespAccept Delay Style. 1=fixed delay : 0=random delay
    paramName = "mrespaccept_fixeddelay";
    if (!(ParamCl<TdataCl>::getBoolOCPConfigValue(myPrefix, paramName, m_respaccept_fixeddelay, m_ParamMap,name())) ) {
        // Could not find the parameter so we must set it to a default
#ifndef NDEBUG 
        cout << "Warning: master paramter \"" << paramName << "\" for Master \"" << name() << "\" was not found in the parameter map." << endl;
        cout << "         setting missing parameter to 1 (fixed delay)." << endl;
#endif
        m_respaccept_fixeddelay = true;
    }

    // Command Set Repeat Count: Number of times to repeat the set of Master Commands. >0 = fixed number of cycles. zero = infinite repeats
    paramName = "command_cycles";
    if (!(ParamCl<TdataCl>::getIntOCPConfigValue(myPrefix, paramName, m_command_cycles, m_ParamMap,name())) ) {
        // Could not find the parameter so we must set it to a default
#ifndef NDEBUG 
        cout << "Warning: master paramter \"" << paramName << "\" for Master \"" << name() << "\" was not found in the parameter map." << endl;
        cout << "         setting missing parameter to 0 (continuously repeating)." << endl;
#endif
        m_command_cycles = 0;
    }
}


// --------------------------------------------------------------------------
// SetConfiguration
// --------------------------------------------------------------------------
template<typename TdataCl> 
void MasterSysC<TdataCl>::setConfiguration( MapStringType& passedMap )
{
    // would be nice to process the Map immediately to avoid copying it and
    // storing it, but some of the parameters may be based on an OCP parameter.
    //
    // As long as Master parameters could be functions of OCP parameters, they will
    // have to be processed at the end of elaboration when the OCP parameters
    // are available.
    //
    // Save the Map until then
    m_ParamMap = passedMap;
}

// ----------------------------------------------------------------------------
// SystemC Thread Processes
// ----------------------------------------------------------------------------

template<typename TdataCl>
void MasterSysC<TdataCl>::requestThreadProcess()
{
    Ta Addr[4];
    bool isWriteCmd;

    // start time of requests
    int NumWait[NUM_TESTS][4] = {
        {100,   3, 0xF, 0xF},
        {7,   1,   3, 0xF},
        {6, 0xF, 0xF, 0xF},
        {10,  2,   1, 0xF},
        {7,   1,   3, 0xF},
        {6,   1,   1,   1},
        {7,   2, 0xF, 0xF},
        {8,   2,   1, 0xF},
        {7,   2,   2,   2}
    };

    // specifies the command to use
    OCPMCmdType Commands[NUM_TESTS][4]  = {
        {OCP_MCMD_WR, OCP_MCMD_RD, OCP_MCMD_IDLE, OCP_MCMD_IDLE},
        {OCP_MCMD_WR, OCP_MCMD_WR, OCP_MCMD_WR, OCP_MCMD_IDLE},
        {OCP_MCMD_RD, OCP_MCMD_IDLE, OCP_MCMD_IDLE, OCP_MCMD_IDLE},
        {OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_IDLE},
        {OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_IDLE},
        {OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_RD},
        {OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_IDLE, OCP_MCMD_IDLE},
        {OCP_MCMD_WR, OCP_MCMD_WR, OCP_MCMD_WR, OCP_MCMD_IDLE},
        {OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_RD}
    };

    // number of specified transactions in a test
    int NumTr[] = {2, 3, 1, 3, 3, 4, 2, 3, 4};

    // The thread to use (if available) for each test
    unsigned int TestThread[] = {0, 1, 2, 3, 4, 5, 6, 7, 8};

    // -----------------------------------
    // (1) processing and preparation step
    // -----------------------------------

    // initialize data
    OCPRequestGrp<Td,Ta> req;
    OCPDataHSGrp<Td> dataHs;
    int              Count = 0;
    int              Nr = 0;
    sc_time          old_time;
    sc_time          current_time;
    bool             sthreadbusy;
    unsigned int     my_data = 0;
    unsigned int     myThread = 0;

    // to allow legal addresses sequence and cmd sequence for bursts
    if (m_burstseq) {
        // Set addresses for INCR bursts
        Addr[0] = 0x20;
        Addr[1] = Addr[0] + m_address_incr;
        Addr[2] = Addr[1] + m_address_incr;
        Addr[3] = Addr[2] + m_address_incr;
        Commands[0][1] = OCP_MCMD_WR;
    } else {
        Addr[0] = 0x1780;
        Addr[1] = 0x20;
        Addr[2] = 0x20;
        Addr[3] = 0x40;
    }

    // calculate the new waiting time
    double  wait_for = NumWait[Nr][Count];

    // Do requests contain data (or will it be sent separately)
    if (m_datahandshake) {
        req.HasMData = false;        
    } else {
        req.HasMData = true;        
    }

    ipP->ocpWait();

    while (true) {

        if (ipP->getReset()) {
            // clear the Master state and wait for the channel to come out of reset
            m_dataQ.clear();  // This thread writes to it, so this thread clears it.
            wait( ipP->ResetEndEvent() );
            // Reset our state
            Count = 0;
            Nr = 0;
            wait_for = NumWait[Nr][Count];
            my_data = 0;
            myThread = 0;
            m_dataQ.clear();  // This thread writes to it, so this thread clears it.
        }

        // main loop
        while (!(ipP->getReset())) {
            // wait for the time to send the current request

            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "master wait_for = " << wait_for << endl;
            }

            if ( !(ipP->ocpSafeWait(wait_for))) {
                // Channel reset while we were waiting
                // Give up.
                break;
            }

            // remember the time
            old_time = sc_time_stamp();

            // Set the thread for this test
            myThread = TestThread[Nr];
            if (int(myThread) >= m_threads) {
                // The thread for this test is out of the range of available threads.
                // Use "old reliable" thread 0 instead.
                myThread = 0;
            }

            // ------------------------------------------------
            // (2) is SThreadBusy?
            // ------------------------------------------------

            // use SThreadBusy to limit the flow of requests if
            // cmdaccept is not part of the channel. (This is the case
            // for SThreadBusy Exact as well).
            if (m_must_use_sthreadbusy) {
                unsigned int threadBit = 1 << myThread;
                sthreadbusy = ipP->getSThreadBusy();
                while (sthreadbusy & threadBit) {

                    if (m_debug_os_ptr) {
                        (*m_debug_os_ptr) << "DB (" << name() << "): "
                            << "waiting on sthreadbusy for Thread#" << myThread << " with sthreadbusy = " << sthreadbusy << " and t = " << sc_simulation_time() << endl;
                    }

                    ipP->ocpWait();
                    sthreadbusy = ipP->getSThreadBusy();
                }
            }

            // ------------------------------------------------
            // (3) send a request
            // ------------------------------------------------

            // Compute the next request
            req.MCmd = Commands[Nr][Count];
            req.MThreadID = myThread;

            // is this an extended command to be sent over a basic channel?
            if ( (!m_readex_enable) && (req.MCmd == OCP_MCMD_RDEX) ) {
                // channel cannot handle READ-EX. Send simple READ.
                req.MCmd = OCP_MCMD_RD;
            } else if ( (!m_writenonpost_enable) && (req.MCmd == OCP_MCMD_WRNP) ) {
                // channel cannout handle WRITE-NP. Send simple WRITE.
                req.MCmd = OCP_MCMD_WR;
            }

            // compute the address
            req.MAddr = Addr[Count] + m_ID*0x40;
            req.MByteEn = 0xf;

            // for burst
            if (m_burstseq) {
                req.MBurstSeq = OCP_MBURSTSEQ_INCR;
                req.MBurstPrecise = 1;
                req.MBurstLength = NumTr[Nr];
                if (m_burstsinglereq) {
                    req.MBurstSingleReq = 1;
                } else {
                    req.MBurstSingleReq = 0;
                }
            }

            if (m_addrspace) {
                req.MAddrSpace = 0x1;
            }
            // compute the data
            switch (req.MCmd) {
                case OCP_MCMD_WR:
                case OCP_MCMD_WRNP:
                case OCP_MCMD_WRC:
                case OCP_MCMD_BCST:
                    // This is a write command - it has data
                    isWriteCmd = true;
                    my_data++;
                    // put the data into the request
                    req.MData = my_data + m_ID*0x40;
                    break;
                case OCP_MCMD_RD:
                case OCP_MCMD_RDEX:
                case OCP_MCMD_RDL:
                    // this is a read command - no data.
                    isWriteCmd = false;
                    req.MData = 0;
                    break;
                default:
                    cout << "ERROR: Master \"" << name() << "\" generates unknown command #"<< req.MCmd << endl;
                    sc_stop();
            }

            // send the request & data (if any)
            if ( (m_datahandshake) && (isWriteCmd) ) {
                // Data Handshake is being used on the channel. Pull out the data and send it separately.
                dataHs.MData = req.MData;
                dataHs.MDataThreadID = req.MThreadID;
                dataHs.MDataByteEn = req.MByteEn;
                dataHs.MDataLast = req.MReqLast;
                // Send out the request (block until it is on the channel)
                if (m_burstsinglereq) {
                    // This is a burst with single request (multiple data)
                    // Only send the request for the start of the burst
                    if (Count == 0) {
                        ipP->startOCPRequestBlocking(req);
                        // TODO: could have had reset here

                        // DEBUG logging
                        if (m_debug_os_ptr) {
                            (*m_debug_os_ptr) << "DB (" << name() << "): " << "sent SRMD Burst request." << endl;
                            (*m_debug_os_ptr) << "DB (" << name() << "): " << "    t = " << sc_simulation_time() << " MCmd: " << req.MCmd << endl;
                            (*m_debug_os_ptr) << "DB (" << name() << "): " << "    MData: " << req.MData << " MByteEn: " << req.MByteEn << endl;
                            (*m_debug_os_ptr) << "DB (" << name() << "): " << "    MBurstSeq: " << req.MBurstSeq << " MBurstLength: " << req.MBurstLength << " MBurstSingleReq: " << req.MBurstSingleReq << endl;
                        }

                    }
                } else {
                    // No single request multiple data burst - always send request
                    ipP->startOCPRequestBlocking(req);
                    // TODO: could have had reset here

                    // DEBUG logging
                    if (m_debug_os_ptr) {
                        (*m_debug_os_ptr) << "DB (" << name() << "): " << "sent request with data handshake." << endl;
                        (*m_debug_os_ptr) << "DB (" << name() << "): " << "    t = " << sc_simulation_time() << " MCmd: " << req.MCmd << endl;
                        (*m_debug_os_ptr) << "DB (" << name() << "): " << "    MData: " << req.MData << " MByteEn: " << req.MByteEn << endl;
                        (*m_debug_os_ptr) << "DB (" << name() << "): " << "    MBurstSeq: " << req.MBurstSeq << " MBurstLength: " << req.MBurstLength << " MBurstSingleReq: " << req.MBurstSingleReq << endl;
                    }

                }
                // Now that request has started, send the data
                m_dataQ.push_back(dataHs);
                // notify the DataHS Tread that new data has arrived
                m_newDataEvent.notify();
            } else {
                // no data handshake needed - just send the command
                ipP->startOCPRequestBlocking(req);
                // TODO: could have had reset here

                // DEBUG logging
                if (m_debug_os_ptr) {
                    (*m_debug_os_ptr) << "DB (" << name() << "): " << "sent request with data." << endl;
                    (*m_debug_os_ptr) << "DB (" << name() << "): " << "    t = " << sc_simulation_time() << " MCmd: " << req.MCmd << endl;
                    (*m_debug_os_ptr) << "DB (" << name() << "): " << "    MData: " << req.MData << " MByteEn: " << req.MByteEn << endl;
                    (*m_debug_os_ptr) << "DB (" << name() << "): " << "    MBurstSeq: " << req.MBurstSeq << " MBurstLength: " << req.MBurstLength << " MBurstSingleReq: " << req.MBurstSingleReq << endl;
                }

            }

            // -------------------------------
            // (1) processing and preparation step
            // -------------------------------

            // Go on to the next test if we have processed all of the requests for this
            // test OR if we are sending a read request with single request / multiple data.
            if ( (++Count >= NumTr[Nr]) || ((m_burstsinglereq)&&(!isWriteCmd)) ) {
                Count = 0;
                if (++Nr >= NUM_TESTS) {
                    // we have finished the cycle.  Shoule we start another?
                    m_command_cycles--;
                    if (m_command_cycles == 0) {
                        // no, that was our last cycle of commands
                        // Work for this thread is finished. Now just sit here and rest.
                        while (true) {
                            // Done sending commands. Sleep.
                            ipP->ocpWait(10000);
                        }
                    } else {
                        // Start another cycle
                        Nr = 1;
                    }
                }
            }

            // calculate the new waiting time
            wait_for = NumWait[Nr][Count];
            current_time = sc_time_stamp();
            double delta_time = (current_time.value() - old_time.value()) / 1000;
            if (delta_time >= wait_for) {
                wait_for = 0;
            } else {
                wait_for = wait_for - delta_time;
            }
        }
    }
}

template<typename TdataCl>
void MasterSysC<TdataCl>::dataHSThreadProcess()
{
    // initialize data
    OCPDataHSGrp<Td> dataHs;

    ipP->ocpWait();

    while (true) {
        if (ipP->getReset()) {
            // wait for the channel to come out of reset
            wait( ipP->ResetEndEvent() );
            // Allow the Request thread a chance to supply us with new data
            wait(SC_ZERO_TIME);
        }

        // main loop
        while ( !(ipP->getReset()) ) {

            // ------------------------------------------------
            // (1) Get the next data to send
            // ------------------------------------------------
            if (m_dataQ.empty()) {
                // No data to send. Wait for some to be queued
                // Note that if no dataHS, then will wait here forever.
                if (m_debug_os_ptr) {
                    (*m_debug_os_ptr) << "DB (" << name() << "): " << "waiting for data to send." << endl;
                }
                // NOTE: won't deadlock at reset. Okay to let this wait through the reset.
                wait(m_newDataEvent);
            }
            dataHs = m_dataQ.front();
            m_dataQ.pop_front();


            // ------------------------------------------------
            // (2) is SDataThreadBusy?
            // ------------------------------------------------

            // NOTE: In the case of a multi-threaded model with sdatathreadbusy on, it is possible to
            //       send data in a different thread order than the associated requests. 
            //       However, just because it is allowed, does not mean it is always a good idea, 
            //       and that is not being done with this master.

            if (m_sdatathreadbusy_exact) {
                // is our data's thread busy?
                unsigned int sdatathreadbusy = ipP->getSDataThreadBusy();
                unsigned int my_threadbit = 1 << dataHs.MDataThreadID;
                while (sdatathreadbusy & my_threadbit) {
                    if (m_debug_os_ptr) {
                        (*m_debug_os_ptr) << "DB (" << name() << "): "
                            << "waiting on sdatathreadbusy." << endl;
                    }
                    ipP->ocpWait();

                    if (ipP->getReset()) {
                        // the channel reset while we were waiting. Bail out of this loop.
                        break;
                    }

                    sdatathreadbusy = ipP->getSThreadBusy();
                }
            }

            // ------------------------------------------------
            // (3) send out the data
            // ------------------------------------------------

            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "send data." << endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "    t = " << sc_simulation_time() << endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "    MData: " << dataHs.MData << endl;
            }

            ipP->startOCPDataHSBlocking(dataHs);
        }
    }
}

template<typename TdataCl>
void MasterSysC<TdataCl>::responseThreadProcess()
{

    // initialization
    OCPResponseGrp<Td> resp;
    double         wait_for;

    // Set the initial value of MThreadBusy
    if (m_mthreadbusy) {
        ipP->putMThreadBusy(m_curMThreadBusy);
    }

    ipP->ocpWait();

    while (true) {

        if (ipP->getReset()) {
            m_curMThreadBusy = 0;
            // wait for the channel to come out of reset
            wait( ipP->ResetEndEvent() );
            // Allow the Request & Data thread a chance to supply us with new responses 
            wait(SC_ZERO_TIME);
            wait(SC_ZERO_TIME);
            m_curMThreadBusy = 0;
        }

        // main loop
        while ( !(ipP->getReset()) ) {
            // ------------------------------------------------
            // (1) wait for a response (blocking wait)
            // ------------------------------------------------

            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "waiting for response." << endl;
            }

            // get the next response
            ipP->getOCPResponseBlocking(resp);

            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "got response." << endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "    t = " << sc_simulation_time() << endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "    SData: " << resp.SData << endl;
            }

            // ------------------------
            // (2) process the response
            // ------------------------

            // compute the response acceptance time
            if (m_respaccept_fixeddelay) {
                wait_for = m_respaccept_delay;
            } else {
                // Go random up to max delay
                wait_for = (int)((m_respaccept_delay+1) * rand() / (RAND_MAX + 1.0));
            }

            // --------------------------------------------------
            // (3) accept the response
            // --------------------------------------------------

            if (m_respaccept) {
                if (wait_for == 0) {
                    // Accept immediately (this cycle)
                    ipP->putMRespAccept();
                } else {
                    // Accept later
                    if (!(ipP->ocpSafeWait(wait_for))) {
                        // Channel reset while we waited. Give up.
                        break;
                    }
                    ipP->putMRespAccept();
                }
            }

            if (m_must_use_mthreadbusy) {
                // use the MThreadBusy signal instead of resp accept
                if (wait_for > 0) {
                    // Set MThreadBusy
                    unsigned int threadBit = 1 << resp.SThreadID;
                    m_curMThreadBusy |= threadBit;
		    wait(0.1, SC_NS);
                    ipP->putMThreadBusy(m_curMThreadBusy);

                    // TODO: Remove
                    if (m_debug_os_ptr) {
                        (*m_debug_os_ptr) << "DB (" << name() << "): "
                            << "delaying response accept: setting mthreadbusy for Thread#" << resp.SThreadID << " with mthreadbusy = " << m_curMThreadBusy << " and t = " << sc_simulation_time() << endl;
                    }

                    // keep MThreadBusy on 
                    if ( !(ipP->ocpSafeWait(wait_for))) {
                        // Channel went into resest while we waited. Give up.
                        break;
                    }
                    // now release it
                    m_curMThreadBusy &= (~threadBit);
		    wait(0.1, SC_NS);
                    ipP->putMThreadBusy(m_curMThreadBusy);

                    // TODO: Remove
                    if (m_debug_os_ptr) {
                        (*m_debug_os_ptr) << "DB (" << name() << "): "
                            << "releasing mthreadbusy for Thread#" << resp.SThreadID << " with mthreadbusy = " << m_curMThreadBusy << " and t = " << sc_simulation_time() << endl;
                    }

                }
            }
        }
    }
}

template<typename TdataCl>
void MasterSysC<TdataCl>::exerciseSidebandThreadProcess(void)
{
    // Systematically send out sideband signals on any signals that are attached to us.

    ipP->ocpWait(10);
    int tweakCounter =0;
    bool hasMError = m_OCPParamP->merror;
    bool nextMError = false;
    bool hasMFlag = m_OCPParamP->mflag;
    int numMFlag = m_OCPParamP->mflag_wdth;
    unsigned int nextMFlag = 0;
    unsigned int maxMFlag = (1 << numMFlag) -1; 

    while (true) {
        if (ipP->getReset()) {
            // The channel is in reset.
            // wait until it is out of reset
            wait(ipP->ResetEndEvent());
            // reset our state
            tweakCounter =0;
            nextMError = false;
            nextMFlag = 0;
        }

        // main loop
        while (!(ipP->getReset())) {
            // wait 10 cycles
            if ( !(ipP->ocpSafeWait(10))) {
                // The channel reset while we were waiting. Give up.
                break;
            }

            // Now count through my sideband changes
            tweakCounter++;

            // Drive MError
            if (hasMError) {
                if (tweakCounter%2 == 0) {
                    // Toggle MERROR
                    nextMError = !nextMError;
                    ipP->MputMError(nextMError);
                }
            }

            // Drive MFlags
            if (hasMFlag) {
                if (tweakCounter%1 == 0) {
                    // go to next MFlag
                    nextMFlag += 1;
                    if (nextMFlag > maxMFlag) {
                        nextMFlag = 0;
                    }
                    ipP->MputMFlag(nextMFlag);
                }
            }
        }
    }
}

template class MasterSysC< OCP_TL1_SIGNAL_CL >;
