// 
//  Copyright 2003 OCP-IP
//  OCP-IP Confidential & Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Alan Kamas based on previous work and data traffic pattern by Yann Bajot
//         $Id:
//
//  Description:  Simple Point to point TL2 example using the OCP-specific TL2
//                API using static events only - no threads.
// ============================================================================

//#define DEBUG_C
//#define TRACE_C

#include <systemc.h>
#include "ocp_tl2_channel.h"  
#include "ocp_tl2_master_port.h"  
#include "ocp_tl2_slave_port.h"  

// ----------------------------------------------------------------------------
// SIMPLE MASTER MODULE:
//
//  * non-pipelined module: no threads - events only. 
//  * uses an OCP-specific TL2 master port
//  * uses a'OCPRequestGrp' structure to pass all the request signals to the
//      channel, and a 'OCPResponseGrp' structure to store the response
//      signals
//  * uses a one OCP Thread.
//  * uses three SC_Methods and one internal event.
//  * performs the following actions in a loop:
//      - sends a 10-length WRITE burst to the slave using
//        sendOCPRequest(). 
//      - sends a 10-length READ burst to the slave using
//        sendOCPRequest(). 
//      - gets the corresponding responses.
//      - performs a 20-length WRITE request.
//      - performs a 20-length READ request and gets the corresponding response.
// 
// ----------------------------------------------------------------------------

// const int LOOP_COUNT = 5;
const int LOOP_COUNT = 10000000;
// const int LOOP_COUNT = 1000000;
const int END_TIME = 1000000000;

SC_MODULE(master) {

    // this module has SC processes
    SC_HAS_PROCESS(master);

    // OCP-specific TL2 specialized master port 
    OCP_TL2_MasterPort<unsigned int, unsigned int> ocp;

    // ----------------------------------------------------------------------------
    // Request Method
    // ----------------------------------------------------------------------------

    // This method is called when the ocp connection is clear for a new request
    void send_request_method () 
    {
        // cout << "Master starts sending request " << m_req_num << " -- at time " << sc_time_stamp() << endl;
        switch(m_req_num) 
        {
            case 0:
                cout << "master DONE. \n" << endl;
                return;

            case 1:
                // Send the first reqeust
                m_req.reset();
                // Set the members of the request group stucture
                m_req.MCmd            =     OCP_MCMD_WR;    // this is a WRITE transaction
                m_req.MDataPtr        =     wdata;          // pass a pointer to the master data array
                // Sets TL2-specific fields
                // this request is sent using a single TL2 chunk
                ReqChunkLen = 10; ReqChunkLast = true;
                // set request to send next time
                m_req_num = 2;
                break;

            case 2:
                // Send the second request
                // Set the members of the request group stucture that have changed
                m_req.MCmd            =     OCP_MCMD_RD;    // this is a READ transaction
                // Sets TL2-specific fields
                ReqChunkLen = 10; ReqChunkLast = true;
                // set request to send next time
                m_req_num=3;
                break;

            case 3:
                // Send the third request
                // Set the members of the request group stucture that have changed
                m_req.MCmd            =     OCP_MCMD_WR;    // this is a READ transaction
                m_req.MDataPtr        =     wdata;          // pass a pointer to the master data array
                // Sets TL2-specific fields
                ReqChunkLen = 20; ReqChunkLast = true;
                // set request to send next time
                m_req_num=4;
                break;

            case 4:
                // Send the fourth request
                // Set the members of the request group stucture that have changed
                m_req.MCmd            =     OCP_MCMD_RD;    // this is a READ transaction
                // Sets TL2-specific fields
                ReqChunkLen = 20; ReqChunkLast = true;
                // set request to send next time
                m_loops_done++;
                if (m_loops_done >= LOOP_COUNT)
                {
                    m_req_num = 0;
                } 
                else 
                {
                    m_req_num = 1;
                }
                break;

            default:
                cout << "ERROR: master reached unknown request number = " << m_req_num << endl;
        }
        // Now that the request has been created, send it.
        bool ret_value=ocp->sendOCPRequest(m_req,ReqChunkLen,ReqChunkLast);
        assert(ret_value);
        // cout << "Master ends sending request at time " << sc_time_stamp() << endl;
    }
        
    // when this method is called there should be a waiting response on the ocp connection
    void get_response_method () 
    {
        switch( m_resp_num )
        {
            case 1:
                // cout << "Master starts getting response 1, chunk 1 -- " << sc_time_stamp() << endl;
                release_immediately = false;
                ret_value=ocp->getOCPResponse(m_resp,release_immediately,RespChunkLen,RespChunkLast);
                assert(ret_value);
                // This a small regression test. 
                assert(RespChunkLen == 5); assert(RespChunkLast == false);
                // Display the received data
                // for ( unsigned int i=0; i< RespChunkLen; i+=1 ) {
                    // cout << m_resp.SDataPtr[i] << " " ;
                // }
                // cout << endl;
                // release the response after some delay
                m_releaseResponseEvent.notify(100,SC_NS);
                // set up the next response
                m_resp_num =2;
                break;

            case 2:
                // cout << "Master starts getting response 1, chunk 2 -- " << sc_time_stamp() << endl;
                release_immediately = false;
                ret_value=ocp->getOCPResponse(m_resp,release_immediately,RespChunkLen,RespChunkLast);
                assert(ret_value);
                // This a small regression test. 
                assert(RespChunkLen == 5); assert(RespChunkLast == true);
                // Display the received data
                // for ( unsigned int i=0; i< RespChunkLen; i+=1 ) {
                    // cout << m_resp.SDataPtr[i] << " " ;
                // }
                // cout << endl;
                // release the response after some delay
                m_releaseResponseEvent.notify(100,SC_NS);
                // set up the next response
                m_resp_num =3;
                break;

            case 3:
                // cout << "Master starts getting response 2  -- " << sc_time_stamp() << endl;
                release_immediately = true;
                ret_value=ocp->getOCPResponse(m_resp,release_immediately,RespChunkLen,RespChunkLast);
                assert(ret_value);
                // This a small regression test. 
                assert(RespChunkLen == 20); assert(RespChunkLast == true);
                // Display the received data
                // for ( unsigned int i=0; i< RespChunkLen; i+=1 ) {
                    // cout << m_resp.SDataPtr[i] << " " ;
                // }
                // cout << endl;
                // response should be released automatically.
                // set up the next response
                // set response to send next time
                m_resp_num = 1;
                break;

            default:
                cout << "ERROR: master reached unknown response number = " << m_resp_num << endl;
        }

    }

    // when this method is called there should be a waiting response on the ocp connection
    void release_response_method () 
    {
        ocp->putMRespAccept();
    }

    // ----------------------------------------------------------------------------
    // Constuctor
    // ----------------------------------------------------------------------------
    master(sc_module_name mod):
        sc_module(mod), ocp("Master_Port")
    {
        // declare my three sensitive methods
        SC_METHOD(send_request_method);
          sensitive << ocp.RequestEndEvent();
        SC_METHOD(get_response_method); 
          sensitive << ocp.ResponseStartEvent();
          dont_initialize();
        SC_METHOD(release_response_method); 
          sensitive << m_releaseResponseEvent;
          dont_initialize();

        // Init Master Data array
        for (unsigned int i=0;i<100;i++)
        {
            wdata[i] = i;
        }
        m_req_num = 1;   // First request to send
        m_loops_done = 0;
        m_resp_num = 1;   // First response to get
        ReqChunkLen = 10; 
        ReqChunkLast = true;
        ret_value = false;
        release_immediately = false;
        RespChunkLen = 10;
        RespChunkLast = true;
    }

    // ----------------------------------------------------------------------------
    // Internal class members 
    // ----------------------------------------------------------------------------
    
    public:

    private:

    // Array storing data to be sent
    unsigned int wdata[100];

    // Release event for delayed releases
    sc_event m_releaseResponseEvent;

    // OCP Request/Response structures
    int m_req_num;
    int m_loops_done;
    unsigned int ReqChunkLen; 
    bool ReqChunkLast;
    bool ret_value;
    int m_resp_num;
    bool release_immediately;
    unsigned int RespChunkLen;
    bool RespChunkLast;
    OCPRequestGrp<unsigned int, unsigned int> m_req;
    OCPResponseGrp<unsigned int> m_resp;

};

// ----------------------------------------------------------------------------
// SIMPLE SLAVE MODULE:
// 
//  * non-pipelined module: event drven only - no SC_Thread
//  * uses an OCP-specific TL2 slave port
//  * uses an'OCPRequestGrp' structure to store all the request signals from the
//      channel, and an 'OCPResponseGrp' structure to send response signals
//  * performs the following actions:
//      - receives a 10-length WRITE burst from the master, and stores the
//          received data in an internal array.
//      - receives a 10-length READ burst from the master, and sends the
//        response using two consecutive response chunks (5-length each) with a
//        different 'SRespInfo' signal value.
//      - receives a 20-length WRITE burst from the master, and stores the
//          received data in an internal array.
//      - receives a 20-length READ burst from the master,and sends the
//        response using one response call
// 
// ----------------------------------------------------------------------------

SC_MODULE(slave) {

    // OCP-specific TL2 specialized master port 
    OCP_TL2_SlavePort<unsigned int,unsigned int> ocp;

    // this module has SC processes
    SC_HAS_PROCESS(slave);

    // Get request and generate response
    // when this method is called there should be a waiting request on the ocp connection
    void get_request_method () 
    {
        switch( m_req_num )
        {
            case 1:
                // ---------------------------------------------------------
                // Gets the first request (should be a 10-length WRITE)
                // ---------------------------------------------------------
                //  We will release the request channel after some delay
                release_immediately = false;
                // get the first request chunk and retrieves chunk parameters
                // cout << "Slave starts getting request 1 -- " << sc_time_stamp() << endl;
                ret_value=ocp->getOCPRequest(m_req,release_immediately,ReqChunkLen,ReqChunkLast);
                // This a small regression test. 
                assert(ret_value); assert(m_req.MCmd == OCP_MCMD_WR);
                assert(ReqChunkLen == 10); assert(ReqChunkLast == true);
                // Display the received data
                // for (unsigned int i=0; i< ReqChunkLen; i+=1 ) {
                    // cout << m_req.MDataPtr[i] << " " ;
                // }
                // cout << endl;
                // Copy the received data into the internal buffer
                for (unsigned int i=0; i< ReqChunkLen; i+=1 ) {
                    data[i] = m_req.MDataPtr[i];
                }
                // releases the request after some delay 
                m_releaseRequestEvent.notify(100,SC_NS);
                // set up the next request
                m_req_num =2;
                break;

            case 2:
                // ---------------------------------------------------------
                // Gets the second request (10-length READ) 
                // ---------------------------------------------------------
                // Now we release the request immediately
                release_immediately = true;
                // get the first request chunk and retrieves the chunk parameters
                // cout << "Slave starts getting request 2 -- " << sc_time_stamp() << endl;
                ret_value=ocp->getOCPRequestBlocking(m_req,release_immediately,ReqChunkLen,ReqChunkLast);
                // cout << "Slave ends getting request 2 -- " << sc_time_stamp() << endl;
                // This a small regression test. 
                assert(ret_value); assert(m_req.MCmd == OCP_MCMD_RD);
                assert(ReqChunkLen == 10); assert(ReqChunkLast == true);

                // ---------------------------------------------------------
                // Sends the response using 2 chunks (Resp 1 & Resp 2)
                // ---------------------------------------------------------

                // Setup the first chunk 
                // Set the members of the response group stucture
                m_resp[1].SResp          =     OCP_SRESP_DVA;      // Response is VALID
                m_resp[1].SDataPtr       =     data;               // pass a pointer to the slave data
                m_resp[1].SRespInfo      =     1;                  // Info for the first chunk
                assert(RespChunkLen[1] == 0);
                RespChunkLen[1] = 5; RespChunkLast[1] = false;

                // Setup the second chunk 
                m_resp[2].SDataPtr       =     &data[5];           // pass a pointer to the second part of the slave data
                m_resp[2].SRespInfo      =     2;                  // Info for the second chunk
                assert(RespChunkLen[2] == 0);
                RespChunkLen[2] = 5; RespChunkLast[2] = true;         // this is the last chunk of the response burst

                // try to ship these  new respones out
                tryToSendResponse();
                // set up the next request
                m_req_num =3;
                break;

            case 3:

                // ---------------------------------------------------------
                // gets the third request (20-length WRITE)
                // ---------------------------------------------------------
                //  We will release the request channel after some delay
                release_immediately = false;


                // get the first request chunk and retrieves chunk parameters
                // cout << "Slave starts getting request 3 -- " << sc_time_stamp() << endl;
                ret_value=ocp->getOCPRequest(m_req,release_immediately,ReqChunkLen,ReqChunkLast);
                // This a small regression test. 
                assert(ret_value); assert(m_req.MCmd == OCP_MCMD_WR);
                assert(ReqChunkLen == 20); assert(ReqChunkLast == true);
                // Display the received data
                // for (unsigned int i=0; i< ReqChunkLen; i+=1 ) {
                    // cout << m_req.MDataPtr[i] << " " ;
                // }
                // cout << endl;
                // Copy the received data into the internal buffer
                for (unsigned int i=0; i< ReqChunkLen; i+=1 ) {
                    data[i] = m_req.MDataPtr[i];
                }
                // releases the request after some delay 
                m_releaseRequestEvent.notify(100,SC_NS);
                // cout << "Slave ends getting request 3 -- " << sc_time_stamp() << endl;
                // set up the next request
                m_req_num = 4;
                break;

            case 4:
                // ---------------------------------------------------------
                // gets the fourth request (20-length READ) 
                // ---------------------------------------------------------
                
                // Now we release the request immediately
                release_immediately = true;
    
                // gets the first request chunk and retrieves chunk parameters
                // cout << "Slave starts getting request 4 -- " << sc_time_stamp() << endl;
                ret_value=ocp->getOCPRequest(m_req,release_immediately,ReqChunkLen,ReqChunkLast);
                assert(ret_value);
                // Regression test
                assert(m_req.MCmd == OCP_MCMD_RD);
                assert(ReqChunkLen == 20); assert(ReqChunkLast == true);
                // cout << "Slave ends getting request 4 -- " << sc_time_stamp() << endl;
                // Delay before issuing the response
                // wait(100,SC_NS);
                // ---------------------------------------------------------
                // Sends the response using only one chunk 
                // ---------------------------------------------------------
                // Set the members of the response group stucture
                m_resp[3].SResp          =     OCP_SRESP_DVA;      // Response is VALID
                m_resp[3].SDataPtr       =     data;               // pass a pointer to the slave data
                m_resp[3].SRespInfo      =     1;                  // Info for the first chunk

                // Sends the first and only chunk 
                assert(RespChunkLen[3] == 0);
                RespChunkLen[3] = 20; RespChunkLast[3] = true;

                // try to ship this new respone out
                tryToSendResponse();
                m_req_num = 1;
                break;

            default:
                cout << "ERROR: slave reached unknown request number = " << m_req_num << endl;
        }

    }

    void releaseRequest_method()
    {
        ret_value = ocp->putSCmdAccept();
        assert(ret_value);
    }

    void send_response_method()
    {
        // Response Channel is free - try to send next response
        tryToSendResponse();
    }

    void tryToSendResponse()
    {
        if (RespChunkLen[m_resp_num] == 0)
        {
            // Give up - our response is not ready yet
            return;
        }
        // cout << "Slave trying to send response " << m_resp_num << " at time " << sc_time_stamp() << endl;
        if ( ocp->sendOCPResponse(m_resp[m_resp_num],RespChunkLen[m_resp_num],RespChunkLast[m_resp_num]) )
        {
            // Success - response is on the channel
            // cout << "Slave was able to send response " << m_resp_num << " at time " << sc_time_stamp() << endl;
            RespChunkLen[m_resp_num]=0;
            m_resp_num++;
            if (m_resp_num == 4)
            {
                m_resp_num = 1;
                m_loops_complete++;
                if( m_loops_complete == LOOP_COUNT)
                {
                    cout << "Slave DONE" << endl;
                }
            }
        }
    }

    // ----------------------------------------------------------------------------
    // constuctor
    // ----------------------------------------------------------------------------
    slave(sc_module_name mod):
        sc_module(mod),
    ocp("Sale_Port")
    {
        SC_METHOD(get_request_method);
          sensitive << ocp.RequestStartEvent();
          dont_initialize();
        SC_METHOD(releaseRequest_method);
          sensitive << m_releaseRequestEvent;
          dont_initialize();
        SC_METHOD(send_response_method);
          sensitive << ocp.ResponseEndEvent();
          dont_initialize();

        // initialize variables
        for (int i=0;i<4;i++)
        {
            m_resp[i].reset();  
            RespChunkLen[i] =0;
            RespChunkLast[i] =true;
        }
        for (int i=0;i<100;i++)
        {
            data[i]=9000+i;
        }
        m_req_num=1;
        m_resp_num=1;
        m_loops_complete=0;
    }

    // ----------------------------------------------------------------------------
    // Internal class members 
    // ----------------------------------------------------------------------------
    private:
    // slave memory 
    unsigned int data[100];

    // OCP Request/Response structures
    OCPRequestGrp<unsigned int, unsigned int> m_req;
    OCPResponseGrp<unsigned int> m_resp[4];
    unsigned int RespChunkLen[4];
    bool RespChunkLast[4];
    sc_event m_releaseRequestEvent;
    sc_event m_newResponseAvailableEvent;
    bool release_immediately;
    bool ret_value;
    unsigned int ReqChunkLen;
    bool ReqChunkLast;
    int m_req_num;
    int m_resp_num;
    int m_loops_complete;
    
};

// ----------------------------------------------------------------------------
// MAIN PROGRAM
// ----------------------------------------------------------------------------

void readMapFromFile(const string &myFileName, MapStringType &myParamMap) 
{
    // read pairs of data from the passed file
    string leftside;
    string rightside;
    
    // (1) open the file
    ifstream inputfile(myFileName.c_str());
    assert( inputfile );

    // set the formatting
    inputfile.setf(std::ios::skipws);

    // Now read through all the pairs of values and add them to the passed map
    while ( inputfile ) {
        inputfile >> leftside;
        inputfile >> rightside;
        myParamMap.insert(std::make_pair(leftside,rightside));
    }

    // All done, close up
    inputfile.close();
}

  // Performs some basic checks on the parameter values
  // To be completed....
  bool ConfigChecking(OCPParameters* pcl) {
     bool configError = false;
      
     // 'Exact ThreadBusy' conditions (OCP 2.0 spec. page 49)
     if (pcl->sthreadbusy_exact==1 && !(pcl->cmdaccept == 0) )
     {
         // cout <<  " OCP parameter error: " << endl;
         // cout << "      sthreadbusy_exact==1 and cmdaccept parameter is not 0" ;
         configError = true;
     }
     if (pcl->sthreadbusy_exact==1 && pcl->datahandshake==1 && pcl->sdatathreadbusy==0 && !(pcl->dataaccept == 0) )
     {
         // cout <<  " OCP parameter error: " << endl;
         // cout << "      sthreadbusy_exact==1,datahandshake==1,sdatathreadbusy==0 and dataaccept parameter is not 0" ;
         configError = true;
     }
     if (pcl->sdatathreadbusy_exact==1 && !(pcl->dataaccept == 0) )
     {
         // cout <<  " OCP parameter error: " << endl;
         // cout << "      sdatathreadbusy_exact==1 and dataaccept parameter is not 0" ;
         configError = true;
     }
     if (pcl->mthreadbusy_exact==1 && !(pcl->respaccept == 0) )
     {
         // cout <<  " OCP parameter error: " << endl;
         // cout << "      mthreadbusy_exact==1 and respaccept parameter is not 0" ;
         configError = true;
     }
     if (pcl->sthreadbusy_exact==0 && pcl->cmdaccept== 0 && pcl->sthreadbusy==1 )
     {
         // cout <<  " OCP parameter error: " << endl;
         // cout << "      sthreadbusy_exact is set to 0, cmdaccept is set to 0, but sthreadbusy is set to 1" ;
         configError = true;
     }
     if (pcl->mthreadbusy_exact==0 && pcl->respaccept== 0 && pcl->mthreadbusy==1 )
     {
         // cout <<  " OCP parameter error: " << endl;
         // cout << "      mthreadbusy_exact is set to 0, respaccept is set to 0, but mthreadbusy is set to 1" ;
         configError = true;
     }

     return !configError;
      
  };



int sc_main(int, char*[])
{

    // Creates the OCP TL2 channel
    OCP_TL2_Channel<unsigned int, unsigned int>  ch0("ch0");

    // Set the OCP parameters for this channel
    MapStringType  ocpParamMap;
    readMapFromFile("ocpParams", ocpParamMap);
    ch0.setConfiguration(ocpParamMap);

    // performs some basic checks on the OCP parameter values
    // Old style OCP parameter class:
    // ParamCl<OCP_TL2_DataCl<unsigned int, unsigned int> > *pcl = ch0.GetParamCl();
    // New style OCP parameter class (no template):
    OCPParameters *pcl = ch0.GetParamCl(); 
    assert(ConfigChecking(pcl));

    // Creates masters and slaves 
    slave sl1("sl1");
    master ms1("ms1");

    // Connect masters and slaves using channels
    ms1.ocp(ch0);
    sl1.ocp(ch0);

    // Starts simulation 
    sc_start(END_TIME, SC_MS);

    return(0);
}
