///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2009-2010
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : Herve Alexanian - Sonics, inc.
//
//          $Id:
//
//  Description :  This file defines a layer adapter module for the TL1 slave
//                 socket. Designed to handler incoming TL1 traffic with certain
//                 user controls and a plug-in for OCP slave behavior.
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

template<unsigned int BUSWIDTH> OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::ocp_tl1_tl3_master_adapter (sc_core::sc_module_name name_):
  sc_core::sc_module (name_)
  , m_timing_guard( "timing_guard" )
  , clk           ( "clk")
  , tl1_socket    ( "slave_tl1", this,  &ocp_tl1_tl3_master_adapter::set_tl1_master_timing )
  , master_socket ( "master_socket", ocp_tl3)
  , m_invariant_ext_pool       (10)
  , m_position_ext_pool        (10)
  , m_req_accept               ( "req_accept"  )
  , m_data_accept              ( "data_accept" )
  , m_p_resp_delay_calc        ( 0 )
  , m_p_req_acc_delay_calc     ( 0 )
  , m_p_data_acc_delay_calc    ( 0 )
  , m_phase_delay_peq          ( "phase_delay_peq" )
  , m_cur_mthreadbusy          ( 0 )
  , m_pipelined_mthreadbusy    ( 0 )
  , m_tl3_peq                  ( this, &ocp_tl1_tl3_master_adapter::nb_process_tl3, &master_socket )
{
    tl1_socket.register_nb_transport_fw(this, &ocp_tl1_tl3_master_adapter::nb_transport_tl1);
    tl1_socket.activate_synchronization_protection();
    m_tb_txn=tl1_socket.get_tb_transaction();
    m_tb=tl1_socket.template get_extension<cmd_thread_busy>(*m_tb_txn);
    m_dtb=tl1_socket.template get_extension<data_thread_busy>(*m_tb_txn);
    master_socket.register_nb_transport_bw(this, &ocp_tl1_tl3_master_adapter::nb_transport_tl3);

    SC_METHOD( exert_req_flow_control      );
    SC_METHOD( exert_data_flow_control     );
    SC_METHOD( pipeline_threadbusy_signals );

    SC_THREAD( process_arb_th );
    SC_THREAD( resp_arb_th    );

    SC_METHOD( phase_delay_peq_out );
    sensitive << m_phase_delay_peq.get_event();
}

template<unsigned int BUSWIDTH>
void
OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::reset()
{
    m_resp_issued           = NULL;
    m_req_pending_acc.txn   = NULL;
    m_data_pending_acc.txn  = NULL;
    m_next_sthreadbusy      = m_cur_sthreadbusy     = 0;
    m_next_sdatathreadbusy  = m_cur_sdatathreadbusy = 0;
    m_num_active_req        = std::vector<uint32_t>( ocp_params.threads, 0 );
    m_num_active_resp_data  = std::vector<uint32_t>( ocp_params.threads, 0 );
    m_num_active_write_data = std::vector<uint32_t>( ocp_params.threads, 0 );
    m_phase_delay_peq.reset();

    // clear all queues
    for ( typename std::vector<std::vector<phase_queue*> >::iterator it = m_req_txn.begin();
          it != m_req_txn.end(); ++it ) {
        for ( typename std::vector<phase_queue*>::iterator vit = it->begin(); vit != it->end(); ++vit )
            (*vit)->clear();
    }
    for ( typename std::vector<std::vector<phase_queue*> >::iterator it = m_resp_txn.begin();
          it != m_resp_txn.end(); ++it ) {
        for ( typename std::vector<phase_queue*>::iterator vit = it->begin(); vit != it->end(); ++vit )
	    (*vit)->clear();
    }
    for ( typename std::vector<phase_queue*>::iterator vit = m_data_txn.begin(); vit != m_data_txn.end(); ++vit ) {
	(*vit)->clear();
    }
}

template<unsigned int BUSWIDTH>
OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::~ocp_tl1_tl3_master_adapter()
{
    // delete queues
    for ( typename std::vector<std::vector<phase_queue*> >::iterator it = m_req_txn.begin();
          it != m_req_txn.end(); ++it ) {
        for ( typename std::vector<phase_queue*>::iterator vit = it->begin(); vit != it->end(); ++vit )
            delete *vit;
    }
    for ( typename std::vector<std::vector<phase_queue*> >::iterator it = m_resp_txn.begin();
          it != m_resp_txn.end(); ++it ) {
        for ( typename std::vector<phase_queue*>::iterator vit = it->begin(); vit != it->end(); ++vit )
            delete *vit;
    }
    for ( typename std::vector<phase_queue*>::iterator vit = m_data_txn.begin(); vit != m_data_txn.end(); ++vit ) {
	delete *vit;
    }
}

template<unsigned int BUSWIDTH>
void OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::before_end_of_elaboration()
{
    ocp_params = tl1_socket.get_ocp_config();
    int effective_tags = ocp_params.tags + ocp_params.taginorder;
    if ( effective_tags == 0 ) // should not happen as 1 is the min number of tags
	effective_tags = 1;    // just to be defensive

    m_req_txn        .resize( ocp_params.threads, std::vector<phase_queue*>( effective_tags) );
    m_data_txn       .resize( ocp_params.threads );
    m_resp_txn       .resize( ocp_params.threads, std::vector<phase_queue*>( effective_tags) );
    m_tl3_txn        .resize( ocp_params.threads, std::vector<ocp_tl1_phase_pos>( effective_tags ) );
    m_tag_process_arb.resize( ocp_params.threads );
    m_tag_resp_arb   .resize( ocp_params.threads );
    m_burst_tracker  .resize( ocp_params.threads, std::vector<ocp_txn_track>(
				 effective_tags, ocp_txn_track( &this->ocp_params, true ) ) );

    // arbiter setup and filters
    m_thread_process_arb.set_range( 0, ocp_params.threads );
    m_thread_resp_arb.set_range   ( 0, ocp_params.threads );
    if ( ocp_params.mthreadbusy ) {
	 uint32_t& tb_word = ( ocp_params.mthreadbusy_pipelined ) ? m_pipelined_mthreadbusy : m_cur_mthreadbusy;
	 m_mthreadbusy_filt.set_busy_word( tb_word );
	 m_thread_resp_arb.add_filter( m_mthreadbusy_filt );
    }
    for ( int thread=0; thread < ocp_params.threads; ++thread ) {
	m_data_txn       [thread] = new phase_queue( 0 );
	m_tag_process_arb[thread].set_range ( 0, effective_tags );
	m_tag_resp_arb   [thread].set_range ( 0, effective_tags );
	m_tag_resp_arb   [thread].add_filter( arbiter_interleave_filter<tag_type>(
					       ocp_params.tag_interleave_size ) );
	// per thread/per tag queues
	for ( int tag=0; tag < effective_tags; ++tag ) {
	    m_req_txn [thread][tag] = new phase_queue( 0 );
	    m_resp_txn[thread][tag] = new phase_queue( 0 );
	}
    }

    // phase accept
    if ( ocp_params.cmdaccept ) {
	m_req_accept.register_callback ( this, &ocp_tl1_tl3_master_adapter<BUSWIDTH>::accept_req );
    }
    if ( ocp_params.dataaccept ) {
	m_data_accept.register_callback( this, &ocp_tl1_tl3_master_adapter<BUSWIDTH>::accept_data );
    }

    m_timing_guard   .m_clk( clk );
    m_phase_delay_peq.m_clk( clk );
    m_req_accept     .m_clk( clk );
    m_data_accept    .m_clk( clk );

    update_timing();
    reset();
}

template<unsigned int BUSWIDTH>
void
OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::update_timing()
{
    ocp_tl1_slave_timing my_tl1_timing;
    my_tl1_timing.ResponseGrpStartTime =
	sc_core::sc_get_time_resolution() + m_timing_guard.m_master_timing.MThreadBusyStartTime;

    my_tl1_timing.SThreadBusyStartTime = sc_core::SC_ZERO_TIME;
    my_tl1_timing.SCmdAcceptStartTime  = sc_core::SC_ZERO_TIME;

    if ( my_tl1_timing != m_my_tl1_timing ) {
	m_my_tl1_timing = my_tl1_timing;
	tl1_socket.set_slave_timing( m_my_tl1_timing );
    }
}

template<unsigned int BUSWIDTH>
void
OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::set_tl1_master_timing(
    OCPIP_VERSION::ocp_tl1_master_timing ms_timing )
{
    m_timing_guard.receive_master_timing( ms_timing );
    update_timing();
}

template<unsigned int BUSWIDTH>
tlm::tlm_sync_enum
OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::nb_transport_tl1(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim) {

    if (ph==RESP_THREAD_BUSY_CHANGE){
	resp_thread_busy* p_tb = tl1_socket.template get_extension<resp_thread_busy>( txn );
        m_cur_mthreadbusy = p_tb->value;
    } else if ( ph==BEGIN_RESET || ph==END_RESET ) {
	reset();
    } else { // dataflow
	ocp_txn_burst_invariant* p_inv = check_ispec_extension<ocp_txn_burst_invariant>( txn, acc );
	ocp_txn_position*        p_pos = check_ispec_extension<ocp_txn_position>       ( txn, acc );
	if ( p_pos == NULL ) {
	    // first time seen: make new tracker and invariant inst extensions
	    assert( p_inv == NULL );
	    p_inv  = m_invariant_ext_pool.create();
	    p_pos  = m_position_ext_pool.create();
	    p_pos->clear();
	    *p_inv = ocp_txn_burst_invariant::init_from( txn, ocp_params );
	    acc(txn).set_extension( p_inv );
	    acc(txn).set_extension( p_pos );
	    assert( ph == tlm::BEGIN_REQ );
	    txn.acquire();
	}
	assert( p_inv );
	assert( p_pos );
	const thread_type thread = p_inv->threadid;
	const tag_type    tag    = p_inv->taginorder ? ocp_params.tags : p_inv->tagid;

	if ( ph == tlm::END_RESP ) {
	    handle_end_resp( txn );
	} else if ( ph == tlm::BEGIN_REQ ) {
	    m_req_pending_acc.txn = &txn;
	    m_req_pending_acc.delay = 0;
	    m_req_pending_acc.position = m_burst_tracker[thread][tag].track_phase( txn, ph );
	    if ( !ocp_params.cmdaccept ) { // mus t end req phase on return
		accept_req( m_req_pending_acc );
		m_req_pending_acc.txn = NULL;
		ph = tlm::END_REQ;
		return tlm::TLM_UPDATED;
	    } else {
		// schedule acceptance, will result in callback to accept_req()
		if ( m_p_req_acc_delay_calc )
		    m_req_pending_acc.delay = m_p_req_acc_delay_calc->calc( m_req_pending_acc.position, &txn );
		m_req_accept.accept( m_req_pending_acc, m_req_pending_acc.delay );
	    }
	} else if ( ph == BEGIN_DATA ) {
	    m_data_pending_acc.txn = &txn;
	    m_data_pending_acc.delay = 0;
	    m_data_pending_acc.position = m_burst_tracker[thread][tag].track_phase( txn, ph );
	    if ( !ocp_params.dataaccept ) { // must end data phase on return
		accept_data( m_data_pending_acc );
		ph = END_DATA;
		return tlm::TLM_UPDATED;
	    } else {
		// schedule acceptance, will result in callback to accept_data()
		if ( m_data_pending_acc.txn == m_req_pending_acc.txn &&
		     ( m_req_pending_acc.position.count == m_data_pending_acc.position.count ) ) {
		    // same phase is pending request accept, accept_req() will schedule data accept.
		    // this should also catch all reqdata_together cases
		} else {
		    if ( m_p_data_acc_delay_calc )
			m_data_pending_acc.delay = m_p_data_acc_delay_calc->calc( m_data_pending_acc.position, &txn );
		    m_data_accept.accept( m_data_pending_acc, m_data_pending_acc.delay );
		}
	    }
	} else {
	    std::cerr<<"Error in "<<name()<<" : got unexpected phase "<<ph<<" in nb_transport_fw"<<std::endl;
	    exit(1);
	}
    }
    return tlm::TLM_ACCEPTED;
}

template<unsigned int BUSWIDTH>
tlm::tlm_sync_enum
OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::nb_transport_tl3(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim)
{
    // per TL3 Communication guidelines of modeling kit, only handle BEGIN_RESP and complete txn
    if ( ph == tlm::BEGIN_RESP ) {
	txn.acquire();
	m_tl3_peq.notify( txn, ph, tim );
	ph = tlm::END_RESP;
	return tlm::TLM_COMPLETED;
    }
    return tlm::TLM_ACCEPTED;
}

template<unsigned int BUSWIDTH>
void
OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::nb_process_tl3(tlm::tlm_generic_payload& txn, const tlm::tlm_phase&)
{
    ocp_txn_burst_invariant* p_inv = require_ispec_extension<ocp_txn_burst_invariant>( txn, acc );
    const thread_type thread = p_inv->threadid;
    const tag_type    tag    = p_inv->taginorder ? ocp_params.tags : p_inv->tagid;
    assert( m_tl3_txn[thread][tag].txn == &txn );
    process_phase( m_tl3_txn[thread][tag] );
    m_tl3_txn[thread][tag].txn = NULL;
    txn.release();
}

template<unsigned int BUSWIDTH>
void OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::process_arb_th()
{
    const tag_type no_tag_winner = static_cast<tag_type>( -1 );
    const thread_type threads = m_req_txn.size();
    const tag_type    tags    = m_req_txn[0].size();

    while ( true ) {
	// after the nb_transport accounting for synchronization PEQ at input
	if ( !m_timing_guard.is_request_stable() ) {
	    wait( m_timing_guard.time_to_request_stable() );
	    wait( sc_core::sc_get_time_resolution() );
	}

	bool all_empty = true;
	for ( thread_type thread=0; thread < threads; ++thread ) {
	    for ( tag_type tag=0; tag < tags; ++tag ) {
		if ( m_req_txn[thread][tag]->nb_can_peek() ) {
		    all_empty = false; break;
		}
	    }
	}
	if ( all_empty ) {
	    wait( m_new_process_input_ev );
	    continue; // something just came in, it can go in this cycle
	}
	wait( sc_core::SC_ZERO_TIME );

	// Tag arbitration
	std::vector<tag_type> tag_winner( threads, no_tag_winner );
	std::set<thread_type> thread_candidates;
	for ( thread_type thread=0; thread < threads; ++thread ) {
	    std::set<tag_type> tag_candidates;
	    for ( tag_type tag=0; tag < tags; ++tag ) {
		if ( !m_req_txn[thread][tag]->nb_can_peek() ) {
		    continue;
		}
		// 1) not ready: dh write request with pending data
		ocp_tl1_phase_pos req_phase;
		m_req_txn[thread][tag]->nb_peek( req_phase );
		if ( req_phase.txn->is_write() && ocp_params.datahandshake ) {
		    if ( !m_data_txn[thread]->nb_can_peek() ) {
			continue;
		    }
		}
		// 2) not ready, this thread/tag is pending on TL3 socket
		if ( m_tl3_txn[thread][tag].txn ) {
		    continue;
		}
		// TODO: handle address overlapping here
		tag_candidates.insert( tag );
	    }
	    if ( !tag_candidates.empty() && m_tag_process_arb[thread]( tag_candidates, tag_winner[thread] ) ) {
		thread_candidates.insert( thread );
	    }
	}

	// Thread arbitration
	thread_type winner;
	if ( m_thread_process_arb( thread_candidates, winner ) ) {
	    // arbitration yielded a winner, send it to processing and update arb winners
	    ocp_tl1_phase_pos req_phase;
	    m_req_txn[winner][tag_winner[winner]]->nb_peek( req_phase );
	    assert( req_phase.txn != NULL );
	    if ( req_phase.txn->is_write() && ocp_params.datahandshake ) {
		// process the data phase. By protocol, it must be at the head of the data queue
		ocp_tl1_phase_pos data_phase;
		bool success = m_data_txn[winner]->nb_peek( data_phase );
		assert( success );
		assert( data_phase.txn == req_phase.txn );
		dispatch_phase( data_phase );
	    } else {
		dispatch_phase( req_phase );
	    }
	    m_thread_process_arb.winner( winner );
	    m_tag_process_arb[winner].winner( tag_winner[winner] );
	}
	wait ( clk->posedge_event() );
    }
}

template<unsigned int BUSWIDTH>
void OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::resp_arb_th()
{
    const tag_type no_tag_winner = static_cast<tag_type>( -1 );
    const thread_type threads = m_resp_txn.size();
    const tag_type    tags    = m_resp_txn[0].size();

    while ( true ) {
	// after the nb_transport accounting for synchronization PEQ at input
	if ( !m_timing_guard.is_request_stable() ) {
	    wait( m_timing_guard.time_to_request_stable() );
	    wait( sc_core::sc_get_time_resolution() );
	}
	bool all_empty = true;
	for ( thread_type thread=0; thread < threads; ++thread ) {
	    for ( tag_type tag=0; tag < tags; ++tag ) {
		if ( m_resp_txn[thread][tag]->nb_can_peek() ) {
		    all_empty = false; break;
		}
	    }
	}
	if ( all_empty ) {
	    wait( m_new_resp_input_ev );
	    continue; // something just came in, it can go in this cycle
	}
	if ( m_resp_issued != NULL ) {
	    wait( m_resp_accept_ev );
	    wait( clk->posedge_event() );
	    continue;
	}

	wait( sc_core::SC_ZERO_TIME );
	if ( ocp_params.mthreadbusy && ocp_params.mthreadbusy_exact )
	    wait_mthreadbusy_stable();

	// Tag arbitration
	std::vector<tag_type> tag_winner( threads, no_tag_winner );
	std::set<thread_type> thread_candidates;
	for ( thread_type thread=0; thread < threads; ++thread ) {
	    std::set<tag_type> tag_candidates;
	    for ( tag_type tag=0; tag < tags; ++tag ) {
		ocp_tl1_phase_pos resp_phase;
		if ( !m_resp_txn[thread][tag]->nb_peek( resp_phase ) ) {
		    continue;
		}
		// not ready: elapsed time
		if ( resp_phase.delay != 0 ) {
		    continue;
		}
		tag_candidates.insert( tag );
	    }
	    if ( !tag_candidates.empty() && m_tag_resp_arb[thread]( tag_candidates, tag_winner[thread] ) ) {
		thread_candidates.insert( thread );
	    }
	}

	// Thread arbitration
	thread_type winner;
	if ( m_thread_resp_arb( thread_candidates, winner ) ) {
	    // arbitration yielded a winner, send it and update arb winners
	    send_response( winner, tag_winner[winner] );
	    m_thread_resp_arb.winner( winner );
	    m_tag_resp_arb[winner].winner( tag_winner[winner] );
	}
	wait ( clk->posedge_event() );
    }
}

template<unsigned int BUSWIDTH>
void OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::pipeline_threadbusy_signals() {
    if ( !ocp_params.mthreadbusy_exact )
	return;
    next_trigger( clk->posedge_event() );
    m_pipelined_mthreadbusy     = m_cur_mthreadbusy;
}

template<unsigned int BUSWIDTH>
void OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::wait_mthreadbusy_stable()
{
    if ( ocp_params.mthreadbusy && ocp_params.mthreadbusy_exact ) {
	sc_core::sc_time wait_time = sc_core::sc_get_time_resolution(); // because of peq protection
	if ( !m_timing_guard.is_mthreadbusy_stable() ) {
	    sc_core::sc_time time_to_stable = m_timing_guard.time_to_mthreadbusy_stable();
	    if ( time_to_stable > wait_time )
		wait_time = time_to_stable;
	}
	wait( wait_time + sc_core::sc_get_time_resolution() ); // exceed stable time by 1 res to be safe
    }
}

template<unsigned int BUSWIDTH>
void OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::dispatch_phase( ocp_tl1_phase_pos& tl1_phase )
{
    tlm::tlm_generic_payload& txn = *tl1_phase.txn;
    // execute the TL3 read on first phase, write on last
    // TODO: for imprecise or non-deterministic address sequences, the early
    // read may not work. Either read on each beat, or chop to single transfers
    if ( ( txn.is_read()  && tl1_phase.position.count  == 1 ) ||
	 ( txn.is_write() && tl1_phase.position.remain == 0 ) ) {
	phase = tlm::BEGIN_REQ;
	time  = sc_core::SC_ZERO_TIME;
	tlm::tlm_sync_enum status = master_socket->nb_transport_fw( txn, phase, time );
	ocp_txn_burst_invariant* p_inv = require_ispec_extension<ocp_txn_burst_invariant>( txn, acc );
	const thread_type thread = p_inv->threadid;
	const tag_type    tag    = p_inv->taginorder ? ocp_params.tags : p_inv->tagid;
	m_tl3_txn[thread][tag]   = tl1_phase;
	// phase processing handled by TL3 return or backward call
	if ( status == tlm::TLM_COMPLETED || phase == tlm::BEGIN_RESP ) {
	    phase = tlm::BEGIN_RESP; // only relevant phase at TL3
	    txn.acquire();
	    m_tl3_peq.notify( txn, phase, time );
	}
    } else {
	// TL1-only phase
	process_phase( tl1_phase );
    }
}


template<unsigned int BUSWIDTH>
void OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::process_phase( ocp_tl1_phase_pos& tl1_phase )
{
    tlm::tlm_generic_payload& txn = *tl1_phase.txn;
    ocp_txn_burst_invariant* p_inv = require_ispec_extension<ocp_txn_burst_invariant>( txn, acc );
    const thread_type thread = p_inv->threadid;
    const tag_type    tag    = p_inv->taginorder ? ocp_params.tags : p_inv->tagid;

    bool has_resp = ( p_inv->cmd != ocpip_legacy::OCP_MCMD_WR || ocp_params.writeresp_enable );
    if ( txn.is_write() && p_inv->srmd ) { // only last data phase of srmd write can yield a response.
	has_resp &= tl1_phase.position.remain == 0;
    }
    bool last_read_resp = false;
    if ( has_resp ) { // enqueue 1 response phase or N for SRMD read
	ocp_tl1_phase_pos resp_phase;
	resp_phase.txn = tl1_phase.txn;
	uint32_t num_resp_phases = ( ( p_inv->srmd && txn.is_read() ) ?
				     p_inv->burst_length : 1 );
	for ( uint32_t i=0; i<num_resp_phases; ++i ) {
	    resp_phase.position = m_burst_tracker[thread][tag].track_phase(
		txn, tlm::BEGIN_RESP );
	    last_read_resp = resp_phase.position.remain == 0;
	    resp_phase.delay = 0;
	    if ( m_p_resp_delay_calc ) {
		resp_phase.delay = m_p_resp_delay_calc->calc( resp_phase.position, resp_phase.txn );
	    }
	    m_resp_txn[thread][tag]->nb_put( resp_phase );
	    if ( i == 0 && resp_phase.delay > 0 ) {
		m_phase_delay_peq.notify( m_resp_txn[thread][tag]->back().delay,
					  resp_phase.delay );
	    }
	    ++m_num_active_resp_data[thread];
	}
	if ( m_num_active_resp_data[thread] >= config.max_active_resp_data ) {
	    m_req_flow_control_ev.notify( sc_core::SC_ZERO_TIME );
	}
	m_new_resp_input_ev.notify( sc_core::SC_ZERO_TIME );
    }

    ocp_tl1_phase_pos completed_phase;
    if ( txn.is_write() && ocp_params.datahandshake ) {
	m_data_txn[thread]->nb_get( completed_phase );
	assert( m_num_active_write_data[thread] > 0 );
	if ( m_num_active_write_data[thread]-- == config.max_active_write_data ) {
	    m_data_flow_control_ev.notify( sc_core::SC_ZERO_TIME );
	}
	assert( completed_phase.txn == tl1_phase.txn );
	assert( completed_phase.position.count == tl1_phase.position.count );
    }
    // dequeue req phase, but wait for last data on SRMD write
    bool dequeue_req = true;
    if ( txn.is_write() && p_inv->srmd ) {
	// tl1_phase is a DH phase
	dequeue_req = tl1_phase.position.remain == 0;
    }
    if ( dequeue_req ) {
	m_req_txn[thread][tag]->nb_get( completed_phase );
	assert( completed_phase.txn == tl1_phase.txn );
	assert( m_num_active_req[thread] > 0 );
	if ( m_num_active_req[thread]-- == config.max_active_req ) {
	    m_req_flow_control_ev.notify( sc_core::SC_ZERO_TIME );
	}
    }
    if ( !has_resp && tl1_phase.position.remain == 0 ) {
	release( txn );
    }
}


template<unsigned int BUSWIDTH>
void OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::send_response( thread_type thread, tag_type tag )
{
    phase_queue::iterator respIt = m_resp_txn[thread][tag]->begin();
    assert( respIt != m_resp_txn[thread][tag]->end() );
    ocp_tl1_phase_pos resp_phase = *respIt;
    assert( resp_phase.txn != NULL );
    tlm::tlm_generic_payload& txn = *resp_phase.txn;
    ocp_txn_position*         p_pos = require_ispec_extension<ocp_txn_position>( txn, acc );
    ocp_txn_burst_invariant*  p_inv = require_ispec_extension<ocp_txn_burst_invariant>( txn, acc );

    // handle TL1 response
    txn.set_response_status( tlm::TLM_OK_RESPONSE );
    time=sc_core::SC_ZERO_TIME;
    phase=tlm::BEGIN_RESP;
    p_pos->resp_position = resp_phase.position;

    // kick off the next resp phase delay if more responses in burst
    if ( m_p_resp_delay_calc && p_inv->srmd && p_pos->resp_position.remain > 0 ) {
	respIt++;
	assert( respIt != m_resp_txn[thread][tag]->end() );
	m_phase_delay_peq.notify( respIt->delay, respIt->delay );
    }

    tlm::tlm_sync_enum retVal = tl1_socket->nb_transport_bw(txn, phase, time);
    switch(retVal){
    case tlm::TLM_ACCEPTED: m_resp_issued = &txn; break;
    case tlm::TLM_UPDATED:
	switch (phase){
	case tlm::END_RESP: {
	    handle_end_resp( txn );
	    break;
	}
	default:
	    std::cerr<<"Error in "<<name()<<" : got unexpected phase update to "<<phase<<" in response to BEGIN_RESP."<<std::endl;
	    exit(1);
	}
	break;
    case tlm::TLM_COMPLETED:;
    }
}

template<unsigned int BUSWIDTH>
void OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::phase_delay_peq_out()
{
    int* pit;
    while ( ( pit = m_phase_delay_peq.get_next_transaction() ) != NULL ) {
	(*pit) = 0; // points to a delay
    }
}

template<unsigned int BUSWIDTH>
void
OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::accept_req( OCPIP_VERSION::ocp_tl1_phase_pos& req_phase )
{
    tlm::tlm_generic_payload& txn = *req_phase.txn;
    ocp_txn_burst_invariant* p_inv = require_ispec_extension<ocp_txn_burst_invariant>( txn, acc );
    ocp_txn_position*        p_pos = require_ispec_extension<ocp_txn_position>       ( txn, acc );
    const thread_type thread = p_inv->threadid;
    const tag_type    tag    = p_inv->taginorder ? ocp_params.tags : p_inv->tagid;

    p_pos->req_position = req_phase.position;
    if ( ocp_params.cmdaccept ) {
	if ( !can_accept_req( req_phase ) ) {
            // off the delay clock, accept will have to be triggered by exert_req_flow_control()
	    m_req_pending_acc.delay = -1;
	    return;
	}
	tlm::tlm_phase ph = tlm::END_REQ;
	sc_core::sc_time zero;
	tl1_socket->nb_transport_bw( txn, ph, zero );
    }
    // make an entry for processing
    m_req_txn[thread][tag]->nb_put( req_phase );
    m_new_process_input_ev.notify( sc_core::SC_ZERO_TIME );
    ++m_num_active_req[thread];
    if ( ocp_params.sthreadbusy && m_num_active_req[thread] == config.max_active_req ) {
	m_req_flow_control_ev.notify( sc_core::SC_ZERO_TIME );
    }

    if ( m_data_pending_acc.txn == m_req_pending_acc.txn &&
	 ( m_req_pending_acc.position.count == m_data_pending_acc.position.count ) ) {
	// data accept for the same phase was not spawned for risk that it would go ahead
	// of this one. Spawn it now. Make sure there is no delay for reqdata_together
	assert( ocp_params.dataaccept );
	m_data_pending_acc.delay = 0;
	if ( m_p_data_acc_delay_calc && !ocp_params.reqdata_together )
	    m_data_pending_acc.delay = m_p_data_acc_delay_calc->calc( m_data_pending_acc.position, &txn );
	m_data_accept.accept( m_data_pending_acc, m_data_pending_acc.delay );
    }
    m_req_pending_acc.txn = NULL;
}

template<unsigned int BUSWIDTH>
void
OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::accept_data( OCPIP_VERSION::ocp_tl1_phase_pos& data_phase )
{
    ocp_txn_burst_invariant* p_inv = require_ispec_extension<ocp_txn_burst_invariant>( *data_phase.txn, acc );
    ocp_txn_position*        p_pos = require_ispec_extension<ocp_txn_position>       ( *data_phase.txn, acc );
    const thread_type thread = p_inv->threadid;
    p_pos->dh_position = data_phase.position;

    if ( ocp_params.dataaccept ) {
	if ( !can_accept_data( data_phase ) ) {
            // off the delay clock, accept will have to be triggered by exert_req_flow_control()
	    m_data_pending_acc.delay = -1;
	    return;
	}
	tlm::tlm_phase ph = END_DATA;
	sc_core::sc_time zero;
	tl1_socket->nb_transport_bw( *data_phase.txn, ph, zero );
    }

    // data phase bookkeeping
    m_data_txn[thread]->nb_put( data_phase );
    ++m_num_active_write_data[thread];
    if ( ocp_params.sdatathreadbusy && m_num_active_write_data[thread] == config.max_active_write_data ) {
	m_data_flow_control_ev.notify( sc_core::SC_ZERO_TIME );
    }

    m_data_pending_acc.txn = NULL;
}

template<unsigned int BUSWIDTH>
void
OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::exert_req_flow_control()
{
    // this gets notified when m_num_active_req or m_num_active_resp_data changes
    next_trigger( m_req_flow_control_ev );
    if ( ocp_params.cmdaccept ) {
	if ( m_req_pending_acc.txn && m_req_pending_acc.delay < 0 && can_accept_req( m_req_pending_acc ) ) {
	    accept_req( m_req_pending_acc );
	}
    } else if ( ocp_params.sthreadbusy ) {
	phase                 = CMD_THREAD_BUSY_CHANGE;
	sc_core::sc_time zero = sc_core::SC_ZERO_TIME;
	// calculate next sthreadbusy
	m_next_sthreadbusy = 0;
	for ( int thread=0; thread < ocp_params.threads; ++thread ) {
	    bool bit_busy = ( m_num_active_req[thread] >= config.max_active_req ||
			      m_num_active_resp_data[thread] >= config.max_active_resp_data );
	    m_next_sthreadbusy |= ( bit_busy << thread );
	}

	if ( m_next_sthreadbusy != m_cur_sthreadbusy ) {
	    if ( ocp_params.sthreadbusy_pipelined ) {
		// drive new value immediately for pipelined threadbusy
		m_cur_sthreadbusy = m_next_sthreadbusy;
		m_tb->value       = m_next_sthreadbusy;
		tl1_socket->nb_transport_bw(*m_tb_txn, phase, zero);
	    } else {
		next_trigger( m_req_flow_control_ev | clk.posedge_event() );
	    }
	}
	if ( clk.posedge() ) { // only drive non-pipelined threadbusy at clk edge
	    assert( !ocp_params.sthreadbusy_pipelined );
	    m_cur_sthreadbusy = m_next_sthreadbusy;
	    m_tb->value       = m_next_sthreadbusy;
	    tl1_socket->nb_transport_bw(*m_tb_txn, phase, zero);
	}
    } else {
	next_trigger(); // no flow control, don't wake up again
    }
}

template<unsigned int BUSWIDTH>
void
OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::exert_data_flow_control()
{
    // this gets notified when m_num_active_write_data
    next_trigger( m_data_flow_control_ev );
    if ( ocp_params.dataaccept ) {
	// accept data that is still pending.
	if ( m_data_pending_acc.txn && m_data_pending_acc.delay < 0 && can_accept_data( m_data_pending_acc ) ) {
	    // unless this is a reqdata_together, in which case accept_req will trigger accept_data
	    if ( !ocp_params.reqdata_together ||
		 m_data_pending_acc.txn != m_req_pending_acc.txn ||
		 m_data_pending_acc.position.count != m_req_pending_acc.position.count )
		accept_data( m_data_pending_acc );
	}
    } else if ( ocp_params.sdatathreadbusy ) {
	phase                 = DATA_THREAD_BUSY_CHANGE;
	sc_core::sc_time zero = sc_core::SC_ZERO_TIME;
	// calculate next sdatathreadbusy
	m_next_sdatathreadbusy = 0;
	for ( int thread=0; thread < ocp_params.threads; ++thread ) {
	    bool bit_busy = ( m_num_active_write_data[thread] >= config.max_active_write_data );
	    m_next_sdatathreadbusy |= ( bit_busy << thread );
	}

	if ( m_next_sdatathreadbusy != m_cur_sdatathreadbusy ) {
	    if ( ocp_params.sdatathreadbusy_pipelined ) {
		// drive new value immediately for pipelined threadbusy
		m_cur_sdatathreadbusy = m_next_sdatathreadbusy;
		m_dtb->value  = m_next_sdatathreadbusy;
		tl1_socket->nb_transport_bw(*m_tb_txn, phase, zero);
	    } else {
		next_trigger( m_data_flow_control_ev | clk.posedge_event() );
	    }
	}
	if ( clk.posedge() ) { // only drive non-pipelined threadbusy at clk edge
	    assert( !ocp_params.sdatathreadbusy_pipelined );
	    m_cur_sdatathreadbusy = m_next_sdatathreadbusy;
	    m_dtb->value          = m_next_sdatathreadbusy;
	    tl1_socket->nb_transport_bw(*m_tb_txn, phase, zero);
	}
    } else {
	next_trigger(); // no flow control, don't wake up again
    }
}

template<unsigned int BUSWIDTH>
bool
OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::can_accept_req( const ocp_tl1_phase_pos& req_phase ) const
{
    ocp_txn_burst_invariant* p_inv = require_ispec_extension<ocp_txn_burst_invariant>( *req_phase.txn, acc );
    const thread_type thread = p_inv->threadid;
    if ( m_num_active_req[thread] >= config.max_active_req )
	return false;

    if ( req_phase.txn->is_write() ) {
	if ( !ocp_params.datahandshake && ( m_num_active_write_data[thread] >= config.max_active_write_data ) )
	    return false;
	if ( ( p_inv->cmd != ocpip_legacy::OCP_MCMD_WR || ocp_params.writeresp_enable ) &&
	     ( m_num_active_resp_data[thread] >= config.max_active_resp_data ) )
	    return false;
    } else if ( m_num_active_resp_data[thread] >= config.max_active_resp_data ) {
	return false;
    }
    return true;
}

template<unsigned int BUSWIDTH>
bool
OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::can_accept_data( const ocp_tl1_phase_pos& data_phase ) const
{
    ocp_txn_burst_invariant* p_inv = require_ispec_extension<ocp_txn_burst_invariant>( *data_phase.txn, acc );
    const thread_type thread = p_inv->threadid;
    return m_num_active_write_data[thread] < config.max_active_write_data;
}

template<unsigned int BUSWIDTH>
void
OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::handle_end_resp(tlm::tlm_generic_payload& txn)
{
    ocp_txn_burst_invariant* p_inv = require_ispec_extension<ocp_txn_burst_invariant>( txn, acc );
    const thread_type thread = p_inv->threadid;
    const tag_type    tag    = p_inv->taginorder ? ocp_params.tags : p_inv->tagid;

    if ( ocp_params.respaccept ) {
	m_resp_issued=NULL;
	m_resp_accept_ev.notify();
    }

    ocp_tl1_phase_pos resp_phase;
    m_resp_txn[thread][tag]->nb_get( resp_phase );
    assert( m_num_active_resp_data[thread] > 0 );
    if ( m_num_active_resp_data[thread]-- == config.max_active_resp_data ) {
	m_req_flow_control_ev.notify( sc_core::SC_ZERO_TIME );
    }
    assert( resp_phase.txn == &txn );
    if ( resp_phase.position.remain == 0 ) {
	release( txn );
    }
}

template<unsigned int BUSWIDTH>
void
OCPIP_VERSION::ocp_tl1_tl3_master_adapter<BUSWIDTH>::release(tlm::tlm_generic_payload& txn)
{
    ocp_txn_burst_invariant* p_inv = require_ispec_extension<ocp_txn_burst_invariant>( txn, acc );
    ocp_txn_position*        p_pos = require_ispec_extension<ocp_txn_position>       ( txn, acc );
    acc(txn).clear_extension( p_inv );
    m_invariant_ext_pool.recycle( p_inv );
    acc(txn).clear_extension( p_pos );
    m_position_ext_pool.recycle( p_pos );
    txn.release();
}
