///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2009-2011
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis, Texas Instruments
//
//          $Id:
//
//  Description :  TL1/TL0 OCP Master Adapter
//
// SystemC Module Adapting between a TLM2-based OCP-TL1 master interface
// and a TL0 (signal-level) OCP interface
// Typical use is between a TL1 OCP initiator and a TL0 OCP target
//
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "ocpip_adapters_dev.h"
// namespace
namespace ocpip_dev { namespace tl1_tl0 {

// WARNING: do not use the character sequence "_dev" in this file
// because it will be replaced by the installation process


using namespace tlm;
using namespace sc_core;
using sc_dt::uint64;


// private namespace for support infrastructure
namespace support_10 {

// base class for OCP burst tracker classes
struct tracker_base {
  virtual ~tracker_base() {}
  virtual void init(tlm_generic_payload &pl, const ocp_parameters &pm) = 0;

  // Req
  virtual unsigned BurstSeq() = 0;
  virtual bool Precise() = 0;
  virtual uint64 OCPAddr() = 0;
  virtual bool ReqDone(bool track) = 0;
  virtual bool ReqAcc() = 0;
  virtual bool ReqRowDone() {return ReqDone(false);};
  virtual bool ReqBE(unsigned bytelane) = 0;
  virtual unsigned char *ReqDataPtr() = 0;
  virtual unsigned BurstLen() = 0;
  virtual unsigned BlockHeight() {return 1;}
  virtual uint64 BlockStride() {return 0;}
  virtual unsigned ReqBeat() {return req.nr;}

  // WR data
  virtual bool WrDone(bool track) = 0;
  virtual bool WrAcc() = 0;
  virtual bool WrRowDone() {return WrDone(false);};
  virtual bool WrBE(unsigned bytelane) = 0;
  virtual unsigned char *WrDataPtr() = 0;
  virtual unsigned WrBeat() {return data.nr;}

  // RD data
  virtual bool RdDone(bool track) = 0;
  virtual bool RdAcc() = 0;
  virtual bool RdBE(unsigned bytelane) = 0;
  virtual unsigned char *RdDataPtr() = 0;
  virtual unsigned RdBeat() {return resp.nr;}

  // management
  virtual void return_to_pool() = 0;

  // members
  struct handshake {unsigned nr, done, acc;};
  handshake req, data, resp;
  uint64 ocp_addr;
  unsigned byte_wdth;
  tlm_generic_payload *txn;
};


// extension object used to store adapter state for each transaction
struct ext: tlm_extension<ext> {
  // TLM extension stuff
  void free() {delete this;}
  void copy_from(tlm_extension_base const &) {}
  tlm_extension_base *clone() const {return 0;}
  static ext &get(tlm_generic_payload &pl) {return *(pl.get_extension<ext>());}
  // transaction ordering support
  unsigned cmd, addr_space, conn_id, atomic_length;
  tlm_generic_payload *next, *prev;
  int tag; // use -1 for tag-in-order
  unsigned thread;
  // burst tracking
  tracker_base *tracker;
};


// function for initialising an extension, returns true on first call
bool init_ext(tlm_generic_payload &, const ocp_parameters&);


// wrapper for processes
struct process {
  void operator()() {(adapter->*pf)();}
  tl1_to_tl0_base *adapter;
  tl1_to_tl0_base::callable pf;
  process(
    tl1_to_tl0_base::callable p, tl1_to_tl0_base *a,
    const char *n, const sc_core::sc_event &e)
    : adapter(a), pf(p)
  {
    sc_spawn_options so;
    so.spawn_method();
    so.dont_initialize();
    so.set_sensitivity(&e);
    sc_spawn(*this, n, &so);
  }
};

// end of support namespace
} using namespace support_10;


// class methods
//////////////////////////////////////////


// process function for critical sc_signal writing
void tl1_to_tl0_base::writer_process() {
  if(mcmd_wr) mcmd(new_mcmd);
  if(mdatavalid_wr) mdatavalid(new_mdatavalid);
  if(mrespaccept_wr) mrespaccept(new_mrespaccept);
  mcmd_wr = mdatavalid_wr = mrespaccept_wr = false;
}


// constructor
tl1_to_tl0_base::tl1_to_tl0_base(sc_module_name n):
  sc_module(n),
  mcmd_wr(false),
  mdatavalid_wr(false),
  mrespaccept_wr(false),
  response_t(sc_get_time_resolution()),
  accept_t(sc_get_time_resolution()),
  data_accept_t(sc_get_time_resolution()),
  threadbusy_t(sc_get_time_resolution()),
  data_threadbusy_t(sc_get_time_resolution()),
  cmd_t(SC_ZERO_TIME),
  data_t(SC_ZERO_TIME),
  oldest_txn(0),
  newest_txn(0),
  cmdaccept_state(false),
  dataaccept_state(false),
  curr_tl0_cmd(0),
  curr_tl0_data(0),
  curr_tl0_resp(0),
  curr_sthreadbusy(~0),
  curr_sdatathreadbusy(~0)
{
}


// destructor
tl1_to_tl0_base::~tl1_to_tl0_base()
{
}


void tl1_to_tl0_base::end_of_elaboration() {
  byte_wdth = ocp_config.data_wdth / 8;

  // set up clocked process
  process(&tl1_to_tl0_base::clk_rising_edge, this, "clk_edge",
    clk_port->posedge_event());

  // signal writer process
  process(&tl1_to_tl0_base::writer_process, this, "signal_writer",
    writer_trigger);

  // set up other processes that will fire at sample times
  if(ocp_config.resp) {
    process(&tl1_to_tl0_base::response_sample, this, "response_sample",
      response_e);
  } else curr_tl0_resp = reinterpret_cast<tlm_generic_payload *>(-1);

  if(ocp_config.cmdaccept) {
    process(&tl1_to_tl0_base::accept_sample, this, "accept_sample",
      accept_e);
  } else cmdaccept_state = true;

  if(ocp_config.dataaccept) {
    process(&tl1_to_tl0_base::data_accept_sample, this, "data_accept_sample",
      data_accept_e);
  } else dataaccept_state = true;

  if(ocp_config.sthreadbusy) {
    process(&tl1_to_tl0_base::threadbusy_sample, this, "threadbusy_sample",
      threadbusy_e);
  }

  if(ocp_config.sdatathreadbusy) {
    process(&tl1_to_tl0_base::data_threadbusy_sample, this, "data_threadbusy_sample",
      data_threadbusy_e);
  }

  if(ocp_config.sflag) {
    process(&tl1_to_tl0_base::handle_sflag, this, "handle_sflag", sflag_event());
  }

  if(ocp_config.interrupt) {
    process(&tl1_to_tl0_base::handle_sinterrupt, this, "handle_sinterrupt", sinterrupt_event());
  }

  if(ocp_config.serror) {
    process(&tl1_to_tl0_base::handle_serror, this, "handle_serror", serror_event());
  }

  if(ocp_config.sreset) {
    process(&tl1_to_tl0_base::handle_sreset, this, "handle_sreset", sreset_event());
  }
}


// clock edge behaviour
void tl1_to_tl0_base::clk_rising_edge() {
  if(!enableclk()) return;

  // pull critical TL0 signals to IDLE
  bool do_critical_writes = ocp_config.respaccept;
  if(cmdaccept_state) {
    new_mcmd = CMD_IDLE;
    do_critical_writes = mcmd_wr = true;
  } else mcmd_wr = false;
  if(dataaccept_state) {
    new_mdatavalid = false;
    do_critical_writes = mdatavalid_wr = true;
  } else mdatavalid_wr = false;
  new_mrespaccept = false;
  mrespaccept_wr = true;
  if(do_critical_writes) writer_trigger.notify();

  // trigger sampling processes
  if(curr_tl0_resp == 0) response_e.notify(response_t);
  if(ocp_config.cmdaccept) {
    accept_e.notify(accept_t);
    // prevent auto-accept of new TL1 requests until after SCmdAccept sample
    cmdaccept_state = false;
  }
  if(ocp_config.dataaccept) {
    data_accept_e.notify(data_accept_t);
    // prevent auto-accept of new TL1 data until after SDataAccept sample
    dataaccept_state = false;
  }
  if(ocp_config.sthreadbusy) threadbusy_e.notify(threadbusy_t);
  if(ocp_config.sdatathreadbusy) data_threadbusy_e.notify(data_threadbusy_t);
};


// TLM transport into this module
tlm_sync_enum tl1_to_tl0_base::nb_transport_fw(
  tlm_generic_payload &pl, tlm_phase &ph, sc_time &ta)
{
  if(ph == BEGIN_REQ) {
    // if new, insert txn into linked list and initialise
    if(init_ext(pl, ocp_config)) add_transaction(pl);
    ext &e(ext::get(pl));
    tracker_base *t = e.tracker;

    // put the request onto the TL0 interface
    // writes to non-existent ports are ignored
    maddr(t->OCPAddr());
    uint64 be = 0;
    for(unsigned b = 0; b < byte_wdth; b++) if(t->ReqBE(b)) be |= (1 << b);
    if(pl.is_write()) {
      if(!ocp_config.datahandshake) {
        unsigned char *d = t->ReqDataPtr();
        for(unsigned b = 0; b < byte_wdth; b++, d++)
          if(be & (1 << b)) mdata(*d, b);
        mdata();
        mdatabyteen(be);
        mdatainfo(&pl, t->ReqBeat());
      }
      lock *lk;
      if(extension_api::get_extension<lock>(lk, pl))
        lk->value->atomic_txn_completed();
    }
    new_mcmd = e.cmd;
    mcmd_wr = true;
    writer_trigger.notify();
    mburstseq(t->BurstSeq());
    mburstlength(t->BurstLen());
    mburstprecise(t->Precise());
    mburstsinglereq(extension_api::get_extension<srmd>(pl) != 0);
    mtaginorder(e.tag < 0);
    mtagid(unsigned(e.tag));
    mthreadid(e.thread);
    mbyteen(be);
    mblockheight(t->BlockHeight());
    mblockstride(t->BlockStride());
    maddrspace(e.addr_space);
    matomiclength(e.atomic_length);
    mconnid(e.conn_id);
    mreqinfo(&pl);
    mreqlast(t->ReqDone(true));
    mreqrowlast(t->ReqRowDone());

    // return: can only accept _after_ cmd-accept has been sampled
    if(cmdaccept_state) {
      if(t->ReqAcc()) remove_from_list(pl);
      ph = END_REQ;
      return TLM_UPDATED;
    } else {
      curr_tl0_cmd = &pl;
      return TLM_ACCEPTED;
    }
  }

  if(ph == BEGIN_DATA) {
    ext &e(ext::get(pl));
    tracker_base *t = e.tracker;
    unsigned char *d = t->WrDataPtr();
    uint64 be = 0;
    for(unsigned b = 0; b < byte_wdth; b++, d++) if(t->WrBE(b)) {
      mdata(*d, b);
      be |= (1 << b);
    }
    mdata();
    new_mdatavalid = true;
    mdatavalid_wr = true;
    writer_trigger.notify();
    mdatatagid(unsigned(e.tag));
    mdatathreadid(e.thread);
    mdatabyteen(be);
    mdatainfo(&pl, t->WrBeat());
    mdatalast(t->WrDone(true));
    mdatarowlast(t->WrRowDone());

    // return: can only accept _after_ data-accept has been sampled
    if(dataaccept_state) {
      if(t->WrAcc()) remove_from_list(pl);
      ph = END_DATA;
      return TLM_UPDATED;
    } else {
      curr_tl0_data = &pl;
      return TLM_ACCEPTED;
    }
  }

  if(ph == END_RESP) {
    end_resp(pl);
    return TLM_ACCEPTED;
  }

  if(ph == RESP_THREAD_BUSY_CHANGE) {
    resp_thread_busy *tbex;
    extension_api::get_extension<resp_thread_busy>(tbex, pl);
    mthreadbusy(tbex->value);
    return TLM_ACCEPTED;
  }

  if(ph == BEGIN_ERROR) {
    merror(true);
    return TLM_ACCEPTED;
  }

  if(ph == END_ERROR) {
    merror(false);
    return TLM_ACCEPTED;
  }

  if(ph == BEGIN_RESET) {
    mreset_n(false);
    return TLM_ACCEPTED;
  }

  if(ph == END_RESET) {
    mreset_n(true);
    exit_reset();
    return TLM_ACCEPTED;
  }

  sc_assert(ph == MFLAG_CHANGE); {
    mflag(&pl);
    return TLM_ACCEPTED;
  }
}


// response accept may be started from return to bw transport call or
// from fw call.  same behaviour in both cases
void tl1_to_tl0_base::end_resp(tlm_generic_payload &pl) {
  curr_tl0_resp = 0;
  new_mrespaccept = true;
  mrespaccept_wr = true;
  if(ocp_config.respaccept) writer_trigger.notify();
  if(ext::get(pl).tracker->RdAcc()) remove_from_list(pl);
}


void tl1_to_tl0_base::add_transaction(tlm_generic_payload &pl) {
  ext &e(ext::get(pl));
  // add the transaction to the linked-list
  e.next = 0;
  e.prev = newest_txn;
  if(newest_txn == 0) oldest_txn = &pl;
  else ext::get(*newest_txn).next = &pl;
  newest_txn = &pl;
  pl.acquire();
}


void tl1_to_tl0_base::remove_from_list(tlm_generic_payload &pl) {
  ext &e(ext::get(pl));
  if(e.prev == 0) oldest_txn = e.next;
  else ext::get(*(e.prev)).next = e.next;
  if(e.next == 0) newest_txn = e.prev;
  else ext::get(*(e.next)).prev = e.prev;
  pl.release();
}


// sample time behaviour: response
void tl1_to_tl0_base::response_sample() {
  unsigned response = sresp();
  if(response != RESP_NULL) {
    int tag = (staginorder() ? -1 : stagid());
    unsigned thread = sthreadid();
    curr_tl0_resp = oldest_txn;
    ext *e;
    while(true) {
      sc_assert(curr_tl0_resp != 0);
      // exit here if no OCP response with this thread & tag is expected
      e = &ext::get(*curr_tl0_resp);
      if((e->tag == tag) && (e->thread == thread) && !e->tracker->RdDone(false)) break;
      curr_tl0_resp = e->next;
    }

    // set the OCP response code
    if(response == RESP_FAIL) {
      semaphore *sf;
      sc_assert(extension_api::get_extension<semaphore>(sf, *curr_tl0_resp));
      sf->value = 1;
      curr_tl0_resp->set_response_status(TLM_OK_RESPONSE);
    } else curr_tl0_resp->set_response_status(
      response != RESP_DVA ? TLM_GENERIC_ERROR_RESPONSE : TLM_OK_RESPONSE);

    // copy data
    if(curr_tl0_resp->is_read() && (response == RESP_DVA)) {
      unsigned char *d = e->tracker->RdDataPtr();
      for(unsigned b = 0; b < byte_wdth; b++, d++)
        if(e->tracker->RdBE(b)) *d = sdata(b);
      sdatainfo(curr_tl0_resp, e->tracker->RdBeat());
    }

    srespinfo(curr_tl0_resp);

    e->tracker->RdDone(true);
    tlm_phase ph(BEGIN_RESP);
    sc_time t(SC_ZERO_TIME);
    if(nb_transport_bw(*curr_tl0_resp, ph, t) == TLM_UPDATED)
      end_resp(*curr_tl0_resp);
  }
}


// sample time behaviour: command accept
void tl1_to_tl0_base::accept_sample() {
  cmdaccept_state = scmdaccept();  // enables transport to accept commands
  if(cmdaccept_state && (curr_tl0_cmd != 0)) {
    tlm_phase ph(END_REQ);
    sc_time t(SC_ZERO_TIME);
    nb_transport_bw(*curr_tl0_cmd, ph, t);
    tracker_base *tk = ext::get(*curr_tl0_cmd).tracker;
    if(tk->ReqAcc()) remove_from_list(*curr_tl0_cmd);
    curr_tl0_cmd = 0;
  }
}


// sample time behaviour: data accept
void tl1_to_tl0_base::data_accept_sample() {
  dataaccept_state = sdataaccept();  // enables transport to accept data hs
  if(dataaccept_state && (curr_tl0_data != 0)) {
    tlm_phase ph(END_DATA);
    sc_time t(SC_ZERO_TIME);
    nb_transport_bw(*curr_tl0_data, ph, t);
    tracker_base *tk = ext::get(*curr_tl0_data).tracker;
    if(tk->WrAcc()) remove_from_list(*curr_tl0_data);
    curr_tl0_data = 0;
  }
}


// sample time behaviour: command thread-busy
void tl1_to_tl0_base::threadbusy_sample() {
  unsigned tb = sthreadbusy();
  if(tb != curr_sthreadbusy) {
    curr_sthreadbusy = tb;
    tlm_phase ph(CMD_THREAD_BUSY_CHANGE);
    sc_time t(SC_ZERO_TIME);
    cmd_thread_busy *tbex;
    tlm_generic_payload *tbpl = get_tb_transaction();
    extension_api::get_extension<cmd_thread_busy>(tbex, *tbpl);
    tbex->value = tb;
    nb_transport_bw(*tbpl, ph, t);
  }
}

// sample time behaviour: data HS thread-busy
void tl1_to_tl0_base::data_threadbusy_sample() {
  unsigned tb = sdatathreadbusy();
  if(tb != curr_sdatathreadbusy) {
    curr_sdatathreadbusy = tb;
    tlm_phase ph(DATA_THREAD_BUSY_CHANGE);
    sc_time t(SC_ZERO_TIME);
    data_thread_busy *tbex;
    tlm_generic_payload *tbpl = get_tb_transaction();
    extension_api::get_extension<data_thread_busy>(tbex, *tbpl);
    tbex->value = tb;
    nb_transport_bw(*tbpl, ph, t);
  }
}

// sideband signalling support
void tl1_to_tl0_base::handle_sflag() {
  tlm_generic_payload *pl = get_flag_transaction();
  sflag(pl);
  tlm_phase ph(SFLAG_CHANGE);
  sc_time t(SC_ZERO_TIME);
  nb_transport_bw(*pl, ph, t);
}

void tl1_to_tl0_base::handle_sinterrupt() {
  tlm_generic_payload *pl = get_interrupt_transaction();
  tlm_phase ph;
  if(sinterrupt()) ph = BEGIN_INTERRUPT; else ph = END_INTERRUPT;
  sc_time t(SC_ZERO_TIME);
  nb_transport_bw(*pl, ph, t);
}

void tl1_to_tl0_base::handle_serror() {
  tlm_generic_payload *pl = get_error_transaction();
  tlm_phase ph;
  if(serror()) ph = BEGIN_ERROR; else ph = END_ERROR;
  sc_time t(SC_ZERO_TIME);
  nb_transport_bw(*pl, ph, t);
}

void tl1_to_tl0_base::handle_sreset() {
  tlm_generic_payload *pl = get_reset_transaction();
  tlm_phase ph;
  if(sreset_n()) ph = END_RESET; else ph = BEGIN_RESET;
  sc_time t(SC_ZERO_TIME);
  nb_transport_bw(*pl, ph, t);
  if(sreset_n()) exit_reset();
}

void tl1_to_tl0_base::exit_reset() {
  ocp_socket_reset();
  tlm_generic_payload *pl = oldest_txn;
  while(pl != 0) {
    tlm_generic_payload *to_release = pl;
    pl = ext::get(*pl).next;
    if((ext::get(*to_release).cmd == CMD_WRC)
      || ((pl == 0) && (ext::get(*to_release).cmd == CMD_RDEX))) {
      lock *lk;
      extension_api::get_extension(lk, *to_release);
      lk->value->atomic_txn_completed();
    }
    to_release->release();
  }
  oldest_txn = newest_txn = 0;
  new_mcmd = CMD_IDLE;
  new_mdatavalid = false;
  new_mrespaccept = false;
  mcmd_wr = mdatavalid_wr = mrespaccept_wr = true;
  writer_trigger.notify();
  curr_tl0_resp = curr_tl0_cmd = curr_tl0_data = 0;
  curr_sthreadbusy = curr_sdatathreadbusy = ~0;
  if(ocp_config.cmdaccept) cmdaccept_state = false;
  if(ocp_config.dataaccept) dataaccept_state = false;
}


// implementation of burst trackers and initialisation functions
namespace support_10 {

// class to implement pooling for burst tracker objects
template<class T, class BASE> struct pooled_tracker: BASE {
  static pooled_tracker *pool(pooled_tracker *used = 0) {
    static pooled_tracker *pool = 0;
    if(used == 0) {
      if(pool == 0) {pool = new T; pool->next = 0;}
      pooled_tracker *rv = pool;
      pool = pool->next;
      return rv;
    } else {
      used->next = pool;
      pool = used;
      return 0;
    }
  }
  pooled_tracker *next;
  void return_to_pool() {pool(this);}
};


struct precise_incr_tracker_base: tracker_base {
  uint64 addr_align;
  virtual void init(tlm_generic_payload &pl, const ocp_parameters &pm) {
    init(pl, pm, false);
  }
  void init(tlm_generic_payload &pl, const ocp_parameters &pm, bool wrap_xor) {
    txn = &pl;
    byte_wdth = pm.data_wdth / 8;
    unsigned align = byte_wdth;
    // set initial OCP addr
    uint64 pla = pl.get_address();
    ocp_addr = (pla / align) * align;
    addr_align = pla - ocp_addr;
    // burst alignment to OCP or burst length
    unsigned len;
    burst_length *bl;
    if(extension_api::get_extension<burst_length>(bl, pl)) {
      len = bl->value;
    } else {
      if(pm.burst_aligned || wrap_xor) {
        for(len = byte_wdth; ; len *= 2) {
          if((len + (pla / len) * len) >= (pla + pl.get_data_length())) break;
        }
        align = len;
        len = len / byte_wdth;
        ocp_addr = (pla / align) * align;
        addr_align = pla - ocp_addr;
      } else {
        len = (pl.get_data_length() + addr_align + byte_wdth - 1) / byte_wdth;
      }
    }
    // expected numbers of transfers
    req.done = data.done = resp.done = 0;
    req.acc = data.acc = resp.acc = 0;
    bool wresp = pm.writeresp_enable || (ext::get(pl).cmd != WR);
    if(pm.burstsinglereq && extension_api::get_extension<srmd>(pl)) {
      req.nr = 1;
      if(pl.is_read()) {resp.nr = len; data.nr = 0;}
      else {resp.nr = (wresp ? 1 : 0); data.nr = len;}
    } else {
      req.nr = len; resp.nr = (wresp ? len : 0);
      data.nr = (pm.datahandshake ? len : 0);
    }
  }

  // Req
  virtual unsigned BurstSeq() {return INCR;}
  virtual bool Precise() {return true;}
  virtual uint64 OCPAddr() {return ocp_addr + req.done * byte_wdth;}
  virtual bool ReqDone(bool track) {return XDone(track, req);}
  virtual bool ReqAcc() {return XAcc(req);}
  virtual bool ReqBE(unsigned bytelane) {return XBE(bytelane, req);}
  virtual unsigned char *ReqDataPtr() {return XDataPtr(req);}
  virtual unsigned BurstLen() {
    return (data.nr == 0 ? (resp.nr == 0 ? req.nr : resp.nr) : data.nr);
  }

  // WR data
  virtual bool WrDone(bool track) {return XDone(track, data);}
  virtual bool WrAcc() {return XAcc(data);}
  virtual bool WrBE(unsigned bytelane) {return XBE(bytelane, data);}
  virtual unsigned char *WrDataPtr() {return XDataPtr(data);}

  // RD data
  virtual bool RdDone(bool track) {return XDone(track, resp);}
  virtual bool RdAcc() {return XAcc(resp);}
  virtual bool RdBE(unsigned bytelane) {return XBE(bytelane, resp);}
  virtual unsigned char *RdDataPtr() {return XDataPtr(resp);}

  virtual bool XDone(bool track, handshake &hs) {
    if(track) ++hs.done;
    return(hs.done == hs.nr);
  }
  virtual bool XAcc(handshake &hs) {
    ++hs.acc;
    return((req.acc == req.nr) && (data.acc == data.nr) && (resp.acc == resp.nr));
  }
  virtual unsigned char *XDataPtr(handshake &hs) {
    return txn->get_data_ptr() + hs.done * byte_wdth - addr_align;
  }
  virtual bool XBE(unsigned bytelane, handshake &hs) {
    int idx = bytelane + hs.done * byte_wdth - addr_align;
    if((idx < 0) || (idx >= int(txn->get_data_length()))) return false;
    unsigned char *bep = txn->get_byte_enable_ptr();
    return((bep == 0) ||
      (bep[idx % txn->get_byte_enable_length()] == TLM_BYTE_ENABLED));
  }
};


struct precise_incr_tracker:
  pooled_tracker<precise_incr_tracker, precise_incr_tracker_base>
{
};

struct precise_strm_tracker:
  pooled_tracker<precise_strm_tracker, precise_incr_tracker_base>
{
  void init(tlm_generic_payload &pl, const ocp_parameters &pm) {
    txn = &pl;
    unsigned align = pm.data_wdth / 8;
    byte_wdth = pl.get_streaming_width();
    // burst length
    unsigned len;
    burst_length *bl;
    uint64 pla = pl.get_address();
    // burst alignment to OCP
    if(extension_api::get_extension<burst_length>(bl, pl)) {
      len = bl->value;
    } else {
      len = (pl.get_data_length() + byte_wdth - 1) / byte_wdth;
    }
    // set initial OCP addr
    ocp_addr = (pla / align) * align;
    addr_align = pla - ocp_addr;
    // expected numbers of transfers
    req.done = data.done = resp.done = 0;
    req.acc = data.acc = resp.acc = 0;
    bool wresp = pm.writeresp_enable || (ext::get(pl).cmd != WR);
    if(pm.burstsinglereq && extension_api::get_extension<srmd>(pl)) {
      req.nr = 1;
      if(pl.is_read()) {resp.nr = len; data.nr = 0;}
      else {resp.nr = (wresp ? 1 : 0); data.nr = len;}
    } else {
      req.nr = len;
      resp.nr = (wresp ? len : 0);
      data.nr = (pm.datahandshake ? len : 0);
    }
  }
  unsigned BurstSeq() {return STRM;}
  uint64 OCPAddr() {return ocp_addr;}
  bool XBE(unsigned bytelane, handshake &hs) {
    if((addr_align > bytelane) ||
      (addr_align + txn->get_streaming_width() <= bytelane)) return false;
    int idx = bytelane + hs.done * byte_wdth - addr_align;
    unsigned char *bep = txn->get_byte_enable_ptr();
    return((bep == 0) ||
      (bep[idx % txn->get_byte_enable_length()] == TLM_BYTE_ENABLED));
  }
};

struct precise_wrap_tracker:
  pooled_tracker<precise_wrap_tracker, precise_incr_tracker_base>
{
  unsigned wrap_len, wrap_start;
  void init(tlm_generic_payload &pl, const ocp_parameters &pm) {
    precise_incr_tracker_base::init(pl, pm, true);
    wrap_len = BurstLen();
    burst_sequence *bs;
    extension_api::get_extension<burst_sequence>(bs, pl);
    wrap_start = (bs->value.xor_wrap_address - ocp_addr) / byte_wdth;
  }
  unsigned BurstSeq() {return WRAP;}
  unsigned char *XDataPtr(handshake &hs) {
    unsigned off = (hs.done + wrap_start) % wrap_len;
    return txn->get_data_ptr() + off * byte_wdth - addr_align;
  }
  bool XBE(unsigned bytelane, handshake &hs) {
    unsigned off = (hs.done + wrap_start) % wrap_len;
    int idx = bytelane + off * byte_wdth - addr_align;
    if((idx < 0) || (idx >= int(txn->get_data_length()))) return false;
    unsigned char *bep = txn->get_byte_enable_ptr();
    return((bep == 0) ||
      (bep[idx % txn->get_byte_enable_length()] == TLM_BYTE_ENABLED));
  }
  uint64 OCPAddr() {
    unsigned off = (req.done + wrap_start) % wrap_len;
    return ocp_addr + off * byte_wdth;
  }
};

struct precise_xor_tracker:
  pooled_tracker<precise_xor_tracker, precise_incr_tracker_base>
{
  unsigned xor_len, xor_start;
  void init(tlm_generic_payload &pl, const ocp_parameters &pm) {
    precise_incr_tracker_base::init(pl, pm, true);
    xor_len = BurstLen();
    burst_sequence *bs;
    extension_api::get_extension<burst_sequence>(bs, pl);
    xor_start = (bs->value.xor_wrap_address - ocp_addr) / byte_wdth;
  }
  unsigned BurstSeq() {return XOR;}
  unsigned char *XDataPtr(handshake &hs) {
    unsigned off = hs.done ^ xor_start;
    return txn->get_data_ptr() + off * byte_wdth - addr_align;
  }
  bool XBE(unsigned bytelane, handshake &hs) {
    unsigned off = hs.done ^ xor_start;
    int idx = bytelane + off * byte_wdth - addr_align;
    if((idx < 0) || (idx >= int(txn->get_data_length()))) return false;
    unsigned char *bep = txn->get_byte_enable_ptr();
    return((bep == 0) ||
      (bep[idx % txn->get_byte_enable_length()] == TLM_BYTE_ENABLED));
  }
  uint64 OCPAddr() {
    unsigned off = req.done ^ xor_start;
    return ocp_addr + off * byte_wdth;
  }
};

struct precise_dflt_tracker:
  pooled_tracker<precise_dflt_tracker, precise_incr_tracker_base>
{
  burst_sequence *seq;
  unsigned beats_per_addr;
  void init(tlm_generic_payload &pl, const ocp_parameters &pm) {
    extension_api::get_extension<burst_sequence>(seq, pl);
    txn = &pl;
    byte_wdth = pm.data_wdth / 8;
    unsigned pack = seq->value.unkn_dflt_bytes_per_address;
    // set initial OCP addr
    uint64 pla = pl.get_address();
    ocp_addr = (pla / byte_wdth) * byte_wdth;
    addr_align = pla - ocp_addr;
    beats_per_addr = (pack + addr_align + byte_wdth - 1) / byte_wdth;
    // burst alignment to OCP and burst length
    unsigned len;
    burst_length *bl;
    if(extension_api::get_extension<burst_length>(bl, pl)) {
      len = bl->value;
    } else {
      // very unlikely that a precise DFLT burst can avoid having the
      // burst_length extension set, especially if addr is not OCP-aligned.
      // assume same addr align for each address
      len = beats_per_addr * pl.get_data_length() / pack;
    }
    // expected numbers of transfers
    req.done = data.done = resp.done = 0;
    req.acc = data.acc = resp.acc = 0;
    bool wresp = pm.writeresp_enable || (ext::get(pl).cmd != WR);
    if(pm.burstsinglereq && extension_api::get_extension<srmd>(pl)) {
      req.nr = 1;
      if(pl.is_read()) {resp.nr = len; data.nr = 0;}
      else {resp.nr = (wresp ? 1 : 0); data.nr = len;}
    } else {
      req.nr = len;
      resp.nr = (wresp ? len : 0);
      data.nr = (pm.datahandshake ? len : 0);
    }
  }
  unsigned BurstSeq() {return seq->value.sequence;}
  uint64 OCPAddr() {
    if(!seq->value.unkn_dflt_addresses_valid) return ocp_addr;
    return seq->value.unkn_dflt_addresses[req.done / beats_per_addr];
  }
  unsigned char *XDataPtr(handshake &hs) {
    return txn->get_data_ptr() + (hs.done % beats_per_addr) * byte_wdth
      + (hs.done / beats_per_addr) * seq->value.unkn_dflt_bytes_per_address
      - addr_align;
  }
  bool XBE(unsigned bytelane, handshake &hs) {
    int idxinaddr = (hs.done % beats_per_addr) * byte_wdth + bytelane - addr_align;
    if((idxinaddr < 0) || (idxinaddr >= int(seq->value.unkn_dflt_bytes_per_address))) return false;
    int idx = idxinaddr + (hs.done / beats_per_addr) * seq->value.unkn_dflt_bytes_per_address;
    unsigned char *bep = txn->get_byte_enable_ptr();
    return((bep == 0) ||
      (bep[idx % txn->get_byte_enable_length()] == TLM_BYTE_ENABLED));
  }
};

struct precise_blck_tracker:
  pooled_tracker<precise_blck_tracker, precise_incr_tracker_base>
{
  burst_sequence *seq;
  unsigned width;
  void init(tlm_generic_payload &pl, const ocp_parameters &pm) {
    txn = &pl;
    extension_api::get_extension<burst_sequence>(seq, pl);
    byte_wdth = pm.data_wdth / 8;
    // set initial OCP addr
    uint64 pla = pl.get_address();
    ocp_addr = (pla / byte_wdth) * byte_wdth;
    addr_align = pla - ocp_addr;
    // burst alignment to OCP or burst length
    burst_length *bl;
    if(extension_api::get_extension<burst_length>(bl, pl)) {
      width = bl->value;
    } else {
      width = (seq->value.blck_row_length_in_bytes + addr_align + byte_wdth - 1) / byte_wdth;
    }
    // expected numbers of transfers
    req.done = data.done = resp.done = 0;
    req.acc = data.acc = resp.acc = 0;
    bool wresp = pm.writeresp_enable || (ext::get(pl).cmd != WR);
    if(pm.burstsinglereq && extension_api::get_extension<srmd>(pl)) {
      req.nr = 1;
      if(pl.is_read()) {resp.nr = width * seq->value.block_height; data.nr = 0;}
      else {resp.nr = (wresp ? 1 : 0); data.nr = width * seq->value.block_height;}
    } else {
      req.nr = width * seq->value.block_height;
      resp.nr = (wresp ? req.nr : 0);
      data.nr = (pm.datahandshake ? width * seq->value.block_height : 0);
    }
  }
  unsigned BurstLen() {return width;}
  unsigned BurstSeq() {return BLCK;}
  uint64 OCPAddr() {
    return ocp_addr + (req.done % width) * byte_wdth + (req.done / width) * seq->value.block_stride;
    // warning: assumes stride is aligned to OCP data width
  }
  unsigned char *XDataPtr(handshake &hs) {
    unsigned char *row = txn->get_data_ptr() + (hs.done / width) * seq->value.blck_row_length_in_bytes;
    unsigned off = hs.done % width;
    return row + off * byte_wdth - addr_align;
  }
  bool XBE(unsigned bytelane, handshake &hs) {
    int idxinrow = (hs.done % width) * byte_wdth + bytelane - addr_align;
    if((idxinrow < 0) || (idxinrow >= int(seq->value.blck_row_length_in_bytes))) return false;
    int idx = idxinrow + (hs.done / width) * seq->value.blck_row_length_in_bytes;
    unsigned char *bep = txn->get_byte_enable_ptr();
    return((bep == 0) ||
      (bep[idx % txn->get_byte_enable_length()] == TLM_BYTE_ENABLED));
  }
  bool ReqRowDone() {return (req.done % width) == 0;}
  bool WrRowDone() {return (data.done % width) == 0;}
  unsigned BlockHeight() {return seq->value.block_height;}
  uint64 BlockStride() {return seq->value.block_stride;}
};

struct imprecise_incr_tracker_base: precise_incr_tracker_base
{
  burst_length *bl;
  virtual void init(tlm_generic_payload &pl, const ocp_parameters &pm) {
    txn = &pl;
    byte_wdth = pm.data_wdth / 8;
    extension_api::get_extension<burst_length>(bl, pl);
    // set initial OCP addr
    uint64 pla = pl.get_address();
    ocp_addr = (pla / byte_wdth) * byte_wdth;
    addr_align = pla - ocp_addr;
    // expected numbers of transfers
    bool wresp = pm.writeresp_enable || (ext::get(pl).cmd != WR);
    req.done = data.done = resp.done = 0;
    req.acc = data.acc = resp.acc = 0;
    req.nr = unsigned(-1);
    resp.nr = (wresp ? req.nr : 0);
    data.nr = (pm.datahandshake ? unsigned(-1) : 0);
  }
  bool ReqDone(bool track) {
    if(track) {
      ++req.done;
      if(bl->value == 1) {
        req.nr = req.done;
        if(resp.nr != 0) resp.nr = req.done;
        if(data.nr != 0) data.nr = req.done;
        return true;
      }
    }
    return (req.done == req.nr);
  }
  bool Precise() {return false;}
  unsigned BurstLen() {return bl->value;}
};

struct imprecise_incr_tracker:
  pooled_tracker<imprecise_incr_tracker, imprecise_incr_tracker_base>
{};

struct imprecise_strm_tracker:
  pooled_tracker<imprecise_strm_tracker, imprecise_incr_tracker_base>
{
  void init(tlm_generic_payload &pl, const ocp_parameters &pm) {
    imprecise_incr_tracker_base::init(pl, pm);
    byte_wdth = pl.get_streaming_width();
  }
  unsigned BurstSeq() {return STRM;}
  uint64 OCPAddr() {return ocp_addr;}
  bool XBE(unsigned bytelane, handshake &hs) {
    if((addr_align > bytelane) ||
      (addr_align + txn->get_streaming_width() <= bytelane)) return false;
    int idx = bytelane + hs.done * byte_wdth - addr_align;
    unsigned char *bep = txn->get_byte_enable_ptr();
    return((bep == 0) ||
      (bep[idx % txn->get_byte_enable_length()] == TLM_BYTE_ENABLED));
  }
};

struct imprecise_unkn_tracker:
  pooled_tracker<imprecise_unkn_tracker, imprecise_incr_tracker_base>
{
  burst_sequence *seq;
  unsigned beats_per_addr;
  void init(tlm_generic_payload &pl, const ocp_parameters &pm) {
    imprecise_incr_tracker_base::init(pl, pm);
    extension_api::get_extension<burst_sequence>(seq, pl);
    beats_per_addr = (seq->value.unkn_dflt_bytes_per_address + addr_align + byte_wdth - 1) / byte_wdth;
  }
  unsigned BurstSeq() {return seq->value.sequence;}
  uint64 OCPAddr() {
    if(!seq->value.unkn_dflt_addresses_valid) return ocp_addr;
    return seq->value.unkn_dflt_addresses[req.done / beats_per_addr];
  }
  unsigned char *XDataPtr(handshake &hs) {
    return txn->get_data_ptr() + (hs.done % beats_per_addr) * byte_wdth
      + (hs.done / beats_per_addr) * seq->value.unkn_dflt_bytes_per_address
      - addr_align;
  }
  bool XBE(unsigned bytelane, handshake &hs) {
    int idxinaddr = (hs.done % beats_per_addr) * byte_wdth + bytelane - addr_align;
    if((idxinaddr < 0) || (idxinaddr >= int(seq->value.unkn_dflt_bytes_per_address))) return false;
    int idx = idxinaddr + (hs.done / beats_per_addr) * seq->value.unkn_dflt_bytes_per_address;
    unsigned char *bep = txn->get_byte_enable_ptr();
    return ((bep == 0) ||
      (bep[idx % txn->get_byte_enable_length()] == TLM_BYTE_ENABLED));
  }
};


// function to initialise an adapter's extension, if necessary
// called only during BEGIN_REQ transport
// returns true only on first call
bool init_ext(tlm_generic_payload &pl, const ocp_parameters &ocp_config) {
  // make sure there's an extension object
  ext *e = pl.get_extension<ext>();
  if(e == 0) {
    e = new ext;
    e->tracker = 0;
    pl.set_extension<ext>(e);
  } else {
    // don't re-initialise if called multiple times (for MRMD)
    if(!e->tracker->ReqDone(false)) return false;
  }
  if(e->tracker != 0) e->tracker->return_to_pool();

  // now choose a type of tracker
  // burst sequence
  unsigned seq = INCR;
  if(pl.get_streaming_width() < pl.get_data_length()) seq = STRM;
  burst_sequence *bs;
  if(extension_api::get_extension<burst_sequence>(bs, pl)) seq = bs->value.sequence;

  if(!ocp_config.burstprecise || !extension_api::get_extension<imprecise>(pl)) {
    switch(seq) {
      case INCR: e->tracker = new precise_incr_tracker; break;
      case STRM: e->tracker = new precise_strm_tracker; break;
      case WRAP: e->tracker = new precise_wrap_tracker; break;
      case XOR: e->tracker = new precise_xor_tracker; break;
      case DFLT1:
      case DFLT2: e->tracker = new precise_dflt_tracker; break;
      case BLCK: e->tracker = new precise_blck_tracker; break;
    }
  } else {
    switch(seq) {
      case INCR: e->tracker = new imprecise_incr_tracker; break;
      case STRM: e->tracker = new imprecise_strm_tracker; break;
      case DFLT1:
      case DFLT2:
      case UNKN: e->tracker = new imprecise_unkn_tracker; break;
    }
  }

  // determine the OCP command
  if(pl.is_read()) {
    e->cmd = CMD_RD;
    if(ocp_config.readex_enable) {
      lock *lk;
      if(extension_api::get_extension<lock>(lk, pl)) {
        e->cmd = CMD_RDEX;
        lk->value->lock_is_understood_by_slave =
          (ocp_config.readex_enable && (lk->value->number_of_txns == 2));
      }
    }
    if(ocp_config.rdlwrc_enable) {
      semaphore *sf;
      if(extension_api::get_extension<semaphore>(sf, pl)) e->cmd = CMD_RDL;
    }
  } else {
    // writes
    if(ocp_config.write_enable && ocp_config.writenonpost_enable)
      e->cmd = (extension_api::get_extension<posted>(pl) ? CMD_WR : CMD_WRNP);
    else
      e->cmd = (ocp_config.write_enable ? CMD_WR : CMD_WRNP);
    if(ocp_config.rdlwrc_enable) {
      semaphore *sf;
      if(extension_api::get_extension<semaphore>(sf, pl)) e->cmd = CMD_WRC;
    }
  }

  e->tracker->init(pl, ocp_config);

  address_space *as;
  e->addr_space = (extension_api::get_extension<address_space>(as, pl) ?
    as->value : 0);
  conn_id *ci;
  e->conn_id = (extension_api::get_extension<conn_id>(ci, pl) ?
    ci->value : 0);
  atomic_length *al;
  e->atomic_length = (extension_api::get_extension<atomic_length>(al, pl) ?
    al->value : 1);

  // set up ordering management
  tag_id *tgid;
  e->tag = (ocp_config.tags <= 1 ? 0 :
    (extension_api::get_extension<tag_id>(tgid, pl) ? tgid->value : -1));
  thread_id *thid;
  e->thread =
    (extension_api::get_extension<thread_id>(thid, pl) ? thid->value : 0);

  return true;
}


// end of all namespaces
}}}
