///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2009-2011
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis, Texas Instruments
//
//          $Id:
//
//  Description :  Testbench for TL1/TL0 adapters: precise aligned SRMD
//                 with tags and non-blocking threads
//
// TL1 and TL0 initiators and targets are created with identical behaviour for
// the different abstraction levels.
// Six simulations are run, 4 with adapters and 2 without, for 0, 1 and 2
//   cascaded adapters.
// All sets of results should be the same, allowing for a few ps of timing
// delay in some cases.
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////


#include "ocpip_adapters.h"
#include <iostream>
#include <queue>


using namespace std;
using namespace ocpip;
using namespace sc_core;
using namespace tlm;


#define NR_THREADS 4

// all OCPs in this testbench have the same configuration, or an extended
// one with threads
ocp_parameters gen_ocp_config() {
  ocp_parameters p;
  // aligned INCR bursts length in {1,2,4,8}
  p.burst_aligned = true;
  p.burstlength = true;
  p.burstlength_wdth = 4;
  p.burstsinglereq = true;
  p.datahandshake = true;
  p.dataaccept = true;
  p.datalast = true;
  p.respaccept = true;
  p.resplast = true;
  p.tags = 5;
  // base config
  p.writeresp_enable = true;
  p.addr_wdth = 32;
  p.data_wdth = 32;
  p.mreset = true;
  return p;
}

ocp_parameters gen_ocp_config_threads() {
  ocp_parameters p = gen_ocp_config();
  p.dataaccept = false;
  p.respaccept = false;
  // multi-threaded; all transfers are thread-busy-exact
  p.threads = NR_THREADS;
  p.mthreadbusy = true;
  p.sdatathreadbusy = true;
  p.sthreadbusy = true;
  p.mthreadbusy_exact = true;
  p.sdatathreadbusy_exact = true;
  p.sthreadbusy_exact = true;
  return p;
}


// traits class for TL0 signal types
class THREAD_TYPES: public tl1_tl0::POD_DEFAULT_TL0_TYPES<> {
public:
  typedef tl1_tl0::TIEOFF<1> SCMDACCEPT_T;
  typedef unsigned MBURSTLENGTH_T;
  typedef bool MBURSTSINGLEREQ_T;
  typedef bool MDATAVALID_T;
  typedef bool MDATALAST_T;
  typedef bool SRESPLAST_T;
  typedef unsigned MDATATAGID_T;
  typedef unsigned MTAGID_T;
  typedef unsigned STAGID_T;
  typedef unsigned MDATATHREADID_T;
  typedef unsigned MTHREADBUSY_T;
  typedef unsigned MTHREADID_T;
  typedef unsigned SDATATHREADBUSY_T;
  typedef unsigned STHREADBUSY_T;
  typedef unsigned STHREADID_T;
};


// simulation duration determined by these parameters - leave space for
// final burst to complete
#define MAX_ADDR 100000
#define MEM_SIZE 100064
#include "tb_common.h"

#define LATENCY 60


class ocp_tl0_initiator_inner: public sc_module {
SC_HAS_PROCESS(ocp_tl0_initiator_inner);
public:
  ocp_tl0_initiator_inner(sc_module_name n, unsigned index, ostream *osp):
    sc_module(n),
    Clk("Clk"),
    MAddr("MAddr"),
    MCmd("MCmd"),
    MData("MData"),
    SData("SData"),
    SCmdAccept("SCmdAccept"),
    SResp("SResp"),
    MReset_n("MReset_n"),
    MBurstLength("MBurstLength"),
    MBurstSingleReq("MBurstSingleReq"),
    MDataValid("MDataValid"),
    MDataLast("MDataLast"),
    SDataAccept("SDataAccept"),
    MRespAccept("MRespAccept"),
    SRespLast("SRespLast"),
    MDataTagID("MDataTagID"),
    MTagID("MTagID"),
    STagID("STagID"),
    cycle(index * 10001),
    next_req_cycle(cycle + 1),
    next_data_cycle(cycle + 1),
    resps_rxd(0),
    resps_wanted(0),
    curr_addr(0),
    curr_burst_len(1),
    tag(0),
    dtag(0),
    req_active(false),
    data_active(false),
    curr_is_read(true),
    os(*osp)
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    MReset_n.initialize(true);
    MCmd.initialize(0);
    MDataValid.initialize(false);
    MRespAccept.initialize(false);

    fill_initiator_memory(memory, index);
  }

  sc_in<bool> Clk;
  sc_out<unsigned> MAddr;
  sc_out<unsigned> MCmd;
  sc_out<unsigned> MData;
  sc_in<unsigned> SData;
  sc_in<bool> SCmdAccept;
  sc_in<unsigned> SResp;
  sc_out<bool> MReset_n;

  sc_out<unsigned> MBurstLength;
  sc_out<bool> MBurstSingleReq;
  sc_out<bool> MDataValid;
  sc_out<bool> MDataLast;
  sc_in<bool> SDataAccept;
  sc_out<bool> MRespAccept;
  sc_in<bool> SRespLast;
  sc_out<unsigned> MDataTagID;
  sc_out<unsigned> MTagID;
  sc_in<unsigned> STagID;

private:
  void on_Clk_rising() {
    if((curr_addr >= MAX_ADDR) && (resps_rxd == resps_wanted)) {
      stop_sim(NR_THREADS * 6);
      return;
    }

    // request side
    if(req_active && SCmdAccept)
      os << cycle << " I: req accepted " << resps_wanted << endl;

    if(SCmdAccept || !req_active) {
      if((cycle >= next_req_cycle) && (curr_addr < MAX_ADDR)) {
        // SRMD bursts
        if(curr_burst_len == 1) {
          if((curr_addr & 31) == 0) curr_burst_len = 8;
        } else curr_burst_len /= 2;

        // alternate WR and RD
        curr_is_read = !curr_is_read;

        MBurstLength = curr_burst_len;
        MAddr = curr_addr;
        MBurstSingleReq = true;
        MTagID = tag;
        resp_tag.push_back(tag);
        tag = (tag + 1) % 5;
        if(curr_is_read) {
          MCmd = 2;
          rd_addr.push_back(curr_addr);
          resps_wanted += curr_burst_len;
          os << cycle << " I: R req sent @ " << curr_addr << ", Tag " << resp_tag.back() << endl;
        } else {
          MCmd = 1;
          rd_addr.push_back(-1);
          wr_addr.push(curr_addr);
          wr_len.push(curr_burst_len);
          resps_wanted += 1;
          os << cycle << " I: W req sent @ " << curr_addr << ", Tag " << resp_tag.back() << endl;
        }
        os << cycle << " I: SRMD req length " << curr_burst_len << endl;

        curr_addr += curr_burst_len * 4;
        req_active = true;

        // wait 1-33 cycles between requests roughly randomly
        unsigned sel = (cycle & 7) ^ ((cycle >> 3) & 7) ^ ((cycle >> 6) & 7);
        next_req_cycle = cycle + 1 + (1 << sel) / 4;
      } else {
        MCmd = 0;
        req_active = false;
      }
    }

    // data side
    if(data_active && SDataAccept)
      os << cycle << " I: data accepted " << resps_wanted << endl;

    if(SDataAccept || !data_active) {
      if((wr_addr.size() > 0) && (cycle >= next_data_cycle)) {
        MDataValid = true;
        data_active = true;
        int &addr(wr_addr.front());
        int &len(wr_len.front());
        unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
        MData = d;
        os << cycle << " I: W data: " << d << endl;
        os << cycle << " I: W data sent @ " << addr << ", Tag " << dtag << endl;
        os << cycle << " I: W length " << len << endl;
        MDataTagID = dtag;
        if(1 == len) {
          wr_addr.pop();
          wr_len.pop();
          dtag = (dtag + 1) % 5;
          MDataLast = true;
        } else {
          addr += 4;
          len--;
          MDataLast = false;
        }
        // wait 1,1,2,1,1,2,1,1,2,1,... cycles between data
        next_data_cycle = cycle + 1 + ((cycle / 2) & 1);
      } else {
        MDataValid = false;
        data_active = false;
      }
    }

    // response side - accept in 8 cycles out of 10
    if(MRespAccept && (SResp != 0)) {
      unsigned rtag = STagID;
      resps_rxd++;
      list<int>::iterator ai = rd_addr.begin();
      list<unsigned>::iterator ti = resp_tag.begin();
      while((*ti) != rtag) {++ti; ++ai;}
      int &addr(*ai);
      if(addr >= 0) {
        *reinterpret_cast<unsigned *>(memory + addr) = SData;
        os << cycle << " I: R data: " <<  SData << endl;
        os << cycle << " I: R resp received @ " << addr << ", Tag " << rtag << endl;
      } else {
        os << cycle << " I: W resp received" << ", Tag " << rtag << endl;
      }
      if(SRespLast) {
        rd_addr.erase(ai);
        resp_tag.erase(ti);
        os << cycle << " I: resp last" << endl;
      } else {
        addr += 4;
      }
    }
    MRespAccept = ((cycle % 10) < 8);

    cycle++;
  }

  char memory[MEM_SIZE];
  unsigned cycle;
  unsigned next_req_cycle, next_data_cycle, resps_rxd, resps_wanted;
  unsigned curr_addr, curr_burst_len, tag, dtag;
  bool req_active, data_active, curr_is_read;
  queue<int> wr_addr, wr_len;
  list<int> rd_addr;
  list<unsigned> resp_tag;
  ostream &os;
};


class ocp_tl1_initiator_inner: public sc_module {
SC_HAS_PROCESS(ocp_tl1_initiator_inner);
public:
  ocp_tl1_initiator_inner(sc_module_name n, unsigned index, ostream *osp):
    sc_module(n),
    Clk("Clk"),
    ocp("ocp", this, &ocp_tl1_initiator_inner::ocp_timing_update),
    cycle(index * 10001),
    next_req_cycle(cycle + 1),
    next_data_cycle(cycle + 1),
    resps_rxd(0),
    resps_wanted(0),
    curr_addr(0),
    curr_burst_len(1),
    tag(0),
    dtag(0),
    wr_len(0),
    rd_len(0),
    req_active(false),
    data_active(false),
    curr_is_read(true),
    scmdaccept(false),
    sdataaccept(false),
    mrespaccept(false),
    sresp(0),
    os(*osp),
    srespT(SC_ZERO_TIME)
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    ocp.set_ocp_config(gen_ocp_config());
    ocp.activate_synchronization_protection();
    ocp.register_nb_transport_bw(this, &ocp_tl1_initiator_inner::nb_transport_bw);

    fill_initiator_memory(memory, index);
  }

  sc_in<bool> Clk;
  ocp_master_socket_tl1<32> ocp;

  void end_of_elaboration() {set_master_timing();}

private:
  struct ext: tlm_extension<ext> {
    void free() {delete this;}
    void copy_from(tlm_extension_base const &) {}
    tlm_extension_base *clone() const {return 0;}
    static ext &get(tlm_generic_payload &pl) {
      ext *e = pl.get_extension<ext>();
      if(e == 0) {
        e = new ext;
        pl.set_extension<ext>(e);
      }
      return *e;
    }
    unsigned addr, tag;
  };

  void on_Clk_rising() {
    if((curr_addr >= MAX_ADDR) && (resps_rxd == resps_wanted)) {
      stop_sim(NR_THREADS);
      return;
    }

    // request side
    if(req_active && scmdaccept)
      os << cycle << " I: req accepted " << resps_wanted << endl;

    if(scmdaccept || !req_active) {
      if((cycle >= next_req_cycle) && (curr_addr < MAX_ADDR)) {
        // SRMD bursts
        if(curr_burst_len == 1) {
          if((curr_addr & 31) == 0) curr_burst_len = 8;
        } else curr_burst_len /= 2;

        // alternate WR and RD
        curr_is_read = !curr_is_read;

        tlm_generic_payload *curr = ocp.get_transaction();
        leak_test(*curr);
        extension_api::validate_extension<srmd>(*curr);
        curr->set_byte_enable_ptr(0);
        curr->set_response_status(TLM_INCOMPLETE_RESPONSE);
        curr->set_address(curr_addr);
        curr->set_data_ptr((unsigned char*)memory + curr_addr);
        curr->set_data_length(4 * curr_burst_len);
        curr->set_streaming_width(4 * curr_burst_len);
        tag_id *tgid;
        extension_api::get_extension<tag_id>(tgid, *curr);
        tgid->value = tag;
        extension_api::validate_extension<tag_id>(*curr);
        ext &e(ext::get(*curr));
        e.tag = tag;
        e.addr = curr_addr;
        if(curr_is_read) {
          curr->set_read();
          resps_wanted += curr_burst_len;
          os << cycle << " I: R req sent @ " << curr_addr << ", Tag " << tag << endl;
        } else {
          curr->set_write();
          ocp.validate_extension<posted>(*curr);
          wr_addr.push(curr);
          resps_wanted += 1;
          os << cycle << " I: W req sent @ " << curr_addr << ", Tag " << tag << endl;
        }
        tag = (tag + 1) % 5;
        os << cycle << " I: SRMD req length " << curr_burst_len << endl;

        tlm_phase ph = BEGIN_REQ;
        sc_time t = SC_ZERO_TIME;
        if(ocp->nb_transport_fw(*curr, ph, t) == TLM_UPDATED) {
          sc_assert(ph == END_REQ);
          scmdaccept = true;
        } else {
          scmdaccept = false;
        }

        curr_addr += curr_burst_len * 4;
        req_active = true;

        // wait 1-33 cycles between requests roughly randomly
        unsigned sel = (cycle & 7) ^ ((cycle >> 3) & 7) ^ ((cycle >> 6) & 7);
        next_req_cycle = cycle + 1 + (1 << sel) / 4;
      } else {
        req_active = false;
      }
    }

    // data side
    if(data_active && sdataaccept)
      os << cycle << " I: data accepted " << resps_wanted << endl;

    if(sdataaccept || !data_active) {
      if((wr_addr.size() > 0) && (cycle >= next_data_cycle)) {
        tlm_generic_payload *curr = wr_addr.front();

        int addr = curr->get_address() +  4 * wr_len;
        unsigned len = curr->get_data_length() / 4;
        unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
        os << cycle << " I: W data: " << d << endl;
        os << cycle << " I: W data sent @ " << addr << ", Tag " << dtag << endl;
        os << cycle << " I: W length " << len - wr_len << endl;

        tlm_phase ph = BEGIN_DATA;
        sc_time t = SC_ZERO_TIME;
        if(ocp->nb_transport_fw(*curr, ph, t) == TLM_UPDATED) {
          sc_assert(ph == END_DATA);
          sdataaccept = true;
        } else {
          sdataaccept = false;
        }
        data_active = true;

        if(len == ++wr_len) {
          wr_addr.pop();
          wr_len = 0;
          dtag = (dtag + 1) % 5;
        }

        // wait 1,1,2,1,1,2,1,1,2,1,... cycles between data
        next_data_cycle = cycle + 1 + ((cycle / 2) & 1);
      } else {
        data_active = false;
      }
    }

    // response side
    if(mrespaccept && (sresp != 0)) {
      ext &e(ext::get(*sresp));
      resps_rxd++;
      if(sresp->is_read()) {
        int addr = e.addr + 4 * rd_len;
        unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
        os << cycle << " I: R data: " << d << endl;
        os << cycle << " I: R resp received @ " << addr << ", Tag " << e.tag << endl;
        if(4 * (++rd_len) == sresp->get_data_length()) {
          rd_len = 0;
          ocp.release_transaction(sresp);
          os << cycle << " I: resp last" << endl;
        }
      } else {
        os << cycle << " I: W resp received" << ", Tag " << e.tag << endl;
        os << cycle << " I: resp last" << endl;
        ocp.release_transaction(sresp);
      }
      sresp = 0;
    }
    mrespaccept = ((cycle % 10) < 8);
    if(mrespaccept && (sresp != 0)) {
      tlm_phase ph = END_RESP;
      sc_time t = SC_ZERO_TIME;
      ocp->nb_transport_fw(*sresp, ph, t);
    }

    cycle++;
  }

  tlm_sync_enum nb_transport_bw(
    tlm_generic_payload &pl, tlm_phase &ph, sc_time &ta)
  {
    if(ph == END_REQ) {
      scmdaccept = true;
      return TLM_ACCEPTED;
    }
    if(ph == END_DATA) {
      sdataaccept = true;
      return TLM_ACCEPTED;
    }
    sc_assert(ph == BEGIN_RESP);
    sresp = &pl;
    if(mrespaccept) {
      ph = END_RESP;
      return TLM_UPDATED;
    } else {
      return TLM_ACCEPTED;
    }
  }

  char memory[MEM_SIZE];
  unsigned cycle;
  unsigned next_req_cycle, next_data_cycle, resps_rxd, resps_wanted;
  unsigned curr_addr, curr_burst_len, tag, dtag, wr_len, rd_len;
  bool req_active, data_active, curr_is_read;
  bool scmdaccept, sdataaccept, mrespaccept;
  tlm_generic_payload *sresp;
  queue<tlm_generic_payload *> wr_addr;
  ostream &os;
  sc_time srespT;

  // not sure if this is redundant when there is no resp-accept in the OCP
  // configuration.  It still exists in the TLM2
  void ocp_timing_update(ocp_tl1_slave_timing times) {
    if(srespT != times.ResponseGrpStartTime) {
      srespT = times.ResponseGrpStartTime;
      set_master_timing();
    }
  }

  void set_master_timing() {
    ocp_tl1_master_timing mytimes;
    mytimes.MRespAcceptStartTime = srespT + sc_get_time_resolution();
    ocp.set_master_timing(mytimes);
  }
};


class ocp_tl0_target_inner: public sc_module {
SC_HAS_PROCESS(ocp_tl0_target_inner);
public:
  ocp_tl0_target_inner(sc_module_name n, unsigned index, ostream *osp):
    sc_module(n),
    Clk("Clk"),
    MAddr("MAddr"),
    MCmd("MCmd"),
    MData("MData"),
    SData("SData"),
    SCmdAccept("SCmdAccept"),
    SResp("SResp"),
    MReset_n("MReset_n"),
    MBurstLength("MBurstLength"),
    MBurstSingleReq("MBurstSingleReq"),
    MDataValid("MDataValid"),
    MDataLast("MDataLast"),
    SDataAccept("SDataAccept"),
    MRespAccept("MRespAccept"),
    SRespLast("SRespLast"),
    MDataTagID("MDataTagID"),
    MTagID("MTagID"),
    STagID("STagID"),
    cycle(index * 10001),
    resp_active(false),
    os(*osp)
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos() << MReset_n.neg();

    SResp.initialize(0);
    SCmdAccept.initialize(false);
    SDataAccept.initialize(false);

    fill_target_memory(memory, index);
  }

  sc_in<bool> Clk;
  sc_in<unsigned> MAddr;
  sc_in<unsigned> MCmd;
  sc_in<unsigned> MData;
  sc_out<unsigned> SData;
  sc_out<bool> SCmdAccept;
  sc_out<unsigned> SResp;
  sc_in<bool> MReset_n;

  sc_in<unsigned> MBurstLength;
  sc_in<bool> MBurstSingleReq;
  sc_in<bool> MDataValid;
  sc_in<bool> MDataLast;
  sc_out<bool> SDataAccept;
  sc_in<bool> MRespAccept;
  sc_out<bool> SRespLast;
  sc_in<unsigned> MDataTagID;
  sc_in<unsigned> MTagID;
  sc_out<unsigned> STagID;

private:
  void on_Clk_rising() {
    if(!MReset_n) {
      // reset all state
      while(rd_addr.size() > 0) rd_addr.pop_front();
      while(resp_tag.size() > 0) resp_tag.pop_front();
      while(rd_len.size() > 0) rd_len.pop_front();
      while(wr_addr.size() > 0) wr_addr.pop();
      while(rd_cycle.size() > 0) rd_cycle.pop();
      while(wr_cycle.size() > 0) wr_cycle.pop();
    } else {
      // response side
      if(resp_active && MRespAccept)
        os << cycle << " T: resp accepted " << endl;

      if(MRespAccept || !resp_active) {
        SResp = 0;
        resp_active = false;
        if(rd_addr.size() > 0) {
          bool is_read = (rd_addr.front() >= 0);
          if(is_read && (rd_cycle.size() > 0) && (cycle >= rd_cycle.front())) {
            // do a read response
            SResp = 1;
            resp_active = true;
            STagID = resp_tag.front();
            int &addr(rd_addr.front());
            unsigned &len(rd_len.front());
            unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
            SData = d;
            os << cycle << " T: R data: " <<  d << endl;
            os << cycle << " T: R resp sent @ " << addr << ", Tag " << resp_tag.front() << endl;
            if(len == 1) {
              SRespLast = true;
              rd_addr.pop_front();
              rd_len.pop_front();
              resp_tag.pop_front();
            } else {
              SRespLast = false;
              addr += 4;
              len--;
            }
            rd_cycle.pop();
          }
          if((!is_read) && (wr_cycle.size() > 0) && (cycle >= wr_cycle.front())) {
            // do a write response
            SResp = 1;
            resp_active = true;
            STagID = resp_tag.front();
            os << cycle << " T: W resp sent" << ", Tag " << resp_tag.front() << endl;
            SRespLast = true;
            rd_addr.pop_front();
            resp_tag.pop_front();
            wr_cycle.pop();
          }
        }
      }

      // request side
      // accept in cycles 5,10,15,... and impose LATENCY cycle minimum latency
      unsigned cmd = MCmd;
      if(SCmdAccept && (cmd != 0)) {
        unsigned tag = MTagID;
        // insert into response queue not necessarily in order
        bool overtake = cycle & 1;
        list<int>::iterator ain = rd_addr.end();
        list<int>::iterator ai = (ain--);
        list<unsigned>::iterator tin = resp_tag.end();
        list<unsigned>::iterator ti = (tin--);
        list<unsigned>::iterator li = rd_len.end();
        while((tag != *tin) && overtake && (resp_tag.begin() != tin) &&
          ((cmd == 2) || ((*ain) >= 0))) {
          ai = (ain--);
          ti = (tin--);
          if((*ai) >= 0) --li;
        }
        if(cmd == 1) {
          sc_assert(MBurstSingleReq);
          rd_addr.insert(ai, -1);
          resp_tag.insert(ti, tag);
          wr_addr.push(MAddr);
          os << cycle << " T: W req received @ " << MAddr << ", Tag " << tag << endl;
          os << cycle << " T: W req length " << MBurstLength << endl;
        } else {
          sc_assert(MBurstSingleReq);
          rd_addr.insert(ai, MAddr);
          resp_tag.insert(ti, tag);
          rd_len.insert(li, MBurstLength);
          os << cycle << " T: R req received @ " << MAddr << ", Tag " <<  tag << endl;
          os << cycle << " T: R req length " << MBurstLength << endl;
          for(unsigned i=0; i<MBurstLength; i++)
            rd_cycle.push((i/4)*4 + cycle + LATENCY);
        }
      }
      bool new_scmdaccept = ((cycle % 5) == 0) || ((cycle % 17) == 0);
      SCmdAccept = new_scmdaccept;

      // data side
      // accept in cycles 3,6,9,12,...
      // but do not accept data before accepting request
      if(SDataAccept && MDataValid) {
        unsigned &addr(wr_addr.front());
        unsigned d = MData;
        os << cycle << " T: W data: " << d << endl;
        *reinterpret_cast<unsigned *>(memory + addr) = d;
        if(MDataLast) {
          wr_addr.pop();
          wr_cycle.push(cycle + LATENCY);
          os << cycle << " T: W data last" << endl;
        } else {
          addr += 4;
        }
      }
      SDataAccept = (((cycle % 3) == 0) || ((cycle % 13) == 0))
        && ((wr_addr.size() > 0) || new_scmdaccept);

      cycle++;
    }
  }

  unsigned cycle;
  bool resp_active;
  char memory[MEM_SIZE];
  list<int> rd_addr;
  list<unsigned> resp_tag, rd_len;
  queue<unsigned> rd_cycle, wr_cycle, wr_addr;
  ostream &os;
};


class ocp_tl1_target_inner: public sc_module {
SC_HAS_PROCESS(ocp_tl1_target_inner);
public:
  ocp_tl1_target_inner(sc_module_name n, unsigned index, ostream *osp):
    sc_module(n),
    Clk("Clk"),
    ocp("ocp", this, &ocp_tl1_target_inner::ocp_timing_update),
    cycle(index * 10001),
    in_reset(false),
    scmdaccept(false),
    sdataaccept(false),
    mrespaccept(false),
    resp_active(false),
    mcmd(0),
    mdatahs(0),
    wr_len(0),
    rd_len(0),
    wr_addr_size(0),
    os(*osp),
    cmdT(SC_ZERO_TIME),
    dataT(SC_ZERO_TIME)
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    ocp.set_ocp_config(gen_ocp_config());
    ocp.activate_synchronization_protection();
    ocp.register_nb_transport_fw(this, &ocp_tl1_target_inner::nb_transport_fw);

    fill_target_memory(memory, index);
  }

  sc_in<bool> Clk;
  ocp_slave_socket_tl1<32> ocp;

  void end_of_elaboration() {set_slave_timing();}

private:
  void on_Clk_rising() {
    if(in_reset) return;

    // response side
    if(resp_active && mrespaccept)
      os << cycle << " T: resp accepted " << endl;

    if(mrespaccept || !resp_active) {
      resp_active = false;
      if(rd_addr.size() > 0) {
        tlm_generic_payload *pl = rd_addr.front();
        if(pl->is_read() && (rd_cycle.size() > 0) && (cycle >= rd_cycle.front()))
        {
          // do a read response
          pl->set_response_status(TLM_OK_RESPONSE);
          int addr = pl->get_address() + 4 * rd_len;
          unsigned *local_addr = reinterpret_cast<unsigned *>(memory + addr);
          unsigned *init_addr =
            reinterpret_cast<unsigned *>(pl->get_data_ptr() + 4 * rd_len);
          *init_addr = *local_addr;
          os << cycle << " T: R data: " <<  *local_addr << endl;
            os << cycle << " T: R resp sent @ " << addr << ", Tag " << resp_tag.front() << endl;
          tlm_phase ph = BEGIN_RESP;
          sc_time t = SC_ZERO_TIME;
          mrespaccept = (ocp->nb_transport_bw(*pl, ph, t) == TLM_UPDATED);
          resp_active = true;
          if((++rd_len) * 4 == pl->get_data_length()) {
            rd_addr.pop_front();
            resp_tag.pop_front();
            pl->release();
            rd_len = 0;
          }
          rd_cycle.pop();
        }
        if(pl->is_write() && (wr_cycle.size() > 0) && (cycle >= wr_cycle.front()))
        {
          // do a write response
          pl->set_response_status(TLM_OK_RESPONSE);
            os << cycle << " T: W resp sent" << ", Tag " << resp_tag.front() << endl;
          tlm_phase ph = BEGIN_RESP;
          sc_time t = SC_ZERO_TIME;
          mrespaccept = (ocp->nb_transport_bw(*pl, ph, t) == TLM_UPDATED);
          resp_active = true;
          rd_addr.pop_front();
          resp_tag.pop_front();
          pl->release();
          wr_cycle.pop();
        }
      }
    }

    // request side
    // accept in cycles 5,10,15,... and impose LATENCY cycle minimum latency
    if(scmdaccept && (mcmd != 0)) {
      sc_assert(extension_api::get_extension<srmd>(*mcmd) != 0);
      tag_id *tgid;
      sc_assert(extension_api::get_extension<tag_id>(tgid, *mcmd));
      unsigned tag = tgid->value;
      // insert into response queue not necessarily in order
      // do not allow writes to overtake writes or we may send a response
      // before getting all data
      bool overtake = cycle & 1;
      list<tlm_generic_payload *>::iterator ain = rd_addr.end();
      list<tlm_generic_payload *>::iterator ai = (ain--);
      list<unsigned>::iterator tin = resp_tag.end();
      list<unsigned>::iterator ti = (tin--);
      while((tag != *tin) && overtake && (resp_tag.begin() != tin) &&
        ((mcmd->is_read()) || ((*ain)->is_read()))) {
        ai = (ain--);
        ti = (tin--);
      }
      rd_addr.insert(ai, mcmd);
      resp_tag.insert(ti, tag);
      unsigned len = mcmd->get_data_length() / 4;
      unsigned addr = mcmd->get_address();
      if(mcmd->is_write()) {
        os << cycle << " T: W req received @ " << addr << ", Tag " << tag << endl;
        os << cycle << " T: W req length " << len << endl;
        ++wr_addr_size;
      } else {
        os << cycle << " T: R req received @ " << addr << ", Tag " <<  tag << endl;
        os << cycle << " T: R req length " << len << endl;
        for(unsigned i=0; i<len; i++) rd_cycle.push((i/4)*4 + cycle + LATENCY);
      }
      mcmd = 0;
    }
    scmdaccept = ((cycle % 5) == 0) || ((cycle % 17) == 0);
    if(scmdaccept && (mcmd != 0)) {
      tlm_phase ph = END_REQ;
      sc_time t = SC_ZERO_TIME;
      ocp->nb_transport_bw(*mcmd, ph, t);
    }

    // data side
    // accept in cycles 3,6,9,12,...
    // but do not accept data before accepting request
    if(sdataaccept && (mdatahs != 0)) {
      int addr = mdatahs->get_address() + wr_len * 4;
      unsigned *local_addr = reinterpret_cast<unsigned *>(memory + addr);
      unsigned *init_addr =
        reinterpret_cast<unsigned *>(mdatahs->get_data_ptr() + 4 * wr_len);
      *local_addr = *init_addr;
      os << cycle << " T: W data: " << *init_addr << endl;
      if(4 * (++wr_len) == mdatahs->get_data_length()) {
        wr_cycle.push(cycle + LATENCY);
        wr_len = 0;
        --wr_addr_size;
        os << cycle << " T: W data last" << endl;
      }
      mdatahs = 0;
    }
    sdataaccept = (((cycle % 3) == 0) || ((cycle % 13) == 0))
        && ((wr_addr_size > 0) || scmdaccept);
    if(sdataaccept && (mdatahs != 0)) {
      tlm_phase ph = END_DATA;
      sc_time t = SC_ZERO_TIME;
      ocp->nb_transport_bw(*mdatahs, ph, t);
    }

    cycle++;
  }

  tlm_sync_enum nb_transport_fw(
    tlm_generic_payload &pl, tlm_phase &ph, sc_time &ta)
  {
    if(ph == END_RESET) {
      in_reset = false;
      while(rd_addr.size() > 0) {
        rd_addr.front()->release();
        rd_addr.pop_front();
      }
      return TLM_ACCEPTED;
    }
    if(ph == BEGIN_RESET) {
      in_reset = true;
      mcmd = 0;
      ocp.reset();
      while(rd_cycle.size() > 0) rd_cycle.pop();
      while(wr_cycle.size() > 0) wr_cycle.pop();
      return TLM_ACCEPTED;
    }
    if(ph == BEGIN_REQ) {
      leak_test(pl);
      mcmd = &pl;
      pl.acquire();
      if(scmdaccept) {
        ph = END_REQ;
        return TLM_UPDATED;
      } else {
        return TLM_ACCEPTED;
      }
    }
    if(ph == BEGIN_DATA) {
      mdatahs = &pl;
      if(sdataaccept) {
        ph = END_DATA;
        return TLM_UPDATED;
      } else {
        return TLM_ACCEPTED;
      }
    }
    sc_assert(ph == END_RESP);
    mrespaccept = true;
    return TLM_ACCEPTED;
  }

  unsigned cycle;
  bool in_reset;
  bool scmdaccept, sdataaccept, mrespaccept, resp_active;
  tlm_generic_payload *mcmd, *mdatahs;
  unsigned wr_len, rd_len, wr_addr_size;
  char memory[MEM_SIZE];
  list<tlm_generic_payload *> rd_addr;
  list<unsigned> resp_tag;
  queue<unsigned> rd_cycle, wr_cycle;
  ostream &os;
  sc_time cmdT, dataT;

  void ocp_timing_update(ocp_tl1_master_timing times) {
    if((cmdT != times.RequestGrpStartTime) ||
       (dataT != times.DataHSGrpStartTime)) {
      cmdT = times.RequestGrpStartTime;
      dataT = times.DataHSGrpStartTime;
      set_slave_timing();
    }
  }

  void set_slave_timing() {
    ocp_tl1_slave_timing mytimes;
    mytimes.SCmdAcceptStartTime = cmdT + sc_get_time_resolution();
    mytimes.SDataAcceptStartTime = dataT + sc_get_time_resolution();
    ocp.set_slave_timing(mytimes);
  }
};


class ocp_tl0_initiator: public sc_module {
SC_HAS_PROCESS(ocp_tl0_initiator);
public:
  ocp_tl0_initiator(sc_module_name n):
    sc_module(n),
    req_priority(0),
    data_priority(0),
    os(sc_module::name())
  {
    SC_METHOD(request);
    // choose a request from a non-busy thread, copy it out, set scmdaccept
    dont_initialize();
    sensitive << SThreadBusy;
    for(unsigned i = 0; i < NR_THREADS; i++) sensitive << MCmdInt[i];

    SC_METHOD(datahs);
    // like request, but be careful about ordering
    dont_initialize();
    sensitive << SDataThreadBusy;
    for(unsigned i = 0; i < NR_THREADS; i++) sensitive << MDataValidInt[i];
    sensitive << MCmd << MThreadID << MBurstLength;


    SC_METHOD(on_Clk_rising);
    // for responses, one register per thread
    dont_initialize();
    sensitive << Clk.pos();

    MReset_n.initialize(true);
    MCmd.initialize(0);
    MDataValid.initialize(false);
    MThreadBusy.initialize(~0);
    req_change = false;
    data_change = false;

    for(unsigned i = 0; i < NR_THREADS; i++) {
      SRespInt[i] = 0;
      char inm[7] = "inner?";
      inm[5] = '0' + i;
      ocp_tl0_initiator_inner &ni(*new ocp_tl0_initiator_inner(inm, i, &oss[i]));
      initiators[i] = &ni;
      ni.Clk(Clk);
      ni.MReset_n(MReset_nInt[i]);
      ni.MAddr(MAddrInt[i]);
      ni.MCmd(MCmdInt[i]);
      ni.SCmdAccept(SCmdAcceptInt[i]);
      ni.MBurstLength(MBurstLengthInt[i]);
      ni.MBurstSingleReq(MBurstSingleReqInt[i]);
      ni.MTagID(MTagIDInt[i]);
      ni.MData(MDataInt[i]);
      ni.MDataValid(MDataValidInt[i]);
      ni.MDataLast(MDataLastInt[i]);
      ni.SDataAccept(SDataAcceptInt[i]);
      ni.MDataTagID(MDataTagIDInt[i]);
      ni.SData(SDataInt[i]);
      ni.SResp(SRespInt[i]);
      ni.SRespLast(SRespLastInt[i]);
      ni.MRespAccept(MRespAcceptInt[i]);
      ni.STagID(STagIDInt[i]);
    }

    SC_METHOD(write_trace);
    sensitive << trace_e;
  }

  ~ocp_tl0_initiator() {
    for(unsigned i = 0; i < NR_THREADS; i++) delete initiators[i];
  }

  sc_in<bool> Clk;
  sc_out<unsigned> MAddr;
  sc_out<unsigned> MCmd;
  sc_out<unsigned> MData;
  sc_in<unsigned> SData;
  sc_in<unsigned> SResp;
  sc_out<bool> MReset_n;

  sc_out<unsigned> MBurstLength;
  sc_out<bool> MBurstSingleReq;
  sc_out<bool> MDataValid;
  sc_out<bool> MDataLast;
  sc_in<bool> SRespLast;
  sc_out<unsigned> MDataTagID;
  sc_out<unsigned> MTagID;
  sc_in<unsigned> STagID;
  sc_out<unsigned> MThreadID;
  sc_out<unsigned> MDataThreadID;
  sc_in<unsigned> SThreadBusy;
  sc_in<unsigned> SDataThreadBusy;
  sc_out<unsigned> MThreadBusy;
  sc_in<unsigned> SThreadID;

private:
  ocp_tl0_initiator_inner *initiators[NR_THREADS];

  sc_signal<bool> MReset_nInt[NR_THREADS];

  sc_signal<unsigned> MAddrInt[NR_THREADS];
  sc_signal<unsigned> MCmdInt[NR_THREADS];
  sc_signal<bool> SCmdAcceptInt[NR_THREADS];
  sc_signal<unsigned> MBurstLengthInt[NR_THREADS];
  sc_signal<bool> MBurstSingleReqInt[NR_THREADS];
  sc_signal<unsigned> MTagIDInt[NR_THREADS];

  sc_signal<unsigned> MDataInt[NR_THREADS];
  sc_signal<bool> MDataValidInt[NR_THREADS];
  sc_signal<bool> MDataLastInt[NR_THREADS];
  sc_signal<bool> SDataAcceptInt[NR_THREADS];
  sc_signal<unsigned> MDataTagIDInt[NR_THREADS];

  sc_signal<unsigned> SDataInt[NR_THREADS];
  sc_signal<unsigned> SRespInt[NR_THREADS];
  sc_signal<bool> SRespLastInt[NR_THREADS];
  sc_signal<bool> MRespAcceptInt[NR_THREADS];
  sc_signal<unsigned> STagIDInt[NR_THREADS];

  sc_signal<unsigned> req_priority, data_priority;
  sc_signal<unsigned> data_owed[NR_THREADS];
  sc_signal<bool> req_change, data_change;

  void request() {
    unsigned cmd_l = 0;
    unsigned tb = SThreadBusy;
    for(unsigned i = 0; i < NR_THREADS; i++) {
      unsigned idx = (i + req_priority) % NR_THREADS;
      if(((~tb) & (1 << idx)) && (MCmdInt[idx] != 0) && (cmd_l == 0)) {
        cmd_l = MCmdInt[idx];
        MAddr = MAddrInt[idx];
        SCmdAcceptInt[idx] = true;
        MBurstLength = MBurstLengthInt[idx];
        MBurstSingleReq = MBurstSingleReqInt[idx];
        MTagID = MTagIDInt[idx];
        MThreadID = idx;
      } else {
        SCmdAcceptInt[idx] = false;
      }
    }
    MCmd = cmd_l;
    req_change = (cmd_l != 0);
  }

  void datahs() {
    bool datavalid_l = false;
    unsigned tb = SDataThreadBusy;
    for(unsigned i = 0; i < NR_THREADS; i++) {
      unsigned idx = (i + data_priority) % NR_THREADS;
      if(((~tb) & (1 << idx)) && MDataValidInt[idx] && !datavalid_l &&
        (data_owed[idx] +
          ((MCmd != 0) && (MThreadID == idx) ? MBurstLength : 0) > 0)) {
        MData = MDataInt[idx];
        datavalid_l = true;
        MDataLast = MDataLastInt[idx];
        SDataAcceptInt[idx] = true;
        MDataTagID = MDataTagIDInt[idx];
        MDataThreadID = idx;
      } else {
        SDataAcceptInt[idx] = false;
      }
    }
    MDataValid = datavalid_l;
    data_change = datavalid_l;
  }

  void on_Clk_rising() {
    // round robin
    if(req_change) req_priority = (req_priority + 1) % NR_THREADS;
    if(data_change) data_priority = (data_priority + 1) % NR_THREADS;

    // responses
    unsigned tin = SThreadID;
    unsigned tout = 0;
    for(unsigned i = 0; i < NR_THREADS; i++) {
      // set the threadbusy
      if(!MRespAcceptInt[i] && (SRespInt[i] != 0)) tout |= (1 << i);
      // clear any register that's accepted
      if(MRespAcceptInt[i]) SRespInt[i] = 0;
    }
    // capture new resp
    if(SResp != 0) {
      SDataInt[tin] = SData;
      SRespInt[tin] = SResp;
      SRespLastInt[tin] = SRespLast;
      STagIDInt[tin] = STagID;
      tout |= (1 << tin);
    }
    MThreadBusy = tout;

    // record the number of data handshakes owed for each thread
    for(unsigned i = 0; i < NR_THREADS; i++) {
      unsigned do_l = data_owed[i];
      if((MCmd == 1) && (MThreadID == i)) do_l += MBurstLength;
      if(MDataValid && (MDataThreadID == i)) do_l -= 1;
      data_owed[i] = do_l;
    }

    trace_e.notify(SC_ZERO_TIME);
  }

  ofstream os;
  ostringstream oss[NR_THREADS];

  sc_event trace_e;
  void write_trace() {
    for(unsigned i = 0; i < NR_THREADS; i++) {
      os << oss[i].str();
      oss[i].str("");
    }
  }
};


class ocp_tl0_target: public sc_module {
SC_HAS_PROCESS(ocp_tl0_target);
public:
  ocp_tl0_target(sc_module_name n):
    sc_module(n),
    resp_priority(0),
    os(sc_module::name())
  {
    SC_METHOD(response);
    // choose a response from a non-busy thread, copy it out, set mrespaccept
    dont_initialize();
    sensitive << MThreadBusy;
    for(unsigned i = 0; i < NR_THREADS; i++) sensitive << SRespInt[i];

    SC_METHOD(reset);
    // broadcast MReset_n to all inner targets
    dont_initialize();
    sensitive << MReset_n;

    SC_METHOD(on_Clk_rising);
    // for requests and datas, one register per thread, aware of MReset_n
    dont_initialize();
    sensitive << Clk.pos();

    SResp.initialize(0);
    SThreadBusy.initialize(~0);
    SDataThreadBusy.initialize(~0);
    resp_change = false;

    for(unsigned i = 0; i < NR_THREADS; i++) {
      data_owed[i] = 0;
      MReset_nInt[i] = true;
      MCmdInt[i] = 0;
      MDataValidInt[i] = false;
      data_valid_waiting[i] = false;
      char inm[7] = "inner?";
      inm[5] = '0' + i;
      ocp_tl0_target_inner &ni(*new ocp_tl0_target_inner(inm, i, &oss[i]));
      targets[i] = &ni;
      ni.Clk(Clk);
      ni.MReset_n(MReset_nInt[i]);
      ni.MAddr(MAddrInt[i]);
      ni.MCmd(MCmdInt[i]);
      ni.SCmdAccept(SCmdAcceptInt[i]);
      ni.MBurstLength(MBurstLengthInt[i]);
      ni.MBurstSingleReq(MBurstSingleReqInt[i]);
      ni.MTagID(MTagIDInt[i]);
      ni.MData(MDataInt[i]);
      ni.MDataValid(MDataValidInt[i]);
      ni.MDataLast(MDataLastInt[i]);
      ni.SDataAccept(SDataAcceptInt[i]);
      ni.MDataTagID(MDataTagIDInt[i]);
      ni.SData(SDataInt[i]);
      ni.SResp(SRespInt[i]);
      ni.SRespLast(SRespLastInt[i]);
      ni.MRespAccept(MRespAcceptInt[i]);
      ni.STagID(STagIDInt[i]);
    }

    SC_METHOD(write_trace);
    sensitive << trace_e;
  }

  ~ocp_tl0_target() {
    for(unsigned i = 0; i < NR_THREADS; i++) delete targets[i];
  }

  sc_in<bool> Clk;

  sc_in<unsigned> MAddr;
  sc_in<unsigned> MCmd;
  sc_in<unsigned> MData;
  sc_out<unsigned> SData;
  sc_out<unsigned> SResp;
  sc_in<bool> MReset_n;

  sc_in<unsigned> MBurstLength;
  sc_in<bool> MBurstSingleReq;
  sc_in<bool> MDataValid;
  sc_in<bool> MDataLast;
  sc_out<bool> SRespLast;
  sc_in<unsigned> MDataTagID;
  sc_in<unsigned> MTagID;
  sc_out<unsigned> STagID;
  sc_in<unsigned> MThreadID;
  sc_in<unsigned> MDataThreadID;
  sc_out<unsigned> SThreadBusy;
  sc_out<unsigned> SDataThreadBusy;
  sc_in<unsigned> MThreadBusy;
  sc_out<unsigned> SThreadID;

private:
  ocp_tl0_target_inner *targets[NR_THREADS];

  sc_signal<bool> MReset_nInt[NR_THREADS];

  sc_signal<unsigned> MAddrInt[NR_THREADS];
  sc_signal<unsigned> MCmdInt[NR_THREADS];
  sc_signal<bool> SCmdAcceptInt[NR_THREADS];
  sc_signal<unsigned> MBurstLengthInt[NR_THREADS];
  sc_signal<bool> MBurstSingleReqInt[NR_THREADS];
  sc_signal<unsigned> MTagIDInt[NR_THREADS];

  sc_signal<unsigned> MDataInt[NR_THREADS];
  sc_signal<bool> MDataValidInt[NR_THREADS];
  sc_signal<bool> MDataLastInt[NR_THREADS];
  sc_signal<bool> SDataAcceptInt[NR_THREADS];
  sc_signal<unsigned> MDataTagIDInt[NR_THREADS];

  sc_signal<unsigned> SDataInt[NR_THREADS];
  sc_signal<unsigned> SRespInt[NR_THREADS];
  sc_signal<bool> SRespLastInt[NR_THREADS];
  sc_signal<bool> MRespAcceptInt[NR_THREADS];
  sc_signal<unsigned> STagIDInt[NR_THREADS];

  sc_signal<unsigned> resp_priority;
  sc_signal<bool> resp_change;
  unsigned data_owed[NR_THREADS];
  bool data_valid_waiting[NR_THREADS];

  void response() {
    unsigned resp_l = 0;
    bool resp_change_l = false;
    unsigned tb = MThreadBusy;
    for(unsigned i = 0; i < NR_THREADS; i++) {
      unsigned idx = (i + resp_priority) % NR_THREADS;
      if(((~tb) & (1 << idx)) && (SRespInt[idx] != 0) && !resp_change_l) {
        resp_change_l = true;
        resp_l = SRespInt[idx];
        SData = SDataInt[idx];
        MRespAcceptInt[idx] = true;
        SRespLast = SRespLastInt[idx];
        STagID = STagIDInt[idx];
        SThreadID = idx;
      } else {
        MRespAcceptInt[idx] = false;
      }
    }
    SResp = resp_l;
    resp_change = resp_change_l;
  }

  void reset() {
    for(unsigned i = 0; i < NR_THREADS; i++) MReset_nInt[i] = MReset_n;
  }

  void on_Clk_rising() {
    if(!MReset_n) {
      SResp = 0;
      SThreadBusy = (~0);
      SDataThreadBusy = (~0);
      for(unsigned i = 0; i < NR_THREADS; i++) {
        MCmdInt[i] = 0;
        MDataValidInt[i] = false;
      }
      return;
    }

    // round robin
    if(resp_change) resp_priority = (resp_priority + 1) % NR_THREADS;

    // requests
    unsigned tin = MThreadID;
    unsigned tout = 0;
    // clear old reqs
    for(unsigned i = 0; i < NR_THREADS; i++) {
      // set the threadbusy
      if(!SCmdAcceptInt[i] && (MCmdInt[i] != 0)) tout |= (1 << i);
      // clear any register that's accepted
      if(SCmdAcceptInt[i]) MCmdInt[i] = 0;
    }
    // capture new req
    if(MCmd != 0) {
      MAddrInt[tin] = MAddr;
      MCmdInt[tin] = MCmd;
      MBurstLengthInt[tin] = MBurstLength;
      MBurstSingleReqInt[tin] = MBurstSingleReq;
      MTagIDInt[tin] = MTagID;
      if(MCmd == 1) data_owed[tin] += MBurstLength;
      tout |= (1 << tin);
    }
    SThreadBusy = tout;

    // data handshakes
    tin = MDataThreadID;
    tout = 0;
    // clear old data
    for(unsigned i = 0; i < NR_THREADS; i++) {
      // set the threadbusy
      if(data_valid_waiting[i] && (!MDataValidInt[i] || !SDataAcceptInt[i]))
        tout |= (1 << i);
      // set data valid if we were stalled waiting for a write command
      if(data_valid_waiting[i] && (data_owed[i] > 0) && !MDataValidInt[i]) {
        MDataValidInt[i] = true;
        data_owed[tin] -= 1;
      }
      // clear any register that's accepted
      if(SDataAcceptInt[i] && MDataValidInt[i])
        MDataValidInt[i] = data_valid_waiting[i] = false;
    }
    // capture new data
    if(MDataValid) {
      data_valid_waiting[tin] = true;
      if(data_owed[tin] > 0) {
        MDataValidInt[tin] = true;
        data_owed[tin] -= 1;
      }
      MDataInt[tin] = MData;
      MDataLastInt[tin] = MDataLast;
      MDataTagIDInt[tin] = MDataTagID;
      tout |= (1 << tin);
    }
    SDataThreadBusy = tout;

    trace_e.notify(SC_ZERO_TIME);
  }

  ofstream os;
  ostringstream oss[NR_THREADS];

  sc_event trace_e;
  void write_trace() {
    for(unsigned i = 0; i < NR_THREADS; i++) {
      os << oss[i].str();
      oss[i].str("");
    }
  }
};


class ocp_tl1_initiator: public sc_module {
SC_HAS_PROCESS(ocp_tl1_initiator);
public:
  ocp_tl1_initiator(sc_module_name n):
    sc_module(n),
    ocp("ocp", this, &ocp_tl1_initiator::ocp_timing_update),
    ocp_int("ocp_int", this, &ocp_tl1_initiator::ocp_timing_update_int),
    sthreadbusy(~0),
    sdatathreadbusy(~0),
    req_priority(0),
    data_priority(0),
    os(sc_module::name())
  {
    ocp.set_ocp_config(gen_ocp_config_threads());
    ocp.activate_synchronization_protection();
    ocp.register_nb_transport_bw(this, &ocp_tl1_initiator::nb_transport_bw);

    ocp_int.set_ocp_config(gen_ocp_config());
    ocp_int.activate_synchronization_protection();
    ocp_int.register_nb_transport_fw(this, &ocp_tl1_initiator::nb_transport_fw);

    SC_METHOD(request);
    // choose a request from a non-busy thread, copy it out, set scmdaccept
    dont_initialize();
    sensitive << request_e;

    SC_METHOD(datahs);
    // choose a datahs from a non-busy thread, copy it out, set sdataaccept
    dont_initialize();
    sensitive << data_e;

    SC_METHOD(on_Clk_rising);
    // for responses, one register per thread
    dont_initialize();
    sensitive << Clk.pos();

    for(unsigned i = 0; i < NR_THREADS; i++) {
      data_owed[i] = 0;
      req_int[i] = 0;
      data_int[i] = 0;
      resp_int[i] = 0;
      resp_stalled[i] = false;
      char inm[7] = "inner?";
      inm[5] = '0' + i;
      ocp_tl1_initiator_inner &ni(*new ocp_tl1_initiator_inner(inm, i, &oss[i]));
      initiators[i] = &ni;
      ni.ocp(ocp_int);
      ni.Clk(Clk);
    }

    SC_METHOD(write_trace);
    sensitive << trace_e;
  }

  ~ocp_tl1_initiator() {
    for(unsigned i = 0; i < NR_THREADS; i++) delete initiators[i];
  }

  sc_in<bool> Clk;
  ocp_master_socket_tl1<32,1> ocp;

  void end_of_elaboration() {
    set_timing();
  }

private:
  struct ext: tlm_extension<ext> {
    void free() {delete this;}
    void copy_from(tlm_extension_base const &) {}
    tlm_extension_base *clone() const {return 0;}
    static ext &get(tlm_generic_payload &pl) {
      ext *e = pl.get_extension<ext>();
      if(e == 0) {
        e = new ext;
        pl.set_extension<ext>(e);
      }
      return *e;
    }
    unsigned thread;
  };

  ocp_tl1_initiator_inner *initiators[NR_THREADS];
  ocp_slave_socket_tl1<32,0> ocp_int;

  tlm_generic_payload *req_int[NR_THREADS];
  tlm_generic_payload *data_int[NR_THREADS];
  tlm_generic_payload *resp_int[NR_THREADS];
  unsigned sthreadbusy, sdatathreadbusy;

  unsigned req_priority, data_priority;
  unsigned data_owed[NR_THREADS];
  bool resp_stalled[NR_THREADS];

  sc_event request_e, data_e;

  void request() {
    // called once MCmds and SThreadBusy are known to be stable
    // have captured an SThreadBusy from nb_transport_bw
    // have captured a set of MCmd from nb_transport_fw
    for(unsigned i = 0; i < NR_THREADS; i++) {
      unsigned idx = (i + req_priority) % NR_THREADS;
      if(((~sthreadbusy) & (1 << idx)) && (req_int[idx] != 0)) {
        tlm_phase ph(BEGIN_REQ);
        sc_time t(SC_ZERO_TIME);
        ocp->nb_transport_fw(*req_int[idx], ph, t);
        tlm_phase ph1(END_REQ);
        sc_time t1(SC_ZERO_TIME);
        ocp_int[idx]->nb_transport_bw(*req_int[idx], ph1, t1);
        if(req_int[idx]->is_write())
          data_owed[idx] += req_int[idx]->get_data_length();
        // round robin
        req_priority = (req_priority + 1) % NR_THREADS;
        req_int[idx]->release();
        req_int[idx] = 0;
        return;
      }
    }
  }

  void datahs() {
    // called once MDataValids and SDataThreadBusy are known to be stable
    // have captured an SDataThreadBusy from nb_transport_bw
    // have captured a set of MDataValids from nb_transport_fw
    for(unsigned i = 0; i < NR_THREADS; i++) {
      unsigned idx = (i + data_priority) % NR_THREADS;
      if(((~sdatathreadbusy) & (1 << idx)) && (data_int[idx] != 0) && (data_owed[idx] >= 4)) {
        tlm_phase ph(BEGIN_DATA);
        sc_time t(SC_ZERO_TIME);
        ocp->nb_transport_fw(*data_int[idx], ph, t);
        tlm_phase ph1(END_DATA);
        sc_time t1(SC_ZERO_TIME);
        ocp_int[idx]->nb_transport_bw(*data_int[idx], ph1, t1);
        data_owed[idx] -= 4;
        // round robin
        data_priority = (data_priority + 1) % NR_THREADS;
        data_int[idx]->release();
        data_int[idx] = 0;
        return;
      }
    }
  }

  void on_Clk_rising() {
    // requests and write data
    request_e.notify(reqT);
    data_e.notify(dataT);

    // responses
    // have captured response from external OCP, now send to internal target
    unsigned tout = 0;
    for(unsigned i = 0; i < NR_THREADS; i++) {
      // block new responses if no space
      if(resp_stalled[i]) tout |= (1 << i);
      // launch responses if possible
      if(!resp_stalled[i] && (resp_int[i] != 0)) {
        tlm_phase ph(BEGIN_RESP);
        sc_time t(SC_ZERO_TIME);
        if(ocp_int[i]->nb_transport_bw(*resp_int[i], ph, t) == TLM_ACCEPTED)
          resp_stalled[i] = true;
        resp_int[i]->release();
        resp_int[i] = 0;
        tout |= (1 << i);
      }
    }
    // signal new m-thread-busy
    tlm_phase ph(RESP_THREAD_BUSY_CHANGE);
    sc_time t(SC_ZERO_TIME);
    resp_thread_busy *tbex;
    tlm_generic_payload *tbpl = ocp.get_tb_transaction();
    leak_test(*tbpl);
    extension_api::get_extension<resp_thread_busy>(tbex, *tbpl);
    tbex->value = tout;
    ocp->nb_transport_fw(*tbpl, ph, t);

    trace_e.notify(SC_ZERO_TIME);
  }

  tlm_sync_enum nb_transport_bw(
    tlm_generic_payload &pl, tlm_phase &ph, sc_time &t)
  {
    if(ph == CMD_THREAD_BUSY_CHANGE) {
      cmd_thread_busy *tbex;
      extension_api::get_extension<cmd_thread_busy>(tbex, pl);
      sthreadbusy = tbex->value;
    }
    if(ph == DATA_THREAD_BUSY_CHANGE) {
      data_thread_busy *tbex;
      extension_api::get_extension<data_thread_busy>(tbex, pl);
      sdatathreadbusy = tbex->value;
    }
    if(ph == BEGIN_RESP) {
      resp_int[ext::get(pl).thread] = &pl;
      pl.acquire();
      ph = END_RESP;
      return TLM_UPDATED;
    }
    if(ph == END_REQ) {
    }
    if(ph == END_DATA) {
    }

    // default return value if not otherwise specified
    return TLM_ACCEPTED;
  }

  tlm_sync_enum nb_transport_fw(unsigned thread,
    tlm_generic_payload &pl, tlm_phase &ph, sc_time &t)
  {
    if(ph == END_RESP) {
      resp_stalled[thread] = false;
    }
    if(ph == BEGIN_DATA) {
      data_int[thread] = &pl;
      pl.acquire();
    }
    if(ph == BEGIN_REQ) {
      req_int[thread] = &pl;
      ext::get(pl).thread = thread;
      thread_id *thid;
      extension_api::get_extension<thread_id>(thid, pl);
      thid->value = thread;
      extension_api::validate_extension<thread_id>(pl);
      pl.acquire();
    }
    return TLM_ACCEPTED;
  }

  ofstream os;
  ostringstream oss[NR_THREADS];

  sc_event trace_e;
  void write_trace() {
    for(unsigned i = 0; i < NR_THREADS; i++) {
      os << oss[i].str();
      oss[i].str("");
    }
  }

  sc_time innerReqT, extThreadBusyT, reqT;
  sc_time innerDataT, extDataThreadBusyT, dataT;

  void set_timing() {
    sc_time newReqT =
      sc_dt::sc_max(innerReqT, extThreadBusyT) + 2 * sc_get_time_resolution();
    sc_time newDataT =
      sc_dt::sc_max(sc_dt::sc_max(innerDataT, extDataThreadBusyT) + 2 * sc_get_time_resolution(), newReqT + sc_get_time_resolution());
    if((reqT != newReqT) || (dataT != newDataT)) {
      reqT = newReqT;
      dataT = newDataT;
      ocp_tl1_slave_timing sTimes;
      sTimes.SCmdAcceptStartTime = reqT;
      sTimes.SDataAcceptStartTime = dataT;
      ocp_int.set_slave_timing(sTimes);
      ocp_tl1_master_timing mTimes;
      mTimes.RequestGrpStartTime = reqT;
      mTimes.DataHSGrpStartTime = dataT;
      ocp.set_master_timing(mTimes);
    }
  }

  void ocp_timing_update(ocp_tl1_slave_timing times) {
    extThreadBusyT = times.SThreadBusyStartTime;
    extDataThreadBusyT = times.SDataThreadBusyStartTime;
    set_timing();
  }

  void ocp_timing_update_int(ocp_tl1_master_timing times) {
    // may come from any of the inner initiators; take the biggest not the latest
    innerReqT = sc_dt::sc_max(innerReqT, times.RequestGrpStartTime);
    innerDataT = sc_dt::sc_max(innerDataT, times.DataHSGrpStartTime);
    set_timing();
  }
};


class ocp_tl1_target: public sc_module {
SC_HAS_PROCESS(ocp_tl1_target);
public:
  ocp_tl1_target(sc_module_name n):
    sc_module(n),
    ocp("ocp", this, &ocp_tl1_target::ocp_timing_update),
    ocp_int("ocp_int", this, &ocp_tl1_target::ocp_timing_update_int),
    mthreadbusy(~0),
    reset(false),
    resp_priority(0),
    os(sc_module::name())
  {
    ocp.set_ocp_config(gen_ocp_config_threads());
    ocp.activate_synchronization_protection();
    ocp.register_nb_transport_fw(this, &ocp_tl1_target::nb_transport_fw);

    ocp_int.set_ocp_config(gen_ocp_config());
    ocp_int.activate_synchronization_protection();
    ocp_int.register_nb_transport_bw(this, &ocp_tl1_target::nb_transport_bw);

    SC_METHOD(response);
    // choose a response from a non-busy thread, copy it out, set mrespaccept
    dont_initialize();
    sensitive << response_e;

    SC_METHOD(on_Clk_rising);
    // for requests and datas, one register per thread, aware of MReset_n
    dont_initialize();
    sensitive << Clk.pos();

    for(unsigned i = 0; i < NR_THREADS; i++) {
      data_owed[i] = 0;
      req_int[i] = 0;
      data_int[i] = 0;
      req_stalled[i] = false;
      data_stalled[i] = false;
      char inm[7] = "inner?";
      inm[5] = '0' + i;
      ocp_tl1_target_inner &ni(*new ocp_tl1_target_inner(inm, i, &oss[i]));
      targets[i] = &ni;
      ocp_int(ni.ocp);
      ni.Clk(Clk);
    }

    SC_METHOD(write_trace);
    sensitive << trace_e;
  }

  ~ocp_tl1_target() {
    for(unsigned i = 0; i < NR_THREADS; i++) delete targets[i];
  }

  sc_in<bool> Clk;
  ocp_slave_socket_tl1<32,1> ocp;

  void end_of_elaboration() {
    set_timing();
  }

private:
  struct ext: tlm_extension<ext> {
    void free() {delete this;}
    void copy_from(tlm_extension_base const &) {}
    tlm_extension_base *clone() const {return 0;}
    static ext &get(tlm_generic_payload &pl) {
      ext *e = pl.get_extension<ext>();
      if(e == 0) {
        e = new ext;
        pl.set_extension<ext>(e);
      }
      return *e;
    }
    unsigned thread;
  };

  ocp_tl1_target_inner *targets[NR_THREADS];
  ocp_master_socket_tl1<32,0> ocp_int;

  unsigned mthreadbusy;
  bool reset;
  tlm_generic_payload *req_int[NR_THREADS];
  tlm_generic_payload *data_int[NR_THREADS];
  tlm_generic_payload *resp_int[NR_THREADS];

  unsigned resp_priority;
  unsigned data_owed[NR_THREADS];
  bool req_stalled[NR_THREADS], data_stalled[NR_THREADS];

  sc_event response_e;

  void response() {
    // called once SResps and MThreadBusy are known to be stable
    // have captured an MThreadBusy from nb_transport_fw
    // have captured a set of SResp from nb_transport_bw
    for(unsigned i = 0; i < NR_THREADS; i++) {
      unsigned idx = (i + resp_priority) % NR_THREADS;
      if(((~mthreadbusy) & (1 << idx)) && (resp_int[idx] != 0)) {
        tlm_phase ph(BEGIN_RESP);
        sc_time t(SC_ZERO_TIME);
        ocp->nb_transport_bw(*resp_int[idx], ph, t);
        tlm_phase ph1(END_RESP);
        sc_time t1(SC_ZERO_TIME);
        ocp_int[idx]->nb_transport_fw(*resp_int[idx], ph1, t1);
        // round robin
        resp_priority = (resp_priority + 1) % NR_THREADS;
        resp_int[idx]->release();
        resp_int[idx] = 0;
        return;
      }
    }
  }

  void on_Clk_rising() {
    if(reset) return;

    // responses
    response_e.notify(respT);

    // requests
    // have captured request from external OCP, now send to internal target
    unsigned tout = 0;
    for(unsigned i = 0; i < NR_THREADS; i++) {
      // block new requests if no space
      if(req_stalled[i]) tout |= (1 << i);
      // launch requests if possible
      if(!req_stalled[i] && (req_int[i] != 0)) {
        tlm_phase ph(BEGIN_REQ);
        sc_time t(SC_ZERO_TIME);
        if(ocp_int[i]->nb_transport_fw(*req_int[i], ph, t) == TLM_ACCEPTED)
          req_stalled[i] = true;
        if(req_int[i]->is_write())
          data_owed[i] += req_int[i]->get_data_length();
        req_int[i]->release();
        req_int[i] = 0;
        tout |= (1 << i);
      }
    }
    // signal new s-thread-busy
    tlm_phase ph(CMD_THREAD_BUSY_CHANGE);
    sc_time t(SC_ZERO_TIME);
    cmd_thread_busy *tbex;
    tlm_generic_payload *tbpl = ocp.get_tb_transaction();
    extension_api::get_extension<cmd_thread_busy>(tbex, *tbpl);
    tbex->value = tout;
    ocp->nb_transport_bw(*tbpl, ph, t);

    // data handshakes
    tout = 0;
    for(unsigned i = 0; i < NR_THREADS; i++) {
      // block new data if no space
      if(data_stalled[i] || (data_int[i] != 0)) tout |= (1 << i);
      // launch data if possible
      if(!data_stalled[i] && (data_owed[i] >= 4) && (data_int[i] != 0)) {
        tlm_phase ph(BEGIN_DATA);
        sc_time t(SC_ZERO_TIME);
        if(ocp_int[i]->nb_transport_fw(*data_int[i], ph, t) == TLM_ACCEPTED)
          data_stalled[i] = true;
        data_int[i]->release();
        data_int[i] = 0;
        tout |= (1 << i);
        data_owed[i] -= 4;
      }
    }
    // signal new s-data-thread-busy
    tlm_phase dph(DATA_THREAD_BUSY_CHANGE);
    sc_time dt(SC_ZERO_TIME);
    tlm_generic_payload *dtbpl = ocp.get_tb_transaction();
    data_thread_busy *dtbex;
    extension_api::get_extension<data_thread_busy>(dtbex, *dtbpl);
    dtbex->value = tout;
    ocp->nb_transport_bw(*dtbpl, dph, dt);

    trace_e.notify(SC_ZERO_TIME);
  }

  tlm_sync_enum nb_transport_fw(
    tlm_generic_payload &pl, tlm_phase &ph, sc_time &t)
  {
    if(reset) {
      if(ph == END_RESET) {
        reset = false;
        for(unsigned i = 0; i < NR_THREADS; i++) {
          req_int[i] = data_int[i] = resp_int[i] = 0;
          data_owed[i] = 0;
          req_stalled[i] = data_stalled[i] = false;
        }
        mthreadbusy = ~0;
      }
    }
    if(ph == BEGIN_RESET) {
      reset = true;
      ocp.reset();
    }

    if(ph == RESP_THREAD_BUSY_CHANGE) {
      resp_thread_busy *tbex;
      extension_api::get_extension<resp_thread_busy>(tbex, pl);
      mthreadbusy = tbex->value;
    }

    if(ph == BEGIN_REQ) {
      thread_id *thid;
      extension_api::get_extension<thread_id>(thid, pl);
      unsigned thread = thid->value;
      ext::get(pl).thread = thread;
      req_int[thread] = &pl;
      pl.acquire();
      ph = END_REQ;
      return TLM_UPDATED;
    }
    if(ph == BEGIN_DATA) {
      data_int[ext::get(pl).thread] = &pl;
      pl.acquire();
      ph = END_DATA;
      return TLM_UPDATED;
    }
    if(ph == END_RESP) {
    }

    // default return value if not otherwise specified
    return TLM_ACCEPTED;
  }

  tlm_sync_enum nb_transport_bw(unsigned thread,
    tlm_generic_payload &pl, tlm_phase &ph, sc_time &t)
  {
    if(ph == END_REQ) {
      req_stalled[thread] = false;
    }
    if(ph == END_DATA) {
      data_stalled[thread] = false;
    }
    if(ph == BEGIN_RESP) {
      resp_int[thread] = &pl;
      pl.acquire();
    }
    return TLM_ACCEPTED;
  }

  ofstream os;
  ostringstream oss[NR_THREADS];
  sc_time innerRespT, extThreadBusyT, respT;

  sc_event trace_e;
  void write_trace() {
    for(unsigned i = 0; i < NR_THREADS; i++) {
      os << oss[i].str();
      oss[i].str("");
    }
  }

  void set_timing() {
    sc_time
      newRespT = sc_dt::sc_max(innerRespT, extThreadBusyT) + 2 * sc_get_time_resolution();
    if(respT != newRespT) {
      respT = newRespT;
      ocp_tl1_master_timing mTimes;
      mTimes.MRespAcceptStartTime = respT;
      ocp_int.set_master_timing(mTimes);
      ocp_tl1_slave_timing sTimes;
      sTimes.ResponseGrpStartTime = respT;
      ocp.set_slave_timing(sTimes);
    }
  }

  void ocp_timing_update(ocp_tl1_master_timing times) {
    extThreadBusyT = times.MThreadBusyStartTime;
    set_timing();
  }

  void ocp_timing_update_int(ocp_tl1_slave_timing times) {
    // may come from any of the inner targets; take the biggest not the latest
    innerRespT = sc_dt::sc_max(innerRespT, times.ResponseGrpStartTime);
    set_timing();
  }
};


class testbench00: public sc_module {
public:
  testbench00(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target")
  {
    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);

    // TL0 interface
    initiator.MAddr(MAddr);
    initiator.MCmd(MCmd);
    initiator.MData(MData);
    initiator.SData(SData);
    initiator.SResp(SResp);
    initiator.MReset_n(MReset_n);
    target.MAddr(MAddr);
    target.MCmd(MCmd);
    target.MData(MData);
    target.SData(SData);
    target.SResp(SResp);
    target.MReset_n(MReset_n);

    initiator.MBurstLength(MBurstLength);
    initiator.MBurstSingleReq(MBurstSingleReq);
    initiator.MDataValid(MDataValid);
    initiator.MDataLast(MDataLast);
    initiator.SRespLast(SRespLast);
    initiator.MDataTagID(MDataTagID);
    initiator.MTagID(MTagID);
    initiator.STagID(STagID);
    target.MBurstLength(MBurstLength);
    target.MBurstSingleReq(MBurstSingleReq);
    target.MDataValid(MDataValid);
    target.MDataLast(MDataLast);
    target.SRespLast(SRespLast);
    target.MDataTagID(MDataTagID);
    target.MTagID(MTagID);
    target.STagID(STagID);

    initiator.MThreadBusy(MThreadBusy);
    initiator.SThreadBusy(SThreadBusy);
    initiator.SDataThreadBusy(SDataThreadBusy);
    initiator.MThreadID(MThreadID);
    initiator.MDataThreadID(MDataThreadID);
    initiator.SThreadID(SThreadID);
    target.MThreadBusy(MThreadBusy);
    target.SThreadBusy(SThreadBusy);
    target.SDataThreadBusy(SDataThreadBusy);
    target.MThreadID(MThreadID);
    target.MDataThreadID(MDataThreadID);
    target.SThreadID(SThreadID);
  }

  sc_in<bool> Clk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl0_target target;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> SData;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;

  sc_signal<unsigned> MBurstLength;
  sc_signal<bool> MBurstSingleReq;
  sc_signal<bool> MDataValid;
  sc_signal<bool> MDataLast;
  sc_signal<bool> SRespLast;
  sc_signal<unsigned> MDataTagID;
  sc_signal<unsigned> MTagID;
  sc_signal<unsigned> STagID;

  sc_signal<unsigned> MThreadBusy;
  sc_signal<unsigned> SThreadBusy;
  sc_signal<unsigned> SDataThreadBusy;
  sc_signal<unsigned> MThreadID;
  sc_signal<unsigned> MDataThreadID;
  sc_signal<unsigned> SThreadID;
};


class testbench01: public sc_module {
public:
  testbench01(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    t_print(&cout),
    adapter("adapter", &t_print)
  {
    // OCP configuration
    adapter.set_ocp_config(gen_ocp_config_threads());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter.Clk(Clk);

    // TL0 interface
    initiator.MAddr(MAddr);
    initiator.MCmd(MCmd);
    initiator.MData(MData);
    initiator.SData(SData);
    initiator.SResp(SResp);
    initiator.MReset_n(MReset_n);
    adapter.MAddr(MAddr);
    adapter.MCmd(MCmd);
    adapter.MData(MData);
    adapter.SData(SData);
    adapter.SResp(SResp);
    adapter.MReset_n(MReset_n);

    initiator.MBurstLength(MBurstLength);
    initiator.MBurstSingleReq(MBurstSingleReq);
    initiator.MDataValid(MDataValid);
    initiator.MDataLast(MDataLast);
    initiator.SRespLast(SRespLast);
    initiator.MDataTagID(MDataTagID);
    initiator.MTagID(MTagID);
    initiator.STagID(STagID);
    adapter.MBurstLength(MBurstLength);
    adapter.MBurstSingleReq(MBurstSingleReq);
    adapter.MDataValid(MDataValid);
    adapter.MDataLast(MDataLast);
    adapter.SRespLast(SRespLast);
    adapter.MDataTagID(MDataTagID);
    adapter.MTagID(MTagID);
    adapter.STagID(STagID);

    initiator.MThreadBusy(MThreadBusy);
    initiator.SThreadBusy(SThreadBusy);
    initiator.SDataThreadBusy(SDataThreadBusy);
    initiator.MThreadID(MThreadID);
    initiator.MDataThreadID(MDataThreadID);
    initiator.SThreadID(SThreadID);
    adapter.MThreadBusy(MThreadBusy);
    adapter.SThreadBusy(SThreadBusy);
    adapter.SDataThreadBusy(SDataThreadBusy);
    adapter.MThreadID(MThreadID);
    adapter.MDataThreadID(MDataThreadID);
    adapter.SThreadID(SThreadID);

    // TL1 interface
    adapter.ocpTL1(target.ocp);

    // timing:
    adapter.set_sample_times(
      sc_time(2.0, SC_PS),
      // TL1 target sends SThreadBusy at 0, adapter sees it at +1 ps because
      // of PEQ, sends it to TL0 at +1 ps.  TL0 initiator sends MCmd at +1 ps on
      // receipt of SThreadBusy, so adapter must sample at +2 ps.
      sc_time(2.0, SC_PS)
      // Same for write-data
      // all other signals are default timing, sampled at +1 ps
    );
  }

  sc_in<bool> Clk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl1_target target;
  tl1_tl0::get_TL0_timing_example<ostream> t_print;
  tl0_initiator_to_tl1_target<32, THREAD_TYPES> adapter;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> SData;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;

  sc_signal<unsigned> MBurstLength;
  sc_signal<bool> MBurstSingleReq;
  sc_signal<bool> MDataValid;
  sc_signal<bool> MDataLast;
  sc_signal<bool> SRespLast;
  sc_signal<unsigned> MDataTagID;
  sc_signal<unsigned> MTagID;
  sc_signal<unsigned> STagID;

  sc_signal<unsigned> MThreadBusy;
  sc_signal<unsigned> SThreadBusy;
  sc_signal<unsigned> SDataThreadBusy;
  sc_signal<unsigned> MThreadID;
  sc_signal<unsigned> MDataThreadID;
  sc_signal<unsigned> SThreadID;
};


class testbench10: public sc_module {
public:
  testbench10(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    t_print(&cout),
    adapter("adapter", &t_print)
  {
    // OCP configuration
    adapter.set_ocp_config(gen_ocp_config_threads());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter.Clk(Clk);

    // TL1 interface
    initiator.ocp(adapter.ocpTL1);

    // TL0 interface
    adapter.MAddr(MAddr);
    adapter.MCmd(MCmd);
    adapter.MData(MData);
    adapter.SData(SData);
    adapter.SResp(SResp);
    adapter.MReset_n(MReset_n);
    target.MAddr(MAddr);
    target.MCmd(MCmd);
    target.MData(MData);
    target.SData(SData);
    target.SResp(SResp);
    target.MReset_n(MReset_n);

    adapter.MBurstLength(MBurstLength);
    adapter.MBurstSingleReq(MBurstSingleReq);
    adapter.MDataValid(MDataValid);
    adapter.MDataLast(MDataLast);
    adapter.SRespLast(SRespLast);
    adapter.MDataTagID(MDataTagID);
    adapter.MTagID(MTagID);
    adapter.STagID(STagID);
    target.MBurstLength(MBurstLength);
    target.MBurstSingleReq(MBurstSingleReq);
    target.MDataValid(MDataValid);
    target.MDataLast(MDataLast);
    target.SRespLast(SRespLast);
    target.MDataTagID(MDataTagID);
    target.MTagID(MTagID);
    target.STagID(STagID);

    adapter.MThreadBusy(MThreadBusy);
    adapter.SThreadBusy(SThreadBusy);
    adapter.SDataThreadBusy(SDataThreadBusy);
    adapter.MThreadID(MThreadID);
    adapter.MDataThreadID(MDataThreadID);
    adapter.SThreadID(SThreadID);
    target.MThreadBusy(MThreadBusy);
    target.SThreadBusy(SThreadBusy);
    target.SDataThreadBusy(SDataThreadBusy);
    target.MThreadID(MThreadID);
    target.MDataThreadID(MDataThreadID);
    target.SThreadID(SThreadID);

    // timing:
    adapter.set_sample_times(
      sc_time(2.0, SC_PS)
      // TL1 initiator sends MThreadBusy at 0, adapter sees it at +1 ps because
      // of PEQ, sends it to TL0 at +1 ps.  TL0 target sends SResp at +1 ps on
      // receipt of MThreadBusy, so adapter must sample at +2 ps.
      // all other signals are default timing, sampled at +1 ps
    );
  }

  sc_in<bool> Clk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl0_target target;
  tl1_tl0::get_TL0_timing_example<ostream> t_print;
  tl1_initiator_to_tl0_target<32, THREAD_TYPES> adapter;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> SData;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;

  sc_signal<unsigned> MBurstLength;
  sc_signal<bool> MBurstSingleReq;
  sc_signal<bool> MDataValid;
  sc_signal<bool> MDataLast;
  sc_signal<bool> SRespLast;
  sc_signal<unsigned> MDataTagID;
  sc_signal<unsigned> MTagID;
  sc_signal<unsigned> STagID;

  sc_signal<unsigned> MThreadBusy;
  sc_signal<unsigned> SThreadBusy;
  sc_signal<unsigned> SDataThreadBusy;
  sc_signal<unsigned> MThreadID;
  sc_signal<unsigned> MDataThreadID;
  sc_signal<unsigned> SThreadID;
};


class testbench11: public sc_module {
public:
  testbench11(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target")
  {
    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);

    // TL1 interface
    initiator.ocp(target.ocp);
  }

  sc_in<bool> Clk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl1_target target;
};


class testbench101: public sc_module {
public:
  testbench101(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    t_print(&cout),
    adapter10("adapter10", &t_print),
    adapter01("adapter01", &t_print)
  {
    // OCP configuration
    adapter10.set_ocp_config(gen_ocp_config_threads());
    adapter01.set_ocp_config(gen_ocp_config_threads());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter10.Clk(Clk);
    adapter01.Clk(Clk);

    // TL1 interface
    initiator.ocp(adapter10.ocpTL1);
    adapter01.ocpTL1(target.ocp);

    // TL0 interface
    adapter10.MAddr(MAddr);
    adapter10.MCmd(MCmd);
    adapter10.MData(MData);
    adapter10.SData(SData);
    adapter10.SResp(SResp);
    adapter10.MReset_n(MReset_n);
    adapter01.MAddr(MAddr);
    adapter01.MCmd(MCmd);
    adapter01.MData(MData);
    adapter01.SData(SData);
    adapter01.SResp(SResp);
    adapter01.MReset_n(MReset_n);

    adapter10.MBurstLength(MBurstLength);
    adapter10.MBurstSingleReq(MBurstSingleReq);
    adapter10.MDataValid(MDataValid);
    adapter10.MDataLast(MDataLast);
    adapter10.SRespLast(SRespLast);
    adapter10.MDataTagID(MDataTagID);
    adapter10.MTagID(MTagID);
    adapter10.STagID(STagID);
    adapter01.MBurstLength(MBurstLength);
    adapter01.MBurstSingleReq(MBurstSingleReq);
    adapter01.MDataValid(MDataValid);
    adapter01.MDataLast(MDataLast);
    adapter01.SRespLast(SRespLast);
    adapter01.MDataTagID(MDataTagID);
    adapter01.MTagID(MTagID);
    adapter01.STagID(STagID);

    adapter10.MThreadBusy(MThreadBusy);
    adapter10.SThreadBusy(SThreadBusy);
    adapter10.SDataThreadBusy(SDataThreadBusy);
    adapter10.MThreadID(MThreadID);
    adapter10.MDataThreadID(MDataThreadID);
    adapter10.SThreadID(SThreadID);
    adapter01.MThreadBusy(MThreadBusy);
    adapter01.SThreadBusy(SThreadBusy);
    adapter01.SDataThreadBusy(SDataThreadBusy);
    adapter01.MThreadID(MThreadID);
    adapter01.MDataThreadID(MDataThreadID);
    adapter01.SThreadID(SThreadID);

    // timing:
    adapter01.set_sample_times(
      sc_time(6.0, SC_PS),  // Cmd
      sc_time(6.0, SC_PS),  // Data
      sc_time(0.0, SC_PS),  // RespAccept
      sc_time(2.0, SC_PS)   // ThreadBusy
    );
    adapter10.set_sample_times(
      sc_time(6.0, SC_PS),  // Resp
      sc_time(0.0, SC_PS),  // CmdAccept
      sc_time(0.0, SC_PS),  // DataAccept
      sc_time(2.0, SC_PS),  // ThreadBusy
      sc_time(2.0, SC_PS)   // DataThreadBusy
    );
  }

  sc_in<bool> Clk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl1_target target;

  tl1_tl0::get_TL0_timing_example<ostream> t_print;
  tl1_initiator_to_tl0_target<32, THREAD_TYPES> adapter10;
  tl0_initiator_to_tl1_target<32, THREAD_TYPES> adapter01;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> SData;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;

  sc_signal<unsigned> MBurstLength;
  sc_signal<bool> MBurstSingleReq;
  sc_signal<bool> MDataValid;
  sc_signal<bool> MDataLast;
  sc_signal<bool> SRespLast;
  sc_signal<unsigned> MDataTagID;
  sc_signal<unsigned> MTagID;
  sc_signal<unsigned> STagID;

  sc_signal<unsigned> MThreadBusy;
  sc_signal<unsigned> SThreadBusy;
  sc_signal<unsigned> SDataThreadBusy;
  sc_signal<unsigned> MThreadID;
  sc_signal<unsigned> MDataThreadID;
  sc_signal<unsigned> SThreadID;
};


class testbench010: public sc_module {
public:
  testbench010(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    t_print(&cout),
    adapter01("adapter01", &t_print),
    adapter10("adapter10", &t_print)
  {
    // OCP configuration
    adapter01.set_ocp_config(gen_ocp_config_threads());
    adapter10.set_ocp_config(gen_ocp_config_threads());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter01.Clk(Clk);
    adapter10.Clk(Clk);

    // TL1 interface
    adapter01.ocpTL1(adapter10.ocpTL1);

    // TL0 interfaces
    initiator.MAddr(MAddr[0]);
    initiator.MCmd(MCmd[0]);
    initiator.MData(MData[0]);
    initiator.SData(SData[0]);
    initiator.SResp(SResp[0]);
    initiator.MReset_n(MReset_n[0]);
    adapter01.MAddr(MAddr[0]);
    adapter01.MCmd(MCmd[0]);
    adapter01.MData(MData[0]);
    adapter01.SData(SData[0]);
    adapter01.SResp(SResp[0]);
    adapter01.MReset_n(MReset_n[0]);

    initiator.MBurstLength(MBurstLength[0]);
    initiator.MBurstSingleReq(MBurstSingleReq[0]);
    initiator.MDataValid(MDataValid[0]);
    initiator.MDataLast(MDataLast[0]);
    initiator.SRespLast(SRespLast[0]);
    initiator.MDataTagID(MDataTagID[0]);
    initiator.MTagID(MTagID[0]);
    initiator.STagID(STagID[0]);
    adapter01.MBurstLength(MBurstLength[0]);
    adapter01.MBurstSingleReq(MBurstSingleReq[0]);
    adapter01.MDataValid(MDataValid[0]);
    adapter01.MDataLast(MDataLast[0]);
    adapter01.SRespLast(SRespLast[0]);
    adapter01.MDataTagID(MDataTagID[0]);
    adapter01.MTagID(MTagID[0]);
    adapter01.STagID(STagID[0]);

    initiator.MThreadBusy(MThreadBusy[0]);
    initiator.SThreadBusy(SThreadBusy[0]);
    initiator.SDataThreadBusy(SDataThreadBusy[0]);
    initiator.MThreadID(MThreadID[0]);
    initiator.MDataThreadID(MDataThreadID[0]);
    initiator.SThreadID(SThreadID[0]);
    adapter01.MThreadBusy(MThreadBusy[0]);
    adapter01.SThreadBusy(SThreadBusy[0]);
    adapter01.SDataThreadBusy(SDataThreadBusy[0]);
    adapter01.MThreadID(MThreadID[0]);
    adapter01.MDataThreadID(MDataThreadID[0]);
    adapter01.SThreadID(SThreadID[0]);

    adapter10.MAddr(MAddr[1]);
    adapter10.MCmd(MCmd[1]);
    adapter10.MData(MData[1]);
    adapter10.SData(SData[1]);
    adapter10.SResp(SResp[1]);
    adapter10.MReset_n(MReset_n[1]);
    target.MAddr(MAddr[1]);
    target.MCmd(MCmd[1]);
    target.MData(MData[1]);
    target.SData(SData[1]);
    target.SResp(SResp[1]);
    target.MReset_n(MReset_n[1]);

    adapter10.MBurstLength(MBurstLength[1]);
    adapter10.MBurstSingleReq(MBurstSingleReq[1]);
    adapter10.MDataValid(MDataValid[1]);
    adapter10.MDataLast(MDataLast[1]);
    adapter10.SRespLast(SRespLast[1]);
    adapter10.MDataTagID(MDataTagID[1]);
    adapter10.MTagID(MTagID[1]);
    adapter10.STagID(STagID[1]);
    target.MBurstLength(MBurstLength[1]);
    target.MBurstSingleReq(MBurstSingleReq[1]);
    target.MDataValid(MDataValid[1]);
    target.MDataLast(MDataLast[1]);
    target.SRespLast(SRespLast[1]);
    target.MDataTagID(MDataTagID[1]);
    target.MTagID(MTagID[1]);
    target.STagID(STagID[1]);

    adapter10.MThreadBusy(MThreadBusy[1]);
    adapter10.SThreadBusy(SThreadBusy[1]);
    adapter10.SDataThreadBusy(SDataThreadBusy[1]);
    adapter10.MThreadID(MThreadID[1]);
    adapter10.MDataThreadID(MDataThreadID[1]);
    adapter10.SThreadID(SThreadID[1]);
    target.MThreadBusy(MThreadBusy[1]);
    target.SThreadBusy(SThreadBusy[1]);
    target.SDataThreadBusy(SDataThreadBusy[1]);
    target.MThreadID(MThreadID[1]);
    target.MDataThreadID(MDataThreadID[1]);
    target.SThreadID(SThreadID[1]);

    // timing:
    adapter01.set_sample_times(
      sc_time(3.0, SC_PS),  // Cmd
      sc_time(3.0, SC_PS)   // Data
    );
    adapter10.set_sample_times(
      sc_time(3.0, SC_PS)  // Resp
    );
  }

  sc_in<bool> Clk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl0_target target;
  tl1_tl0::get_TL0_timing_example<ostream> t_print;
  tl0_initiator_to_tl1_target<32, THREAD_TYPES> adapter01;
  tl1_initiator_to_tl0_target<32, THREAD_TYPES> adapter10;

  sc_signal<unsigned> MAddr[2];
  sc_signal<unsigned> MCmd[2];
  sc_signal<unsigned> MData[2];
  sc_signal<unsigned> SData[2];
  sc_signal<unsigned> SResp[2];
  sc_signal<bool> MReset_n[2];

  sc_signal<unsigned> MBurstLength[2];
  sc_signal<bool> MBurstSingleReq[2];
  sc_signal<bool> MDataValid[2];
  sc_signal<bool> MDataLast[2];
  sc_signal<bool> SRespLast[2];
  sc_signal<unsigned> MDataTagID[2];
  sc_signal<unsigned> MTagID[2];
  sc_signal<unsigned> STagID[2];

  sc_signal<unsigned> MThreadBusy[2];
  sc_signal<unsigned> SThreadBusy[2];
  sc_signal<unsigned> SDataThreadBusy[2];
  sc_signal<unsigned> MThreadID[2];
  sc_signal<unsigned> MDataThreadID[2];
  sc_signal<unsigned> SThreadID[2];
};


class testbench: public sc_module {
public:
  testbench(sc_module_name n, bool run_01 = true, bool run_10 = true):
    sc_module(n),
    tb00("00"),
    tb01("01"),
    tb10("10"),
    tb11("11"),
    tb101("101"),
    tb010("010"),
    Clk("Clk", sc_time(10.0, SC_NS))
  {
    tb00.Clk(Clk);
    tb11.Clk(Clk);

    if(run_01) tb01.Clk(Clk);
    else tb01.Clk(tieoff);

    if(run_10) tb10.Clk(Clk);
    else tb10.Clk(tieoff);

    if(run_01 && run_10) {
      tb010.Clk(Clk);
      tb101.Clk(Clk);
    } else {
      tb101.Clk(tieoff);
      tb010.Clk(tieoff);
    }
  }
private:
  testbench00 tb00;
  testbench01 tb01;
  testbench10 tb10;
  testbench11 tb11;
  testbench101 tb101;
  testbench010 tb010;
  sc_clock Clk;
  sc_signal<bool> tieoff;
};


#ifndef OMIT_SC_MAIN
int sc_main(int argc, char **argv) {
  sc_report_handler::set_actions(SC_ERROR, SC_DISPLAY | SC_ABORT);
  bool run_01 = ((argc < 2) || ('A' == *argv[1]) || ('0' == *argv[1]));
  bool run_10 = ((argc < 2) || ('A' == *argv[1]) || ('1' == *argv[1]));
  testbench theTestbench("testbench", run_01, run_10);
  sc_start();
  return 0;
}
#endif

