///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2008
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : Robert Guenzel (from TU of Braunschweig) for Greensocs Ltd.
//
//          $Id:
//
//  Description :  This class converts the OCP TLM2 transfers into
//                 'old' style Req/Resp/DHS groups, so that an 'old' style
//                 monitor can analyze what is going on, as if it was looking
//                 at an 'old' style TL1 channel.
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#ifndef OCPIP_VERSION
  #error ocp_tl2_txn_monitor.h may not be included directly. Use #inculde "ocpip.h" or #include "ocpip_X_X_X.h" (where desired ocp version is X.X.X)
#endif

namespace OCPIP_VERSION{

// Utility classes for display
namespace ocp_txn_format
{
struct DisplayableBase {
    virtual std::ostream& streamIt( std::ostream& ) const = 0;
    virtual ~DisplayableBase() {}

    friend std::ostream& operator<<(std::ostream& os, const DisplayableBase& d ) {
	return ( d.streamIt( os ) );
    }
};

template<typename T>
struct Displayable: public DisplayableBase
{
    Displayable& operator=( T data );
    virtual ~Displayable() {}
    virtual std::ostream& streamIt( std::ostream& os ) const;
    T m_data;
};

enum DisplayFormatMode { FORMAT_DEC=1, FORMAT_HEX=2, FORMAT_STR=4 };
struct DisplayableFormat: public DisplayableBase
{
    virtual std::ostream& streamIt( std::ostream& os ) const {
	char fill    = ' ';
	if ( m_mode & FORMAT_HEX )
	    fill = '0';
	if ( m_mode & FORMAT_DEC )
	    os << std::dec;
	else if ( m_mode & FORMAT_HEX )
	    os << std::hex;
	return ( os << std::resetiosflags(std::ios::left)
		 << std::setiosflags(std::ios::right)
		 << std::setfill( fill )
		 << std::setw( m_width ) );
    }
    virtual ~DisplayableFormat() {}
    int               m_mode;
    uint32_t      m_width;
};

template <typename objT>
struct DisplayEntryBase
{
    DisplayEntryBase( const char* name, const char* reference, const char* legend );
    virtual const DisplayableBase& display( const objT& ) const = 0;
    virtual const DisplayableBase& format () const {
        return m_format;
    }
    void setFormat( int mode, int width );
    virtual ~DisplayEntryBase() {}

    const   char*     m_name;
    const   char*     m_reference;
    const   char*     m_legend;
    DisplayableFormat m_format;
};

template <typename objT, typename ownerT, typename dataT>
class DisplayEntry: public DisplayEntryBase<objT>
{
public:
    typedef dataT ( ownerT::*AccessFunc )( const objT& ) const;
    DisplayEntry( const ownerT* pOwner, const char* name, const char* reference,
                  AccessFunc f    , const char* legend );
    virtual const DisplayableBase& display( const objT& o ) const;
private:
    const ownerT*              m_pOwner;
    AccessFunc                 m_func;
    mutable Displayable<dataT> m_display;
};
}

template <uint32_t BUSWIDTH, uint32_t ADDRWIDTH> class ocp_tl2_txn_monitor_trans_access;
template <uint32_t BUSWIDTH, uint32_t ADDRWIDTH>
class ocp_tl2_txn_monitor : public sc_core::sc_module,
			    private OCPIP_VERSION::infr::ocp_observer_base
{
  typedef typename OCPIP_VERSION::infr::bind_checker<tlm::tlm_base_protocol_types>::ext_support_type ext_support_base;
  typedef typename ocp_data_class_unsigned<BUSWIDTH,ADDRWIDTH>::DataType data_type;
  typedef typename ocp_data_class_unsigned<BUSWIDTH,ADDRWIDTH>::AddrType addr_type;
  friend class ocp_tl2_txn_monitor_trans_access<BUSWIDTH,ADDRWIDTH>;

private:
  
  
  //extend the p2p data structure for the OCP-PEQ
  struct ocp_p2p_data_with_mon_info
  : public ocp_p2p_data
  {
    bool fwNbw;
    bool callNreturn;
    tlm::tlm_sync_enum retVal;
  };

  struct ext_support : public ext_support_base { 
      typedef ext_support_base ext_support_type;
      ext_support( const char* name ) :
	    m_name( name ) {}
      const char* name(){return m_name;}
      const char* m_name;
  };
  
  struct ocp_mon_peq :
    public ocp_peq<ext_support, //fake port class (see above) to allow access to a name and the ext_support typename
                   false,       //the PEQ does not use indexes
                   ocp_p2p_data_with_mon_info, //we use our extended p2p data struct
                   true> //we activate the write back of the p2p data that comes out of the PEQ, such that the callback will update tmp_p2p
  {
    typedef ocp_peq<ext_support, false, ocp_p2p_data_with_mon_info, true> base_peq_type;

    template <typename MODULE>
    ocp_mon_peq(MODULE* mod, typename nb_cb_type_selector<MODULE, false>::peq_cb_type cb, ext_support* socket)
      : base_peq_type(mod,cb,socket)
    {
    }
    
    //call prior to notify
    void set_mon_info(bool fwNbw, bool callNreturn, tlm::tlm_sync_enum retVal){
      base_peq_type::p2p_tmp.fwNbw=fwNbw;
      base_peq_type::p2p_tmp.callNreturn=callNreturn;
      base_peq_type::p2p_tmp.retVal=retVal;
    }
    
    //call after PEQ callback
    bool get_fwNbw(){return base_peq_type::p2p_tmp.fwNbw;}
    bool get_callNreturn(){return base_peq_type::p2p_tmp.callNreturn;}
    tlm::tlm_sync_enum get_retVal(){return base_peq_type::p2p_tmp.retVal;}
  };
  
  ext_support m_ext_support;
  tlm_utils::instance_specific_extension_accessor acc;
  ocp_mon_peq m_peq;
  sc_core::sc_time m_null_time;
  void peq_cb(tlm::tlm_generic_payload&, const tlm::tlm_phase&);

  std::ostream& m_os;
  ocp_parameters m_ocp_params;

  //ocp_extension_pool<mon_delay_info> m_monInfoPool;

  virtual void end_of_elaboration();
  virtual void start_of_simulation();
  void nb_call_callback(bool fwNbw
                       ,tlm::tlm_generic_payload& txn
                       ,const tlm::tlm_phase& phase
                       ,const sc_core::sc_time& time);
                       
  void nb_return_callback(bool fwNbw
                         ,tlm::tlm_generic_payload& txn
                         ,const tlm::tlm_phase& phase
                         ,const sc_core::sc_time& time
                         ,tlm::tlm_sync_enum retVal);

  void b_call_callback(tlm::tlm_generic_payload& txn, const sc_core::sc_time& time);
  
  void b_return_callback(tlm::tlm_generic_payload& txn, const sc_core::sc_time& time);

  uint32_t m_byte_width;
  tl2_master_timing_group m_master_timing;
  tl2_slave_timing_group m_slave_timing;

  ocp_extension_pool<ocp_txn_burst_invariant> m_invariantExtPool;
  ocp_extension_pool<ocp_txn_position>        m_positionExtPool;

  void get_thread_and_tag( tlm::tlm_generic_payload&, uint32_t&, uint32_t& );

  struct Transfer
  {
      Transfer() {
	  *m_burstRepr = *m_respLatRepr = *m_respAccRepr = '\0';
      }

      tlm::tlm_generic_payload*      m_txn;
      ocp_txn_burst_invariant*       m_p_invariant;
      ocpip_legacy::OCPSRespType     m_resp;
      data_type                      m_data;
      addr_type                      m_addr;
      uint32_t                       m_byteen;
      uint32_t                       m_burst_length;
      sc_core::sc_time               m_startTime;
      int                            m_burstCount;
      char                           m_burstRepr[10];
      char                           m_respLatRepr[10];
      char                           m_respAccRepr[10];
      bool                           m_complete;
      bool                           m_needsData;
      bool                           m_needsResponse;
      int                            m_RqAL;
      int                            m_RqDL;
  };
  ocp_tl2_txn_monitor_trans_access<BUSWIDTH,ADDRWIDTH>*       m_p_trans_access;
  std::vector<ocp_txn_format::DisplayEntryBase<Transfer>* >   m_entries;
  std::deque<Transfer>                                        m_timed_transfers;
  std::vector<std::vector<std::deque<Transfer*> > >           m_transfers_pending_data;
  std::vector<std::vector<std::deque<Transfer*> > >           m_transfers_pending_resp;
  std::vector<ocpip_legacy::OcpIp::BurstSequence<addr_type> > m_addr_sequence;
  std::vector<std::vector<ocp_txn_track> >                    m_burst_tracker;
  void init_display_fields();
  void t_log_header();
  void t_log_new_req ( tlm::tlm_generic_payload& );
  void t_log_new_data( tlm::tlm_generic_payload& );
  void t_log_new_resp( tlm::tlm_generic_payload& );
  void print_transfer( const Transfer& );
  void flush_queue( bool force = false );

public: 

  template <uint32_t BUSWIDTH_>
  ocp_tl2_txn_monitor(OCPIP_VERSION::infr::monitor<BUSWIDTH_, tlm::tlm_base_protocol_types>&,
		      sc_core::sc_module_name,
		      std::ostream&);
  ~ocp_tl2_txn_monitor();
  sc_core::sc_time m_period;
  void error(const char* func) const ;
};
}

#include __MACRO_STRINGYFY__(../mon/src/OCPIP_VERSION/ocp_tl2_txn_monitor.tpp)
