///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2008
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis, Texas Instruments
//                Robert Guenzel (from TU of Braunschweig) for Greensocs Ltd.
//
//          $Id:
//
//  Description :  Master for the TL1 timing example
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#ifndef __TIMING_MASTER_H__
#define __TIMING_MASTER_H__

#include "timing_common.h"

// master derives from slave-timing-if, because it is
// single-ported and timing-sensitive.
class timing_master : 
  public sc_core::sc_module
{
public:
  SC_HAS_PROCESS(timing_master);
  ocpip::ocp_master_socket_tl1<> ocp;
  sc_core::sc_in<bool> clk;  
  
  
    timing_master(sc_core::sc_module_name name) :
      sc_core::sc_module(name), ocp("ocp", this, &timing_master::setOCPTL1SlaveTiming, ocpip::ocp_master_socket_tl1<>::mm_txn_with_data()) {

      SC_THREAD(clock_rising);
      sensitive<<clk.pos();
      dont_initialize();

      requests_sent = 0;
      responses_got = 0;
      requests_outstanding = 0;

      resp_expected = new unsigned int[max_requests_outstanding];
      rd_ptr = 0;
      wr_ptr = 0;
      
      m_stb=0;

      time_quantum = sc_core::sc_get_time_resolution();
      sthreadbusy_sample_time = time_quantum;  // initial guess
      
      tb_ph=ocpip::THREAD_BUSY_CHANGE;
      
      std::cout << "<<<< E-O-E >>>> " << sc_core::sc_module::name() << std::endl;
      ocpip::ocp_tl1_master_timing my_timing;
      my_timing.RequestGrpStartTime = sthreadbusy_sample_time;
      ocp.set_master_timing(my_timing);
      
      ocp.register_nb_transport_bw(this, &timing_master::nb_transport);
      //we do NOT activate the DCP, because we are doing thread busy control
      // and so every one waits at least on ps into the cycle before
      // issueing any reqs or resps, so we have stable values at the clock
      
      
#ifndef USE_EXTERNAL_PARAM_TYPE
      std::cout<<sc_core::sc_module::name()<<" is configuring its socket by itself, since there is no external configuration environment."<<std::endl;
      ocpip::map_string_type config_map=get_config_map(1,32);
      ocpip::ocp_parameters  config;
      config.set_ocp_configuration(ocp.name(), config_map);
      ocp.set_ocp_config(config);
#else
      std::cout<<sc_core::sc_module::name()<<" is NOT configuring its socket, since there is an external configuration environment."<<std::endl;
#endif

      has_resp=false;
      resp=0;
    }

    // destructor to print some stats
    ~timing_master() {
      std::cout << "Deleting master: " << name() << std::endl;
      std::cout << "  Requests sent: " << requests_sent << std::endl;
      std::cout << "  Responses got: " << responses_got << std::endl;
      std::cout << "  SThreadBusy sample time: " << sthreadbusy_sample_time << std::endl;
    }

    void check_resp(unsigned int resp){
      requests_outstanding--;
      responses_got++;
      if((resp & 0xFFFF) != resp_expected[rd_ptr]) {
        std::cout << name() << " " << sc_core::sc_time_stamp() <<
          " ERROR: out of order response\n";
        std::cout << "expected: " << std::hex << resp_expected[rd_ptr] << std::dec << std::endl;
        std::cout << "got:      " << std::hex << resp << std::dec << std::endl;
        sc_core::sc_stop();
      }
      rd_ptr = (rd_ptr+1) % max_requests_outstanding;    
    }

    // processes
    void clock_rising() {
      tlm::tlm_generic_payload* tb_txn=ocp.get_tb_transaction();
      ocp.get_extension<ocpip::thread_busy>(*tb_txn)->value.type=ocpip::M_THREAD;
      tlm::tlm_phase ph;
      sc_core::sc_time t;
      while(true) {
        sc_core::wait();
        // assert MThreadBusy directly after clock edge
        int mtb = (unit_rand(name()) < pr_busy ? 1 : 0);
        
        ocp.get_extension<ocpip::thread_busy>(*tb_txn)->value.mask=mtb;
        ocp->nb_transport_fw(*tb_txn, tb_ph, zero_time);

        // get any response from previous cycle
        if(has_resp) {
          has_resp=false;
          check_resp(resp);
        }

        // wait for long enough that SThreadBusy is stable
        sc_core::wait(sthreadbusy_sample_time);

        m_stb&=1;        
          
        double doub_tmp=unit_rand(name());

        // start a new request if we want to
        if((doub_tmp < pr_request) && (m_stb == 0) &&
           (requests_outstanding < max_requests_outstanding)) {
          tlm::tlm_generic_payload* gp=ocp.get_transaction();
          gp->set_command(tlm::TLM_READ_COMMAND);
          gp->set_address(get_address_random(name()));
          gp->set_byte_enable_ptr(NULL);
          gp->set_streaming_width(sizeof(unsigned int));
          gp->set_response_status(tlm::TLM_INCOMPLETE_RESPONSE);

          ocp.reserve_data_size(*gp, sizeof(unsigned int));
          
          
          requests_outstanding++;
          requests_sent++;
          resp_expected[wr_ptr] = gp->get_address() & 0xFFFF;
          wr_ptr = (wr_ptr+1) % max_requests_outstanding;
             
          ph=tlm::BEGIN_REQ;
          t=sc_core::SC_ZERO_TIME;
          tlm::tlm_sync_enum retVal=ocp->nb_transport_fw(*gp, ph, t);
          switch (retVal){
            case tlm::TLM_UPDATED:
              if (ph==tlm::END_REQ); //great! just an immediate accept!
              else{
                std::cerr<<"Unexpected phase "<<ph<<" on return path. In "<<name()<<" at "<<sc_core::sc_time_stamp()<<std::endl;
              }
              break;
            case tlm::TLM_COMPLETED:; //socket explodes on COMPLETE so we can just do nothing about it
            default:
              std::cerr<<"When using thread busy exact, I expect the use of the return path. In "<<name()<<" at "<<sc_core::sc_time_stamp()<<std::endl;
              exit(1);
          }
        }
      }
    }

    // when informed of slave timing, master must re-inform the OCP channel
    // if anything changed
    void setOCPTL1SlaveTiming(ocpip::ocp_tl1_slave_timing slave_timing) {
      std::cout << "  << S-S-T >>   " << name() << std::endl;
     
      // calculate the sample time for sthreadbusy based on the new timing
      // information from the channel
      sc_core::sc_time stb_sample = slave_timing.SThreadBusyStartTime + time_quantum;
      // if larger than before, update this module's configuration and
      // then inform the slave via the channel
      if(stb_sample > sthreadbusy_sample_time) {
        sthreadbusy_sample_time = stb_sample;
        ocpip::ocp_tl1_master_timing my_timing;
        my_timing.RequestGrpStartTime = sthreadbusy_sample_time;
        ocp.set_master_timing(my_timing);
      }
    }
    
    tlm::tlm_sync_enum nb_transport(tlm::tlm_generic_payload& gp, tlm::tlm_phase& ph, sc_core::sc_time& t){
      switch (ph){
        case tlm::BEGIN_RESP:
          resp=(*((unsigned int*)gp.get_data_ptr()));
          has_resp=true;
          ph=tlm::END_RESP; //we use thread busy so we accept immediately
          ocp.release_transaction(&gp);
          return tlm::TLM_UPDATED;
          break;
        case tlm::END_REQ: //END_REQ is wrong as we expect to get it piggy bagged when using thread busy
        case tlm::BEGIN_REQ:
        case tlm::END_RESP:
          std::cerr<<"WRONG PHASE "<<ph<<" at "<<sc_core::sc_time_stamp()<<" in "<<name()<<std::endl;
          exit(1);
        default:
          if (ph==ocpip::THREAD_BUSY_CHANGE){
            assert(ocp.get_extension<ocpip::thread_busy>(gp)->value.type==ocpip::S_THREAD);
            m_stb=ocp.get_extension<ocpip::thread_busy>(gp)->value.mask;
          }
          else{
            std::cerr<<"Unexpected phase "<<ph<<" at "<<sc_core::sc_time_stamp()<<" in "<<name()<<std::endl;
            exit(1);
          }
      }
      return tlm::TLM_ACCEPTED;
    }

  private:
    sc_core::sc_time sthreadbusy_sample_time;
    sc_core::sc_time time_quantum;
    sc_core::sc_time zero_time;
    tlm::tlm_phase tb_ph;

    int requests_outstanding;
    unsigned int *resp_expected;
    int rd_ptr, wr_ptr;
    unsigned int m_stb;
    static const int max_requests_outstanding = 4;

    int requests_sent, responses_got;

    static const double pr_request;
    static const double pr_busy;
    
    unsigned int resp;
    bool has_resp;
    sc_core::sc_event                go_on;
};

const double timing_master::pr_request=0.125;
const double timing_master::pr_busy=0.5;

// end of multiple inclusion protection
#endif

