///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2008
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis, Texas Instruments
//                Robert Guenzel (from TU of Braunschweig) for Greensocs Ltd.
//
//          $Id:
//
//  Description :  Slave for the TL1 timing example
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#ifndef __TIMING_SLAVE_H__
#define __TIMING_SLAVE_H__


#include "timing_common.h"

class timing_slave : 
  public sc_core::sc_module
{
  public:
  // ports
  sc_core::sc_in<bool> clk;
  ocpip::ocp_slave_socket_tl1<> ocp;
  
  ocpip::ocp_extension_pool<local_thread_ID> m_extPool;
  tlm_utils::instance_specific_extension_accessor acc;
  typedef tlm::tlm_generic_payload* ocp_handle;
  typedef ocp_handle* ocp_handle_ptr;

  SC_HAS_PROCESS(timing_slave);

    timing_slave(sc_core::sc_module_name name, int nr_threads) :
      sc_core::sc_module(name), ocp("ocp", this, &timing_slave::setOCPTL1MasterTiming), m_extPool(10) {

      threads = nr_threads;

      if (nr_threads<2){
        std::cerr<<sc_core::sc_module::name()<<" can only operate correctly with more than one thread"<<std::endl;
        exit(1);
      }

      // array of FIFOs, one per thread, for responses
      rd_ptr = new int[threads];
      wr_ptr = new int[threads];
      full = new bool[threads];
      responses_waiting = new ocp_handle_ptr[threads];
      for(int i=0; i<threads; i++) {
        rd_ptr[i] = 0;
        wr_ptr[i] = 0;
        full[i] = false;
        responses_waiting[i] = new ocp_handle[max_responses_waiting];
      }

      requests_got = 0;
      responses_sent = 0;

      SC_THREAD(main_thread);
      sensitive<<clk.pos();
      dont_initialize();

      time_quantum = sc_core::sc_get_time_resolution();
      request_sample_time = time_quantum;  // initial guess
      mthreadbusy_sample_time = time_quantum;  // initial guess
      ocp.register_nb_transport_fw(this, &timing_slave::nb_transport_fw); 
      
      std::cout << "<<<< E-O-E >>>> " << sc_core::sc_module::name() << std::endl;

      // non-default timing
      ocpip::ocp_tl1_slave_timing mytiming;
      mytiming.ResponseGrpStartTime = my_max(request_sample_time,
        mthreadbusy_sample_time);
      ocp.set_slave_timing(mytiming);
      
      ocpip::map_string_type config_map=get_config_map(nr_threads,32);
      ocpip::ocp_parameters  config;
      config.set_ocp_configuration(ocp.name(), config_map);
      ocp.set_ocp_config(config);
      
      req=NULL;
    };

    ~timing_slave() {
      std::cout << "Deleting slave:   " << name() << std::endl;
      std::cout << "  Total requests got:   " << requests_got << std::endl;
      std::cout << "  Total responses sent: " << responses_sent << std::endl;
      std::cout << "  Per thread outstanding:  ";
      for(int i=0; i<threads; i++) {
        std::cout << " " << (full[i] ? max_responses_waiting :
          (wr_ptr[i] - rd_ptr[i] + max_responses_waiting) % max_responses_waiting);
      }
      std::cout << std::endl;
      std::cout << "  MThreadBusy sample time:  " << mthreadbusy_sample_time << std::endl;
      std::cout << "  Request sample time:       " << request_sample_time << std::endl;
    }


    // processes
    void main_thread() {

      tlm::tlm_generic_payload* tb_txn=ocp.get_tb_transaction();
      ocp.get_extension<ocpip::thread_busy>(*tb_txn)->value.type=ocpip::S_THREAD;
      tlm::tlm_phase ph;
      tlm::tlm_phase tb_ph=ocpip::THREAD_BUSY_CHANGE;
      sc_core::sc_time t, tb_time;  

      while(true) {
        sc_core::wait();  // clock rising edge

        // generate sthreadbusy
        int mask = 1;
        int stb_out = 0;
        for(int i=0; i<threads; i++) {
          if(full[i]) {
            stb_out |= mask;
          }
          mask <<= 1;
        }
        ocp.get_extension<ocpip::thread_busy>(*tb_txn)->value.mask=stb_out;
        ocp->nb_transport_bw(*tb_txn, tb_ph, tb_time);

        // wait for requests and capture them
        sc_core::wait(request_sample_time);

        if(req) {
          requests_got++;
          local_thread_ID* tid;
          acc(*req).get_extension(tid);
          assert(tid);
          int t = tid->local_id;
          ((responses_waiting[t])[wr_ptr[t]])=req;
          req->set_response_status(tlm::TLM_OK_RESPONSE);
          unsigned int* tmp_data_ptr=(unsigned int*)req->get_data_ptr();
          *tmp_data_ptr=req->get_address();
          wr_ptr[t] = (wr_ptr[t] + 1) % max_responses_waiting;
          if(wr_ptr[t] == rd_ptr[t]) {
            full[t] = true;
          }
          req=NULL;
        }

        // wait for mthreadbusy if necessary
        if(request_sample_time < mthreadbusy_sample_time) {
          sc_core::wait(mthreadbusy_sample_time - request_sample_time);
        }
        //int mtb = ocp->getMThreadBusy();

        // generate responses
        if(unit_rand(name()) < pr_response) {
          // priority goes to lowest thread
          for(int i=0; i<threads; i++) {
            if((full[i] || (wr_ptr[i] != rd_ptr[i])) && !(mtb & 1)) {
              responses_sent++;
              ph=tlm::BEGIN_RESP;
              t=sc_core::SC_ZERO_TIME;
              ocp_handle ocp_p=((responses_waiting[i])[rd_ptr[i]]);

              local_thread_ID* threadID;
              acc(*ocp_p).get_extension(threadID);
              assert(threadID);
              
              acc(*ocp_p).clear_extension(threadID);
              m_extPool.recycle(threadID);
              
              switch (ocp->nb_transport_bw(*ocp_p, ph, t)){
                case tlm::TLM_UPDATED:
                  assert(ph==tlm::END_RESP);
                case tlm::TLM_COMPLETED:
                  break;
                default:
                  std::cerr<<"I expect to get updated or completed when using thread busy"<<std::endl;
                  exit(1);
              }
              rd_ptr[i] = (rd_ptr[i] + 1) % max_responses_waiting;
              full[i] = false;
              break;
            }
            mtb >>= 1;
          }
        }
      }
    }

    // callback from the channel to indicate master timing
    // need to capture mthreadbusy and response sample times,
    // and inform channel if this changes the response timing
    void setOCPTL1MasterTiming(ocpip::ocp_tl1_master_timing master_timing) {
      std::cout << "  << S-M-T >>   " << name() << std::endl;

      sc_core::sc_time current_resp_out_t = my_max(request_sample_time,
        mthreadbusy_sample_time);

      if(master_timing.MThreadBusyStartTime + time_quantum >
            mthreadbusy_sample_time) {
        mthreadbusy_sample_time =
            master_timing.MThreadBusyStartTime + time_quantum;
      }

      if(master_timing.RequestGrpStartTime + time_quantum >
            request_sample_time) {
        request_sample_time =
            master_timing.RequestGrpStartTime + time_quantum;
      }

      sc_core::sc_time new_resp_out_t = my_max(request_sample_time,
        mthreadbusy_sample_time);

      if(new_resp_out_t > current_resp_out_t) {
        ocpip::ocp_tl1_slave_timing sto;
        sto.ResponseGrpStartTime = new_resp_out_t;
        ocp.set_slave_timing(sto);
      }
    }

    tlm::tlm_sync_enum nb_transport_fw(tlm::tlm_generic_payload& gp, tlm::tlm_phase& ph, sc_core::sc_time& tim){
      if (ph==ocpip::THREAD_BUSY_CHANGE){
        assert(ocp.get_extension<ocpip::thread_busy>(gp)->value.type==ocpip::M_THREAD);
        mtb=ocp.get_extension<ocpip::thread_busy>(gp)->value.mask;
        return tlm::TLM_ACCEPTED;
      }
    
      if (ph!=tlm::BEGIN_REQ){
        std::cerr<<"I only expect BEGIN_REQ on the forward path, but got "<<ph<<" In "<<name()<<" at "<<sc_core::sc_time_stamp()<<std::endl;
        exit(1);
      }
      
      ocpip::thread_id* tmp;
      bool thread_id_available=ocp.get_extension<ocpip::thread_id>(tmp, gp);
      assert(thread_id_available); //there has to be a valid thread ID

      local_thread_ID* threadID;
      acc(gp).get_extension(threadID);
      assert(!threadID); //make sure we did not see this thing before
      threadID=m_extPool.create();
      threadID->local_id=tmp->value;
      acc(gp).set_extension(threadID);
      
      req=&gp;
      ph=tlm::END_REQ;
      return tlm::TLM_UPDATED;
    }

  private:
    sc_core::sc_time time_quantum;
    sc_core::sc_time request_sample_time;
    sc_core::sc_time mthreadbusy_sample_time;

    static const double pr_response;
    static const int max_responses_waiting = 4;
    int *rd_ptr, *wr_ptr;
    bool *full;
    ocp_handle_ptr *responses_waiting;
    int threads, mtb;

    int requests_got, responses_sent;

    tlm::tlm_generic_payload* req;
};

const double timing_slave::pr_response=0.25;

#endif

