// 
//  Copyright 2008 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Anssi Haverinen, Nokia Inc.
//                Robert Guenzel (from TU of Braunschweig) for Greensocs Ltd.
//           $Id: master_w.cc,v 1.3 2006/02/03 18:04:56 tkogel Exp $
//
//  Description : OCP API - TL1 packet CPU bus profile example
// ============================================================================

#include "master_w.h"

// ----------------------------------------------------------------------------
// Constructor
// ----------------------------------------------------------------------------
Master_w::Master_w (sc_core::sc_module_name name_): 
  sc_core::sc_module (name_), ipP("ipPort", ocpip::ocp_master_socket_tl1<32>::mm_txn_with_data())
  , ongoing_req(false)
  , ongoing_dhs(false)
  , resp(NULL) {

  // initialize common members
  cnt = 0;
  wraplen = 0;
  addr = 0;
  burstcnt = 0;
  SC_METHOD(proc);
  sensitive<<clk.pos(); 
  dont_initialize();
  
  ipP.register_nb_transport_bw(this, &Master_w::nb_transport);
  ipP.activate_delta_cycle_protection();
  dhscnt=0;
}

// ----------------------------------------------------------------------------
// Destructor
// ----------------------------------------------------------------------------
Master_w::~Master_w() {}

// ----------------------------------------------------------------------------
//  Method : Master_w::proc()
//
//  Synchronous Master process
//
// ----------------------------------------------------------------------------
void Master_w::proc() {

  // check if request data channel is free
  //tmp = !ipP->getSBusyDataHS();
  if (!ongoing_dhs) {
    if (req_data.size() && req_data.front().currently_allowed_chunks>dhscnt){
      // Send data if request group has been already sent

      phase=ocpip::BEGIN_DATA;
      time=sc_core::SC_ZERO_TIME;
      tlm::tlm_sync_enum retVal=ipP->nb_transport_fw(*req_data.front().txn, phase, time);
#ifdef DEBUG_G1
      std::cout << "Master_w sent data "
	   << "  time  = " << sc_core::sc_time_stamp().to_seconds()
	   << "  MData  = " << (*(((Td*)(req_data.front().txn->get_data_ptr()))+dhscnt)) << std::endl;
#endif
      switch (retVal){
        case tlm::TLM_ACCEPTED: ongoing_dhs=true; break;
        case tlm::TLM_UPDATED:{
          if (phase!=ocpip::END_DATA){ //that's the only update allowed for BEGIN_DATA
            std::cerr<<"Error in "<<name()<<": Got updated phase "<<phase<<" in response to BEGIN_DATA"<<std::endl;
            exit(1);
          }
          if ((++dhscnt)==req_data.front().max_chunks) {
            req_data.pop_front();
            dhscnt=0;
          }          
          break;
          //we can leave ongoing_dhs at false, as we can do a dhs at the next clock edge
        }
        case tlm::TLM_COMPLETED:;
      }
    }
  }

  // check if request channel is free
  //tmp = !ipP->getSBusy();
  if (!ongoing_req) {
    if (cnt==0 | cnt==8 | cnt==16){
      req=ipP.get_transaction();
      ipP.reserve_data_size(*req, 8*sizeof(Td));
      req->set_command(tlm::TLM_WRITE_COMMAND);
      ocpip::burst_sequence* b_seq;
      ipP.get_extension<ocpip::burst_sequence>(b_seq, *req);
      req->set_streaming_width(8*sizeof(Td));
      req->set_address(addr);      
      if (cnt==0)
        b_seq->value.sequence=ocpip::INCR;
      else
      if (cnt==8){
        b_seq->value.sequence=ocpip::WRAP;
        b_seq->value.xor_wrap_address=addr; //wrap=incr in this example
        req->set_address(addr); //first burst went from 0 to 31, so that one starts at 32
      }
      else{
        b_seq->value.sequence=ocpip::STRM;
        req->set_streaming_width(sizeof(Td));
      }
      ipP.validate_extension<ocpip::burst_sequence>(*req);
      ocpip::burst_length* b_len;
      ipP.get_extension<ocpip::burst_length>(b_len, *req);
      b_len->value=8;
      ipP.validate_extension<ocpip::burst_length>(*req); 
      req->set_byte_enable_ptr(NULL);
      req->set_response_status(tlm::TLM_INCOMPLETE_RESPONSE);           
    }
    if (cnt<8) {
      wraplen = 0;
      phase=tlm::BEGIN_REQ;
      // Set OCP command type
      req->set_address(addr);
      if (burstcnt==7) {
        burstcnt=0;
      }
      else {
        burstcnt++;
      }	
    }
    else if (cnt<16) {
      wraplen = 8;
      phase=tlm::BEGIN_REQ;
      req->set_address(addr);
      if (burstcnt==7) {
        burstcnt=0;
      }
      else {
        burstcnt++;
      }
    }
    else if (cnt<24){
      wraplen = 0;
      phase=tlm::BEGIN_REQ;
      req->set_address(0);
      if (burstcnt==7) {
        burstcnt=0;
      }
      else {
        burstcnt++;
      }
    }
    else {
      phase=tlm::UNINITIALIZED_PHASE;
    }
  

    // Send request
    if (phase!=tlm::UNINITIALIZED_PHASE){
      time=sc_core::SC_ZERO_TIME;
      tlm::tlm_sync_enum retVal = ipP->nb_transport_fw(*req, phase, time);
      switch(retVal){
        case tlm::TLM_ACCEPTED: ongoing_req=true; break;
        case tlm::TLM_UPDATED: 
          switch (phase){
            case tlm::END_REQ: 
              break;
            default:
              std::cerr<<"Error in "<<name()<<" : got unexpected phase update to "<<phase<<" in response to BEGIN_REQ."<<std::endl;
              exit(1);                
          }
          break;
        case tlm::TLM_COMPLETED:;
      }


#ifdef DEBUG_G1
      std::cout << "Master_w sent request "
	   << "  time  = " << sc_core::sc_time_stamp().to_seconds()
	   << "  MAddr  = " << req->get_address() << std::endl;
#endif

      if (req->get_command() == tlm::TLM_WRITE_COMMAND){
        *(((Td*)(req->get_data_ptr()))+(((burstcnt)?burstcnt:8)-1))=addr;
        if ((req_data.size()==0) ||req!=req_data.back().txn){
          assert(burstcnt==1);
          req_data.push_back(outstanding_dhs(req, 8, burstcnt));
        }
        else
          req_data.back().currently_allowed_chunks++;
        expected_rsps.push_back(expected_rsp(req, (((burstcnt)?burstcnt:8)-1), burstcnt==0));
      }
      // Address generator
      if (wraplen) {
        if ((addr+4) < ((addr & (~(wraplen-1)*4))+ wraplen*4))
          addr = addr+4; 
        else
          addr = addr & (~((wraplen-1)*4));
      }
      else
        if (addr < 124)
          addr += 4;
        else
          addr = 0;    
      cnt++;
    }
  }

  if (resp) {
    time=sc_core::SC_ZERO_TIME;
    phase=tlm::END_RESP;
    ipP->nb_transport_fw(*resp, phase, time);
    assert(expected_rsps.front().txn==resp);
    if (resp->get_response_status() == tlm::TLM_OK_RESPONSE) {
#ifdef DEBUG_G1
      std::cout << "Master_w got valid response "
       << "  time  = " << sc_core::sc_time_stamp().to_seconds()
       << "  SResp  = " << resp->get_response_status()
       << "  SData  = " << (*(((Td*)(resp->get_data_ptr()))+expected_rsps.front().chunk)) << std::endl;
#endif
      if (expected_rsps.front().last) {
        ipP.release_transaction(resp); //we are done with this txn now
      }
      resp=NULL;
      expected_rsps.pop_front();
    }
  }

  
  
} // end of method

tlm::tlm_sync_enum Master_w::nb_transport(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim) {
  switch(ph){
    case tlm::END_REQ:
      ongoing_req=false;
      break;
    case tlm::BEGIN_RESP: resp=&txn; break;
    default:
      if (ph==ocpip::END_DATA){
        ongoing_dhs=false;
        assert(req_data.size());
        if ((++dhscnt)==req_data.front().max_chunks) {
          req_data.pop_front();
          dhscnt=0;
        }
      }
      else{
        std::cerr<<"Error in "<<name()<<" : got unexpected phase "<<ph<<" in nb_transport_bw"<<std::endl;
        exit(1);
      }
  }
  return tlm::TLM_ACCEPTED;
}


