// 
//  Copyright 2008 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Albert Chitemyan, Sonics Inc. 
//                
//         $Id: 
//
//  Description : OCP API - Layer-2 Delay Example slave
//    Features:
//    - Non-blocking methods are used
//    - Not pipelined (only one pending burst at a time)
//    - Support for sc_time argument in nb_transport
// ============================================================================

#include "slave.h"

#define CLKPERIOD 1

using namespace std;
// ----------------------------------------------------------------------------
// Constructor
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> Slave<BUSWIDTH, ADDRWIDTH>::Slave (sc_core::sc_module_name name_)
  : sc_core::sc_module (name_), tp_socket("tp_socket"), req(NULL), ongoing_rsp(false) {
  
  scv_random::set_global_seed(scv_random::pick_random_seed());
  
  // initialize common members
  last_request = tlm::TLM_IGNORE_COMMAND;

  SC_METHOD(receiveBurst);
  sensitive << receive_burst_ev;
  dont_initialize(); // purely reactive

  SC_METHOD(sendResp);
  sensitive << send_resp_ev;
  dont_initialize(); // purely reactive

  tp_socket.register_nb_transport_fw(this, &Slave::nb_transport);
}


// ----------------------------------------------------------------------------
// Destructor
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> Slave<BUSWIDTH, ADDRWIDTH>::~Slave(){}


// ----------------------------------------------------------------------------
//  Method : Slave::receiveBurst
//
// ----------------------------------------------------------------------------

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
void Slave<BUSWIDTH, ADDRWIDTH>::receiveBurst()
{   
    scv_smart_ptr< unsigned int > rand;
    rand->next();

    // If a previous read-burst has been served, we check for new burst 
    if (last_request == tlm::TLM_IGNORE_COMMAND) {
	if (req){
	    if (req->get_command() == tlm::TLM_READ_COMMAND) {
		rsp = req;
		last_request = req->get_command();
		send_resp_ev.notify( (*rand) % 3 * CLKPERIOD, sc_core::SC_NS );
#ifdef DEBUG_TL2                  
	    cout << "Slave got READ burst " 
		 << "                     addr=" << hex << rsp->get_address() 
		 << dec << "                             time=" << sc_core::sc_time_stamp() << endl;	
#endif
	    } else {
#ifdef DEBUG_TL2
		cout << "Slave got WRITE burst " 
		     << "                               first_data=" << hex << (*((Td*)(req->get_data_ptr()))) 
		     << "                    time=" << dec << sc_core::sc_time_stamp() << endl;
#endif
	    }
	}

	time = sc_core::SC_ZERO_TIME;
	phase = tlm::END_REQ;
	tlm::tlm_sync_enum retVal = tp_socket->nb_transport_bw(*req, phase, time);

	switch(retVal){
        case tlm::TLM_ACCEPTED: break;
        case tlm::TLM_UPDATED:
	    cerr<<"Error in " << name() << " got unexpected phase update " << phase << " in response to END_REQ" << endl;
	    exit(1);
	    break;
        case tlm::TLM_COMPLETED:;
	}
	req = NULL;
    }
}

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
void Slave<BUSWIDTH, ADDRWIDTH>::sendResp()
{
    scv_smart_ptr< unsigned int > rand;
    rand->next();

    if (last_request == tlm::TLM_READ_COMMAND) {	
	if (!ongoing_rsp) {
	    assert(rsp);

	    // generate some data
	    unsigned int resp_data = 10000 + ((*rand)/100) * 100;

	    // get burst length to know many data should be sent
	    ocpip::burst_length* b_len;
	    bool has_b_len = tp_socket.get_extension<ocpip::burst_length>(b_len, *rsp);
	    assert(has_b_len);
	    unsigned int burst_len = b_len->value;

	    for (unsigned int i = 0; i < burst_len; ++i)
		*(((Td*)(rsp->get_data_ptr())) + i) = resp_data++;

	    rsp->set_response_status(tlm::TLM_OK_RESPONSE);

	    // randomize delay time for response 
	    time = sc_time(((*rand)%5 + 1)*100, SC_NS);
	    phase = tlm::BEGIN_RESP;
	    tlm::tlm_sync_enum retVal = tp_socket->nb_transport_bw(*rsp,phase, time);

	    switch(retVal){
	    case tlm::TLM_ACCEPTED: 
		ongoing_rsp=true; 
		break;
	    case tlm::TLM_UPDATED:
		if (phase != tlm::END_RESP) {
		    std::cerr<<"Error in "<< name() <<" got unexpected phase update "<< phase <<" in response to BEGIN_RESP"<<std::endl;
		    exit(1);
		}
		break;
	    case tlm::TLM_COMPLETED: break;
	    }

#ifdef DEBUG_TL2
        cout << "Slave sent response, delay=" << time
         << "       first_data=" << hex << (*(((Td*)(rsp->get_data_ptr()))))
         << "                      time=" << dec << sc_core::sc_time_stamp() << endl;
#endif


	    last_request = tlm::TLM_IGNORE_COMMAND;
	    rsp = NULL;
	}
    }
    if ( req ) {
	// an incoming burst was pending, receive it now
	receive_burst_ev.notify( sc_core::SC_ZERO_TIME );
    }	
} // end of method

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
tlm::tlm_sync_enum Slave<BUSWIDTH, ADDRWIDTH>::nb_transport(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim)
{
  switch(ph){
    case tlm::BEGIN_REQ:
      if ( !req ) {
	  // get burst if we are not busy
	  receive_burst_ev.notify( 1 * CLKPERIOD, sc_core::SC_NS );
      }
      req = &txn;
      break;
    case tlm::END_RESP:
	if (last_request == tlm::TLM_READ_COMMAND) {
	    // burst waiting to response
	    send_resp_ev.notify( 1 * CLKPERIOD, sc_core::SC_NS );
	}
      ongoing_rsp = false;
      break;
    default:
      std::cerr<<"Error in "<<name()<<" got unexpected phase "<<phase<<" in nb_transport_fw"<<std::endl;
      exit(1);        
  }
  return tlm::TLM_ACCEPTED;
}

// ----------------------------------------------------------------------------
//
//  Instantiation of the Slave
//
// ----------------------------------------------------------------------------
template class Slave< 32,32 >; 

