// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com
//                Stephane Guntz, PROSILOG, guntz@prosilog.com
//         Date : 01/15/2004
//
//  Description : Layer Adapter, TL0-TL1 example, Synchronous TL0 OCP Master
//                with datahandshake support (OCP2.0)
//    Features:
//    - TL0 master implementing OCP2.0 Basic Signals 
//    - Aynchronous Acknowldege signals, with propagation delay emulation  
//    - Synchronous Datahandshake, with propagation delay emulation
//    - Performs 5 WRITE operations followed by 5 READ operations
//    - Checks received responses
//    
//  Parameter    :
//    Template arguments.
//     - TdataCl: Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//     - Td: Data type
//     - Ta: Address type
//                
//   Constructor arguments.
//     - sc_module_name: Name of the module instance
//     - sc_time RequestDelay:
//        delay between clock rising edge and 'MCmd' assertion 
//     - sc_time DataDelay:
//        delay between clock rising edge and 'MDataValid' assertion 
//     - sc_time ResponseDelay:
//        delay between valid response reception ('SResp'==DVA)
//            and 'MRespAccept' signal assertion
// ============================================================================

// Used for debug purposes
#define DEBUG_TL0_MASTER_SYNC_HS

#include "ocp_tl0_master_sync_hs.h"
#include "ocp_tl1_data_cl.h"

// ----------------------------------------------------------------------------
// Process : OCP_TL0_Master_sync_hs::OCP_TL0_Master_sync_hs
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_Master_sync_hs<TdataCl>::OCP_TL0_Master_sync_hs
       (
          sc_module_name name_
          , sc_time RequestDelay
          , sc_time DataDelay
          , sc_time ResponseDelay
        )
          : sc_module(name_)
          , m_RequestDelay(RequestDelay) 
          , m_DataDelay(DataDelay) 
          , m_ResponseDelay(ResponseDelay)
{
  
  // Output ports initializing 
  MCmd.initialize(sc_bv<3>(OCP_MCMD_IDLE));
  MData.initialize(Td(0));
  MAddr.initialize(Ta(0));
  MRespAccept.initialize(false);
  MDataValid.initialize(false);

  // Response Processes
  SC_THREAD(GetResponse);
  sensitive << SResp;
  
  // Request Processes
  SC_THREAD(SendRequest);
  sensitive_pos << Clk;
   
} // end of constructor


// ----------------------------------------------------------------------------
// Process : OCP_TL0_Master_sync_hs::~OCP_TL0_Master_sync_hs
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_Master_sync_hs<TdataCl>::~OCP_TL0_Master_sync_hs()
{
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_Master_sync_hs::GetResponse()
//
//  - Catch 'SResp' variations
//  - Compare Response with expected value
//  - Issue a Response acknowledge after 'ResponseDelay' ns 
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_Master_sync_hs<TdataCl>::GetResponse()
{
  Td CorrectData = 0;
  Td Rdata;
  int i = 0;

  while(true)
  {
    do {
      wait(); // for an event on SResp
    }
    while(SResp.read() != sc_bv<2>(OCP_SRESP_DVA));

    Rdata = SData.read();

#ifdef DEBUG_TL0_MASTER_SYNC_HS
    cout << "TL0 Master got response "
      << " delta " << simcontext()->delta_count()
      << " time " << sc_time_stamp().to_seconds()
      << " received data " << Rdata ;
    if(Rdata != CorrectData++) 
      cout << " is BAD" << endl;
    else
      cout << " is good" << endl;
#endif

    wait(m_ResponseDelay);

    MRespAccept.write(true);

#ifdef DEBUG_TL0_MASTER_SYNC_HS
    cout << "TL0 Master released response "
      << " delta " << simcontext()->delta_count()
      << " time " << sc_time_stamp().to_seconds()
      << " data " << Rdata << endl;
#endif

    wait(Clk.posedge_event());
    MRespAccept.write(false);

    i++;
  } // end of while(true) loop

} // end of SC_THREAD


// ----------------------------------------------------------------------------
//  Method : OCP_TL0_Master_sync_hs::SendRequest()
//  
//  - Sends OCP requests: 5 writes followed by 5 reads
//  - Implements datahandshake for WRITE operations
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_Master_sync_hs<TdataCl>::SendRequest()
{
  int i = 0;
  int j;
  int Pause[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0}; // # of cycles between requests;
  //int Pause[] = {1, 1, 1, 1, 1, 1, 1, 1, 1, 1}; // # of cycles between requests;
  //int Pause[] = {2, 2, 2, 2, 2, 2, 2, 2, 2, 2}; // # of cycles between requests;

  Ta Addr[] =     {0,4,8,12,16,0,4,8,12,16};    // MAddr field for each request
  Td SentData = 0;
  int cnt = 0;

  while(true)
  {
    if (cnt == 0)
    {
      // get new Pause value
      j=i;
      i++;
      cnt = Pause[i%10];

      // Delay Request signals
      // This must work with or without the next line
      wait(m_RequestDelay);

      // Set Request signals
      MAddr.write(Addr[j%10]);
      if(j%10<5) {
        MCmd.write(sc_bv<3>(OCP_MCMD_WR));
      }
      else {
        MCmd.write(sc_bv<3>(OCP_MCMD_RD));
      }

#ifdef DEBUG_TL0_MASTER_SYNC_HS
      cout << "TL0 Master sent request "
        << " delta " << simcontext()->delta_count()
        << " time " << sc_time_stamp().to_seconds() << endl;
#endif
      // Wait for slave acknowledge
      do {
        wait();
      } while (!SCmdAccept.read());

      MCmd.write(sc_bv<3>(OCP_MCMD_IDLE));

      // Datahandshake
      if(j%10<5) {

        // Delay Datahandshake signals
        // This must work with or without the next line
        wait(m_DataDelay);

        // Set Datahandshake signals
        MData.write(SentData++);
        MDataValid.write(true);

#ifdef DEBUG_TL0_MASTER_SYNC_HS
        cout << "TL0 Master sent data "
          << " delta " << simcontext()->delta_count()
          << " time " << sc_time_stamp().to_seconds()
          << " written data " << SentData-1 << endl;
#endif
        // Wait for slave data acknowledge
        do {
          wait();
        } while (!SDataAccept.read());
        MDataValid.write(false);
      }
    }
    else
    {
      cnt--;
      wait();
    }
  } // end of while loop
} // end of SC_THREAD


// ----------------------------------------------------------------------------
//
//  Instantiation of the Master
//
// ----------------------------------------------------------------------------

template class OCP_TL0_Master_sync_hs<OCP_TL1_DataCl<int,int>  >; // see ocp_tl1_globals.h
