// Copyright (c) 2004 Sonics, Inc.
//
// Confidential and Proprietary Information of Sonics, Inc.
// Use, disclosure, or reproduction is prohibited without
// written permission from Sonics, Inc.
//
// $Id: StlConverter.cc,v 1.1 2007/01/25 22:09:07 halexan Exp $
//
// This file implements the command conversion from the wide
// StlTransferCommand that the parser builds to the end-type requested
// for the data and address widths of the OCP

#include "StlParser.h"
#include "OcpBundleDefs.h"
#include "boost/limits.hpp"

using namespace std;
using namespace OcpIp;

namespace {
// Utility function to make a lsb-mask of a given width 
template<typename Td>
Td makeDataMask( uint32_t width )
{
    if ( width == 8*sizeof(Td) )
        return numeric_limits<Td>::max();
    else
        return Td( ( 1ULL << width ) - 1 );
}


// Spell out 128 and 256. I'd like to write
// template<int N> makeDataMask<Sonics::BigInt<N> > but it won't let me
template<>
Sonics::BigInt<128>
makeDataMask<Sonics::BigInt<128> >( uint32_t width )
{
    Sonics::BigInt<128> ret;
    for ( uint32_t i=0; i<width; ++i ) ret.set(i);
    return ret;
}
template<>
Sonics::BigInt<256>
makeDataMask<Sonics::BigInt<256> >( uint32_t width )
{
    Sonics::BigInt<256> ret;
    for ( uint32_t i=0; i<width; ++i ) ret.set(i);
    return ret;
}
}
  
template<typename Td, typename Ta>
StlParser::CommandConverterImpl<Td, Ta>::CommandConverterImpl(
    const StlParser::OcpParams& params,
    const StlParser::MyTransferCommand& ref ) :    
    StlParser::CommandConverter( ref ),
    m_command( params )
{}

template<typename Td, typename Ta>
void
StlParser::CommandConverterImpl<Td, Ta>::convert()
{
    typedef StlParserTransferCommand<Td, Ta> TargetTransferCommand;
    typedef typename TargetTransferCommand::DataInfoType TargetDataInfoType;
    typedef typename TargetTransferCommand::ByteEnType   TargetByteEnType;

    m_command.clear();
    m_command._idleCycles() = m_reference.idleCycles;
    typename TargetTransferCommand::Request& request = m_command._request();
    const MyTransferCommand::Request& refRequest = m_reference.request;
    request.MCmd          = refRequest.MCmd;
    request.MAddr         = refRequest.MAddr;
    request.MAddrSpace    = refRequest.MAddrSpace;
    request.MByteEn       = refRequest.MByteEn;
    request.MReqInfo      = refRequest.MReqInfo;
    request.MDataInfo     = refRequest.MDataInfo;
    request.MThreadID     = refRequest.MThreadID;
    request.MConnID       = refRequest.MConnID;
    request.MTagID        = refRequest.MTagID;
    request.MTagInOrder   = refRequest.MTagInOrder;
    request.MBurstLength  = refRequest.MBurstLength;
    request.MBurstPrecise = refRequest.MBurstPrecise;
    request.MBurstSeq     = refRequest.MBurstSeq;
    request.MBurstSingleReq = refRequest.MBurstSingleReq;
    request.DataLength    = refRequest.DataLength;

    const uint32_t dataWdth     = m_reference.ocpParam( P_data_wdth );
    const uint32_t dataInfoWdth = m_reference.ocpParam( P_mdatainfo_wdth );
    const uint32_t byteenWdth   = m_reference.ocpParam( P_byteen_wdth );
    Td dataLsbMask           = makeDataMask<Td>( dataWdth );
    TargetDataInfoType dataInfoLsbMask =
      makeDataMask<TargetDataInfoType>( dataInfoWdth );
    TargetByteEnType byteenLsbMask =
        makeDataMask<TargetByteEnType>( byteenWdth );
    if (  m_reference.m_transferWidth > dataWdth ) {
        assert( refRequest.DataLength == 1 );
        const MyTransferCommand::DataType& refData     = refRequest.MDataPtr[0];
        const MyTransferCommand::DataType& refDataMask = m_reference.dataMask[0];
        const TargetDataInfoType& refDataInfo     = refRequest.MDataInfoPtr[0];
        const TargetDataInfoType& refDataInfoMask = m_reference.dataInfoMask[0];
        const TargetByteEnType&   refByteen       = refRequest.MByteEn;
        // expand burst data
        request.DataLength = m_reference.m_transferWidth / dataWdth;
        for ( uint32_t i=0; i < request.DataLength; ++i ) {
            MyTransferCommand::DataType  shiftData       = refData;
            MyTransferCommand::DataType  shiftDataMask   = refDataMask;
            TargetDataInfoType shiftDataInfo     = refDataInfo;
            TargetDataInfoType shiftDataInfoMask = refDataInfoMask;
            TargetByteEnType   shiftByteen       = refByteen;
            if ( m_reference.ocpParam( P_endian ) == Definitions::ENDIAN_BIG ) {
                shiftData         >>= ( request.DataLength - i - 1 )*dataWdth;
                shiftDataMask     >>= ( request.DataLength - i - 1 )*dataWdth;
                shiftDataInfo     >>= ( request.DataLength - i - 1 )*dataInfoWdth;
                shiftDataInfoMask >>= ( request.DataLength - i - 1 )*dataInfoWdth;
                shiftByteen       >>= ( request.DataLength - i - 1 )*byteenWdth;
            } else {
                shiftData         >>= i*dataWdth;
                shiftDataMask     >>= i*dataWdth;
                shiftDataInfo     >>= i*dataInfoWdth;
                shiftDataInfoMask >>= i*dataInfoWdth;
                shiftByteen       >>= i*byteenWdth;
            }
            Td data, dataMask;
            TargetDataInfoType dataInfo, dataInfoMask;
            TargetByteEnType byteen;
            data          = shiftData;
            data         &= dataLsbMask;
            dataMask      = shiftDataMask;
            dataMask     &= dataLsbMask;
            dataInfo      = shiftDataInfo;
            dataInfo     &= dataInfoLsbMask;
            dataInfoMask  = shiftDataInfoMask;
            dataInfoMask &= dataInfoLsbMask;
            byteen        = shiftByteen;
            byteen       &= byteenLsbMask;
            m_command.addData( data, dataInfo, byteen );
            assert( m_command.dataMask.size() > i );
            assert( m_reference.dataMask.size() > i );
            m_command._dataMask()[i] = dataMask;
            if ( m_command.ocpParam( P_sdatainfo ) )
                m_command._dataInfoMask()[i] = dataInfoMask;
        }
    } else {
        for ( uint32_t i=0; i < refRequest.DataLength; ++i ) {
            Td data;
            TargetDataInfoType dataInfo;
            // Use the = operator as a data converter
            data          = refRequest.MDataPtr[i];
            data         &= dataLsbMask;
            dataInfo      = refRequest.MDataInfoPtr[i];
            dataInfo     &= dataInfoLsbMask;
            m_command.addData( data, dataInfo );
            assert( m_command.dataMask.size() > i );
            assert( m_reference.dataMask.size() > i );
            m_command._dataMask()[i] = m_reference.dataMask[i];
            if ( m_command.ocpParam( P_sdatainfo ) )
                m_command._dataInfoMask()[i] = m_reference.dataInfoMask[i];
        }
    }
}

StlParser::CommandConverter*
StlParser::CommandConverter::create( 
    const StlParser::OcpParams& params,
    const StlParser::MyTransferCommand& ref )
{
#define TYPE32  uint32_t
#define TYPE64  snx_uint64_t
#define TYPE128 Sonics::BigInt<128>
#define TYPE256 Sonics::BigInt<256>
// asserts to ensure type is the size we think
#define TRYCONVERTERWITH( wd, wa )  \
assert( sizeof( TYPE##wd ) == wd/8 ); \
assert( sizeof( TYPE##wa ) == wa/8 ); \
if ( wd >= dataWdth && wa >= addrWdth ) \
    return new StlParser::CommandConverterImpl<TYPE##wd, TYPE##wa>( params, ref );

    uint32_t dataWdth = params.getParam( P_data_wdth );
    uint32_t addrWdth = params.getParam( P_addr_wdth );

// try in ascending width to get the smallest fit
    TRYCONVERTERWITH( 32, 32 );
    TRYCONVERTERWITH( 32, 64 );
    TRYCONVERTERWITH( 64, 32 );
    TRYCONVERTERWITH( 64, 64 );
    TRYCONVERTERWITH( 128, 32 );
    TRYCONVERTERWITH( 128, 64 );
    TRYCONVERTERWITH( 256, 32 );
    TRYCONVERTERWITH( 256, 64 );

#undef TRYCONVERTERWITH
#undef TYPE32
#undef TYPE64
#undef TYPE128
#undef TYPE256

    assert( false );
    return NULL;
}
