// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level Layer-1
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//                Anssi Haverinen, Nokia Inc., anssi.haverinen@nokia.com
//         Date : 02/15/2003
//
//  Description : Transaction Level - Layer-1 example Master
//    Features:
//    - Asynchronous master using OCP data class
//    - Non-blocking methods are used
//    - Pipelined Request and Response threads
//    - Performs 5 WRITE operations followed by 5 READ operations
//       and checks read data, assuming that the target
//       acts as a RAM
//                
// Parameter    :
//   Template arguments.
//   - TdataCl: Data class containing data members and
//              access methods for the data exchanged between
//              Masters and Slaves
//   - Td: Data type
//   - Ta: Address type
//                
//   Constructor arguments.
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//     - Priority: Positive number specifying the priority relative to the
//                 other Masters for Bus access. Higher numbers means higher
//                 priority. Masters can have the same priority.
//     - RequestDelay: Non-negative number specifying the delay in percent
//                 of a clock cycle, the request signals are transmitted.
//     - ResponseDelay: Non-negative number specifying the delay in percent
//                 of a clock cycle, the response is acknowledged .
// ============================================================================

#include "ocp_tl1_master_async.h"
#include "ocp_tl1_data_cl.h"

// ----------------------------------------------------------------------------
// Process : OCP_TL1_Master_Async::OCP_TL1_Master_Async
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL1_Master_Async<TdataCl>::OCP_TL1_Master_Async
      (
        sc_module_name name_
      , int ID
      , int Priority
      , double RequestDelay
      , double ResponseDelay
      )
        : sc_module         (name_)
        , MasterP           ("MasterPort")
        , m_ID              (ID)
        , m_Priority        (Priority)
        , m_RequestDelay        (RequestDelay)
        , m_ResponseDelay        (ResponseDelay)
{
  // InitParameters();

  SC_THREAD(MasterRequest);
  sensitive_pos(clk); 
  SC_THREAD(MasterResponse);
  sensitive << MasterP; 
}

// ----------------------------------------------------------------------------
// Process : OCP_TL1_Master_Async::~OCP_TL1_Master_Async
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL1_Master_Async<TdataCl>::~OCP_TL1_Master_Async()
{
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Async::MasterRequest()
//
//  Top level request method of the Master
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Master_Async<TdataCl>::MasterRequest()
{
  int i = 0;
  int j;
  int Pause[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0}; // # of cycles between requests;
  //int Pause[] = {1, 1, 1, 1, 1, 1, 1, 1, 1, 1}; // # of cycles between requests;
  //int Pause[] = {2, 2, 2, 2, 2, 2, 2, 2, 2, 2}; // # of cycles between requests;

  Ta Addr[] =     {0,4,8,12,16,0,4,8,12,16};    // MAddr field for each request
  Td SentData[] =     {5,4,3,2,1,0,0,0,0,0};    // MData field for each request
  int cnt = 0;

  while(true)
  {
    wait(); // wait for rising edge of clock

    if (cnt == 0)
    {
      // check if channel is free
      if (!MasterP->MgetSbusy())
      {
        // get new Pause value
        j=i;
        i++;
        cnt = Pause[i%10];

        // Delay data signal
        // This must work with or without the next line
        if (m_RequestDelay > 0)
          wait(TL_CLK_PERIOD * (m_RequestDelay/100.0), TL_TIME_UNIT);

        // Put data into the channel
        m_DataCl->MputMAddr(Addr[j%10]);
        m_DataCl->MputMData(SentData[j%10]);
        if(j%10<5) {
          m_DataCl->MputMCmd(OCP_MCMD_WR);
          MasterP->MputWriteRequest(); // WRITE operations 
        }
        else {
          m_DataCl->MputMCmd(OCP_MCMD_RD);
          MasterP->MputReadRequest(); // READ operations 
        }

#ifdef DEBUG_G1
        cout << "TL1 Master sent request "
          << " delta " << simcontext()->delta_count()
          << " time " << sc_time_stamp().to_seconds();
        if(j%10<5) 
          cout << " written data " << SentData[j%10] << endl;
        else 
          cout << endl;
#endif

      }
    }
    else
    {
      cnt--;
    }
  } // end of while loop
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Async::MasterResponse()
//
//  Top level response method of the Master
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Master_Async<TdataCl>::MasterResponse()
{
  Td CorrectData[] =  {5,4,3,2,1};
  Td Rdata;
  int i = 0;

  while(true)
  {
    wait(); // for default event

    if (MasterP->MgetResponse(false))
    {
      m_DataCl->MgetSData(Rdata);

#ifdef DEBUG_G1
      cout << "TL1 Master got response "
           << " delta " << simcontext()->delta_count()
           << " time " << sc_time_stamp().to_seconds()
           << " received data " << Rdata ;
      if(Rdata != CorrectData[i%5]) 
        cout << " is BAD" << endl;
      else
        cout << " is good" << endl;
#endif

      if (m_ResponseDelay > 0)
        wait(TL_CLK_PERIOD * (m_ResponseDelay/100.0), TL_TIME_UNIT);

      MasterP->Mrelease();

#ifdef DEBUG_G1
      cout << "TL1 Master released response "
           << " delta " << simcontext()->delta_count()
           << " time " << sc_time_stamp().to_seconds()
           << " data " << Rdata << endl;
#endif

      i++;
    }
  } // end of while(true) loop
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Async::SputDirect()
//
//  Debug interface method.
//  Read/Write data, Slave to Master direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl> bool OCP_TL1_Master_Async<TdataCl>::SputDirect(
         int SlaveID, bool IsWrite, Td *Data, Ta Address, int NumWords)

{
  // not implemented
  return(false);
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Async::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Master_Async<TdataCl>::end_of_elaboration()
{
  sc_module::end_of_elaboration();

  // Initialize debug interface
  MasterP->MregisterDirectIF(this);

  // Get data structure
  m_DataCl = MasterP->GetDataCl();

  // Get communication structure
  m_CommCl = MasterP->GetCommCl();

  // Get system parameter structure
  m_ParamCl = MasterP->GetParamCl();

  // Put parameter into Channel
  // No parameters needed
}


// ----------------------------------------------------------------------------
//
//  Instantiation of the Master
//
// ----------------------------------------------------------------------------
template class OCP_TL1_Master_Async<TL1_TEMPL_DATA_CL >; // see ocp_tl1_globals.h
