// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//                Stphane Guntz, PROSILOG, guntz@prosilog.com
//         Date : 03/10/2003
//
//  Description : Layer Adapter, TL0-TL1 example
//              Top level, TL1 Synchronous Master <---> TL0 Synchronous Slave
//
//  This example shows an OCP connexion between a synchronous TL0 Slave and
//    a TL1 Master, where responses occurs one cycle after requests, and 
//    'Accept' signals assertion takes one clock cycle ( + some propagation
//    delays)
//    
//  The master issues 5 writes followed by 5 reads, and checks received data
//    assuming that OCP Target acts as a RAM.
//    
//  Operation timings are controlled by the following parameters: T1, T2, T3,
//  T4 ,T5 and T6
//    - T2, T3, T5 and T6 are expressed as 'sc_time' values
//    - T1 and T4 are expressed as a percentage (double) of the clock cycle 
//    - Clock cycle duration in this example is 10 ns
//  
//  OCP transfers occurs such as follows:
//    * 'T1*clock_cycle/100' ns after the clock rising edge, TL1 Master sends a
//      request 
//    * TL0 Slave adapter catchs the request and set the TL0 Request Signal
//      group
//    * After request reception, TL0 slave waits for:
//                  - Next cycle + 'T2' ns and releases the request
//                  - Next cycle + 'T3' ns and issues the response
//    * As soon as 'SCmdAccept' is set, TL0 adapter releases the TL1 request
//    * As soon as 'SResp' is set, TL0 adapter sends the TL1 response
//    * After Response reception, TL1 Master waits for 'T4*clock_cycle/100' ns
//      and releases the TL1 response
//    * As soon as TL1 Response is released, TL0 Adapter set 'MRespAccept'
//      signal
//
//  T5 & T6 specifies the sample delays for the TL0 Slave Adapter.
//  To get a correct behaviour, T5 must be larger than 'T2' (SCmdAccept)
//  and T6 must be larger than 'T3' (response signals).
//
//  TL0 signals can be viewed using the VCD file 'top_slave_sync2.vcd'
//    
// ============================================================================

#include "ocp_tl1_master_async.h"
#include "ocp_tl0_slave_sync2.h"
#include "ocp_tl0_tl1_slave_adapter.h"
#include "ocp_tl1_data_cl.h"
#include "tl_channel.h"  


int sc_main(int, char*[])
{
    // TL1 channel
    //
    // 
    bool sync = true; // mandatory for OCP data classes
    bool defaultevent = true; // mandatory for the tl1 master used in this example 
    // 'SyncEvent' must be set fot the adapter to work correctly
    bool syncevent = true;
    TL_Channel<TL1_TEMPL_DATA_CL > ch1("ch0", sync, syncevent, defaultevent);

    // TL1 Master
    //
    // 
    // Request group is valid after 'T1'% of the clock cycle
    double T1 = 10;     
    // Response acknowledge is valid after 'T4'% of the clock cycle, following the response arrival 
    // T4 value > 100% is used to emulate a sequential path for 'MRespAccept' signal
    double T4 = 110;     

    OCP_TL1_Master_Async<TL1_TEMPL_DATA_CL > ms1("ms1", 1, 1, T1, T4);

    
    // TL0 Synchronous Slave 
    //
    // 
    sc_time T2 = sc_time(2, SC_NS);   // Request delay
    sc_time T3 = sc_time(1, SC_NS);   // Response delay
    OCP_TL0_Slave_sync2<TL1_TEMPL_DATA_CL > sl0("sl_adapter", T2, T3 );

    // TL0 Slave Adapter
    //
    // Sample delay T5 is used to sample SCmdAccept signal: it must be > T2 in this case 
    // Sample delay T6 is used to sample Response signal: it must be > T3 in this case
    // check_setup_time parameter is actived to check TL0 setup time violations
    sc_time T5 = sc_time(3, SC_NS);
    sc_time T6 = sc_time(2, SC_NS);
    bool check_setup_time = true;
    OCP_TL0_TL1_Slave_Adapter<TL1_TEMPL_DATA_CL > sl_adapter("sl0", 1, T5, T6, check_setup_time );


    // Testbench building
    //
    //
    sc_clock clk("clock", TL_CLK_PERIOD, TL_TIME_UNIT);
    sc_signal<sc_bv<3> > 	s_MCmd;
    sc_signal<int>       	s_MAddr;
    sc_signal<int>       	s_MData;
    sc_signal<bool> 	    s_SCmdAccept;
    sc_signal<int>       	s_SData;
    sc_signal<sc_bv<2> >  s_SResp;
    sc_signal<bool>  	    s_MRespAccept;


    ms1.MasterP(ch1);
    ms1.clk(clk);

    sl_adapter.SlaveP(ch1);

    sl_adapter.Clk(clk);
    sl_adapter.MCmd(s_MCmd);
    sl_adapter.MAddr(s_MAddr);
    sl_adapter.MData(s_MData);
    sl_adapter.SCmdAccept(s_SCmdAccept);
    sl_adapter.SData(s_SData);
    sl_adapter.SResp(s_SResp);
    sl_adapter.MRespAccept(s_MRespAccept);

    sl0.Clk(clk);
    sl0.MCmd(s_MCmd);
    sl0.MAddr(s_MAddr);
    sl0.MData(s_MData);
    sl0.SCmdAccept(s_SCmdAccept);
    sl0.SData(s_SData);
    sl0.SResp(s_SResp);
    sl0.MRespAccept(s_MRespAccept);

    // VCD Trace file for TL0 signals
    //
    //
    sc_trace_file *vcd_file = sc_create_vcd_trace_file("top_slave_sync2"); 

    sc_trace(vcd_file,clk,"Clk");
    sc_trace(vcd_file,s_MCmd,"MCmd");
    sc_trace(vcd_file,s_MAddr,"MAddr");
    sc_trace(vcd_file,s_MData,"MData");
    sc_trace(vcd_file,s_SCmdAccept,"SCmdAccept");
    sc_trace(vcd_file,s_SData,"SData");
    sc_trace(vcd_file,s_SResp,"SResp");
    sc_trace(vcd_file,s_MRespAccept,"MRespAccept");


    sc_start(500, SC_NS);

    sc_close_vcd_trace_file(vcd_file);

    return(0);
}
