// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//                (modified by Joe Chou, Sonics Inc., joechou@sonicsinc.com)
//         Date : 03/25/2003
//
//  Description : Transaction Level - Layer-2 Bus
//                Features: 
//                - Arbitrary number of Masters
//                - Arbitrary number of Slaves
//                - Address decoding
//                - 2 tier arbitration
//                - Debug interface
//                - Bus can be pipelined or non-pipelined.
//
// Parameter    : Template arguments.
//                - TdataCl: Data class containing data members and
//                           access methods for the data exchanged between
//                           Masters and Slaves
//                  - Td: Data type
//                  - Ta: Address type
//                Constructor arguments.
//                - sc_module_name: Name of the module instance
//                - BusID: Unique number identifying the Bus
//                         BusID must be unique among all Busses in a system
//                - Pipelined: Switch to change between non-pipelined and
//                             pipelined operation mode of the Bus.
//                - WriteResponse: Switch to enable/disable the sending of a
//                                 response packet to a Master write request.
//                                 Note that all modules involved in a system
//                                 must use the same value.
//                - ReadWaitCycles: Number of cycles per beat the Bus delays
//                                  the transport of the read request/response.
//                - WriteWaitCycles: Number of cycles per beat the Bus delays
//                                   the transport of the write request/response.
//
// ============================================================================

// ===========================================================================
//
// Bus operation
// -------------
// - Master sends request to Bus and locks the Master-Bus-Request Channel.
// - Bus collects all requests and selects one Master and the addressed Slave.
//   It then copies the data from the Master-Bus Channel to the Bus-Slave
//   Channel.
// - The Bus triggers the addressed Slave and locks the Bus-Slave-Request
//   Channel.
// - The Bus waits for the answer of the addressed Slave.
// - The addressed Slave processes the incomming data and unlocks the
//   Bus-Slave-Request Channel.
// - Bus unlocks the Master-Bus-Request Channel
// - The Slave triggeres the Bus and locks the Bus-Slave-Response Channel.
// - The Bus copies the data from the Bus-Slave Channel to the Master-Bus
//   Channel and then unlocks the Bus-Slave-Response Channel.
// - The Bus then triggers the Master and locks the Master-Bus-Response Channel.
// - The Master processes the response data and unlocks the Master-Bus-Response
//   Channel.
//
// Pipelined versus non-pipelined operation mode
// ---------------------------------------------
// The bus can be pipelined or non-pipelined.
// In the non-pipelined case the Master requests and Slave responses
// are handled sequentially in a single thread. This is pretty secure
// against deadlocks.
// In the piplined case the request and reponse methods are called from
// two different threads. The request method is triggered at each
// Master request and the response method is triggered at each Slave response.
// The bus makes no assumptions on the pipeline structure like pipeline depth
// etc. All data the bus needs (that is in particular the port numbers of the
// granted Master and the addressed Slave) are derived from the Channel data.
// It is therefore the task of the Masters and Slaves to update the Channel
// data and trigger the request/response thread of the Bus at the right time.
// Hence the pipeline behavior is completely determined by the
// timing of the Master and Slave. In our example it is the Slave who
// determines pipeline delay and pipeline depth.
//
// Arbitration
// -----------
// Arbitartion is done according the following 2 tier scheme:
// - First tier arbitration is priority based
// - For Masters with the same (highest) priority the second tier arbitration
//   algorithm is used. The second tier arbitration is a fair among equals
//   algorithm based on the number of grants.
//
//
// Address decoding
// ----------------
// This bus verion only supports one address region per Slave.
// Hence the following featurs are NOT supported:
// - Multiple possibly disconnected address regions per Slave
// - Address remapping between boot and normal mode
//
// The bus sends the same address to the Slave as obtained from the Master.
// That is, a transformation of the address to the Slaves memory index range
// (which is normally [0, EndAddress - StartAddress - 1]) is not performed
// and must be done in the Slave.
//
//
// Timing
// ------
// The bus has the following timing:
// - Write request  transfer: wait NumWords * WriteWaitCycles cycles
// - Read  request  transfer: wait    1     * ReadWaitCycles  cycles
// - Write response transfer: wait    1     * WriteWaitCycles cycles
// - Read  response transfer: wait NumWords * ReadWaitCycles  cycles
// ===========================================================================

#include "ocp_tl2_bus.h"

// ----------------------------------------------------------------------------
// Process : OCP_TL2_Bus::OCP_TL2_Bus
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataClSport, class TdataClMport>
OCP_TL2_Bus<TdataClSport, TdataClMport>::OCP_TL2_Bus(sc_module_name name_,
  int BusID, bool Pipelined, bool WriteResponse, int ReadWaitCycles,
  int WriteWaitCycles)
   : sc_module(name_),
     SlaveP("SlavePort"),
     MasterP("MasterPort"),
     m_BusID(BusID),
     m_Pipelined(Pipelined),
     m_WriteResponse(WriteResponse),
     m_ReadWaitCycles(ReadWaitCycles),
     m_WriteWaitCycles(WriteWaitCycles),
     m_IsInitialized(false),
     m_GrantedMaster(-1),
     m_AddressedSlave(-1),
     m_TaNum(0)
{
  int i;

  // InitParameters();

  //
  m_NumBytesPerWord = sizeof(Td);

  // Initialize some arrays
  for (i = 0; i < TL2_MAX_MASTERS; i++) {
    m_MasterID[i]  = -1;
    m_Sport[i] = -1;
    m_Requests[i]  = 0;
    m_NumGrants[i] = 0;
  }

  // Initialize for dummy master
  m_NumGrants[TL2_MAX_MASTERS] = TL2_MAX_DOUBLE;

  //
  for (i = 0; i < TL2_MAX_SLAVES; i++) {
    m_SlaveID[i] = -1;
    m_Mport[i] = -1;
    m_MemoryMap[i].StartAddr = -1;
    m_MemoryMap[i].EndAddr = -1;
  }

  for (i = 0; i < TL2_ARB_REG_LENGTH; i++) {
    m_ArbiterReg[i] = -1;
  }

  // Default master is dummy master
  m_ArbiterReg[TL2_DF_GRANT_ID] = TL2_DUMMY_MASTER;

  // Safty net for destructor
  m_MemoryMapCache = 0;

  //
  if (m_Pipelined) {
    SC_THREAD(BusRequest);
    sensitive << SlaveP;
    SC_THREAD(BusResponse);
    sensitive << MasterP;
    SC_THREAD(BusError);
    sensitive << m_BusErrorEvent;
  } else {
    SC_THREAD(Bus);
    sensitive << SlaveP;
  }
}

// ----------------------------------------------------------------------------
// Process : OCP_TL2_Bus::~OCP_TL2_Bus
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataClSport, class TdataClMport>
OCP_TL2_Bus<TdataClSport, TdataClMport>::~OCP_TL2_Bus()
{
  cout << "Bus " << m_BusID << " : Number of Transactions = "
       <<  m_TaNum << endl;

#ifdef TL2_USE_MEM_CACHE
  delete [] m_MemoryMapCache;
#endif
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Bus::Bus()
//
//  Top level method of the non-pipelined Bus
//
// ----------------------------------------------------------------------------
template<class TdataClSport, class TdataClMport>
void OCP_TL2_Bus<TdataClSport, TdataClMport>::Bus()
{
  int  i;
  int  ReqPending;
  bool Write;

  //
  if (!m_IsInitialized) Initialize();

  //
  while (true) {
    // give up control and wait for Master requests
    wait();

    // a SlaveP change nodified
    ReqPending = true;

    // Loop until all missed events are collected
    while (ReqPending) { 
      // collect all requests (non-blocking)
      for (i = 0; i < m_NumMasters; i++) {
        if (SlaveP[i]->SgetRequest(false)) {
          // release later

          // remember the receiving of a request
          m_Requests[i] = 1;
        }
      }

      // Arbitration (among all requests, if any)
      m_GrantedMaster = Arbitration();

      // see who wins the arbitration
      if (m_GrantedMaster == TL2_DUMMY_MASTER) {
        // dummy master has the grant (i.e. no pending request)
        ReqPending = false;
      } else {
        // a winner found and the bus is processing the request

        // initialization
        int ErrMsg = 0;

        // Update transaction counter
        m_TaNum++;

        // get the request from the winner
        Write = SlaveP[m_GrantedMaster]->IsWrite();
        Ta addr = m_SdataCl[m_GrantedMaster]->SgetMAddr();

#ifdef DEBUG_G1
        PrintTime("Start request in Bus ",m_BusID,addr);
#endif

        // Address decoding (to which slave port)
        m_AddressedSlave = AddressDecoding(addr);

        // is it a legal slave port?
        if (m_AddressedSlave < TL2_MAX_SLAVES) {
          // a legal slave port found

          // is it a write request?
          if (Write) {
            // a write transfer

            // Request-transfer time of the bus
            int NumWords = m_SdataCl[m_GrantedMaster]->SgetAtomicLen();
            wait(NumWords*m_WriteWaitCycles,TL_TIME_SCALE);

            // forward a request from the Master-Bus Channel to the Bus-Slave
            // Channel, and setup the connection ID
            ForwardRequest(m_GrantedMaster,m_AddressedSlave);

            // Send a write request to Slave
            if ((MasterP[m_AddressedSlave]->MputWriteRequestBlocking())) {
              // Release Master-Bus-Request Channel only when the forwarding
              // request is sent
              SlaveP[m_GrantedMaster]->Srelease();

              // check if write response is enabled
              if (m_WriteResponse) {
                // need to return a write response

                // Get a Slave response
                if ((MasterP[m_AddressedSlave]->MgetResponseBlocking(false))) {
                  // a response is received; release later

                  // add transfer time (could be collapsed with the above timing
                  // to speedup simulation)

                  // Write Response transfer time of the bus
                  wait(1*m_WriteWaitCycles,TL_TIME_SCALE);

                  // forward response from the Bus-Slave Channel to the
                  // Master-Bus Channel
                  ForwardResponse(m_GrantedMaster,m_AddressedSlave);

                  // Send response to Master
                  if (SlaveP[m_GrantedMaster]->SputResponseBlocking()) {
                    // release Bus-Slave Channel
                    MasterP[m_AddressedSlave]->Mrelease();
                  } else {
                    cout << "Blocking errors detected"
                         << " <" << name() << ">" << endl;
                    ErrMsg = 3;
                  }
                } else {
                  // errors found
                  cout << "Blocking errors detected"
                       << " <" << name() << ">" << endl;

                  // Send error response to Master
                  m_SdataCl[m_GrantedMaster]->SputSResp(OCP_SRESP_ERR);
                  ErrMsg = 2;

                  // Send response to Master
                  if (!(SlaveP[m_GrantedMaster]->SputResponseBlocking())) {
                    cout << "Blocking errors detected"
                         << " <" << name() << ">" << endl;
                    ErrMsg = 3;
                  }
                }
              } // end of write response
            } else {
              // errors found
              cout << "Blocking errors detected"
                   << " <" << name() << ">" << endl;

              // Release Master-Bus-Request Channel
              SlaveP[m_GrantedMaster]->Srelease();

              // Send error response to Master
              m_SdataCl[m_GrantedMaster]->SputSResp(OCP_SRESP_ERR);
              ErrMsg = 1;

              // Send response to Master
              if (!(SlaveP[m_GrantedMaster]->SputResponseBlocking())) {
                cout << "Blocking errors detected"
                     << " <" << name() << ">" << endl;
                ErrMsg = 3;
              }
            } // end of write request branch
          } else {
            // a read transfer

            // Request-transfer time of the bus
            wait(1*m_ReadWaitCycles,TL_TIME_SCALE);

            // forward a request from the Master-Bus Channel to the Bus-Slave
            // Channel, and setup the connection ID
            ForwardRequest(m_GrantedMaster,m_AddressedSlave);

            // Send a read request to Slave
            if ((MasterP[m_AddressedSlave]->MputReadRequestBlocking())) {
              // Release Master-Bus-Request Channel only when the forwarding
              // request is sent
              SlaveP[m_GrantedMaster]->Srelease();

              // Get Slave response
              if ((MasterP[m_AddressedSlave]->MgetResponseBlocking(false))) {
                // a response is received

                // add transfer time (could be collapsed with the above timing
                // to speedup simulation)

                // Read Response transfer time of the bus
                int NumWords = m_MdataCl[m_AddressedSlave]->MgetAtomicLen();
                wait(NumWords*m_ReadWaitCycles,TL_TIME_SCALE);

                // forward response from the Bus-Slave Channel to the
                // Master-Bus Channel
                ForwardResponse(m_GrantedMaster,m_AddressedSlave);

                // Send response to Master
                if (SlaveP[m_GrantedMaster]->SputResponseBlocking()) {
                  // release Bus-Slave Channel
	          MasterP[m_AddressedSlave]->Mrelease();
                } else {
                  cout << "Blocking errors detected"
                       << " <" << name() << ">" << endl;
                  ErrMsg = 3;
                }
              } else {
                // errors found

                // Send error response to Master
                m_SdataCl[m_GrantedMaster]->SputSResp(OCP_SRESP_ERR);
                ErrMsg = 2;

                // Send response to Master
                if (!(SlaveP[m_GrantedMaster]->SputResponseBlocking())) {
                  cout << "Blocking errors detected"
                       << " <" << name() << ">" << endl;
                  ErrMsg = 3;
                }
              }
            } else {
              // errors found
              cout << "Blocking errors detected"
                   << " <" << name() << ">" << endl;

              // Release Master-Bus-Request Channel
              SlaveP[m_GrantedMaster]->Srelease();

              // Send error response to Master
              m_SdataCl[m_GrantedMaster]->SputSResp(OCP_SRESP_ERR);
              ErrMsg = 1;

              // Send response to Master
              if (!(SlaveP[m_GrantedMaster]->SputResponseBlocking())) {
                cout << "Blocking errors detected"
                     << " <" << name() << ">" << endl;
                ErrMsg = 3;
              }
            }
          } // end of read branch
        } else {
          // Address decoding failed

          // Release Master-Bus-Request Channel
          SlaveP[m_GrantedMaster]->Srelease();

          // send error response to Master
          m_SdataCl[m_GrantedMaster]->SputSResp(OCP_SRESP_ERR);

          // Send response to Master
          if (m_WriteResponse || (!Write)) {
            if (!(SlaveP[m_GrantedMaster]->SputResponseBlocking())) {
              cout << "Blocking errors detected"
                   << " <" << name() << ">" << endl;
              ErrMsg = 3;
            }
          }
        }

        // Issue error message
        switch (ErrMsg) {
        case 1: // Mput request to Slave failed
          // Deep ship: Should never happen
          cout << "Error: Slave (" << m_SlaveID[m_AddressedSlave]
               << ") dropped Master (" << m_MasterID[m_GrantedMaster]
               << ") request <" << name() << ">" << endl;
          break;
        case 2: // Mget response from Slave failed
          // Deep ship: Should never happen
          cout << "Error: Slave (" << m_SlaveID[m_AddressedSlave]
               << ") did not respond to Master ("
               << m_MasterID[m_GrantedMaster] << ") <"
               << name() << ">" << endl;
          break;
        case 3: // Sput response to Master failed
          // Deep ship: Should never happen
          cout << "Error: Master (" << m_MasterID[m_GrantedMaster]
               << ") dropped Slave (" << m_SlaveID[m_AddressedSlave]
               << ") response <"
               << name() << ">" << endl;
          break;
        default: // No error
          break;
        }
      } // else (still found a pending request)
    } // while (ReqPending) loop
  } // while (true) loop
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Bus::BusRequest()
//
//  Top level method of the pipelined Bus, request thread.
//
// ----------------------------------------------------------------------------
template<class TdataClSport, class TdataClMport>
void OCP_TL2_Bus<TdataClSport, TdataClMport>::BusRequest()
{
  int  i;
  bool bt0;
  int  ReqPending;
  bool Write;
  bool BusError;

  //
  if (!m_IsInitialized) Initialize();

  //
  while (true) {
    // give up control and wait for Master requests
    wait();

    // a SlaveP change notified
    ReqPending = true;

    // Loop until all missed events are collected
    while (ReqPending) {
      // collect all requests (non-blocking)
      for (i = 0; i < m_NumMasters; i++) {
        if (SlaveP[i]->SgetRequest(false)) {
          // release later

          // remember the receiving of a request
          m_Requests[i] = 1;
        }
      }

      // Initialization
      BusError = false;

      // Arbitration (among all requests, if any)
      m_GrantedMaster = Arbitration();

      if (m_GrantedMaster == TL2_DUMMY_MASTER) {
        // dummy master has the grant (i.e. no pending request)
        ReqPending = false;
      } else {
        // a winner found and the bus is processing the request

#ifdef DEBUG_B
        cout << "Bus request from Master " << m_MasterID[m_GrantedMaster]
             << " at " << sc_time_stamp().to_double() << endl;
#endif

        // Update transaction counter
        m_TaNum++;

        // get the request from the winner
        Write = SlaveP[m_GrantedMaster]->IsWrite();
        Ta addr = m_SdataCl[m_GrantedMaster]->SgetMAddr();

        // see whether need a response (used by the error cases) 
        bool ResFlag = m_WriteResponse || (!Write);

#ifdef DEBUG_G1
        PrintTime("Start request in Bus ", m_BusID, addr);
#endif

        // Address decoding (to which slave port)
        m_AddressedSlave = AddressDecoding(addr);

        // is it a legal slave port?
        if (m_AddressedSlave < TL2_MAX_SLAVES) {
          // a legal slave port found

          // check for read/write request
          if (Write) {
            // a write transfer

            // Request-transfer time of the bus
            int NumWords = m_MdataCl[m_AddressedSlave]->SgetAtomicLen();
            wait(NumWords*m_WriteWaitCycles,TL_TIME_SCALE);

            // forward a request from the Master-Bus Channel to the Bus-Slave
            // Channel, and setup the connection ID
            ForwardRequest(m_GrantedMaster,m_AddressedSlave);

            // send a write request to Slave
            bt0 = MasterP[m_AddressedSlave]->MputWriteRequestBlocking();
          } else {
            // a read transfer

            // Request-transfer time of the bus
            wait(1*m_ReadWaitCycles,TL_TIME_SCALE);

            // forward a request from the Master-Bus Channel to the Bus-Slave
            // Channel, and setup the connection ID
            ForwardRequest(m_GrantedMaster,m_AddressedSlave);

            // send a read request to Slave
            bt0 = MasterP[m_AddressedSlave]->MputReadRequestBlocking();
          }

          // blocking function returns error
          if (!bt0) {
            // Deep ship: Should never happen
            cout << "Error: Slave (" << m_SlaveID[m_AddressedSlave]
                 << ") dropped Master (" << m_MasterID[m_GrantedMaster]
                 << ") request <"
                 << name() << ">" << endl;

            // ask the BusError thread to return a request, if needed
            if (ResFlag) {
              // trigger error response to Master
              m_GrantedMasterE  = m_GrantedMaster;
              m_AddressedSlaveE = m_AddressedSlave;
              m_SaddrE = m_SdataCl[m_GrantedMaster]->SgetMAddr();
              m_BusErrorEvent.notify();
            }
          }

          // release Master-Bus Channel
          SlaveP[m_GrantedMaster]->Srelease();
        } else {
          // Address decoding failed

          // release Master-Bus Channel
          SlaveP[m_GrantedMaster]->Srelease();

          // ask the BusError thread to return a request, if needed
          if (ResFlag) {
            // trigger error response to Master
            m_GrantedMasterE  = m_GrantedMaster;
            m_AddressedSlaveE = m_AddressedSlave;
            m_SaddrE = m_SdataCl[m_GrantedMaster]->SgetMAddr();
            m_BusErrorEvent.notify(SC_ZERO_TIME);
          }
        }
      } // else (still found a pending request)
    } // while (ReqPending) loop
  } // while (true) loop
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Bus::BusResponse()
//
//  Top level method of the pipelined Bus, response thread.
//
// ----------------------------------------------------------------------------
template<class TdataClSport, class TdataClMport>
void OCP_TL2_Bus<TdataClSport, TdataClMport>::BusResponse()
{
  int  i;
  int  AddressedSlave;
  int  GrantedMaster;
  bool ResPending;
  bool Write;

  //
  if (!m_IsInitialized) Initialize();

  //
  while (true) {
    // give up control and wait for Slave responses
    wait();

    // a MasterP change nodified
    ResPending = true;

    // Loop until all missed events are collected
    while (ResPending) { 
      ResPending = false;

      // collect all responses (non-blocking)
      for (i = 0; i < m_NumSlaves; i++) {
        if (MasterP[i]->MgetResponse(false)) {
          // release later

          // --------------------
          // process the response
          // --------------------

          //
          ResPending = true;

          // find the return port
          AddressedSlave = i;
          GrantedMaster = GetMasterID(m_MdataCl[i]->MgetSConnID(),
                                      AddressedSlave);
          // setup response type
          Write = m_MdataCl[i]->IsWriteResponse();

          // Sanity check
          if (GrantedMaster < 0 || AddressedSlave < 0) {
            cerr << "ERROR: Wrong Master or Slave ID <"
                 << name() << ">" << endl;
                 sc_stop();
                 return;
          }

          // Another sanity check
          if (Write && (!m_WriteResponse)) {
            cerr << "ERROR: Slave responded to a write request with "
                 << "WriteResponse = false >" << name() << ">" << endl;
                 sc_stop();
                 return;
          }

          // Response-transfer time of the bus
          if (Write) {
            wait(1*m_WriteWaitCycles,TL_TIME_SCALE);
          } else {
            int NumWords = m_MdataCl[m_AddressedSlave]->SgetAtomicLen();
            wait(NumWords*m_ReadWaitCycles,TL_TIME_SCALE);
          }

          // forward response from the Bus-Slave Channel to the Master-Bus
          // Channel
          ForwardResponse(GrantedMaster,AddressedSlave);

#ifdef DEBUG_B
          cout << "Bus response to Master "<< m_MasterID[GrantedMaster]
               << " at " << sc_time_stamp().to_double() << endl;
#endif

          // Send a response to Master
          if (SlaveP[GrantedMaster]->SputResponseBlocking()) {
            // Release Bus-Slave Channel
            MasterP[AddressedSlave]->Mrelease();
          } else {
            // Deep ship: Should never happen
            int temp = (AddressedSlave < TL2_MAX_SLAVES)?
                       m_SlaveID[AddressedSlave]: TL2_MAX_SLAVES;
            cout << "Error: Master (" << m_MasterID[GrantedMaster]
                 << ") dropped Slave (" << temp
                 << ") response <"
                 << name() << ">" << endl;
          }

#ifdef DEBUG_B
          cout << "Bus finished response to Master "<< m_MasterID[GrantedMaster]
               << " at " << sc_time_stamp().to_double() << endl;
#endif

        } // end of Bus to Master response
      } // end of loop over all MasterP ports
    } // end of pending request loop
  } // while (true) loop
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Bus::BusError()
//
//  Top level method of the pipelined Bus, error response thread.
//
// ----------------------------------------------------------------------------
template<class TdataClSport, class TdataClMport>
void OCP_TL2_Bus<TdataClSport, TdataClMport>::BusError()
{
  //
  if (!m_IsInitialized) Initialize();

  //
  while (true) {
    wait();

    // send error response to Master
    m_SdataCl[m_GrantedMasterE]->SputSResp(OCP_SRESP_ERR);
    m_SdataCl[m_GrantedMasterE]->SputSAddr(m_SaddrE);
    if (!(SlaveP[m_GrantedMasterE]->SputResponseBlocking())) {
      // Deep ship: Should never happen
      int temp = (m_AddressedSlaveE < TL2_MAX_SLAVES)?
                 m_SlaveID[m_AddressedSlaveE]: TL2_MAX_SLAVES;
      cout << "Error: Master (" << m_MasterID[m_GrantedMasterE]
           << ") dropped Slave (" << temp
           << ") response <"
           << name() << ">" << endl;
    }

#ifdef DEBUG_B
    cout << "Bus sends Error response to Master " << endl;
#endif
  } // end of while (true)
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Bus::AddressDecoding()
//
//  Decoding of Address
//
//  Returns port number of addressed Slave
// ----------------------------------------------------------------------------
template<class TdataClSport, class TdataClMport>
int OCP_TL2_Bus<TdataClSport, TdataClMport>::AddressDecoding(Ta Addr)
{
#ifdef TL2_USE_MEM_CACHE
  if ((Addr < m_FirstStartAddr) || (Addr > m_LastEndAddr)) {

#ifdef DEBUG_G1
    cout << "Address Decoding (out of boundy) to bus master port: "
         << TL2_MAX_SLAVES
         << " <" << name() << ">" << endl;
#endif

    return (TL2_MAX_SLAVES);
  } else {

    int port =
      m_MemoryMapCache[(Addr - m_FirstStartAddr) >> TL2_ADDR_ALLIGNMENT];

#ifdef DEBUG_G1
    cout << "Address Decoding to bus master port: " << port
         << " <" << name() << ">" << endl;
#endif

    return (port);
  }
#else
  int j = TL2_MAX_SLAVES;

  for (j = 0; j < m_NumSlaves; j++) {
    if ((Addr >= m_MemoryMap[j].StartAddr) &&
        (Addr <=  m_MemoryMap[j].EndAddr)) {
        break;
    }
  }

#ifdef DEBUG_G1
    cout << "Address Decoding to bus master port: " << port
         << " <" << name() << ">" << endl;
#endif

  return (j);
#endif // TL2_USE_MEM_CACHE
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Bus::Arbitration()
//
//  2 tier Arbitration:
//    First tier is priority based
//    Second tier is fair among equals based on number of grants
//
//  Returns port number of Master who gets the grant
// ----------------------------------------------------------------------------
template<class TdataClSport, class TdataClMport>
int OCP_TL2_Bus<TdataClSport, TdataClMport>::Arbitration()
{
  int i;
  int CurPriority = 0;
  int GrantedMaster = m_ArbiterReg[TL2_DF_GRANT_ID]; // which equals to the
                                                     // TL2_DUMMY_MASTER


#ifdef DEBUG_ARB
  for (i = 0; i < m_NumMasters; i++) {
    cout << "Priority of bus slave port " << i
         << " (before arb) is " << m_ArbiterReg[i]
         << " <" << name() << ">" << endl;
  }
#endif

  for (i = 0; i < m_NumMasters; i++) {
    if (m_Requests[i] != 0) {
      // found a request, check the priority
      if (m_ArbiterReg[i] > CurPriority) {
        // first tier arbitration
        GrantedMaster = i;
        CurPriority = m_ArbiterReg[i];
      } else if (m_ArbiterReg[i] == CurPriority) {
        // second tier arbitration
        if (m_NumGrants[i] <= m_NumGrants[GrantedMaster])
          GrantedMaster = i;
      }
    }
  }
  if (GrantedMaster != TL2_DUMMY_MASTER) {
    m_Requests[GrantedMaster] = 0;
    m_NumGrants[GrantedMaster] += 1;
  }

#ifdef DEBUG_ARB
  for (i = 0; i < m_NumMasters; i++) {
    cout << "Priority of bus slave port " << i
         << " (after arb) is " << m_ArbiterReg[i]
         << " <" << name() << ">" << endl;
  }
#endif

#ifdef DEBUG_G1
  cout << "Arbitration from bus slave port " << GrantedMaster
       << " <" << name() << ">" << endl;
#endif

  return (GrantedMaster);
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Bus::SetConnID(int, int)
//
//  Computes the Connection ID (MConnID) which identifies uniquely
//  a Master-Slave connection. It is more convenient to do this in
//  the Bus (as in the Master) because only the Bus can select a unique
//  number.
//
// ----------------------------------------------------------------------------
template<class TdataClSport, class TdataClMport>
int OCP_TL2_Bus<TdataClSport, TdataClMport>::SetConnID(int GrantedMaster,
  int AddressedSlave)
{
  return (GrantedMaster + AddressedSlave * m_NumMasters);
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Bus::GetMasterID(int, int)
//
//  Computes the MasterID from the ConnectionID (SConnID)
//  The algorithm must be inverse to SetConnID()
//
// ----------------------------------------------------------------------------
template<class TdataClSport, class TdataClMport>
int OCP_TL2_Bus<TdataClSport, TdataClMport>::GetMasterID(int SConnID,
  int AddressedSlave)
{
  return (SConnID - AddressedSlave * m_NumMasters);
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Bus::ForwardRequest()
//
//  forward a request from Master-Bus Channel to Bus-Slave Channel
//
// ----------------------------------------------------------------------------
template<class TdataClSport, class TdataClMport>
void OCP_TL2_Bus<TdataClSport, TdataClMport>::ForwardRequest(int GrantedMaster,
  int AddressedSlave)
{
  int  words;
  bool last;
  Td*  data;

  // forward the MAddr field
  m_MdataCl[AddressedSlave]->MputMAddr(m_SdataCl[GrantedMaster]->SgetMAddr());

  // forward the MData, atomic length, and burst last fields
  data = m_SdataCl[GrantedMaster]->SgetMData(words,last);
  m_MdataCl[AddressedSlave]->MputMData(data,words,last);

#ifdef DEBUG_JOE
  cout << "Forward request: MAddr: " << m_SdataCl[GrantedMaster]->SgetMAddr()
       << " MDataPtr: " << data
       << " <" << name() << ">" << endl;
#endif

  // setup the connection ID
  // - the MConnID is used to identify the granted Master for a returned
  //   response
  m_MdataCl[AddressedSlave]->
    MputMConnID(SetConnID(GrantedMaster,AddressedSlave)); 
 
  // Add additional OCP fields, if needed
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Bus::ForwardResponse()
//
//  forward a response from Bus-Slave Channel to Master-Bus Channel
//
// ----------------------------------------------------------------------------
template<class TdataClSport, class TdataClMport>
void OCP_TL2_Bus<TdataClSport, TdataClMport>::ForwardResponse(int GrantedMaster,
  int AddressedSlave)
{
  int  words;
  bool last;
  Td*  data;

  // forward the SData, atomic length, and burst last fields
  data = m_MdataCl[AddressedSlave]->MgetSData(words,last);
  m_SdataCl[GrantedMaster]->SputSData(data,words,last);

  // forward the SResp, SAddr, and SCmd fields
  m_SdataCl[GrantedMaster]->SputSResp(m_MdataCl[AddressedSlave]->MgetSResp());
  m_SdataCl[GrantedMaster]->SputSAddr(m_MdataCl[AddressedSlave]->MgetSAddr());
  m_SdataCl[GrantedMaster]->SputSCmd(m_MdataCl[AddressedSlave]->MgetSCmd());

#ifdef DEBUG_JOE
  cout << "Forward response: SAddr: " << m_MdataCl[AddressedSlave]->MgetSAddr()
       << " MDataPtr: " << data
       << " <" << name() << ">" << endl;
#endif

  // Add additional fields, if needed.
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Bus::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataClSport, class TdataClMport>
void OCP_TL2_Bus<TdataClSport, TdataClMport>::end_of_elaboration()
{
  //
  sc_module::end_of_elaboration();

  int i;

  // We have m_NumSlaves Master ports and m_NumMasters Slave ports at the bus
  m_NumSlaves  = MasterP.size();
  m_NumMasters = SlaveP.size();

  // For each channel attached to a Slave port
  for (i = 0; i < m_NumMasters; i++) {
    // Initialize debug interface
    SlaveP[i]->SregisterDirectIF(this);

    // Get data structure
    m_SdataCl[i] = SlaveP[i]->GetDataCl();

    // Get communication structure
    m_ScommCl[i] = SlaveP[i]->GetCommCl();

    // Get system parameter structure
    m_SparamCl[i] = SlaveP[i]->GetParamCl();
  }

  // For each channel attached to a Master port
  for (i = 0; i < m_NumSlaves; i++) {
    // Initialize debug interface
    MasterP[i]->MregisterDirectIF(this);

    // Get data structure
    m_MdataCl[i] = MasterP[i]->GetDataCl();

    // Get communication structure
    m_McommCl[i] = MasterP[i]->GetCommCl();

    // Get system parameter structure
    m_MparamCl[i] = MasterP[i]->GetParamCl();
  }
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Bus::Initialize()
//
//  This method is activated at the beginning of the simulation.
//  The main purpose is to fetch the Master and Slave parameters
//  from the channel.
//  It assumes that Master and Slave have reported their parameters
//  to the channel in the end_of_elaboration() method.
// ----------------------------------------------------------------------------
template<class TdataClSport, class TdataClMport>
void OCP_TL2_Bus<TdataClSport, TdataClMport>::Initialize()
{
  int i;
  int j;
  int k;
  int it0;
  int it1;
  int it2;
  int ID;
  Ta  adr1;
  Ta  adr2;

  // PARAMETER collection
  // Get Master and Slave parameters from the channel
  //
  // We assume the following Parameter struct members:
  //
  // Master parameter
  // ---------------- 
  // MasterID: Non-negative integer number < TL2_MAX_MASTERS
  // Priority: Non-negative integer number (higher value = higher priority)
  //
  // Slave parameter
  // ---------------
  // SlaveID: Non-negative integer number < TL2_MAX_SLAVES
  // StartAddress: Start address of the memory
  // EndAddress:   End address of the memory
  // Not that for simplicity of code we do not support multiple address
  // regions per slave and address remapping (from boot to normal mode)
  // It is easy to include this as well.

  // Get parameters from the connected Masters
  for (i = 0; i < m_NumMasters; i++) {
    ID = m_SparamCl[i]->MasterID;

    if ((ID < 0) || (ID >= TL2_MAX_MASTERS)) {
      cerr << "ERROR: Master id " << ID << " is out of range. <"
           << name() << ">" << endl;
      sc_stop();
      return;
    }

    for (j = 0; j < i; j++) {
      if (m_MasterID[j] == ID) {
        cerr << "ERROR: Master id " << ID << " is already in use. <"
             << name() << ">" << endl;
        sc_stop();
        return;
      }
    }

    //
    m_MasterID[i] = ID;
    m_Sport[ID]   = i;

    // initialize Master priorities + default grant for the Arbiter
    m_ArbiterReg[i] = m_SparamCl[i]->Priority;

    // if (m_SparamCl[i]->DefaultGrant == true) not supported
    if (0) {
      if (m_ArbiterReg[TL2_DF_GRANT_ID] != -1) {
        cerr << "ERROR: At least 2 Masters ( " << ID << " and "
             << m_ArbiterReg[TL2_DF_GRANT_ID] << " ) have the default grant <"
             << name() << ">" << endl;
        sc_stop();
        return;
      } else {
        m_ArbiterReg[TL2_DF_GRANT_ID] = i;
      }
    }
  } // end of Master parameter collection

  // DISABLE DefaultGrant
  m_ArbiterReg[TL2_DF_GRANT_ID] = TL2_DUMMY_MASTER;

  // Get parameters from the connected Slaves
  for (i = 0; i < m_NumSlaves; i++) {
    ID = m_MparamCl[i]->SlaveID;

    if ((ID < 0) || (ID >= TL2_MAX_SLAVES)) {
      cerr << "ERROR: Slave id " << ID << " is out of range. <"
           << name() << ">" << endl;
      sc_stop();
      return;
    }

    for (j = 0; j < i; j++) {
      if (m_SlaveID[j] == ID) {
        cerr << "ERROR: Slave id " << ID << " is already in use. <"
             << name() << ">" << endl;
        sc_stop();
        return;
      }
    }

    //
    m_SlaveID[i] = ID;
    m_Mport[ID]  = i;

    // initialize Memory map for the Address Decoder
    // we suport only one region
    it0 = 1; // m_MparamCl[i]->NumMemRegions;
    if ((it0 < 0) || (it0 > TL2_MAX_REGIONS)) {
      it1 = TL2_MAX_REGIONS;
      cerr << "ERROR: Invalid number of Memory regions ( " << it0
           << " ). The maximum is " << it1 << " <"
           << name() << ">" << endl;
      sc_stop();
      return;
    }

    for (j = 0; j < it0; j++) {
      adr1 = m_MparamCl[i]->StartAddress;
      if ((adr1 & TL2_SADDR_MASK)) {
        cerr << "ERROR: Start adddress " << adr1 << " of Slave " << ID
             << " is not 1K alligned <"
             << name() << ">" << endl;
        sc_stop();
        return;
      }

      adr2 = m_MparamCl[i]->EndAddress;
      if (((adr2 + 1) & TL2_SADDR_MASK)) {
        cerr << "ERROR: End adddress " << adr2 << " of Slave " << ID
             << " is not 1K alligned <"
             << name() << ">" << endl;
        sc_stop();
        return;
      }

      // Check for overlapping memory regions
      for (k = 0; k < m_NumSlaves; k++) {
        if ((adr1 >= m_MemoryMap[k].StartAddr &&
             adr1 <= m_MemoryMap[k].EndAddr) ||
            (adr2 >= m_MemoryMap[k].StartAddr &&
             adr2 <= m_MemoryMap[k].EndAddr)) {
          cerr << "ERROR: Address range overlapping between Slave " << k
               << " and Slave " << ID << " <"
               << name() << ">" << endl;
          sc_stop();
          return;
        }
      }

      m_MemoryMap[i].StartAddr = adr1;
      m_MemoryMap[i].EndAddr   = adr2;
    }
  } // end of Slave parameter collection

// generate memory cache to speedup address decoder
#ifdef TL2_USE_MEM_CACHE

  // find min/max address
  m_FirstStartAddr = m_MemoryMap[0].StartAddr;
  m_LastEndAddr = m_MemoryMap[0].EndAddr;
  for (i = 1; i < m_NumSlaves; i++) {
    if (m_MemoryMap[i].StartAddr < m_FirstStartAddr)
       m_FirstStartAddr = m_MemoryMap[i].StartAddr;
    if (m_MemoryMap[i].EndAddr > m_LastEndAddr)
       m_LastEndAddr = m_MemoryMap[i].EndAddr;
  }

  // allocate cache table
  it0 = ((m_LastEndAddr + 1 - m_FirstStartAddr) >> TL2_ADDR_ALLIGNMENT);

  // allcoate memory
  if (it0 > 0) {
    try {
       m_MemoryMapCache = new Ta[it0];
    }
    catch(std::bad_alloc ex) {
      cerr << "ERROR: Memory allocation for cache table failed "
           << "(disable TL2_USE_MEM_CACHE) <"
           << name() << ">" << endl;
      cerr << "       Size  is: " << it0 << endl;
      cerr << "       Error is: " << ex.what() << endl;
      sc_stop();
    }
  }

#ifdef DEBUG_B1
  cout << "Using memory map cache table of size " << it0 * sizeof(Ta)
       << " bytes, and " << it0 << " entries"
       << " <" << name() << ">" << endl;
#endif

  // initialize cache table
  for (i = 0; i < it0; i++) {
    adr1 = m_FirstStartAddr + (i << TL2_ADDR_ALLIGNMENT);
    it2 = 1; // sanity check if cache table is correctly build
    for (j = 0; j < m_NumSlaves; j++) {
      if ((adr1 >= m_MemoryMap[j].StartAddr) &&
          (adr1 <=  m_MemoryMap[j].EndAddr)) {
        m_MemoryMapCache[i] = j;
        it2 = 0;

#ifdef DEBUG_B1
  cout << "Cache line " << i << " range: " << m_MemoryMap[j].StartAddr
       << " - " << m_MemoryMap[j].EndAddr << " to slave port " << j
       << " <" << name() << ">" << endl;
#endif

        break;
      }
    }

    if (it2) {
      // should never happen
      cerr << "ERROR: Generation of memory map cache table failed "
           << "(disable TL2_USE_MEM_CACHE) <"
           << name() << ">" << endl;
      sc_stop();
      return;
    }
  }

#endif // TL2_USE_MEM_CACHE

  m_IsInitialized = true;
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Bus::MPutDirect()
//
//  Debug interface method.
//  Read/Write data, Master to Slave direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataClSport, class TdataClMport>
bool OCP_TL2_Bus<TdataClSport, TdataClMport>::MputDirect(int MasterID,
  bool IsWrite, Td* Data, Ta Address, int NumWords)
{
  int AddressedSlave;
  int GrantedMaster;

  // This check is necessary for the case that a master
  // calls the direct method before the bus has been
  // scheduled for the first time
  if (!m_IsInitialized) Initialize();

  //
  if (MasterID >= 0 && MasterID < TL2_MAX_MASTERS)
    GrantedMaster = m_Mport[MasterID];
  else return (false);

  //
  if (GrantedMaster > -1)
    AddressedSlave = AddressDecoding(Address);
  else return (false);

  // call the direct method from the Slave
  if (AddressedSlave < TL2_MAX_SLAVES)
    return (MasterP[AddressedSlave]->MputDirect(m_BusID,IsWrite,Data,Address,
            NumWords));
  else return (false);
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL2_Bus::SPutDirect()
//
//  Debug interface method.
//  Read/Write data, Slave to Master direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataClSport, class TdataClMport>
bool OCP_TL2_Bus<TdataClSport, TdataClMport>::SputDirect(int SlaveID,
  bool IsWrite, Td* Data, Ta Address, int NumWords)
{
  // Not implemented, is a bit more tricky because we do not
  // have a general scheme to address a Master

  //if (SlaveID >= 0 && SlaveID < TL2_MAX_SLAVESS)
  //  GrantedSlave = m_Sport[SlaveID];
  //else return (false);

  //if (GrantedSlave > -1)
  //  AddressedMaster = ???;
  //else return (false);

  // call the direct method from the Master
  //return (SlaveP[AddressedMaster]->SputDirect(m_BusID,IsWrite,Data,Address,
  //        NumWords));

  return (0);
}

// ----------------------------------------------------
//  Method : PrintTime()
//
//  Print time stamp
// ----------------------------------------------------
template<class TdataClSport, class TdataClMport>
void OCP_TL2_Bus<TdataClSport, TdataClMport>::PrintTime(const char* Pstring,
  int ID, Ta Addr)
{
  cout << Pstring << ID << " : Address = " << Addr << " at "
       << sc_time_stamp().to_double()
       << endl;
}

// ----------------------------------------------------------------------------
//
//  Instantiation of the Bus
//
// ----------------------------------------------------------------------------

#include "ocp_tl2_data_cl.h"
template class OCP_TL2_Bus< OCP_TL2_TEMPL_DATA_CL,OCP_TL2_TEMPL_DATA_CL >;
