/*****************************************************************************

  The following code is derived, directly or indirectly, from the SystemC
  source code Copyright (c) 1996-2003 by all Contributors.
  All Rights reserved.

  The contents of this file are subject to the restrictions and limitations
  set forth in the SystemC Open Source License Version 2.4 (the "License");
  You may not use this file except in compliance with such restrictions and
  limitations. You may obtain instructions on how to receive a copy of the
  License at http://www.systemc.org/. Software distributed by Contributors
  under the License is distributed on an "AS IS" basis, WITHOUT WARRANTY OF
  ANY KIND, either express or implied. See the License for the specific
  language governing rights and limitations under the License.

 *****************************************************************************/

/*****************************************************************************

  MODIFICATION LOG - modifiers, enter your name, affiliation, date and
  changes you are making here.

      Name, Affiliation, Date:
  Description of Modification:

 *****************************************************************************/

// ============================================================================
//      Project : Generic SystemC TL Communication Methodology
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//         Date : 11/15/2002
//
//  Description : Transaction Level - Layer-3 example Slave
//    This Slave stores/fetches ATM cells into/from a ring buffer
//    Features: 
//      - Blocking methods are used
//      - Non-pipelined 
//
// Parameter    :
//   Template arguments.
//     - TdataCl: Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//     - Td: Data type
//     - Ta: Address type
//                
//   Constructor arguments.
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Slave.
//           ID must be unique among all Slaves attached to the same Bus.
//
// ============================================================================

#include "tl3_slave.h"

// ----------------------------------------------------------------------------
// Process : TL3_Slave::TL3_Slave
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl> TL3_Slave<TdataCl>::TL3_Slave
       (
          sc_module_name name_
        , int ID
        , bool WriteResponse
        )
          : sc_module          (name_)
          , SlaveP             ("SlavePort")
          , m_ID               (ID)
          , m_WriteResponse     (WriteResponse)
          , m_TaNum            (0)
{
  int i;

  // InitParameters();

  // allcoate memory
  if (TL3_MAX_ATM_CELLS > 0)
  {
    try
    {
      m_SlaveMemory = new Td[TL3_MAX_ATM_CELLS];
    }
    catch(std::bad_alloc ex)
    {
      cerr << "ERROR: Memory allocation for Slave memory failed "
            << name() << ">" << endl;
      cerr << "       Size  is: " << TL3_MAX_ATM_CELLS << endl;
      cerr << "       Error is: " << ex.what() << endl;
      sc_stop();
      return;
    }
  }

  Td B;

  for (i = 0; i < TL3_MAX_ATM_CELLS; i++)
    m_SlaveMemory[i] = B;

  SC_THREAD(Slave);
}


// ----------------------------------------------------------------------------
// Process : TL3_Slave::~TL3_Slave
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> TL3_Slave<TdataCl>::~TL3_Slave()
{
  delete [] m_SlaveMemory;
}


// ----------------------------------------------------------------------------
//  Method : TL3_Slave::Slave()
//
//  Top level method of the Slave (non-pipelined case)
//
// ----------------------------------------------------------------------------
template<class TdataCl> void TL3_Slave<TdataCl>::Slave()
{
  Td *Mdata;
  bool Write;
  int Sresp;
  int NumCells;
  Ta Rindex = 0;
  Ta Windex = 0;

  Mdata = new Td[TL3_MAX_ATM_CELLS];

  while(true)
  {
    // Get Master/Bus request; use blocking methods
    if ((SlaveP->SgetRequestBlocking(true))) // release immediately
    {
      NumCells  = m_DataCl->SgetNumCells();
      // Mdata  = m_DataCl->SgetData(); // pointer passing
      m_DataCl->SgetData(Mdata);        // explicit data copying
      Write     = SlaveP->IsWrite();

      // Put data into the memory (write) or into the Data field (read).
      // The direct method call is just a reuse of an existing method.
      // This has nothing to do with the debug interface.
      Sresp = (MputDirect(1, Write, Mdata, Write?Windex:Rindex, NumCells))
            ? TL_SRESP_OK : TL_SRESP_ERROR;

      m_DataCl->SputStatus(Sresp);

      if (Write)
      {
        // write request
 
#ifdef DEBUG_S
   PrintTrans(m_TaNum, Windex, NumCells, " Blocking write request ");
#endif
#ifdef DEBUG_G1
  PrintTime("Accept write request in Slave ", m_ID, Windex);
#endif

        Windex += NumCells;
        while (Windex >= TL3_MAX_ATM_CELLS) Windex -= TL3_MAX_ATM_CELLS;

        if (m_WriteResponse)
        {
          // Set the write response field
          m_DataCl->SetWriteResponse();
          // Set number of response data
          m_DataCl->SputNumCells(0);

#ifdef DEBUG_G1
  PrintTime("Start write response in Slave ", m_ID, Windex);
#endif
 
          // Send response to Master/Bus
          if (!SlaveP->SputResponseBlocking())
          {
            // Deep ship: Should never happen
            cout << "Error: Master (" << 1 << ") dropped Slave ("
                 << m_ID << ") response <" << name() << ">" << endl;
          }
        }
      } // end of write branch
      else
      {
        // read request

#ifdef DEBUG_G1
  PrintTime("Accept read request in Slave ", m_ID, Rindex);
#endif
#ifdef DEBUG_S
   PrintTrans(m_TaNum, Rindex, NumCells, " Blocking read request ");
#endif

        Rindex += NumCells;
        while (Rindex >= TL3_MAX_ATM_CELLS) Rindex -= TL3_MAX_ATM_CELLS;

        // Response phase

        // Set the read response field
        m_DataCl->SetReadResponse();
        m_DataCl->SputNumCells(NumCells);
        // Set data field
        m_DataCl->SputData(Mdata); // pointer passing

#ifdef DEBUG_G1
  PrintTime("Start read response in Slave ", m_ID, Rindex);
#endif

        // Send response to Master
        if (!SlaveP->SputResponseBlocking())
        {
          // Deep ship: Should never happen
          cout << "Error: Master (" << 1 << ") dropped Slave ("
               << m_ID << ") response <" << name() << ">" << endl;
        }
      } // end of read branch

      // this is the stop condition
      if (++m_TaNum >= TL3_NUM_TA)
      {
        delete [] Mdata;
        sc_stop();
        return;
      }
    } // end of SgetRequest() bransch
  } // end of while(true) loop
} // end of Slave method


// ----------------------------------------------------------------------------
//  Method : TL3_Slave::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl> void TL3_Slave<TdataCl>::end_of_elaboration()
{
  sc_module::end_of_elaboration();

  // Initialize debug interface
  SlaveP->SregisterDirectIF(this);

  // Get data structure
  m_DataCl = SlaveP->GetDataCl();

  // Get communication structure
  m_CommCl = SlaveP->GetCommCl();

  // Get system parameter structure
  m_ParamCl = SlaveP->GetParamCl();

  // Put parameter into Channel
  // No parameters needed
}


// ----------------------------------------------------------------------------
//  Method : TL3_Slave::MPutDirect()
//
//  Debug interface method.
//  Read/Write data, Master to Slave direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl> bool TL3_Slave<TdataCl>::MputDirect(
         int MasterID, bool IsWrite, Td *Data, Ta Index, int NumCells)
{
  int i;

  if (IsWrite)
  {
    // write request
    for (i = 0; i < NumCells; i++)
    {
      m_SlaveMemory[Index] = Data[i];
      if (++Index >= TL3_MAX_ATM_CELLS) Index = 0;
    }
  }
  else
  {
    // read request
    for (i = 0; i < NumCells; i++)
    {
      Data[i] = m_SlaveMemory[Index];
      if (++Index >= TL3_MAX_ATM_CELLS) Index = 0;
    }
  }
  return(true);
}

// ----------------------------------------------------------------------------
//  Method : TL3_Slave::PrintTrans()
//
//  Print transaction
//
// ----------------------------------------------------------------------------
template<class TdataCl> void TL3_Slave<TdataCl>::PrintTrans(
         int TaNum, Ta Index, int NumCells, const char *Method)

{
  cout << "Slave " << m_ID << " : " << "Req = " << TaNum
           << "  Index = " << Index 
	   << "  NumCells = " << NumCells << Method << endl;
}

// ----------------------------------------------------
//  Method : PrintTime()
//
//  Print time stamp
// ----------------------------------------------------
template<class TdataCl> void TL3_Slave<TdataCl>::PrintTime(
         const char *Pstring, int ID, Ta Index)
{
  cout << Pstring << ID << " : Index = " << Index << endl;
}


// ----------------------------------------------------------------------------
//
//  Instantiation of the Slave
//
// ----------------------------------------------------------------------------

#include "tl3_data_cl.h"
template class TL3_Slave<TL3_TEMPL_DATA_CL >; // defined in tl3_globals.h
