///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2004 OCP-IP
// OCP-IP Confidential & Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : Tim Kogel, CoWare, Inc.
//          Date: 11/26/2004
//
//  Description :  OCP Channel Transaction Recording Monitor
//	  This system monitor is based on the transaction recording 
//	  API in the SystemC Verification (SCV) Library. It targets 
//	  performance analysis for architectural modeling.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _OCP_PERF_MONITOR_SYSTEM_SCV_H
#define _OCP_PERF_MONITOR_SYSTEM_SCV_H

#include <map>
#include "ocp_tl2_monitor_observer_if.h"
#include "scv.h"


template <class, class> class OCP_TL2_MonitorIF;


template <class Tdata, class Taddr>
class OCP_Perf_Monitor_System_SCV: 
  public OCP_TL2_Monitor_ObserverIF<Tdata,Taddr>
{
public:
  typedef typename OCP_TL2_Monitor_ObserverIF<Tdata,Taddr>::tl2_peek_type tl2_peek_type;

  OCP_Perf_Monitor_System_SCV(bool enable_realtions = false);
  virtual ~OCP_Perf_Monitor_System_SCV();

  virtual void registerChannel	(tl2_peek_type *channel,
				 bool master_is_node,
				 bool slave_is_node);

  virtual void start_of_simulation();

  virtual void NotifyRequestStart (tl2_peek_type *);
  virtual void NotifyRequestEnd   (tl2_peek_type *);
  virtual void NotifyResponseStart(tl2_peek_type *);
  virtual void NotifyResponseEnd  (tl2_peek_type *);

  virtual void cancelTransaction(long long tr_handle);

  virtual void print_handle_map();

protected:
  struct ChannelStruct{
    string				m_channel_name;
    string				m_master_name;
    string				m_slave_name;
    bool				m_master_is_node;
    bool				m_slave_is_node;
    scv_tr_relation_handle_t		m_channel_relation;
    scv_tr_stream*			m_initiator_stream;
    scv_tr_generator<unsigned int>*	m_initiator_read_gen;
    scv_tr_generator<unsigned int>*	m_initiator_write_gen;
    scv_tr_stream*			m_req_stream;
    scv_tr_generator<unsigned int>*	m_req_read_gen;
    scv_tr_generator<unsigned int>*	m_req_write_gen;
    scv_tr_stream*			m_resp_stream;
    scv_tr_generator<unsigned int>*	m_resp_gen;
    scv_tr_handle			m_req_handle;
    scv_tr_handle			m_resp_handle;
    
    ChannelStruct() : 
      m_initiator_stream(NULL),
      m_initiator_read_gen(NULL),
      m_initiator_write_gen(NULL),
      m_req_stream(NULL),
      m_req_read_gen(NULL),
      m_req_write_gen(NULL),
      m_resp_stream(NULL),
      m_resp_gen(NULL)
    {}
  };

  typedef map<uint64,scv_tr_handle>		handle_map_t;
  typedef map<tl2_peek_type *,ChannelStruct*>	channel_map_t;
  channel_map_t					m_channel_map;
  typename channel_map_t::iterator		m_channel_it;
  handle_map_t					m_handle_map;
  typename handle_map_t::iterator		m_handle_it;
  unsigned int					m_count;
  scv_tr_handle					m_initiator_handle;
  scv_tr_relation_handle_t			m_relation; 
  scv_tr_relation_handle_t*			m_relation_p; // not used
  const bool					m_enable_relations;
  bool						m_enabled;
};


template <class Tdata, class Taddr>
OCP_Perf_Monitor_System_SCV<Tdata,Taddr>::OCP_Perf_Monitor_System_SCV(bool enable_realtions) :
  m_enable_relations(enable_realtions),
  m_enabled(false)
{
  m_count=0;
  if (m_enable_relations) {
    m_relation = scv_tr_db::get_default_db()->create_relation("successor");
    m_relation_p = new scv_tr_relation_handle_t;
  }
}

template <class Tdata, class Taddr>
OCP_Perf_Monitor_System_SCV<Tdata,Taddr>::~OCP_Perf_Monitor_System_SCV()
{
  // delete stuff
  if (m_enable_relations) {
    delete m_relation_p;
  }
  for (m_channel_it = m_channel_map.begin();
       m_channel_it != m_channel_map.end(); 
       m_channel_it++) {
    ChannelStruct* tmp = m_channel_it->second;
    if (m_enabled && m_enable_relations) {
      delete tmp->m_req_stream;
      delete tmp->m_req_read_gen;
      delete tmp->m_req_write_gen;
      delete tmp->m_resp_stream;
      delete tmp->m_resp_gen;
    }
    if (m_enabled && !tmp->m_master_is_node) {
      delete tmp->m_initiator_stream;
      delete tmp->m_initiator_write_gen;
      delete tmp->m_initiator_read_gen;
    }
    delete tmp;
  }
  m_channel_map.erase(m_channel_map.begin(),m_channel_map.end());
}
  
template <class Tdata, class Taddr>
void
OCP_Perf_Monitor_System_SCV<Tdata,Taddr>::registerChannel(tl2_peek_type *tl2_channel, bool master_is_node, bool slave_is_node)
{
  ChannelStruct* tmp = new ChannelStruct();
  string full_master_name;
  string full_slave_name;
  full_master_name = string("sys.") + tl2_channel->peekMasterPortName();
  full_slave_name = string("sys.") + tl2_channel->peekSlavePortName();
  tmp->m_channel_name = tl2_channel->peekChannelName();
  tmp->m_master_name = full_master_name;
  tmp->m_slave_name = full_slave_name;
  tmp->m_master_is_node = master_is_node;
  tmp->m_slave_is_node = slave_is_node;

  m_channel_map[tl2_channel] = tmp;
  m_count++;
}

template <class Tdata, class Taddr>
void
OCP_Perf_Monitor_System_SCV<Tdata,Taddr>::start_of_simulation()
{
  if (m_enabled) return;
  for (m_channel_it = m_channel_map.begin();
       m_channel_it != m_channel_map.end(); 
       m_channel_it++) {
    ChannelStruct* tmp = m_channel_it->second;
    if (m_enable_relations) {
      string req_stream_name = tmp->m_channel_name + string("_req");
      string resp_stream_name= tmp->m_channel_name + string("_resp");
      tmp->m_req_stream = new scv_tr_stream(req_stream_name.c_str(),"request");
      tmp->m_req_read_gen = new scv_tr_generator<unsigned int>("read",*tmp->m_req_stream,"addr");
      tmp->m_req_write_gen = new scv_tr_generator<unsigned int>("write",*tmp->m_req_stream,"addr");
      tmp->m_resp_stream = new scv_tr_stream(resp_stream_name.c_str(),"response");
      tmp->m_resp_gen = new scv_tr_generator<unsigned int>("read",*tmp->m_resp_stream,"addr");
    }

    if (!tmp->m_master_is_node) {
      tmp->m_initiator_stream = new scv_tr_stream(tmp->m_master_name.c_str(),"initiator");
      tmp->m_initiator_write_gen = new scv_tr_generator<unsigned int>("write",*tmp->m_initiator_stream,"addr");
      tmp->m_initiator_read_gen = new scv_tr_generator<unsigned int>("read",*tmp->m_initiator_stream,"addr");
    }
  }
  m_enabled = true;
}

template <class Tdata, class Taddr>
void
OCP_Perf_Monitor_System_SCV<Tdata,Taddr>::NotifyRequestStart (tl2_peek_type *tl2_channel)
{
  m_channel_it = m_channel_map.find(tl2_channel);

#ifndef NDEBUG
  if ( m_channel_it == m_channel_map.end() ) {
    cerr << sc_simulation_time() << ", ERROR NotifyRequestStart: channel " << tl2_channel->peekChannelName() << " not registered\n";
    return;
  }
#endif

  ChannelStruct* channel_struct = m_channel_it->second;
  const OCPTL2RequestGrp<Tdata,Taddr>& req = tl2_channel->peekOCPRequest();

  bool is_read= (req.MCmd==OCP_MCMD_RD)||(req.MCmd==OCP_MCMD_RDEX)||(req.MCmd==OCP_MCMD_RDL);

  if (channel_struct->m_master_is_node) {
    if (m_enable_relations) {
      // continue transaction from node
#ifndef NDEBUG
      if ( req.TrHandle < 0 ) {
	cerr << sc_simulation_time() << ", ERROR: OCP_Perf_Monitor_System_SCV::NotifyRequestStart, channel " << channel_struct->m_channel_name 
	     << ", invalid transaction handle\n";
	return;
      }
#endif
      m_handle_it = m_handle_map.find(req.TrHandle);
#ifndef NDEBUG
      if ( m_handle_it == m_handle_map.end() ) {
	cerr << sc_simulation_time() << ", ERROR: OCP_Perf_Monitor_System_SCV::NotifyRequestStart, channel " << channel_struct->m_channel_name   
	     << ", handle " << req.TrHandle << " not found\n"; 
	return;
      }
#endif
      m_initiator_handle = m_handle_it->second;
    }
  }
  else {
    // start new transaction from initiator
    if (is_read) {
      m_initiator_handle = channel_struct->m_initiator_read_gen->begin_transaction(req.MAddr);
    } else {
      m_initiator_handle = channel_struct->m_initiator_write_gen->begin_transaction(req.MAddr);
    }
    m_handle_map[m_initiator_handle.get_id()] = m_initiator_handle;
    // cout << sc_simulation_time() << ", startTransaction(req) from initiator " << channel_struct->m_master_name << ", handle = " <<  m_initiator_handle.get_id() << endl;
    req.TrHandle = m_initiator_handle.get_id();
    // print_handle_map();
  }

  if (m_enable_relations) {
#ifndef NDEBUG
    if ( channel_struct->m_req_handle.is_active()  ) {
      cerr << sc_simulation_time() << ", ERROR:  OCP_Perf_Monitor_System_SCV::NotifyRequestStart, channel " << channel_struct->m_channel_name 
	   << ", m_req_handle " << req.TrHandle << " is already active\n";
      return;
    }
#endif
    if (is_read) {
      channel_struct->m_req_handle = channel_struct->m_req_read_gen->begin_transaction( req.MAddr, m_relation, m_initiator_handle);
    } else {
      channel_struct->m_req_handle = channel_struct->m_req_write_gen->begin_transaction( req.MAddr,m_relation, m_initiator_handle);
    }
    // cout << sc_simulation_time() << ", startTransaction(req) realted from master " << channel_struct->m_master_name << ", initiator handle " << channel_struct->m_req_handle.get_immediate_related_transaction(m_relation_p)->get_id()  << ", new handle = " << channel_struct->m_req_handle.get_id() << endl;
  }
}

template <class Tdata, class Taddr>
void
OCP_Perf_Monitor_System_SCV<Tdata,Taddr>::NotifyRequestEnd(tl2_peek_type *tl2_channel)
{
  m_channel_it = m_channel_map.find(tl2_channel);
#ifndef NDEBUG
  if ( m_channel_it == m_channel_map.end()) {
    cerr << sc_simulation_time() << ", ERROR: OCP_Perf_Monitor_System_SCV::NotifyRequestEnd"  
	 << ", channel " << tl2_channel->peekChannelName() << " not registered\n"; 
    return;
  }
#endif
  ChannelStruct* channel_struct = m_channel_it->second;
  const OCPTL2RequestGrp<Tdata,Taddr>& req = tl2_channel->peekOCPRequest();

  if (m_enable_relations) {
#ifndef NDEBUG
    if ( !channel_struct->m_req_handle.is_active()  ) {
      cerr << sc_simulation_time() << ", ERROR: OCP_Perf_Monitor_System_SCV::NotifyRequestEnd, channel " 
	   << channel_struct->m_channel_name << ", m_req_handle NOT active\n";
      return;
    }
#endif
    // end transaction and delete handle
    // cout << sc_simulation_time() << ", channel " << channel_struct->m_channel_name << ", cmd " << req.MCmd << ", end handle " <<  channel_struct->m_req_handle.get_id() << endl;
    channel_struct->m_req_handle.end_transaction();
  }
  
  if ( !channel_struct->m_slave_is_node && (req.MCmd == OCP_MCMD_WR || req.MCmd == OCP_MCMD_BCST) )  {
    // this is a posted write or broadcast transaction, which has reached its target, so the initiator transaction ends here

#ifndef NDEBUG
    if ( req.TrHandle < 0 ) {
      cerr << sc_simulation_time() << ", ERROR: OCP_Perf_Monitor_System_SCV::NotifyRequestEnd, channel " 
	   << channel_struct->m_channel_name <<", invalid transaction handle\n";
      return;
    }
#endif

    m_handle_it = m_handle_map.find(req.TrHandle);

#ifndef NDEBUG
    if ( m_handle_it == m_handle_map.end() ) {
      cerr << sc_simulation_time() << ", ERROR OCP_Perf_Monitor_System_SCV::NotifyRequestEnd, channel "
	   << channel_struct->m_channel_name <<", initiator handle " << req.TrHandle << " not found\n"; 
      return;
    }
#endif

    m_initiator_handle = m_handle_it->second;
    //cout << sc_simulation_time() << ", channel " << channel_struct->m_channel_name << ", cmd " << req.MCmd << ", end & delete initiator handle " << m_initiator_handle.get_id() << endl;
    m_initiator_handle.end_transaction();
    m_handle_map.erase(m_handle_it);
    // print_handle_map();
  }
}

template <class Tdata, class Taddr>
void
OCP_Perf_Monitor_System_SCV<Tdata,Taddr>::NotifyResponseStart(tl2_peek_type *tl2_channel)
{
  if (!m_enable_relations) {
    return;
  }

  m_channel_it = m_channel_map.find(tl2_channel);

#ifndef NDEBUG
  if ( m_channel_it == m_channel_map.end() ) {
    cerr << sc_simulation_time() << ", ERROR: OCP_Perf_Monitor_System_SCV::NotifyResponseStart, channel " 
	 << tl2_channel->peekChannelName() << " not registered\n"; 
    return;
  }
#endif

  ChannelStruct* channel_struct = m_channel_it->second;
  const OCPTL2ResponseGrp<Tdata>& resp = tl2_channel->peekOCPResponse();

#ifndef NDEBUG
  if ( channel_struct->m_resp_handle.is_active()  ) {
    cerr << sc_simulation_time() << ", ERROR: OCP_Perf_Monitor_System_SCV::NotifyResponseStart, channel " 
	 << channel_struct->m_channel_name << ", m_resp_handle is active\n";
    return;
  }
  if (resp.TrHandle < 0) {
    cerr << sc_simulation_time() << ", ERROR: OCP_Perf_Monitor_System_SCV::NotifyResponseStart, channel"   
	 << channel_struct->m_channel_name << ", invalid transaction handle\n";
    return;
  }
#endif

  m_handle_it = m_handle_map.find(resp.TrHandle);

#ifndef NDEBUG
  if ( m_handle_it == m_handle_map.end() ) {
    cerr << sc_simulation_time() << ", ERROR: OCP_Perf_Monitor_System_SCV::NotifyResponseStart, channel " 
	 << channel_struct->m_channel_name << ", handle " << resp.TrHandle << " not found\n"; 
    return;
  }
#endif

  m_initiator_handle = m_handle_it->second;

  channel_struct->m_resp_handle = channel_struct->m_resp_gen->begin_transaction( resp.SRespInfo, m_relation, m_initiator_handle);
  
  // cout << sc_simulation_time() << ", startTransaction(resp) from slave " << channel_struct->m_slave_name << ", old handle = " << channel_struct->m_resp_handle.get_immediate_related_transaction(m_relation_p)->get_id() << ", new handle = " <<   channel_struct->m_resp_handle.get_id() << endl;
}

template <class Tdata, class Taddr>
void
OCP_Perf_Monitor_System_SCV<Tdata,Taddr>::NotifyResponseEnd(tl2_peek_type *tl2_channel)
{
  m_channel_it = m_channel_map.find(tl2_channel);
#ifndef NDEBUG
  if ( m_channel_it == m_channel_map.end() ) {
    cerr << sc_simulation_time() << ", ERROR: OCP_Perf_Monitor_System_SCV::NotifyResponseEnd, channel " 
	 << tl2_channel->peekChannelName() << " not registered\n"; 
    return;
  }
#endif

  ChannelStruct* channel_struct = m_channel_it->second;
  const OCPTL2ResponseGrp<Tdata>& resp = tl2_channel->peekOCPResponse();

  if (m_enable_relations) {
#ifndef NDEBUG
    if ( !channel_struct->m_resp_handle.is_active()  ) {
      cerr << sc_simulation_time() << ", ERROR: OCP_Perf_Monitor_System_SCV::NotifyResponseEnd, channel " 
	   << channel_struct->m_channel_name << ", m_resp_handle NOT active\n";
      return;
    }
#endif
    // end response transaction
    // cout << sc_simulation_time() << ", channel " << channel_struct->m_channel_name << ",  resp end " << channel_struct->m_resp_handle.get_id()  << endl;
    channel_struct->m_resp_handle.end_transaction();
  }

  if ( !channel_struct->m_master_is_node )  {
    // finally delete the initiator handle when the transaction reaches the master

#ifndef NDEBUG
    if (resp.TrHandle < 0) {
      cerr << sc_simulation_time() << ", ERROR: OCP_Perf_Monitor_System_SCV::NotifyResponseEnd, channel "
	   << channel_struct->m_channel_name << ", invalid transaction handle\n";
      return;
    }
#endif

    m_handle_it = m_handle_map.find(resp.TrHandle);

#ifndef NDEBUG
    if ( m_handle_it == m_handle_map.end() ) {
      cerr << sc_simulation_time() << ", ERROR OCP_Perf_Monitor_System_SCV::NotifyResponseEnd, channel " 
	   << channel_struct->m_channel_name << ", handle " << resp.TrHandle << " not found\n"; 
      return;
    }
#endif
    m_initiator_handle = m_handle_it->second;

    // end transaction and delete handle
    //cout << sc_simulation_time() << ", channel " << channel_struct->m_channel_name << ", resp end & delete initiator handle " << m_initiator_handle.get_id() << endl;
    m_initiator_handle.end_transaction();
    m_handle_map.erase(m_handle_it);
    // print_handle_map();
  }
}

template <class Tdata, class Taddr>
void
OCP_Perf_Monitor_System_SCV<Tdata,Taddr>::cancelTransaction(long long tr_handle)
{
#ifndef NDEBUG
  if ( tr_handle < 0 ) {
    cerr << sc_simulation_time() << ", ERROR: OCP_Perf_Monitor_System_SCV::cancelTransaction"   
	 << ", invalid transaction handle\n";
    return;
  }
#endif

  m_handle_it = m_handle_map.find(tr_handle);

#ifndef NDEBUG
  if ( m_handle_it == m_handle_map.end() ) {
    cerr << sc_simulation_time() << ", ERROR: OCP_Perf_Monitor_System_SCV::cancelTransaction"  
	 << ", handle " << tr_handle << " not found\n"; 
    return;
  }
#endif

  m_initiator_handle = m_handle_it->second;
  // << sc_simulation_time() << ", cancel, end & delete initiator handle " << m_initiator_handle.get_id() << endl;
  m_initiator_handle.end_transaction();
  m_handle_map.erase(m_handle_it);
  // print_handle_map();
}

template <class Tdata, class Taddr>
void
OCP_Perf_Monitor_System_SCV<Tdata,Taddr>::print_handle_map()
{
  cout << "OCP_Perf_Monitor_System_SCV::print_handle_map, @" << sc_time_stamp() << endl;
  for (m_handle_it = m_handle_map.begin(); 
       m_handle_it != m_handle_map.end(); 
       m_handle_it++) {
    cout << " | " << m_handle_it->first << " - " << m_handle_it->second.get_id() ;
  }
  cout << endl;
}

#endif
