///////////////////////////
//
//  (c) Copyright OCP-IP 2003
//  OCP-IP Confidential and Proprietary
//
// Simple Main to read in Map data from files
// and then use that to configure and connect
// a master and slave.
//
///////////////////////////

#define NDEBUG

#include <map>
#include <set>
#include <string>
#include <algorithm>
#include <stdio.h>
#include <stdlib.h>
#include <iostream>

#include "systemc.h"

#include "channel_types.h"
#include "MasterSysC.h"
#include "SlaveSysC.h"
#include "ocp_tl1_data_cl.h"
#include "ocp_tl_param_cl.h"
#include "ocp_tl1_channel.h"

#define OCP_CLOCK_PERIOD        1
#define OCP_CLOCK_TIME_UNIT     SC_NS

#define MASTER_CLOCK_PERIOD     1
#define MASTER_CLOCK_TIME_UNIT  SC_NS

#define SLAVE_CLOCK_PERIOD      1
#define SLAVE_CLOCK_TIME_UNIT   SC_NS


void process_command_line(int   argc,
                          char* argv[],
                          string& ocp_params_file_name,
                          string& master_params_file_name,
                          string& slave_params_file_name,
                          double& simulation_end_time,
                          bool& debug_dump,
                          string& debug_file_name)
{
    // get the ocp parameters file name
    ocp_params_file_name = "";
    if (argc > 1) {
        string file_name(argv[1]);
        ocp_params_file_name = file_name;
    }

    // get the master parameters file name
    master_params_file_name = "";
    if (argc > 2) {
        string file_name(argv[2]);
        master_params_file_name = file_name;
    }
    // get the slave parameters file name
    slave_params_file_name = "";
    if (argc > 3) {
        string file_name(argv[3]);
        slave_params_file_name = file_name;
    }

    // get the simulation end time
    simulation_end_time = 1000;
    if (argc > 4) {
        simulation_end_time = (double) atol(argv[4]);
    }

    // do we dump out a log file?
    debug_dump= false;
    debug_file_name = "";
    if (argc > 5) {
        string file_name(argv[5]);
        debug_file_name = file_name;
        debug_dump = true;
    }
}

void readMapFromFile(const string &myFileName, MapStringType &myParamMap) 
{
    // read pairs of data from the passed file
    string leftside;
    string rightside;
    
    // (1) open the file
    ifstream inputfile(myFileName.c_str());
    assert( inputfile );

    // set the formatting
    inputfile.setf(std::ios::skipws);

    // Now read through all the pairs of values and add them to the passed map
    while ( inputfile ) {
        inputfile >> leftside;
        inputfile >> rightside;
        myParamMap.insert(std::make_pair(leftside,rightside));
    }

    // All done, close up
    inputfile.close();
}

int
sc_main(int argc, char* argv[])
{
    OCP_TL1_Channel< OCP_TL1_DataCl<OCPCHANNELBit32, OCPCHANNELBit32> >* pOCP;
    MasterSysC< OCP_TL1_DataCl<OCPCHANNELBit32, OCPCHANNELBit32> >*          pMaster;
    SlaveSysC< OCP_TL1_DataCl<OCPCHANNELBit32, OCPCHANNELBit32> >*           pSlave;
    MapStringType                                                ocpParamMap;
    MapStringType                                                masterParamMap;
    MapStringType                                                slaveParamMap;

    double         simulation_end_time;
    bool           debug_dump;
    string         ocpParamFileName;
    string         masterParamFileName;
    string         slaveParamFileName;
    string         dump_file_name;
    ofstream       debugFile;

    // --------------------------------
    // (1) process command line options
    //     and read my parameters
    // --------------------------------
    process_command_line(argc,argv,ocpParamFileName,masterParamFileName,
           slaveParamFileName,simulation_end_time,debug_dump,dump_file_name);

    if ( ! ocpParamFileName.empty() ) {
        readMapFromFile(ocpParamFileName, ocpParamMap);
    }

    if ( ! masterParamFileName.empty() ) {
        readMapFromFile(masterParamFileName, masterParamMap);
    }

    if ( ! slaveParamFileName.empty() ) {
        readMapFromFile(slaveParamFileName, slaveParamMap);
    }

    // open a trace file
    if (debug_dump) {
        cout << "Debug dumpfilename: " << dump_file_name << endl;
        debugFile.open(dump_file_name.c_str());
    }

    sc_clock clk("clk", OCP_CLOCK_PERIOD,OCP_CLOCK_TIME_UNIT) ;
    // ----------------------------------------------------------
    // (2) Create the OCP Channel
    // ----------------------------------------------------------

    // Self-timed channel instantiation
    pOCP = new OCP_TL1_Channel< OCP_TL1_DataCl<OCPCHANNELBit32, OCPCHANNELBit32> >
      ("ocp0",true,true,true,NULL,OCP_CLOCK_PERIOD,OCP_CLOCK_TIME_UNIT,"ocp0.ocp", false);

    // Clocked channel instantiation.
    //pOCP = new OCP_TL1_Channel< OCP_TL1_DataCl<OCPCHANNELBit32, OCPCHANNELBit32> >
    //  ((std::string)"ocp0",(sc_clock *)&clk, (std::string)"ocp0.ocp");

    pOCP->setConfiguration(ocpParamMap);

    // ----------------------------------------------------------
    // (3) Create the Master and Slave
    // ----------------------------------------------------------

    pMaster = new MasterSysC< OCP_TL1_DataCl<OCPCHANNELBit32, OCPCHANNELBit32> >("master", MASTER_CLOCK_PERIOD, MASTER_CLOCK_TIME_UNIT, 0, &debugFile );
    pMaster->setConfiguration(masterParamMap);

    pSlave = new SlaveSysC< OCP_TL1_DataCl<OCPCHANNELBit32, OCPCHANNELBit32> >("slave", SLAVE_CLOCK_PERIOD, SLAVE_CLOCK_TIME_UNIT, 0, 0x3FF, &debugFile );
    pSlave->setConfiguration(slaveParamMap);

    // ----------------------------------------------------------
    // (4) connect channel, master, and slave, & clock
    // ----------------------------------------------------------
    pMaster->ipP(*pOCP);
    pSlave->tpP(*pOCP);

    // ------------------------
    // (5) start the simulation
    // ------------------------
    sc_start(simulation_end_time,SC_NS);

    // -------------------
    // (6) post processing
    // -------------------

    cout << "main program finished at " 
         << sc_time_stamp().to_double() << endl;

    sc_simcontext* sc_curr_simcontext = sc_get_curr_simcontext();
    cout << "delta_count: " << dec << sc_curr_simcontext->delta_count() << endl;
    cout << "next_proc_id: " << dec << sc_curr_simcontext->next_proc_id()
         << endl;

    return (0);
}
