// 
//  (c) Copyright OCP-IP 2003, 2004
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//
//  Description : Transaction Level - Layer-0 to Layer-1 Slave Adapter (OCP 2.0)
//               (with Datahandshake support)
//  $Id: ocp2_tl0_tl1_slave_adapter.cpp,v 1.3 2006/01/30 22:08:35 halexan Exp $
//  
//     Features :
//    - Layer adapter used to connect a TL0 OCP Slave to a TL1 system
//
//    - Uses the OCP-specific TL1 Channel and associated API
//
//    - Interface implements OCP Basic signals : Clk, MCmd, MAddr, MData,
//        SCmdAccept, SData, SResp & MRespAccept.
//        + Datahandshake signals: MDataValid, SDataAccept
//
//    - Glitches and propagation delays for input signals are removed using two
//      'Sample delay' parameters.  Signals coming from the TL0 slave are not
//      supposed to be clocked and therefore can change at any time during the
//      clock cycle. It is also possible that some glitches appear on them,
//      particulary in the case of co-simulation with a HDL-simulated RTL
//      Slave. 
//
//      Once a TL1 interface call has been done, it can not be cancelled. The
//        slave adapter issues three kinds of call to the TL1 channel:
//          * startOCPResponse()
//          * putSDataAccept()
//          * putSCmdAccept() 
//
//      The 'startOCPResponse()' and associated API calls must happen only
//        when TL0 associated signals 'SResp' and 'SData' are stable. Since the
//        propagation delays for these signals are slave dependant, a 'Response
//        Sample Delay' parameter is used to determine when these signals
//        should be sampled during the cycle.  Once sampled,'SResp' is tested;
//        if the response is valid (DVA value), a 'startOCPResponse()' call is done.
//
//      The 'putSCmdAccept()' call must happen only when 'SCmdAccept' is stable.
//        Similarly, 'putSDataAccept()' call must wait 'SDataAccept' to be
//        stable. Since the propagation delay for these signals are slave
//        dependant, the same 'AcceptSampleDelay' parameter is used to
//        determine when 'SDataAccept' AND 'SCmdAccept' should be sampled
//        during the cycle.
//
//      The smallest value for the sample delays is sc_get_time_resolution() ;
//      it should be used in the case when TL0 master has pure 'zero-delay'
//      signals. Users should always use a timed sample delay (and not a
//      defined number of delta-cycles) since it is not possible to predict the
//      number of delta-cycles required for signal stability.
//
//      'Check_setup_time()' function is proposed for debugging purposes, to
//        ensure that TL0 slave output signals don't change after the sample
//        delays, which would cause the adapter to fail. A good way to proceed
//        is to perform the first simulation with checking function actived to
//        determine the appropriate sample delay value.  Once TL0 behaviour is
//        correct, checking function should be disabled to increase simulation
//        speed.
//                
// Parameters   : 
//   Template arguments.
//   - TdataCl: Data class containing data members and
//              access methods for the data exchanged between
//              Masters and Slaves
//   - Td: Data type
//   - Ta: Address type
//                
//   First Constructor arguments (with explicit sampling times):
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Slave.
//           ID must be unique among all Slaves attached to the same Bus.
//     - AcceptSampleDelay: Time number specifying the delay between clock 
//           rising edge and the actual sampling time for 'scmdaccept' and
//           'sdataaccept' signals
//     - ResponseSampleDelay: Time number specifying the delay between clock 
//           rising edge and the actual sampling time for response signals
//     - Check_setup_time: boolean value specifying if 'check_setup_time()' 
//                 function is executed during simulation 
//
//   Second Constructor arguments (sampling times use default values):
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//     - Combinational: Boolean specifying if DataHandshake/Response occurs in the
//                 same clock cycle or sequentially. Sampling times will be
//                 set as follows:
//                    - Combinational:
//                          * RequestSampleDelay = ClockCycle*99/100
//                          * ResponseSampleDelay = ClockCycle*60/100
//                    - Sequential:
//                          * RequestSampleDelay = ClockCycle*99/100
//                          * ResponseSampleDelay = ClockCycle*60/100
//     - ClockCycle: sc_time number specifying the clock cycle
//     - Check_setup_time: boolean value specifying if 'check_setup_time()' 
//                 function is executed during simulation 
//
//
//     NOTE : The adapter uses events of the communication class to
//     synchronize.  TL1 channel connected to the slave port MUST have its
//     'syncevent' parameter set.
//
//     History:
//      06/10/2004: Added PE() support for TL1
// ============================================================================

#include "ocp2_tl0_tl1_slave_adapter.h"
#include "ocp_tl_param_cl.h"
#include "ocp_tl1_data_cl.h"
#include "ocp_compatibility.h"

// Used for debug purposes
//#define DEBUG_SLAVE_ADAPTER

static
inline bool _isWriteCmd( const OCPMCmdType& cmd )
{
    return ( ( cmd == OCP_MCMD_WR   ) ||
             ( cmd == OCP_MCMD_WRNP ) ||
             ( cmd == OCP_MCMD_WRC  ) ||
             ( cmd == OCP_MCMD_BCST ) );
}

// ----------------------------------------------------------------------------
// Process : OCP2_TL0_TL1_Slave_Adapter::OCP2_TL0_TL1_Slave_Adapter
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP2_TL0_TL1_Slave_Adapter<TdataCl>::OCP2_TL0_TL1_Slave_Adapter
(
 sc_module_name name_
 , const OCP_TL0_PortFactory<OCP2_TL0_MasterPorts>& portFactory
 , bool checkSetupTime
 , bool combinational
)
: sc_module         (name_)
  , SlaveP                ( "SlavePort"        )
  , Clk                   ( "clk"              )
  , m_tl0Ports            ( *( portFactory() ) )
  , m_check_active        ( checkSetupTime     ) 
  , m_default_values      ( true               ) 
  , m_request_in_progress ( false              ) 
  , m_datahs_in_progress  ( false              ) 
  , m_PE_mode             ( false              ) 
{
    common_Setup();
}

template<class TdataCl>
void
OCP2_TL0_TL1_Slave_Adapter<TdataCl>::common_Setup()
{
    // Init TL0 output signals:
    //    * 'MCmd' as IDLE
    //    * 'MRespAccept'  and 'MDataValid' as false 
    assert( m_tl0Ports.MCmd != NULL );
    m_tl0Ports.MCmd->initValue( OCP_MCMD_IDLE );
    if ( m_tl0Ports.MRespAccept != NULL )
        m_tl0Ports.MRespAccept->initValue( false );
    if ( m_tl0Ports.MDataValid != NULL )
        m_tl0Ports.MDataValid->initValue( false );

    // Reset the response structure
    TL1_response.reset();

    // SC_METHODs
    if ( m_tl0Ports.SResp != NULL ) {
        SC_METHOD(send_Response);
        sensitive << e_response_sample_event;
    }

    SC_METHOD(set_reset_MRespAccept_PE);
    sensitive << SlaveP.ResponseReleaseEvent();
    dont_initialize();

    if ( m_tl0Ports.MRespAccept != NULL ) {
        SC_METHOD(set_MRespAccept_Normal);
        sensitive << SlaveP.ResponseEndEvent();
        dont_initialize();
        
        SC_METHOD(reset_MRespAccept_Normal);
        sensitive_pos << Clk;
        dont_initialize();
    }

    SC_METHOD(reset_TL0_signals);
    sensitive_pos << Clk;
    dont_initialize();

    SC_METHOD(send_MReset);
    sensitive << SlaveP.ResetStartEvent() << SlaveP.ResetEndEvent();
    dont_initialize();

    if ( m_tl0Ports.SReset_n != NULL ) {
      SC_METHOD(send_SReset);
      sensitive << m_tl0Ports.SReset_n->defaultEvent();
    }

    SC_METHOD(send_Request);
    sensitive << SlaveP.RequestStartEvent();
    dont_initialize();

    SC_METHOD(send_Data);
    sensitive << SlaveP.DataHSStartEvent();
    dont_initialize();

    SC_METHOD(release_Request);
    sensitive << e_accept_sample_event;

    SC_METHOD(release_Data);
    sensitive << e_accept_sample_event;

    SC_METHOD(set_TL1_ThreadBusy);
    sensitive << e_threadbusy_sample_event;
    dont_initialize();

    if ( m_tl0Ports.MThreadBusy != NULL ) {
        SC_METHOD(set_TL0_MThreadBusy);
        sensitive << SlaveP.MThreadBusyEvent();
        dont_initialize();
    }

    SC_METHOD(sample_events_trigger);
    dont_initialize();
    sensitive_pos << Clk;

    if(m_check_active) {
        if ( m_tl0Ports.SCmdAccept  != NULL ||
             m_tl0Ports.SDataAccept != NULL ) {
            SC_METHOD(check_accept_setup_time);        
            if ( m_tl0Ports.SCmdAccept != NULL )
                sensitive << m_tl0Ports.SCmdAccept->defaultEvent();
            if ( m_tl0Ports.SDataAccept != NULL )
                sensitive << m_tl0Ports.SDataAccept->defaultEvent();
        }
        if ( m_tl0Ports.SResp  != NULL ||
             m_tl0Ports.SData  != NULL ) {
            SC_METHOD(check_response_setup_time);
            if ( m_tl0Ports.SResp != NULL )
                sensitive << m_tl0Ports.SResp->defaultEvent();
            if ( m_tl0Ports.SData != NULL )
                sensitive << m_tl0Ports.SData->defaultEvent();
        }
    }

    m_response_accepted = false;
    m_ready_for_response = true;
}

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::setClockPeriod(), setSampleDelays
//
//  Set up the accept/response sample delays relative to the clock cycle
//  or as absolute delays.
//  This is separate from construction because clock period may not be known
//  if embedded in a structural module. However, this must be called before
//  elaboration
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl>
void OCP2_TL0_TL1_Slave_Adapter<TdataCl>::setClockPeriod( const sc_time& period )
{
    // fractions of cycle loosely based on Level2 timing guidelines
    m_response_sample_delay   = sc_time( period * double( 0.60 ) );
    m_accept_sample_delay     = sc_time( period * double( 0.75 ) );
    m_threadbusy_sample_delay = sc_time( period * double( 0.10 ) );
}  

template<class TdataCl>
void OCP2_TL0_TL1_Slave_Adapter<TdataCl>::setSampleDelays( const sc_time& acceptDelay,
                                                           const sc_time& responseDelay,
                                                           const sc_time& threadBusyDelay )
{
    m_accept_sample_delay     = acceptDelay;
    m_response_sample_delay   = responseDelay;
    m_threadbusy_sample_delay = threadBusyDelay;
}

// ----------------------------------------------------------------------------
// Process : OCP2_TL0_TL1_Slave_Adapter::~OCP2_TL0_TL1_Slave_Adapter
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP2_TL0_TL1_Slave_Adapter<TdataCl>::~OCP2_TL0_TL1_Slave_Adapter()
{
}

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::send_Response()
//
// Catch responses from the TL0 Slave, and send them to the TL1 channel 
//
//  Sensitivity: 
//      sensitive << e_response_sample_event;
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Slave_Adapter<TdataCl>::send_Response()
{
    // Test if a valid response is present on TL0 signals
    OCPSRespType resp = static_cast<OCPSRespType>( m_tl0Ports.SResp->getValue() );
    if( m_ready_for_response && !SlaveP->getMBusy() && resp != OCP_SRESP_NULL ) {

        // This is a new response. No more responses sent until the current one
        // has been accepted
        m_ready_for_response = false;

  	// Send a 'DVA' TL1 Response
  	if( resp == OCP_SRESP_DVA ) {
            TL1_response.SResp = OCP_SRESP_DVA;
            if ( m_tl0Ports.SData != NULL )
                m_tl0Ports.SData->getLongValue( TL1_response.SData );
            if ( m_tl0Ports.SDataInfo != NULL )
                m_tl0Ports.SDataInfo->getLongValue( TL1_response.SDataInfo );
            if ( m_tl0Ports.SRespInfo != NULL )
                m_tl0Ports.SRespInfo->getLongValue( TL1_response.SRespInfo );
            if ( m_tl0Ports.SRespLast != NULL )
                TL1_response.SRespLast = m_tl0Ports.SRespLast->getValue();
            if ( m_ParamCl->threads > 1 )
                TL1_response.SThreadID = m_tl0Ports.SThreadID->getValue();
            bool ret = SlaveP->startOCPResponse(TL1_response);
            
            // For Debug purposes only
            assert(ret);
        } else {
            // Send an 'ERR' TL1 Response
            if( resp == OCP_SRESP_ERR ) {
                TL1_response.SResp = OCP_SRESP_ERR;
                if ( m_tl0Ports.SData != NULL )
                    m_tl0Ports.SData->getLongValue( TL1_response.SData );
                if ( m_tl0Ports.SDataInfo != NULL )
                    m_tl0Ports.SDataInfo->getLongValue( TL1_response.SDataInfo );
                if ( m_tl0Ports.SRespInfo != NULL )
                    m_tl0Ports.SRespInfo->getLongValue( TL1_response.SRespInfo );
                if ( m_ParamCl->threads > 1 )
                    TL1_response.SThreadID = m_tl0Ports.SThreadID->getValue();
                bool ret = SlaveP->startOCPResponse(TL1_response);
                
                // For Debug purposes only
                assert(ret);
            } else {
                cout << "Error in 'OCP TL0 TL1 Slave Adapter' class 'send_Response' method " << endl
                     << "       TL0 SResp signal value '" << resp
                     << "' is not supported" << endl;
                exit(-1);
            }
        }
#ifdef DEBUG_SLAVE_ADAPTER
        cout << "TL0-TL1 Slave:"
             << "\tdelta " << simcontext()->delta_count()
             << "\ttime " << sc_time_stamp().to_default_time_units()
             << "\tcalls startOCPResponse()" << sc_time_stamp().to_default_time_units()
             << endl;
#endif

    }
} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::set_MRespAccept_Normal()
//
//  Set 'MRespAccept' TL0 signal in normal mode 
//
//  Sensitivity: 
//      sensitive << MasterP.ResponseEndEvent()
// ----------------------------------------------------------------------------
template<class TdataCl> void
OCP2_TL0_TL1_Slave_Adapter<TdataCl>::set_MRespAccept_Normal()
{
    // The current TL1 response has been accepted
    m_tl0Ports.MRespAccept->putValue( 1 );
    m_response_accepted = true;
} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::reset_MRespAccept_Normal()
//
//  Reset 'MRespAccept' TL0 signal in normal mode 
//
//  Sensitivity: 
//      sensitive_pos << Clk
// ----------------------------------------------------------------------------
template<class TdataCl> void
OCP2_TL0_TL1_Slave_Adapter<TdataCl>::reset_MRespAccept_Normal()
{
    if(!m_PE_mode && m_tl0Ports.MRespAccept != NULL) 
        m_tl0Ports.MRespAccept->putValue( 0 );
} // end of SC_METHOD


// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::set_reset_MRespAccept_PE()
//
//  Set and reset 'MRespAccept' TL0 signal in PE mode
//
//  Sensitivity: 
//      sensitive << MasterP.ResponseReleaseEvent()
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Slave_Adapter<TdataCl>::set_reset_MRespAccept_PE()
{

    if(m_CommCl->ResponseRelease) {
        // Entering the PE mode
        m_PE_mode = true;
        // TL1 slave called putMRespAccept(true)
        m_tl0Ports.MRespAccept->putValue( 1 );
        // The current TL1 response has been accepted
        m_response_accepted = true;
    } else {
        // Leaving the PE mode
        m_PE_mode = false;
        // TL1 slave called putMRespAccept(false)
        m_tl0Ports.MRespAccept->putValue( 0 );
    }
}

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::reset_TL0_signals()
//
//  Reset Request group TL0 signals 
//
//  Sensitivity: 
//      sensitive_pos << Clk;
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Slave_Adapter<TdataCl>::reset_TL0_signals()
{
    if(!m_request_in_progress) {
        m_tl0Ports.MCmd->putValue( OCP_MCMD_IDLE );
    }

    if(!m_datahs_in_progress && m_tl0Ports.MDataValid != NULL) {
        m_tl0Ports.MDataValid->putValue( 0 );
    }

    // The current TL1 response has been accepted by the master, and this is the
    // end of the OCP cycle => return to 'IDLE' state
    OCPSRespType resp = OCP_SRESP_NULL;
    if ( m_tl0Ports.SResp != NULL )
        resp = static_cast<OCPSRespType>( m_tl0Ports.SResp->getValue() );
    if ( resp != OCP_SRESP_NULL && m_response_accepted ) {
        m_ready_for_response = true;
        m_response_accepted = !( m_ParamCl->respaccept );
    }

    if ( m_ParamCl->sthreadbusy ) {
        SlaveP->putSThreadBusy( static_cast<unsigned int>( -1 ) );
    }
    if ( m_ParamCl->sdatathreadbusy ) {
        SlaveP->putSDataThreadBusy( static_cast<unsigned int>( -1 ) );
    }
} // end of SC_METHOD


// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::send_MReset()

//  Pass MReset from TL1 to TL0
//
//  Sensitivity: 
//  sensitive << SlaveP.ResetStartEvent() << SlaveP.ResetEndEvent()
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Slave_Adapter<TdataCl>::send_MReset()
{
    bool activeLowResetValue = !( SlaveP->getReset() );
    if ( m_tl0Ports.MReset_n != NULL )
        m_tl0Ports.MReset_n->putValue( activeLowResetValue );
}

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::send_SReset()
//  Pass SReset from TL0 to TL1
//
//  Sensitivity: 
//  sensitive << m_tl0Ports.SReset_n->default_event();
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Slave_Adapter<TdataCl>::send_SReset()
{
    bool inReset = !( m_tl0Ports.SReset_n->getValue() );
    if ( inReset )
        SlaveP->SResetAssert();
    else
        SlaveP->SResetDeassert();
}

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::send_Request()

//  Catch TL1 request fields and send them to the TL0 slave 
//
//  Sensitivity: 
//  sensitive << SlaveP.RequestStartEvent();
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Slave_Adapter<TdataCl>::send_Request()
{       
    if(SlaveP->getOCPRequest(TL1_request)) {
        m_request_in_progress = true;
        m_tl0Ports.MCmd->putValue( TL1_request.MCmd );
        if ( m_ParamCl->addr )
            m_tl0Ports.MAddr->putValue( TL1_request.MAddr );
        if ( m_ParamCl->addrspace )
            m_tl0Ports.MAddrSpace->putValue( TL1_request.MAddrSpace );
        if ( m_ParamCl->byteen )
            m_tl0Ports.MByteEn->putValue( TL1_request.MByteEn );
        if ( m_ParamCl->threads > 1 )
            m_tl0Ports.MThreadID->putValue( TL1_request.MThreadID );
        if ( m_ParamCl->connid )
            m_tl0Ports.MConnID->putValue( TL1_request.MConnID );
        if ( m_ParamCl->reqinfo )
            m_tl0Ports.MReqInfo->putValue( TL1_request.MReqInfo );
        if ( m_ParamCl->atomiclength )
            m_tl0Ports.MAtomicLength->putValue( TL1_request.MAtomicLength );
        if ( m_ParamCl->burstlength )
            m_tl0Ports.MBurstLength->putValue( TL1_request.MBurstLength );
        if ( m_ParamCl->burstprecise )
            m_tl0Ports.MBurstPrecise->putValue( TL1_request.MBurstPrecise );
        if ( m_ParamCl->burstseq )
            m_tl0Ports.MBurstSeq->putValue( TL1_request.MBurstSeq );
        if ( m_ParamCl->burstsinglereq )
            m_tl0Ports.MBurstSingleReq->putValue( TL1_request.MBurstSingleReq );
        if ( m_ParamCl->reqlast )
            m_tl0Ports.MReqLast->putValue( TL1_request.MReqLast );        

        if ( _isWriteCmd( TL1_request.MCmd ) && !m_ParamCl->datahandshake ) {
            m_tl0Ports.MData->putValue( TL1_request.MData );
            if ( m_ParamCl->mdatainfo ) 
                m_tl0Ports.MDataInfo->putValue( TL1_request.MDataInfo );
            if ( m_ParamCl->datalast )
                m_tl0Ports.MDataLast->putValue( TL1_request.MReqLast );
            if ( m_ParamCl->mdatabyteen ) 
                m_tl0Ports.MDataByteEn->putValue( TL1_request.MByteEn );
        }

#ifdef DEBUG_SLAVE_ADAPTER
        cout << "TL0 Slave Adapter sends a new request "
             << " delta " << simcontext()->delta_count()
             << " time " << sc_time_stamp().to_default_time_units()
             << " TL1_request.MCmd " << TL1_request.MCmd
             << " TL1_request.MAddr " << TL1_request.MAddr;
        if ( TL1_request.HasMData )
            cout << " TL1_request.MData " << TL1_request.MData;
        cout << endl;
#endif
    }
	
} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::send_Data()
//
//  Catch TL1 Data request fields and send them to the TL0 slave 
// 
//  Sensitivity: 
//  sensitive << SlaveP.DataRequestStartEvent();
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Slave_Adapter<TdataCl>::send_Data()
{
    if(SlaveP->getOCPDataHS(TL1_dataHS)) {
        m_datahs_in_progress = true;
        m_tl0Ports.MData->putValue( TL1_dataHS.MData );
        m_tl0Ports.MDataValid->putValue( 1 );
        if ( m_ParamCl->threads > 1 )
            m_tl0Ports.MDataThreadID->putValue( TL1_dataHS.MDataThreadID );
        if ( m_ParamCl->mdatainfo )
            m_tl0Ports.MDataInfo->putValue( TL1_dataHS.MDataInfo );
        if ( m_ParamCl->mdatabyteen )
            m_tl0Ports.MDataByteEn->putValue( TL1_dataHS.MDataByteEn );
        if ( m_ParamCl->datalast )
            m_tl0Ports.MDataLast->putValue( TL1_dataHS.MDataLast );
        
#ifdef DEBUG_SLAVE_ADAPTER
        cout << "TL0 Slave Adapter sends a new dataHS "
             << " delta " << simcontext()->delta_count()
             << " time " << sc_time_stamp().to_default_time_units()
             << " TL1_dataHS.MData " << TL1_dataHS.MData
             << endl;
#endif
    }

} // end of SC_METHOD 


// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::release_Request()
//
//  Release request present on TL1 channel
//
//  Sensitivity: 
//      sensitive << e_accept_sample_event;
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Slave_Adapter<TdataCl>::release_Request()
{
    if(m_request_in_progress) {
        assert( !m_ParamCl->cmdaccept || m_tl0Ports.SCmdAccept != NULL );
        if( !m_ParamCl->cmdaccept ||
            m_tl0Ports.SCmdAccept->getValue() == 1 ) {
#ifdef DEBUG_SLAVE_ADAPTER
            cout << "TL0-TL1 Slave:"
                 << "\tdelta " << simcontext()->delta_count()
                 << "\ttime " << sc_time_stamp().to_default_time_units()
                 << "\tcalls putSCmdAccept()" << sc_time_stamp().to_default_time_units()
                 << endl;
#endif  
            SlaveP->putSCmdAccept(); 
            m_request_in_progress = false;
        }
    }
} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::release_Data()
//
//  Release data request present on TL1 channel
//
//  Sensitivity: 
//      sensitive << e_accept_sample_event;
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Slave_Adapter<TdataCl>::release_Data()
{
    if(m_datahs_in_progress) {
        assert( !m_ParamCl->dataaccept || m_tl0Ports.SDataAccept != NULL );
        if( !m_ParamCl->dataaccept ||
            m_tl0Ports.SDataAccept->getValue() == 1 ) {
            SlaveP->putSDataAccept();
            m_datahs_in_progress = false;
        }
    }
} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::set_TL1_ThreadBusy()
//
//  Set TL1 threadbusy state after sampling TL0 SThreadBusy signal
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Slave_Adapter<TdataCl>::set_TL1_ThreadBusy()
{
    if ( m_tl0Ports.SThreadBusy != NULL ) {
        SlaveP->putSThreadBusy( m_tl0Ports.SThreadBusy->getValue() );
    }
    if ( m_tl0Ports.SDataThreadBusy != NULL ) {
        SlaveP->putSDataThreadBusy( m_tl0Ports.SDataThreadBusy->getValue() );
    }
}

template<class TdataCl> void OCP2_TL0_TL1_Slave_Adapter<TdataCl>::set_TL0_MThreadBusy()
{
    if ( m_tl0Ports.MThreadBusy != NULL  )
        m_tl0Ports.MThreadBusy->putValue( SlaveP->getMThreadBusy() );
}

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::sample_events_trigger()
//
//  Trigger 'e_cmdaccept_sample_event' and 'e_response_sample_event'
//  used to sample input signals 
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Slave_Adapter<TdataCl>::sample_events_trigger()
{
#ifdef DEBUG_SLAVE_ADAPTER
    cout << "TL0-TL1 Slave:"
        << "\tdelta " << simcontext()->delta_count()
        << "\ttime " << sc_time_stamp().to_default_time_units()
        << "\tOCP Clock cycle"
        << endl;
#endif

  e_accept_sample_event.notify(m_accept_sample_delay);
  e_response_sample_event.notify(m_response_sample_delay);
  if ( m_ParamCl->sthreadbusy || m_ParamCl->sdatathreadbusy )
      e_threadbusy_sample_event.notify(m_threadbusy_sample_delay);

  if(m_check_active)
    m_last_rising_edge = sc_time_stamp();

} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::check_response_setup_time()
//
//  Checks timing violations, i.e. if 'SResp' or 'SData' signals
//    change after the sample time
//    
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Slave_Adapter<TdataCl>::check_response_setup_time()
{
  if( (sc_time_stamp()-m_last_rising_edge) >= m_response_sample_delay ) {
    cout << "TL0 Slave Adapter '" << name() << "': Setup Time Violation "
      << " delta " << simcontext()->delta_count()
      << " time " << sc_time_stamp().to_default_time_units() << endl
      << " last rising edge " << m_last_rising_edge.to_default_time_units() << endl
      << " one of the signal 'SResp' or 'SData' had a late change" << endl
      << " You have to adjust 'ResponseSampleDelay' parameter" 
      << endl;

   if(m_default_values) {
     cout << " Default values for sampling times do not work with that design." << endl
      << " You have to explicitly specify sampling times using the appropriate constructor" 
      << endl;
   }
   sc_stop();
  }

} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::check_accept_setup_time()
//
//  Checks timing violations, i.e. if 'SCmdAccept' or 'SDataAccept' signals
//    change after the sample time
//    
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Slave_Adapter<TdataCl>::check_accept_setup_time()
{
  if( (sc_time_stamp()-m_last_rising_edge) >= m_accept_sample_delay ) {
    cout << "TL0 Slave Adapter '" << name() << "': Setup Time Violation "
      << " delta " << simcontext()->delta_count()
      << " time " << sc_time_stamp().to_default_time_units() << endl
      << " last rising edge " << m_last_rising_edge.to_default_time_units() << endl
      << " signal 'SCmdAccept' or 'SDataAccept' had a late change" << endl
      << " You have to adjust 'AcceptSampleDelay' parameter" 
      << endl;

   if(m_default_values) {
     cout << " Default values for sampling times do not work with that design." << endl
      << " You have to explicitly specify sampling times using the appropriate constructor" 
      << endl;
   }
    sc_stop();
  }

} // end of SC_METHOD 



// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Slave_Async::MPutDirect()
//
//  Debug interface method.
//  Read/Write data, Master to Slave direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl> bool OCP2_TL0_TL1_Slave_Adapter<TdataCl>::MputDirect(
    int MasterID, bool IsWrite, Td *Data, Ta Address, int NumWords)
{

  // not implemented
  return(false);
}

// ----------------------------------------------------------------------------
//  Method : OCP2_TL0_TL1_Slave_Adapter::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP2_TL0_TL1_Slave_Adapter<TdataCl>::end_of_elaboration()
{
  sc_module::end_of_elaboration();

  // Get communication structure
  m_CommCl = SlaveP->GetCommCl();

  // Here we check that the TL1 channel has 'cmdaccept' and 'dataaccept'
  m_ParamCl = SlaveP->GetParamCl();

  if(m_response_sample_delay < sc_get_time_resolution() ) {
      cout << "Error: TL0 OCP Slave Adapter constructor (instance '" << name() << "')"<< endl
           << "       'Response Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
           << "       'Response Sample Delay' =" << m_response_sample_delay
           << endl;
      exit(-1);
  }
  
  if(m_accept_sample_delay < sc_get_time_resolution() ) {
      cout << "Error: TL0 OCP Slave Adapter constructor (instance '" << name() << "')"<< endl
             << "       'CmdAccept Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
           << "       'CmdAccept Sample Delay' =" << m_accept_sample_delay
           << endl;
      exit(-1);
  }

  if(m_threadbusy_sample_delay < sc_get_time_resolution() ) {
      cout << "Error: TL0 OCP Slave Adapter constructor (instance '" << name() << "')"<< endl
             << "       'ThreadBusy Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
           << "       'ThreadBusy Sample Delay' =" << m_threadbusy_sample_delay
           << endl;
      exit(-1);
  }
  
  if(!(m_ParamCl->respaccept)) {
      m_response_accepted = true;
  }
}
