// 
//  Copyright 2006 OCP-IP
//  OCP-IP Confidential & Proprietary
//
// ============================================================================
//      Project : OCP SLD WG
//      Authors : Anssi Haverinen, Nokia
//                Herve Alexandrian, Sonics
//         $Id: tl1_CPUMaster.h,v 1.2 2007/08/21 02:23:35 halexan Exp $
//
//  Description:  TL0-TL1 adapter demo
//                
// ============================================================================



#ifndef _OCP_TL1_MASTER_CPU_H
#define _OCP_TL1_MASTER_CPU_H

// User defined types
#include "channel_types.h"

// OCP-IP Channel header files
#include "ocp_globals.h"
#include "ocp_tl1_master_port.h"
#include "ocp_tl_param_cl.h"

#include "sim_params.h"
#include "CPUTraffic.h"

SC_MODULE (tl1_CPUMaster) {

  // channel port
  OCP_TL1_MasterPort<OCP_TL1_DataCl<DType, AType > > tpP;

  // TL1 clock
  sc_in_clk  Clk;

  ~tl1_CPUMaster() {
    delete traffic;
  }

  SC_HAS_PROCESS(tl1_CPUMaster);

  // constructor
  tl1_CPUMaster (sc_module_name name_): 
    sc_module (name_), tpP("tpPort") {
    // initialize common members

    traffic = new CPUTraffic;
    timeslot = 0;
    tmp = 0;
    init = true;
    go_cpu = false;

    req.MData = 0;
    req.MAddr = 0;
    req.MCmd = OCP_MCMD_IDLE;
    req.MByteEn = 0x3;
    
    SC_THREAD(proc);
    sensitive_pos(Clk); 
    dont_initialize();

    SC_THREAD(inth);
  }

  // threads

  // Interrupt task
  void inth() {
    while(1){
      wait(100, SC_US);
      timeslot_start = sc_time_stamp().to_seconds();
#ifdef DEBUG_MSG
      cout << "CPU interrupt at time "
      	   << sc_time_stamp().to_seconds() << endl;
#endif
      go_cpu = true;
      interrupt.notify(SC_ZERO_TIME);
    }
  }

  void proc() {

    OCPParameters* pParam = tpP->GetParameters();
    while (1) {
      int wc, rc, timeslot_wc = 0, timeslot_rc = 0;
      bool high_bytes = true;
      bool writes_done = false;
      bool reads_done = false;
    
      if (!go_cpu) {
        tpP->putMRespAccept( true );
	wait(interrupt);
      
	go_cpu = false;
	
	// CPU port traffic, we do only 16-bit access here
	// to demonstrate byte enable.
	
	while (!writes_done || !reads_done){
	  rc = 0;
	  wc = 0;
	  while (wc < CPU_RW_INTERLEAVE && !writes_done){
	    req.MCmd = OCP_MCMD_WR;
	    req.MAddr = 2*timeslot_wc + 128; 
	    req.MData = timeslot_wc;
	    if (high_bytes)
	      req.MByteEn = 0x3;
	    else
	      req.MByteEn = 0xC;
	    wait();
	    while (tpP->getSBusy())
	      wait();
	  
            int numResponses = 0;
            if ( pParam->burstlength ) {
                numResponses = sendBurst( req );
            }
	    tpP->startOCPRequest(req);
/* 	    while(!tpP->getOCPResponse(resp, true)) */
            wait();
	    high_bytes = !high_bytes;
	    
	    wc++;
	    if (timeslot_wc < traffic->writes[timeslot]) {
	      timeslot_wc++;
	      writes_done = false;
	    }
	    else {
	      timeslot_wc = 0;
	      writes_done = true;
	    }
	  }
	  while (rc < CPU_RW_INTERLEAVE && !reads_done){
	    req.MCmd = OCP_MCMD_RD;
	    req.MAddr = 2*timeslot_rc + 128; 
	    req.MData = timeslot_rc;
	    if (high_bytes)
	      req.MByteEn = 0x3;
	    else
	      req.MByteEn = 0xC;
	    wait();
	    while (tpP->getSBusy())
	      wait();
	    tpP->startOCPRequest(req);
	    while(!tpP->getOCPResponse(resp, true))
	      wait();
	    high_bytes = !high_bytes;
	    
	    rc++;
	    if (timeslot_rc < traffic->reads[timeslot]) {
	      timeslot_rc++;
	      reads_done = false;
	    }
	    else {
	      timeslot_rc = 0;
	      reads_done = true;
	    }
	  }
	}
	timeslot_end = sc_time_stamp().to_seconds();
#ifdef DEBUG_MSG	
	cout << "CPU timeslot load  " << timeslot << "     "
	     << 100 * (timeslot_end - timeslot_start) / 0.000052 << "%" << endl;
#endif	
	if (timeslot < traffic->num_timeslots-1)
	  timeslot++;
	else
	  timeslot = 0;
      }
    }
  }

 private:
  int sendBurst( const OCPRequestGrp<DType,AType>& req ) {
      OCPParameters* pParam = tpP->GetParameters();
      int length = 4;
      int height = 1;
      if ( pParam->blockheight )
          height = 2;

      for ( int i=0; i<length * height; ++i ) {
          bool first = ( i == 0 );
          bool last  = ( i == length*height - 1 );
          OCPRequestGrp<DType,AType> breq( req );
          breq.MBurstLength = length;
          breq.MBlockHeight = height;
          breq.MBlockStride = 0x100;
          breq.MReqLast = last;
          breq.MReqRowLast = ( ( i + 1 ) % length ) == 0;
          breq.MAddr += i * pParam->data_wdth/8;
          tpP->startOCPRequestBlocking( breq );
      }
      // how many responses
      int responses = length * height;
      if ( ( req.MCmd == OCP_MCMD_WR || req.MCmd == OCP_MCMD_WRNP ) &&
           !pParam->writeresp_enable )
          responses = 0;
      return responses;
  }
      
 private:
  bool tmp;
  int timeslot;
  double timeslot_start;
  double timeslot_end;

  // Request group
  OCPRequestGrp<DType,AType> req;
  
  // Response group
  OCPResponseGrp<DType> resp;

  sc_event interrupt;
  bool go_cpu;
  bool init;

  CPUTraffic *traffic;
  
};


#endif
