// 
//  (c) Copyright OCP-IP 2003, 2004, 2005
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//
//      Project : OCP SLD WG, Layer adapters
//      Authors : Herve Alexanian, SONICS, herve@sonicsinc.com
//                Stephane Guntz, PROSILOG, guntz@prosilog.com
//		  Yann Bajot, PROSILOG, bajot@prosilog.com
//
//  Description : Transaction Level - Layer-1 to Layer-2 OCP Master Adapter
//  $Id:
//
// ============================================================================

#include "ocp_tl1_tl2_master_adapter.h"
#include "array_stack.h"
#include "ocp_utils.h"
#include <deque>
#include <algorithm>
#include <assert.h>

#define MAX_THREADS     16
#define POOL_SIZE       1024

//---------------------------------------------------------------------------
//Method: Tl2ResponseExpander constructor
//Note: The TL2 response is copied. This may cost some performance
// but would require some infrastructure to address with shared pointers
// and a memory pool for example.
//---------------------------------------------------------------------------
template<typename Td>
Tl2ResponseExpander<Td>::Tl2ResponseExpander(
    const OCPTL2ResponseGrp<Td>& resp, Td* pDataPtr
    ) :
    m_numExpanded( 0 ),
    m_turnAroundBurstLength( 0 ),
    m_tl2Resp( resp ),
    m_tl1Resp( resp )
{    
    // deep data copy
    if ( resp.SDataPtr != NULL ) {
        for ( unsigned int i=0; i < resp.DataLength; ++i )
            pDataPtr[i] = m_tl2Resp.SDataPtr[i];
    }
    m_tl2Resp.SDataPtr     = pDataPtr;
}

template<typename Td>
OCPResponseGrp<Td>
Tl2ResponseExpander<Td>::next()
{    
    // assign the length sensitive fields
    if ( m_tl2Resp.SDataPtr     != NULL )
        m_tl1Resp.SData          = m_tl2Resp.SDataPtr[m_numExpanded];
    ++m_numExpanded;

    // calculate last of burst/row
    m_tl1Resp.SRespLast = m_tl2Resp.LastOfBurst &&
        ( m_numExpanded == m_tl2Resp.DataLength );
    m_tl1Resp.SRespRowLast = m_tl1Resp.SRespLast;
    if ( m_turnAroundBurstLength > 0 )
        m_tl1Resp.SRespRowLast |= m_tl2Resp.LastOfRow &&
            ( m_numExpanded % m_turnAroundBurstLength ) == 0;
    return m_tl1Resp;
}

//---------------------------------------------------------------------------
//Method: Tl1RequestAggregator constructor
//Note: The TL1 requests are aggregated into one expected TL2 request
//---------------------------------------------------------------------------
template<typename Td, typename Ta>
Tl1RequestAggregator<Td,Ta>::Tl1RequestAggregator( int maxChunkLength,
    bool lastOfBurst )
  : m_numAggregated( 0 ),
    m_numDataAggregated( 0 ),
    m_burstPosition( 0 ),
    m_maxChunkLength( maxChunkLength ),
    m_pParams( NULL )
{
    m_tl2Req.DataLength   = 0;
    m_tl2Req.LastOfBurst  = lastOfBurst;
}

template<typename Td, typename Ta>
bool
Tl1RequestAggregator<Td,Ta>::finishedRequest() const
{
    return ( m_tl2Req.MBurstSingleReq && m_numAggregated > 0 ) ||
        ( m_numAggregated == static_cast<unsigned int>( m_maxChunkLength ) ) ||
        ( !m_tl2Req.MBurstPrecise && m_tl2Req.LastOfBurst ) ||
        ( m_tl2Req.MBurstPrecise &&
          m_tl2Req.DataLength == m_numAggregated );
}

template<typename Td, typename Ta>
bool
Tl1RequestAggregator<Td,Ta>::finished() const
{
    if ( OcpIp::isRead( m_tl2Req.MCmd ) )
        return finishedRequest();
    return finishedRequest() && ( m_numDataAggregated == m_tl2Req.DataLength );
}

template<typename Td, typename Ta>
void
Tl1RequestAggregator<Td,Ta>::next( const OCPRequestGrp<Td, Ta>& req )
{
    assert( !finishedRequest() );
    assert( m_burstPosition > 0 );
    bool dataHandshake = m_pParams->datahandshake;
    unsigned int maxChunkLength = m_maxChunkLength;
    unsigned int remainingBurstLength = req.MBurstLength *
        ( ( req.MBlockHeight > 1 ) ? req.MBlockHeight : 1 );
    if ( req.MBurstPrecise )
        remainingBurstLength -= ( m_burstPosition - 1 );
    if ( m_tl2Req.DataLength == 0 ) {
        // copyFrom clobbers the pointers!
        Td          * pSaveDataPtr      = m_tl2Req.MDataPtr;
        unsigned int* pSaveByteEnPtr    = m_tl2Req.MByteEnPtr;
        uint64      * pSaveDataInfoPtr  = m_tl2Req.MDataInfoPtr;

        // copyFrom
        m_tl2Req.copyFrom ( req, 0, false );

        // restore old pointers
        m_tl2Req.MDataPtr     = pSaveDataPtr;
        m_tl2Req.MByteEnPtr   = pSaveByteEnPtr;
        m_tl2Req.MDataInfoPtr = pSaveDataInfoPtr;

        if ( req.MBurstPrecise ) {
            m_tl2Req.LastOfBurst =
                ( remainingBurstLength <= maxChunkLength ) ||
                ( req.MBurstSingleReq && OcpIp::isRead( req.MCmd ) );
	    m_tl2Req.LastOfRow = m_tl2Req.LastOfBurst;
	    if ( req.MBlockHeight > 1 ) {
                unsigned int nextRowBound = req.MBurstLength *
                    int( ceil( 1.0 * m_burstPosition / req.MBurstLength ) );
                m_tl2Req.LastOfRow |=
                    ( m_burstPosition + maxChunkLength ) >= nextRowBound;
            }
        }
    }

    // copy data group if no datahandshake phase
    if ( OcpIp::isWrite( req.MCmd ) && !dataHandshake ) {
        assert( m_tl2Req.MDataPtr     != NULL );
        m_tl2Req.MDataPtr[m_numDataAggregated] = req.MData;
        if ( m_pParams->mdatainfo ) {
            assert( m_tl2Req.MDataInfoPtr   != NULL );
            m_tl2Req.MDataInfoPtr[m_numDataAggregated] = req.MDataInfo;
        }
        ++m_numDataAggregated;
    }
    if ( m_pParams->byteen && !req.MBurstSingleReq ) {
        assert( m_tl2Req.MByteEnPtr   != NULL );
        m_tl2Req.MByteEnPtr[m_numAggregated] = req.MByteEn;
    }

    // TL2 datalength is how many requests (datawords) we expect to aggregate
    if ( req.MBurstPrecise ) {
        if ( OcpIp::isRead( req.MCmd ) && req.MBurstSingleReq )
            m_tl2Req.DataLength = req.MBurstLength *
                ( ( req.MBlockHeight > 1 ) ? req.MBlockHeight : 1 );
        else
            m_tl2Req.DataLength = ( remainingBurstLength < maxChunkLength ) ?
                remainingBurstLength : maxChunkLength;
    } else {
        m_tl2Req.DataLength++;
        m_tl2Req.LastOfBurst = ( 1 == req.MBurstLength );
    }

    // Compute numAggregated
    if ( req.MBurstSingleReq ) {
        m_numAggregated = req.MBurstLength;
    } else {
        ++m_numAggregated;
    }
}

template<typename Td, typename Ta>
void
Tl1RequestAggregator<Td,Ta>::nextData( const OCPDataHSGrp<Td>& dataHs )
{
    assert( OcpIp::isWrite( m_tl2Req.MCmd ) );
    assert( m_tl2Req.MDataPtr != NULL );
    assert( m_tl2Req.MThreadID == dataHs.MDataThreadID );
    m_tl2Req.MDataPtr[m_numDataAggregated] = dataHs.MData;
    if ( m_pParams->mdatabyteen ) {
        assert( m_tl2Req.MByteEnPtr   != NULL );
        m_tl2Req.MByteEnPtr[m_numDataAggregated] = dataHs.MDataByteEn;
    }
    if ( m_pParams->mdatainfo ) {
        assert( m_tl2Req.MDataInfoPtr   != NULL );
        m_tl2Req.MDataInfoPtr[m_numDataAggregated] = dataHs.MDataInfo;
    }
    ++m_numDataAggregated;
}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter constructor
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::OCP_TL1_TL2_Master_Adapter(
    sc_module_name name, int max_chunk_length,
    int adapter_depth, sc_time timeout): 
    sc_module(name),
    SlaveP("SlaveP"),
    MasterP("MasterP"),
    clk( "clk" ),
    m_ptl2MByteEnPool  ( new ArrayStack<unsigned int>( POOL_SIZE ) ),
    m_ptl2MDataPool    ( new ArrayStack<Td>( POOL_SIZE ) ),
    m_ptl2SDataPool    ( new ArrayStack<Td>( POOL_SIZE ) ),
    m_ptl2MDataInfoPool( new ArrayStack<uint64>( POOL_SIZE ) ),
    m_pAddrSequence    ( NULL ),
    m_tl2RequestThreadId ( 0 ),
    m_tl1ResponseThreadId( 0 ),
    m_maxChunkLength(max_chunk_length),
    m_bindGetSThreadBusy( MasterP ),
    m_bindPutSThreadBusy( SlaveP ),
    m_sThreadBusyRelay( "SThreadBusy", MasterP.SThreadBusyEvent(),
                        m_bindGetSThreadBusy, m_bindPutSThreadBusy ),
    m_bindGetMThreadBusy( SlaveP ),
    m_bindPutMThreadBusy( MasterP ),
    m_mThreadBusyRelay( "MThreadBusy", SlaveP.MThreadBusyEvent(),
                        m_bindGetMThreadBusy, m_bindPutMThreadBusy )

{
    SC_METHOD(SlaveRequest);
    sensitive << SlaveP.RequestStartEvent();
    dont_initialize();

    SC_METHOD(SCmdAccept);
    sensitive << m_tl1CmdAcceptEvent;
    dont_initialize();

    SC_METHOD(SlaveResyncDataHs);
    sensitive << SlaveP.DataHSStartEvent();
    dont_initialize();

    SC_METHOD(SlaveDataHs);
    sensitive << m_slaveDataHsStartEventPlusZero;
    dont_initialize();

    SC_METHOD(SDataAccept);
    sensitive << m_tl1DataAcceptEvent;
    dont_initialize();

// Named SC_THREAD
#define SC_THREADN(func, name)                       \
    declare_thread_process( func ## _handle,         \
                            name,                    \
                            SC_CURRENT_USER_MODULE,  \
                            func )

    for ( int t=0; t < MAX_THREADS; ++t ) {
        char mname[40], sname[40];
        sprintf( mname, "MasterRequest_%d", t );
        sprintf( sname, "SlaveResponse_%d", t );
        SC_THREADN(MasterRequest, mname);
        SC_THREADN(SlaveResponse, sname);
    }

    SC_METHOD(MasterResponse);
    sensitive << MasterP.ResponseStartEvent();
    dont_initialize();

    SC_METHOD(MRespAccept);
    sensitive << m_tl2RespAcceptEvent;
    dont_initialize();

    SC_METHOD(MputReset);
    sensitive << SlaveP.ResetStartEvent() << SlaveP.ResetEndEvent();
    dont_initialize();

    SC_METHOD(SputReset);
    sensitive << MasterP.ResetStartEvent() << MasterP.ResetEndEvent();
    dont_initialize();

    SC_METHOD(SputFlags);
    sensitive << MasterP.SidebandSlaveEvent() ;

    SC_METHOD(SputMFlag);
    sensitive << SlaveP.SidebandMFlagEvent() ;

    SC_METHOD(SputMError);
    sensitive << SlaveP.SidebandMErrorEvent() ;
}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter destructor
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::~OCP_TL1_TL2_Master_Adapter()
{
    delete m_pAddrSequence;
    delete m_ptl2MDataInfoPool;
    delete m_ptl2SDataPool;
    delete m_ptl2MDataPool;
    delete m_ptl2MByteEnPool;
}

//---------------------------------------------------------------------------
// Method: OCP_TL1_TL2_Master_Adapter::SlaveRequest
// get the TL1 request and aggregate into TL2 request
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::SlaveRequest()
{
    if ( !SlaveP->getOCPRequest( m_tl1Request ) )
        return;

    unsigned int threadId = m_tl1Request.MThreadID;
    assert( threadId < m_tl1RequestAggregatorQueues.size() );
    Tl1RequestAggregatorQueue& queue = m_tl1RequestAggregatorQueues[threadId];
    bool isWrite = OcpIp::isWrite( m_tl1Request.MCmd );

    // count position in burst for MRMD requests that get chunked over multiple
    // aggregators.
    unsigned int burstPosition = 1;
    if ( !m_tl1Request.MBurstSingleReq ) {
        OcpIp::BurstCounter&  burstCount = m_burstCount[threadId];
        burstCount.next( m_tl1Request );
        burstPosition = burstCount.count();
    }
    // chunk size 1 for unknown bursts.
    unsigned int maxChunkLength =
        ( m_tl1Request.MBurstSeq == OCP_MBURSTSEQ_UNKN ) ? 1 : m_maxChunkLength;

    // Look for the last incomplete aggregator or make new ones
    Tl1RequestAggregatorQueue newAggregQueue;
    if ( queue.empty() || queue.back().finishedRequest() ) {
        unsigned int numAggreg = 1;
        // except for SRMD WR request
        if ( m_tl1Request.MBurstSingleReq && isWrite ) {
            assert( m_params->datahandshake );
            unsigned int burstLength = m_tl1Request.MBurstLength *
                ( ( m_tl1Request.MBlockHeight > 1 ) ? m_tl1Request.MBlockHeight : 1 );
            numAggreg = int( ceil( 1.0 * burstLength / maxChunkLength ) );
        }

        if ( numAggreg > 1 ) {
            // need to compute starting address for aggregators if request
            // will be chunked over multiple aggregators
            assert( m_tl1Request.MBurstPrecise );
            assert( m_pAddrSequence != NULL );
            OCPMBurstSeqType burstseq = m_tl1Request.MBurstSeq;
            if ( m_tl1Request.MBurstSeq == OCP_MBURSTSEQ_UNKN )
                burstseq = OCP_MBURSTSEQ_INCR; // cannot calculate unkn sequence
            m_pAddrSequence->init( m_tl1Request.MAddr, burstseq,
                                   m_tl1Request.MBurstLength,
                                   m_tl1Request.MBurstPrecise,
                                   m_tl1Request.MBlockHeight,
                                   m_tl1Request.MBlockStride );
            m_pAddrSequence->calculate();
        }
        typename OcpIp::BurstSequence<Ta>::iterator addrIter =
            m_pAddrSequence->begin();
        newAggregQueue.resize( numAggreg, maxChunkLength );
        for ( typename Tl1RequestAggregatorQueue::iterator it = 
                  newAggregQueue.begin(); it != newAggregQueue.end(); ++it ) {
            Tl1RequestAggregator<Td, Ta>& aggregator = *it;
            // burst tracking (for requests chunked over multiple aggregators)
            aggregator.setBurstPosition( burstPosition );
            if ( numAggreg > 1 && burstPosition > 1 ) {
                addrIter += maxChunkLength;
                m_tl1Request.MAddr = *addrIter;
            }
            burstPosition += maxChunkLength;
            // allocate spaces
            aggregator.setDataPtr( m_ptl2MDataPool->getNew() );
            // No Byteen pointer for single req reads. Use MByteEn member
            if ( ( m_params->byteen && !m_tl1Request.MBurstSingleReq )
                 || ( isWrite && m_params->mdatabyteen ) )
                aggregator.setByteEnPtr( m_ptl2MByteEnPool->getNew() );
            if ( m_params->mdatainfo ) 
                aggregator.setDataInfoPtr( m_ptl2MDataInfoPool->getNew() );
            aggregator.setParamCl( m_params );
            aggregator.next( m_tl1Request );
        }
    }
    
    // copy the new aggregators
    copy( newAggregQueue.begin(), newAggregQueue.end(), back_inserter( queue ) );   
    Tl1RequestAggregator<Td, Ta>& aggregator = queue.back();
    if ( newAggregQueue.empty() )
        aggregator.next( m_tl1Request );
    if ( aggregator.finished() ) {
        m_newTl2ReqEvent[threadId]->notify();
    }
    m_tl1CmdAcceptEvent.notify( m_times.RqAL * MasterP->getPeriod() );
}

//---------------------------------------------------------------------------
// Method: OCP_TL1_TL2_Master_Adapter::SlaveResyncDataHs
// delay data hs event by a delta cycle, since we need to have request first
// then datahandshake
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
void
OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::SlaveResyncDataHs()
{
    m_slaveDataHsStartEventPlusZero.notify(SC_ZERO_TIME);
}

//---------------------------------------------------------------------------
// Method: OCP_TL1_TL2_Master_Adapter::SlaveDataHs
// get the TL1 data hs and aggregate into TL2 request
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
void
OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::SlaveDataHs()
{
    if ( !SlaveP->getOCPDataHS( m_tl1DataHs ) )
        return;

    unsigned int threadId = m_tl1DataHs.MDataThreadID;
    assert( threadId < m_tl1RequestAggregatorQueues.size() );
    Tl1RequestAggregatorQueue& queue = m_tl1RequestAggregatorQueues[threadId];

    // Stuff to the back of first unfinished aggregator
    typename Tl1RequestAggregatorQueue::iterator unfinishedIt = queue.begin();
    while ( unfinishedIt != queue.end() && unfinishedIt->finished() )
      ++unfinishedIt;
    // this check also covers if the queue is empty
    if ( unfinishedIt == queue.end() ) {
        cerr << sc_time_stamp() << ": "
             << name() << ": WARNING: Ignoring unexpected TL1 data received\n"
             << hex << m_tl1DataHs.MData << endl;
        return;
    }      
    assert( !unfinishedIt->finished() );
    Tl1RequestAggregator<Td, Ta>& aggregator = *unfinishedIt;
    aggregator.nextData( m_tl1DataHs );
    if ( aggregator.finished() )
        m_newTl2ReqEvent[threadId]->notify();

    m_tl1DataAcceptEvent.notify( m_times.DAL * MasterP->getPeriod() );
}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::MasterRequest
//send TL2 request made of atomic TL1 requests
//
//---------------------------------------------------------------------------
template< class TdataCl_tl1>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::MasterRequest()
{
    int threadId = m_tl2RequestThreadId++;
    if ( threadId >= m_params->threads ) {
        // there is one MasterRequest SC_THREAD per ocp thread
        // but technically, we start them at construction time when we don't
        // know the number of threads yet. So we start a fixed number and let
        // those who exceed the thread count die now
        return;
    }

    assert( threadId < static_cast<int>( m_tl1RequestAggregatorQueues.size() ) );
    Tl1RequestAggregatorQueue& queue = m_tl1RequestAggregatorQueues[threadId];
    sc_event& newReqEvent            = *(m_newTl2ReqEvent[threadId]);
    
    OCPTL2RequestGrp<Td, Ta> tl2req;

    while ( true ) {        
        if ( queue.empty() || !queue.front().finished()) {
	    do wait( newReqEvent );
	    while ( queue.empty() || !queue.front().finished() );
            // re-sync to clock rising edge
            wait( clk->posedge_event() );
        }

        Tl1RequestAggregator<Td, Ta>& aggregator = queue.front();
        assert( aggregator.finished() );
        tl2req = aggregator.get();
        while ( true ) {
            if ( m_params->sthreadbusy ) {
                while ( MasterP->getSThreadBusyBit( threadId ) )
                    wait( MasterP->SThreadBusyEvent() );
            }
            if ( MasterP->sendOCPRequest( tl2req ) )
                break;
            wait( MasterP->RequestEndEvent() );
        }
        // wait for acceptance
        wait( MasterP->RequestEndEvent() );

        // store the first chunk of this burst as turnaround info if response
        // is expected
        bool needsResponse =
            !OcpIp::isWrite( tl2req.MCmd ) || m_params->writeresp_enable;
        if ( needsResponse ) {
            if ( !m_tl2BurstOpen[threadId] )
                m_tl2ReqPendingResp[threadId].push_back( tl2req );
            m_tl2BurstOpen[threadId] = !tl2req.LastOfBurst;
        }

        if ( tl2req.MByteEnPtr )
            m_ptl2MByteEnPool->recycle  ( tl2req.MByteEnPtr );
        if ( tl2req.MDataPtr )
            m_ptl2MDataPool->recycle    ( tl2req.MDataPtr );
        if ( tl2req.MDataInfoPtr )
            m_ptl2MDataInfoPool->recycle( tl2req.MDataInfoPtr );
        queue.pop_front();
    }
} //end of MasterRequest method

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::SlaveResponse
//get the TL2 response from the slave
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::SlaveResponse()
{ 
    int threadId = m_tl1ResponseThreadId++;
    if ( threadId >= m_params->threads ) {
        // there is one SlaveResponse SC_THREAD per ocp thread
        // but technically, we start them at construction time when we don't
        // know the number of threads yet. So we start a fixed number and let
        // those who exceed the thread count die now
        return;
    }

    assert( threadId < static_cast<int>( m_tl2ResponseExpanderQueues.size() ) );
    Tl2ResponseExpanderQueue&     queue = m_tl2ResponseExpanderQueues[threadId];
    sc_event& newResponseEvent          = *(m_newTl2RespEvent[threadId]);
    TdataCl_tl1* pDataCl                = SlaveP->GetDataCl();
    
    OCPResponseGrp<Td> tl1resp;

    while ( true ) {
        if ( queue.empty() ) {
            wait( newResponseEvent );
        }
        Tl2ResponseExpander<Td>& expander = queue.front();
        tl1resp = expander.next();

        if ( m_params->mthreadbusy ) {
            while ( pDataCl->isThisThreadBusy( threadId, SlaveP->getMThreadBusy() ) )
                wait( SlaveP->MThreadBusyEvent() );
        }
        // TODO: warning: the code below will override any thread busy signal
        // that gets set while it waits
        wait( SC_ZERO_TIME );
        while ( !SlaveP->startOCPResponse( tl1resp ) ) {
            wait( SlaveP->ResponseEndEvent() );
            wait( clk->posedge_event() );
        }
        // wait on accept, synchronously
        do wait( clk->posedge_event() );
        while ( !SlaveP->getMRespAccept() );

        if ( expander.finished() ) {
            const OCPTL2ResponseGrp<Td>& tl2Resp = expander.get();
            if ( tl2Resp.SDataPtr )
                m_ptl2SDataPool->recycle( tl2Resp.SDataPtr );
            m_tl2RespAcceptEvent.notify();
            queue.pop_front();
        }
    }
} //end of SlaveResponse method


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::MasterResponse
//send the TL1 response to the master, made from the TL2 response from the slave 
//---------------------------------------------------------------------------
template<class TdataCl_tl1>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::MasterResponse()
{
    if ( !MasterP->getOCPResponse( m_tl2Response ) )
        return;

    // Queue up the TL2 request
    unsigned int threadId = m_tl2Response.SThreadID;
    assert( threadId < m_tl2ResponseExpanderQueues.size() );

    Tl2ResponseExpander<Td> expander( m_tl2Response,
                                      m_ptl2SDataPool->getNew() );
    m_tl2ResponseExpanderQueues[threadId].push_back( expander );
    assert( threadId < m_tl2ReqPendingResp.size() );
    if ( !m_tl2ReqPendingResp[threadId].empty() ) {            
        m_tl2ResponseExpanderQueues[threadId].back().
            setTurnAroundReq( m_tl2ReqPendingResp[threadId].front() );
        if ( m_tl2Response.LastOfBurst )
            m_tl2ReqPendingResp[threadId].pop_front();
    }
    m_newTl2RespEvent[threadId]->notify();
} //end of MasterResponse method


// ----------------------------------------------------------------------------
// Accept methods
// ----------------------------------------------------------------------------
template< class TdataCl_tl1>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::SCmdAccept()
{
    if ( m_params->cmdaccept )
        SlaveP->putSCmdAccept();
}

template< class TdataCl_tl1>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::SDataAccept()
{
    if ( m_params->dataaccept )
        SlaveP->putSDataAccept();
}

template< class TdataCl_tl1>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::MRespAccept()
{
    MasterP->acceptResponse();
}

// ----------------------------------------------------------------------------
// sideband processes
// ----------------------------------------------------------------------------
template<class TdataCl_tl1 >
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::MputReset()
{
    if ( !m_params->mreset ) return;
    // be cautious to only move the reset signal if it affects something
    // or we risk an infinite loop if both mreset and sreset are enabled
    if ( SlaveP->getReset() ) {
        if ( !MasterP->getReset() )
            MasterP->MResetAssert();
    } else {
        if ( MasterP->getReset() )
            MasterP->MResetDeassert();
    }
}

template<class TdataCl_tl1 >
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::SputReset()
{
    if ( !m_params->sreset ) return;
    if ( MasterP->getReset() ) {
        if ( !SlaveP->getReset() )        
            SlaveP->SResetAssert();
    } else {
        if ( SlaveP->getReset() )
            SlaveP->SResetDeassert();
    }
}

template<class TdataCl_tl1 >
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::SputFlags()
{
    SlaveP->SputSFlag(MasterP->MgetSFlag());
    SlaveP->SputSError(MasterP->MgetSError());
    SlaveP->SputSInterrupt(MasterP->MgetSInterrupt());
}
template<class TdataCl_tl1 >
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::SputMFlag()
{
    MasterP->MputMFlag(SlaveP->SgetMFlag());
}
template<class TdataCl_tl1 >
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::SputMError()
{
    MasterP->MputMError(SlaveP->SgetMError());
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL1_TL2_Master_Adapter::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl_tl1>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1>::end_of_elaboration()
{
    sc_module::end_of_elaboration();

    // Get system parameter structure
    m_params = SlaveP->GetParameters();
    MasterP->getSlaveTiming( m_times );

    int threads = m_params->threads;
    m_tl1RequestAggregatorQueues.resize   ( threads );
    m_tl2ResponseExpanderQueues.resize    ( threads );
    m_newTl2ReqEvent.resize               ( threads );
    m_newTl2RespEvent.resize              ( threads );
    m_burstCount.resize                   ( threads );
    m_tl2ReqPendingResp.resize            ( threads );
    m_tl2BurstOpen.resize                 ( threads );

    // Addr sequence is a pointer new'ed and configured now
    m_pAddrSequence = new OcpIp::BurstSequence<Ta>( m_params->data_wdth,
                                                    m_params->addr_wdth, 0, 0 );
    for ( int t=0; t<threads; ++t) {
        m_newTl2ReqEvent[t]   = new sc_event();
        m_newTl2RespEvent[t]  = new sc_event();
        m_tl2BurstOpen[t]     = false;
    }
}
