///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2009-2010
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : Herve Alexanian - Sonics, inc.
//
//          $Id:
//
//  Description :  This file defines a generic OCP master model driven by STL
//                 stimulus for use in TL3 modeling examples
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////
#include "generic_stl_master_tl3.h"

template<unsigned int BUSWIDTH>
ocpip_example::generic_stl_master_tl3<BUSWIDTH>::generic_stl_master_tl3(sc_core::sc_module_name name_):
    sc_core::sc_module (name_)
    , m_socket( "socket", ocpip::ocp_tl3, ocpip::ocp_master_socket<BUSWIDTH>::mm_txn_with_be_and_data() )
{
    using namespace ocpip;
    using namespace ocpip_legacy;

    m_socket.register_nb_transport_bw(this, &generic_stl_master_tl3::nb_transport);

    // here we are dealing with the STL reader's unfriendly sequence for systemc creation
    // 1) the STL stream objects (sc_module) should be created at construction time
    // 2) the STL reader requires the config at construction time
    // 3) the function to find STL files which determines how many STL streams to make
    //    also creates STL readers....
    // So, we will create STL streams now, and throw away the STL readers and re-create
    // them later when the config is known, assuming the STL files are stable.
    OCPParameters params = m_socket.get_ocp_config().
	template legacy_conversion<ocp_data_class_unsigned<BUSWIDTH,32> >();
    std::deque<OcpIp::IStlReader*> stlReaders = OcpIp::IStlReader::findStlPrograms( basename(), false, params );
    if ( stlReaders.empty() ) {
	std::cerr << "Warning: No STL stream found for master " << name() << std::endl;
    } else {
	for ( deque<OcpIp::IStlReader*>::iterator stlIt = stlReaders.begin(); stlIt != stlReaders.end(); ++stlIt ) {
	    OcpIp::IStlReader* pReader = *stlIt;
	    string legalName = pReader->getFileName();
	    assert ( !legalName.empty() );
	    for ( string::iterator it = legalName.begin(); it != legalName.end(); ++it ) {
		if ( *it == '.' ) *it = '_';
	    }
	    m_streams.push_back( new StlStream( legalName.c_str(), *this ) );
	    m_streams.back()->pStl = NULL; // to be replaced later
	    delete pReader;
	}
    }
    m_socket.register_configuration_listener_callback(this, &generic_stl_master_tl3<BUSWIDTH>::when_config_known);
}

template<unsigned int BUSWIDTH>
void
ocpip_example::generic_stl_master_tl3<BUSWIDTH>::before_end_of_elaboration()
{
    if ( m_period == sc_core::SC_ZERO_TIME ) {
	std::cerr << "STL master " << name()
		  << " requires a non-zero clock period to interpret STL time stamps." << std::endl;
	abort();
    }
}

template<unsigned int BUSWIDTH>
void
ocpip_example::generic_stl_master_tl3<BUSWIDTH>::when_config_known(
    const ocpip::ocp_parameters& config, const std::string& )
{
    using namespace ocpip;
    using namespace ocpip_legacy;

    OCPParameters params = config.template legacy_conversion<ocp_data_class_unsigned<BUSWIDTH,32> >();
    std::deque<OcpIp::IStlReader*> stlReaders = OcpIp::IStlReader::findStlPrograms( basename(), false, params );

    // must match the number of expected STL streams determined at construction
    assert( m_streams.size() == stlReaders.size() );
    typename std::vector<StlStream*>::iterator streamIt  = m_streams.begin();
    for ( std::deque<OcpIp::IStlReader*>::iterator stlIt = stlReaders.begin();
	  stlIt != stlReaders.end(); ++stlIt, ++streamIt ) {
	assert( streamIt != m_streams.end() );
	(*streamIt)->pStl = *stlIt;
    }
}

// ----------------------------------------------------------------------------
// Destructor
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH>
ocpip_example::generic_stl_master_tl3<BUSWIDTH>::~generic_stl_master_tl3() {}

// ----------------------------------------------------------------------------
//  Method : Master::proc()
//
//  Synchronous Master process
//
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH>
ocpip_example::generic_stl_master_tl3<BUSWIDTH>::StlStream::StlStream(
    sc_core::sc_module_name name_, ocpip_example::generic_stl_master_tl3<BUSWIDTH>& master ):
    sc_core::sc_module( name_ )
    , master( master )
    , pStl( NULL )
{
    SC_THREAD(proc);
}

template<unsigned int BUSWIDTH>
void
ocpip_example::generic_stl_master_tl3<BUSWIDTH>::StlStream::proc() {

    assert( pStl != NULL );
    using namespace ocpip_legacy::OcpIp;
    ocpip::ocp_master_socket<BUSWIDTH>&   testP    = master.m_socket;

    // convert all addresses to 64 bits
    typedef typename ocpip::unsigned_type<BUSWIDTH>::type Td;
    // typedef uint64_t Ta;
    typedef unsigned long long Ta;
    pStl->IStlReader::setConversionTypes<Td, Ta>();

    while ( true ) {
	bool stleof;
	pStl->next( stleof );
	if ( stleof )
	    break;

        StlCommandType cType = pStl->getCommandType();
        if ( cType == STL_INVALID ) continue;

        // wait for time stamp
	sc_core::sc_time earliest = pStl->getCommandDelay() * master.m_period;
	sc_core::sc_time now      = sc_core::sc_time_stamp();
	if ( now < earliest ) {
	    wait ( earliest - now );
	}

        switch ( cType )
        {
        case STL_TRANSFER:
        {
	    typedef StlTransferCommand<Td, Ta> StlTransfer;
            StlTransfer transfer = pStl->IStlReader::getTransferCommand<Td,Ta>();

            if ( transfer.request.MCmd == ocpip_legacy::OCP_MCMD_IDLE ) {
		if ( transfer.idleCycles > 0 ) {
		    wait( transfer.idleCycles * master.m_period );
		}
	    } else {
		tlm::tlm_generic_payload* txn = testP.get_transaction();
		ocpip::ocp_parameters params = testP.get_ocp_config();
		set_txn_cmd( testP, *txn, (ocpip::mcmd_codes)transfer.request.MCmd );
		txn->set_address( transfer.request.MAddr );
		// data length
		uint32_t total_length = transfer.request.MBurstLength;
		if ( params.burstseq_blck_enable && transfer.request.MBurstSeq == ocpip_legacy::OCP_MBURSTSEQ_BLCK )
		    total_length *= transfer.request.MBlockHeight;
		assert( total_length == transfer.request.DataLength && "TL2 chunks not supported yet" );
		if ( params.burstprecise && params.burstlength && transfer.request.MBurstPrecise == 0 ) {
		    // take a guess: 8 times max representable burst
		    total_length = ( 1 << params.burstlength_wdth ) << 3;
		    testP.template validate_extension<ocpip::imprecise>(*txn);
		}
		txn->set_data_length( total_length * ((BUSWIDTH+7)>>3) );
		testP.reserve_data_size(*txn, ((BUSWIDTH+7)>>3)*total_length);
		txn->set_streaming_width(txn->get_data_length());
		if ( txn->is_write() ) {
		    memcpy( txn->get_data_ptr(), transfer.request.MDataPtr, transfer.request.DataLength*((BUSWIDTH+7)>>3 ) );
		    // unsafe: txn->set_data_ptr( reinterpret_cast<unsigned char*>( transfer.request.MDataPtr ) );
		}
		if ( params.burstlength ) {
		    if ( params.burstseq_strm_enable && transfer.request.MBurstSeq == ocpip_legacy::OCP_MBURSTSEQ_STRM )
			txn->set_streaming_width((BUSWIDTH+7)>>3);
		    else if ( params.burstseq_blck_enable && transfer.request.MBurstSeq == ocpip_legacy::OCP_MBURSTSEQ_BLCK )
			set_txn_block_burst( testP, *txn,
					     transfer.request.MBurstLength,
					     transfer.request.MBlockHeight,
					     transfer.request.MBlockStride );
		    else {
			ocpip::burst_sequence* b_seq = set_txn_row_burst( testP, *txn,
									  transfer.request.MBurstLength,
									  (ocpip::burst_seqs)transfer.request.MBurstSeq );
			if ( transfer.request.MBurstSeq == ocpip_legacy::OCP_MBURSTSEQ_XOR ||
			     transfer.request.MBurstSeq == ocpip_legacy::OCP_MBURSTSEQ_WRAP ) {
			    // alter the txn address to the burst aligned base
			    b_seq->value.xor_wrap_address = transfer.request.MAddr;
			    uint64_t base_address = transfer.request.MAddr;
			    uint32_t lsb = ceilLog2( transfer.request.MBurstLength * ((BUSWIDTH+7)>>3) );
			    fillBitRange( base_address, lsb-1, 0, 0 );
			    txn->set_address( base_address );
			}
		    }

		}
		if ( params.addrspace )
		    set_txn_address_space( testP, *txn, transfer.request.MAddrSpace );
		if ( params.atomiclength )
		    set_txn_atomic_length( testP, *txn, 0 ); // no MAtomicLength at TL2
		if ( params.threads > 1 )
		    set_txn_thread_id( testP, *txn, transfer.request.MThreadID );
		if ( params.tags > 1 )
		    set_txn_tag_id( testP, *txn, transfer.request.MTagID );
		if ( params.connid )
		    set_txn_conn_id( testP, *txn, transfer.request.MConnID );
		if ( params.burstsinglereq && transfer.request.MBurstSingleReq  )
		    testP.template validate_extension<ocpip::srmd>(*txn);

		// Enqueue it
		tlm::tlm_phase begin_req=tlm::BEGIN_REQ;
		sc_core::sc_time zero_time;
		testP->nb_transport_fw( *txn, begin_req, zero_time );
#ifdef DEBUG_G1
		std::cout << "Master queued request " << req->get_command()
			  << " time " << sc_core::sc_time_stamp().to_seconds();
		if (req->get_command() == tlm::TLM_WRITE_COMMAND) {
		    std::cout << " data " << (*((Td*)(req->get_data_ptr())));
		}
		std::cout << std::endl;
#endif
		wait( master.m_period );
	    }
        }
        break;
        case STL_WAIT:
        case STL_SIGNAL:
        case STL_RESET:
	    std::cerr << "Simple master only supports STL transfer commands" << endl;
            break;
        default:
            assert( false );
        }
    }
} // end of method

template<unsigned int BUSWIDTH>
tlm::tlm_sync_enum
ocpip_example::generic_stl_master_tl3<BUSWIDTH>::nb_transport(
    tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time&)
{
    if ( ph == tlm::BEGIN_RESP ) {
	m_socket.release_transaction( &txn );
    } else {
	std::cerr<<"Error in "<<name()<<" : got unexpected phase "<<ph<<" in nb_transport_bw"<<std::endl;
	std::cerr<<"Only BEGIN_RESP is supported."<<std::endl;
	exit(1);
    }
    return tlm::TLM_ACCEPTED;
}

