// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//                Stphane Guntz, PROSILOG, guntz@prosilog.com
//         Date : 06/2003
//
//  Description : Transaction Level - Layer-1 to Layer-2 Master Adapter

//    
//  Parameter    :
//    Template arguments.
//     - TdataCl_tl1: TL1 Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//	   - TdataCl_tl2: TL2 Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//     - Td: Data type 
//     - Ta: Address type
//                
//	Constructor arguments:
//      - sc_module_name: Name of the module instance
//	    - max_burst_length: Maximum size of burst length
//		- adapter_depth: Number of missed events and responses to be cached.
//						 The adapter stores new request/response infos which have arrived during the processing of the current request/response
//						 and sends them when the current request/response is finished
//	 
//	 
//  The TL1 master adapter has a TL2 master interface on one side, and a TL1 slave interface on the other side.
//  It must be connected to:
//      - a TL1 master through a TL1 channel
//      - a TL2 slave through a TL2 channel
//
//	The adapter retrieves TL1 requests and stores them in its internal buffer. When a new request arrived, the adapter
//	compares it with former requests to decide whether it is part of the current burst or the beginning of a new burst
//  (it can be a single request too).
//  When a burst is finished, a TL2 request is sent to the slave.
//  
//  On the response side, the TL1 master adapter retrieves the TL2 response from the slave, and sends corresponding
//  TL1 responses to the master. 
// 
// ============================================================================

#include "ocp_tl1_tl2_master_adapter.h"


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter constructor
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::OCP_TL1_TL2_Master_Adapter(sc_module_name name_, int max_burst_length_, int adapter_depth_): 
	sc_module(name_),
    max_burst_length(max_burst_length_),
	adapter_depth(adapter_depth_)
{

	buffer_size=max_burst_length*adapter_depth;

	m_request_buffer=new request_parameters[buffer_size];
	m_response_buffer= new response_parameters[adapter_depth];
	m_data=new Td_tl1[buffer_size];
	m_response_data= new Td_tl2[buffer_size];
	
	//arrays for storing characteristics of TL2 requests
	burst_request_index=new int[adapter_depth];
	burst_request_length=new int[adapter_depth];

	//arrays for storing characteristics of TL1 responses
	response_index=new int[adapter_depth];
	response_length=new int[adapter_depth];

	//initialize internal request, data and response buffer
	for(int i=0; i<buffer_size; i++)  {
		reset_request_parameters(i);
		m_data[i]=0;
		m_response_data[i]=0;
	}

	beginning_burst_index= current_index=0;

	for(int i=0; i<adapter_depth; i++)   {
		burst_request_index[i]=0;
		burst_request_length[i]=0;
		response_index[i]=0;
		response_length[i]=0;
	}
	last_datum_sent=false;
	release_request_thread=false;
	release_response_thread=false;
	m_pending_request=0;
	m_request_number=0;
	m_pending_response=0;
	m_response_number=0;
	current_processed_request=current_processed_response=0;

	m_NumBytesPerWord = sizeof(Td_tl1);

	//SystemC processes 
	SC_THREAD(SlaveRequest);
	sensitive<<SlaveP;

	SC_THREAD(MasterRequest);
	sensitive<<send_TL2_request_event;

	SC_THREAD(SlaveResponse);

	SC_THREAD(MasterResponse);
	sensitive<<send_TL1_response_event;

}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter destructor
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::~OCP_TL1_TL2_Master_Adapter()    {
	
	delete [] m_request_buffer;
	delete [] m_response_buffer;
	delete [] m_response_data;
	delete [] m_data;
	delete [] burst_request_index;
	delete [] burst_request_length;
	delete [] response_index;
	delete [] response_length;
}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::SlaveRequest 
//get the TL1 requests, store them in the internal buffer, prepare TL2 request to be sent to the TL2 slave
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::SlaveRequest()   {


	while(true)   {

		wait();			//wait for default event of the channel
		
		if(SlaveP->SgetRequest(false))   {  //get a pending request

			//store the request fields in the internal buffer
			m_request_buffer[current_index].m_MAddr=		m_DataCl_tl1->SgetMAddr();
			m_request_buffer[current_index].m_MAddrSpace=	m_DataCl_tl1->SgetMAddrSpace();
			m_request_buffer[current_index].m_MBurst=		m_DataCl_tl1->SgetMBurst();
			m_request_buffer[current_index].m_MByteEn=	m_DataCl_tl1->SgetMByteEn();
			m_request_buffer[current_index].m_MCmd=		m_DataCl_tl1->SgetMCmd();
			m_request_buffer[current_index].m_MConnID=	m_DataCl_tl1->SgetMConnID();
			m_request_buffer[current_index].m_MThreadID=	m_DataCl_tl1->SgetMThreadID();
		
			
			#ifdef DEBUG_MASTER_ADAPTER_TL1   
				cout<<"---\nSlaveRequest(): master adapter gets a TL1 request at time "<<sc_time_stamp()<<endl;
				cout<<"MAddr: "<<m_request_buffer[current_index].m_MAddr<<endl;
				cout<<"MBurst: "<<m_request_buffer[current_index].m_MBurst<<endl;
				cout<<"MThreadID: "<<m_request_buffer[current_index].m_MThreadID<<endl;
				cout<<"MCmd: "<<m_request_buffer[current_index].m_MCmd<<"\n---"<<endl;
			#endif

			//store the data in the internal buffer
			m_data[current_index]=m_DataCl_tl1->SgetMData();

			switch(define_adapter_task())   {

			case SEND_BURST_WITH_DATA:
				//specify burst length and beginning
				burst_request_index[m_request_number]=beginning_burst_index;
				burst_request_length[m_request_number]=abs(current_index-beginning_burst_index+1); //use the abs function because of circular buffer

				//change the internal index
				beginning_burst_index=current_index+1;

				//the following transaction is the beginning of a new burst
				new_burst_transaction=true;

				//last datum of the burst is not part of the burst
				if(m_request_buffer[m_request_number].m_MBurst==OCP_MBURST_LAST)
					last_datum_sent=true;
				else last_datum_sent=false;

				//increment number of pending requests that the adapter has to send
				m_pending_request++;
				//increment number of the next request that is going to be stored
				//check if the request buffer is full: next request is currently being processed
				if((m_request_number+1)%adapter_depth==current_processed_request)
					release_request_thread=true;
				else {
					release_request_thread=false;
					m_request_number=(m_request_number+1)%adapter_depth;
					SlaveP->Srelease();
				}
				
				//notify request event
				send_TL2_request_event.notify();	
				break;

			case SEND_BURST_WITHOUT_DATA:
				//specify burst length and beginning
				burst_request_index[m_request_number]=beginning_burst_index;
				burst_request_length[m_request_number]=abs(current_index-beginning_burst_index); //use the abs function because of circular buffer

				//change the internal index
				beginning_burst_index=current_index;

				//the following transaction is the beginning of a new burst
				new_burst_transaction=true;

				//last datum of the burst is not part of the burst
				last_datum_sent=false;

				//increment number of pending requests that the adapter has to send
				m_pending_request++;
				//increment number of the next request that is going to be stored
				//check if the request buffer is full: next request is not currently being processed
				if((m_request_number+1)%adapter_depth==current_processed_request)
					release_request_thread=true;
				else {
					release_request_thread=false;
					m_request_number=(m_request_number+1)%adapter_depth;
					SlaveP->Srelease();
				}
				
				//notify request event
				send_TL2_request_event.notify();
				break;

			case STORE_NEW_DATA:
				new_burst_transaction=false;
				SlaveP->Srelease();
				release_request_thread=false;
				break;

			case SEND_BURST_AND_SINGLE_REQUEST:
				//specify the characteristics of the current burst
				//specify burst length and beginning
				burst_request_index[m_request_number]=beginning_burst_index;
				burst_request_length[m_request_number]=abs(current_index-beginning_burst_index); //use the abs function because of circular buffer

				//change the internal index
				beginning_burst_index=current_index;
				//change the internal index
				if(current_index!=buffer_size-1)
					current_index++;
				else current_index=0;

				//increment number of pending requests that the adapter has to send
				m_pending_request++;

				//increment number of the next request that is going to be stored
				//check if the request buffer is full: next request is not currently being processed
				if((m_request_number+1)%adapter_depth==current_processed_request)    {
					release_request_thread=true;
					send_TL2_request_event.notify();
					wait(thread_released_event);  //wait for the current request to be finished to store the characteristics of the next request
				}
				else {
					release_request_thread=false;
					m_request_number=(m_request_number+1)%adapter_depth;
				}

				//continue with single request
				burst_request_index[m_request_number]=beginning_burst_index;
				burst_request_length[m_request_number]=1; //use the abs function because of circular buffer

				//change the internal index
				beginning_burst_index=current_index+1;

				//the following transaction is the beginning of a new burst
				new_burst_transaction=true;

				//last datum of the burst is part of the burst
				last_datum_sent=true;

				//increment number of pending requests that the adapter has to send
				m_pending_request++;

				//increment number of the next request that is going to be stored
				//check if the request buffer is full: next request is not currently being processed
				if((m_request_number+1)%adapter_depth==current_processed_request)
					release_request_thread=true;
				else {
					release_request_thread=false;
					m_request_number=(m_request_number+2)%adapter_depth;
					SlaveP->Srelease();
				}
				
				//notify request event
				send_TL2_request_event.notify();
				break;

			default:
				break;
			}		

			//change the internal index
			if(current_index!=buffer_size-1)
				current_index++;
			else current_index=0;

		}  //get a pending request
	} //end of infinite loop
}  //end of SlaveRequest


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::MasterRequest
//send TL2 request made of atomic TL1 requests
//
//---------------------------------------------------------------------------
template< class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::MasterRequest()   {
	
	while(true)   {
		
		if(m_pending_request==0)
			wait();  //wait for notification of send_TL2_request_event

		//put data in the TL2 data class
		m_DataCl_tl2->MputMData( &m_data[burst_request_index[current_processed_request]], burst_request_length[current_processed_request], last_datum_sent);
		//put request fields into the TL2 data class
		put_TL2_request_fields(m_request_buffer[burst_request_index[current_processed_request]]);

		//check if the request is a write or a read request
		switch(m_request_buffer[burst_request_index[current_processed_request]].m_MCmd)   {

			case OCP_MCMD_WR:
				#ifdef DEBUG_MASTER_ADAPTER_TL1
					cout<<"\n**MasterRequest: send a TL2 write request at time t="<<sc_time_stamp()<<endl;
				#endif
				MasterP->MputWriteRequestBlocking();
				current_processed_request=(current_processed_request+1)%adapter_depth;

				//release the request thread, if it was not released before
				if(release_request_thread)  {
					m_request_number=(m_request_number+1)%adapter_depth;
					SlaveP->Srelease();
					release_request_thread=false;
					thread_released_event.notify();
				}
				break;

			case OCP_MCMD_RD:
				#ifdef DEBUG_MASTER_ADAPTER_TL1
					cout<<"\n**MasterRequest: send a TL2 read request at time t="<<sc_time_stamp()<<endl;
				#endif
				MasterP->MputReadRequestBlocking();
				current_processed_request=(current_processed_request+1)%adapter_depth;

				//release the request thread, if it was not released before
				if(release_request_thread)  {
					m_request_number=(m_request_number+1)%adapter_depth;
					SlaveP->Srelease();
					release_request_thread=false;
					thread_released_event.notify();
				}
				break;

			case OCP_MCMD_RDEX:
				#ifdef DEBUG_MASTER_ADAPTER_TL1
					cout<<"\n**MasterRequest: send an exclusive TL2 read request at time t="<<sc_time_stamp()<<endl;
				#endif
				MasterP->MputReadRequestBlocking();
				current_processed_request=(current_processed_request+1)%adapter_depth;

				//release the request thread, if it was not released before
				if(release_request_thread)  {
					m_request_number=(m_request_number+1)%adapter_depth;
					SlaveP->Srelease();
					release_request_thread=false;
					thread_released_event.notify();
				}
				break;

			default:
				cout<<"++++\ndefault\n++++"<<endl;
				break;
		} //end of switch
				
		if(m_pending_request!=0)
			m_pending_request--;

	} //end of infinite loop

} //end of MasterRequest method



//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::SlaveResponse
//get the TL2 response from the slave
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::SlaveResponse()   {

	Td_tl2* temp;

	while(true)    {
		//get TL2 response from the slave 
		if(MasterP->MgetResponseBlocking(false))   {
			#ifdef DEBUG_MASTER_ADAPTER_TL1
				cout<<"\n---Slave has sent a TL2 response at time t="<<sc_time_stamp()<<endl;
			#endif

			//copy response fields from TL2 data class to adapter
			m_response_buffer[m_response_number].m_SResp=m_DataCl_tl2->MgetSResp();
			m_response_buffer[m_response_number].m_SThreadID=m_DataCl_tl2->MgetSThreadID();

			//response
			#ifdef DEBUG_MASTER_ADAPTER_TL1
				cout<<"TL2 response to a read request:"<<endl;
				cout<<"SResp= "<<m_response_buffer[m_response_number].m_SResp<<endl;
				cout<<"SThreadID= "<<m_response_buffer[m_response_number].m_SThreadID<<"\n***"<<endl;
			#endif
			
			//copy the response data from the TL2 data class to the adapter
			if(m_response_number==0)
				response_index[m_response_number]=0;
			else response_index[m_response_number]=response_index[m_response_number-1]+response_length[m_response_number-1];

			temp = m_DataCl_tl2->MgetSData(response_length[m_response_number]);
			for(int i=0; i<response_length[m_response_number]; i++)
				m_response_data[i+response_index[m_response_number]] =temp[i];

			send_TL1_response_event.notify();

			//increment number of pending responses that the adapter has to send
			m_pending_response++;

			//increment next response number
			if((m_response_number+1)%adapter_depth==current_processed_response)
				release_response_thread=true;
			else {
				release_response_thread=false;
				m_response_number=(m_response_number+1)%adapter_depth;
				//release response channel
				MasterP->Mrelease();
			}
			
		}  //end of get Response
	} //end of infinite loop
} //end of SlaveResponse method


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::MasterResponse
//send the TL1 response to the master, made from the TL2 response from the slave 
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::MasterResponse()   {

	while(true)    {
		if(m_pending_response==0)
			wait();  //wait for notification of send_TL1_response_event

		//copies response fields into TL1 data class
		m_DataCl_tl1->SputSResp(m_response_buffer[current_processed_response].m_SResp);
		m_DataCl_tl1->SputSThreadID(m_response_buffer[current_processed_response].m_SThreadID);

		//send atomic TL1 responses
		for(int i=response_index[current_processed_response]; i<response_index[current_processed_response]+response_length[current_processed_response]; i++)  {
			m_DataCl_tl1->SputSData(m_response_data[i]);

			#ifdef DEBUG_MASTER_ADAPTER_TL1
				cout<<"***TL1 response at time= "<<sc_time_stamp()<<endl;
				cout<<"SData= "<<m_response_data[i]<<"\n***"<<endl;
			#endif

			SlaveP->SputResponse();

			wait(clk->posedge_event());  //wait for next clock edge to send next TL1 response;
		}
		current_processed_response=(current_processed_response+1)%adapter_depth;

		//release response_thread if it not was done before
		if(release_response_thread)
			MasterP->Mrelease();

		if(m_pending_response!=0)
			m_pending_response--;

	} //end of infinite loop
} //end of MasterResponse method



//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::define_adapter_task
//
//---------------------------------------------------------------------------
template< class TdataCl_tl1, class TdataCl_tl2>
AdapterTask OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::define_adapter_task()   {
	
	//compare with request parameters in the internal buffer
	if( !new_burst_transaction)   {

		bool same_fields=same_request_fields(m_request_buffer[current_index], m_request_buffer[beginning_burst_index]);
		bool same_burst=same_burst_field(m_request_buffer[current_index], m_request_buffer[beginning_burst_index]);

		//the request fields are the same, and address increment is right		
		if( same_fields && is_address_increment_right(m_request_buffer[current_index], m_request_buffer[beginning_burst_index]) )  {
			//datum is last of burst: send the completed burst
			if(m_request_buffer[current_index].m_MBurst==OCP_MBURST_LAST) 
				return SEND_BURST_WITH_DATA;
			else {
				//the burst field is the same
				if(same_burst)
					//completed burst has maximum length: send it
					if(abs(current_index-beginning_burst_index+1)==max_burst_length)
						return SEND_BURST_WITH_DATA;
					else return STORE_NEW_DATA;
				//burst field has changed and is not last of burst: send the current burst without the new datum
				else return SEND_BURST_WITHOUT_DATA;
			}
		}
		//the fields have changed, or the address increment is wrong
		else {
			//the request is OCP_MBURST_LAST: the adapter has to send 2 requests (current burst+ single last request)
			if(m_request_buffer[current_index].m_MBurst==OCP_MBURST_LAST) 
				return SEND_BURST_AND_SINGLE_REQUEST;
			//or send the current burst and store the new request
			else return SEND_BURST_WITHOUT_DATA;
		}
	}
	//new burst transaction
	else  {
		//datum is the last of a burst (burst of 1 packet!)
		if(m_request_buffer[current_index].m_MBurst==OCP_MBURST_LAST)
			return SEND_BURST_WITH_DATA;
		//or store it
		else return STORE_NEW_DATA;
	}		
}



//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::put_TL2_request_fields
//
//---------------------------------------------------------------------------
template< class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::put_TL2_request_fields(request_parameters& req_param)   {
	m_DataCl_tl2->MputMAddr(req_param.m_MAddr);
	m_DataCl_tl2->MputMAddrSpace(req_param.m_MAddrSpace);
	m_DataCl_tl2->MputMBurst(req_param.m_MBurst);
	m_DataCl_tl2->m_MputMByteEn(req_param.m_MByteEn);
	m_DataCl_tl2->MputMCmd(req_param.m_MCmd);
	m_DataCl_tl2->MputMConnID(req_param.m_MConnID);
	m_DataCl_tl2->MputMThreadID(req_param.m_MThreadID);

} 
	


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::same_request_fields
//
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
bool OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::same_request_fields(request_parameters& new_req_param, request_parameters& current_req_param )  {

	//check if the fields have changed
	if(new_req_param.m_MAddrSpace != current_req_param.m_MAddrSpace)
		return false;
	else if(new_req_param.m_MByteEn != current_req_param.m_MByteEn)
		return false;
	else if(new_req_param.m_MCmd != current_req_param.m_MCmd)
		return false;
	else if(new_req_param.m_MConnID != current_req_param.m_MConnID)
		return false;
	else if(new_req_param.m_MThreadID != current_req_param.m_MThreadID)
		return false;
	
	//fields have not changed:
	return true;

}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::same_burst_field
//
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
bool OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::same_burst_field(request_parameters& new_req_param, request_parameters& current_req_param )  {

	if(new_req_param.m_MBurst != current_req_param.m_MBurst)
		return false;
	else return true;

}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::is_address_increment_right
//
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
bool OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::is_address_increment_right(request_parameters& new_req_param, request_parameters& current_req_param )  {

	//if burst is a streaming burst, verify if the address has not changed
	if(new_req_param.m_MBurst==OCP_MBURST_STRM)  {	
		if( ((current_req_param.m_MBurst==OCP_MBURST_STRM) || (current_req_param.m_MBurst==OCP_MBURST_LAST)) 
			&& (new_req_param.m_MAddr ==current_req_param.m_MAddr))
			return true;
		else return false;
	}
	//if burst is a incremental burst, verify if the address increment has changed
	else if( (new_req_param.m_MBurst==OCP_MBURST_TWO)
		|| (new_req_param.m_MBurst==OCP_MBURST_FOUR)
		|| (new_req_param.m_MBurst==OCP_MBURST_EIGHT)
		|| (new_req_param.m_MBurst==OCP_MBURST_CONT)
		|| (new_req_param.m_MBurst==OCP_MBURST_LAST))  {
		if(new_req_param.m_MAddr == current_req_param.m_MAddr+(current_index-beginning_burst_index)*m_NumBytesPerWord)
			return true;
		else return false;
	}

	//in the other cases (custom bursts), send as single TL2 requests 
	return false;
}


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter::reset_request_parameters
//reset request parameters
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::reset_request_parameters(int i)   {
	m_request_buffer[i].m_MAddr=		0;
	m_request_buffer[i].m_MAddrSpace=	0;
	m_request_buffer[i].m_MBurst=		OCP_MBURST_LAST;
	m_request_buffer[i].m_MByteEn=	0;
	m_request_buffer[i].m_MCmd=		OCP_MCMD_IDLE ;
	m_request_buffer[i].m_MConnID=	0;
	m_request_buffer[i].m_MThreadID=	0;
}



// ----------------------------------------------------------------------------
//  Method : OCP_TL1_TL2_Master_Adapter::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------

template<class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::end_of_elaboration()
{

  sc_module::end_of_elaboration();

  // Initialize debug interface
  MasterP->MregisterDirectIF((MdirectIF<TdataCl_tl2>*)(this));
  SlaveP->SregisterDirectIF((SdirectIF<TdataCl_tl1>*)(this));


  // Get data structure
  m_DataCl_tl1 = SlaveP->GetDataCl();
  m_DataCl_tl2 = MasterP->GetDataCl();


  // Get communication structure
  m_CommCl_tl1 = SlaveP->GetCommCl();
  m_CommCl_tl2 = MasterP->GetCommCl();


  // Get system parameter structure
  m_ParamCl_tl1 = SlaveP->GetParamCl();
  m_ParamCl_tl2 = MasterP->GetParamCl();


  // Put parameter into Channel
  // No parameters needed

}

// ----------------------------------------------------------------------------
//  Method : OCP_TL1_TL2_Master_Adapter::MPutDirect()
//  Debug interface method.
//  Read/Write data, Master to Slave direction
//  Returns true for success and false for failure
//
//	calls Slave's MputDirect function
//
// ----------------------------------------------------------------------------

template<class TdataCl_tl1, class TdataCl_tl2>
bool OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::MputDirect(
    int MasterID, bool IsWrite, Td_tl2 *Data, Ta_tl2 Address, int NumWords)

{
  return(SlaveP->MputDirect(MasterID, IsWrite, Data, Address, NumWords));

}


//----------------------------------------------------------------------------
//  Method : OCP_TL1_TL2_Master_Adapter::SPutDirect()
//  Debug interface method.
//  Read/Write data, Slave to Master direction
//
//	calls Master's SputDirect function
//
// ----------------------------------------------------------------------------

template<class TdataCl_tl1, class TdataCl_tl2>
bool OCP_TL1_TL2_Master_Adapter<TdataCl_tl1, TdataCl_tl2>::SputDirect(
         int SlaveID, bool IsWrite, Td_tl1 *Data, Ta_tl1 Address, int NumWords)
{
	return(MasterP->SputDirect(SlaveID, IsWrite, Data, Address, NumWords));	
}


// ----------------------------------------------------------------------------
//
//  Instantiation of the adapter
//
// ----------------------------------------------------------------------------

template class OCP_TL1_TL2_Master_Adapter<TL1_TEMPL_DATA_CL, OCP_TL2_TEMPL_DATA_CL >; // see ocp_tl1_globals.h and ocp_tl2_globals.h

