// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//                Stphane Guntz, PROSILOG, guntz@prosilog.com
//         Date : 06/2003
//
//  Description : Transaction Level - Layer-1 to Layer-2 Master Adapter

//    
//  Parameter    :
//    Template arguments.
//     - TdataCl_tl1: TL1 Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//	   - TdataCl_tl2: TL2 Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//     - Td: Data type
//     - Ta: Address type
//                
//   Constructor arguments:
//      - sc_module_name: Name of the module instance
//	    - max_burst_length: Maximum size of burst length
//		- adapter_depth: Number of missed events and responses to be cached.
//						 The adapter stores new request/response infos which have arrived during the processing of the current request/response
//						 and sends them when the current request/response is finished
//	 
//	 
//  The TL1 master adapter has a TL2 master interface on one side, and a TL1 slave interface on the other side.
//  It must be connected to:
//      - a TL1 master through a TL1 channel
//      - a TL2 slave through a TL2 channel
//
//	The adapter retrieves TL1 requests and stores them in its internal buffer. When a new request arrived, the adapter
//	compares it with former requests to decide whether it is part of the current burst or the beginning of a new burst
//  (it can be a single request too).
//	When a burst is finished, a TL2 request is sent to the slave.
//  
//  On the response side, the TL1 master adapter retrieves the TL2 response from the slave, and sends corresponding
//  TL1 responses to the master. 
// 
// ============================================================================

#ifndef _OCP_TL1_TL2_MASTER_ADAPTER_H
#define _OCP_TL1_TL2_MASTER_ADAPTER_H

#include "ocp_tl1_globals.h"
#include "ocp_tl2_globals.h"
#include "tl_slave_if.h"
#include "tl_master_if.h"
#include "ocp_tl1_data_cl.h"
#include "ocp_tl2_data_cl.h"

//for debugging purposes, print messages on screen
//comment for no messages
//#define DEBUG_MASTER_ADAPTER_TL1


#ifndef ADAPTER_TASK_ENUM
#define ADAPTER_TASK_ENUM

enum AdapterTask   {
	SEND_BURST_WITH_DATA = 0,
	SEND_BURST_WITHOUT_DATA,
	STORE_NEW_DATA,
	SEND_BURST_AND_SINGLE_REQUEST
};

#endif


template <class TdataCl_tl1, class TdataCl_tl2> class OCP_TL1_TL2_Master_Adapter
  : public sc_module
   , public MdirectIF<TdataCl_tl1>
   , public SdirectIF<TdataCl_tl2>
{
public:  
  //TL1 data type and address type
  typedef typename TdataCl_tl1::DataType Td_tl1;
  typedef typename TdataCl_tl1::AddrType Ta_tl1;

  //TL2 data type and address type
   typedef typename TdataCl_tl2::DataType Td_tl2;
  typedef typename TdataCl_tl2::AddrType Ta_tl2;
  

  //internal structure for data and request fields
  struct response_parameters  {
	OCPSRespType	m_SResp;
	int			    m_SThreadID;
  };

  struct request_parameters  {
	Ta_tl1			m_MAddr;
	int				m_MAddrSpace;
	OCPMBurstType	m_MBurst;
	int				m_MByteEn;
	OCPMCmdType		m_MCmd;
	int				m_MConnID;
	int			    m_MThreadID;
  };


  // TL1 Slave Port and TL2 Master port
  sc_port<TLslaveIF<TdataCl_tl1>, 1> SlaveP;
  sc_port<TLmasterIF<TdataCl_tl2>, 1> MasterP;

  //clock port
  sc_in_clk clk;

  SC_HAS_PROCESS(OCP_TL1_TL2_Master_Adapter);

  // constructor
  OCP_TL1_TL2_Master_Adapter(sc_module_name name, int max_burst_length, int adapter_depth);

  // destructor
  ~OCP_TL1_TL2_Master_Adapter();

  // SystemC processes
  void MasterRequest();
  void SlaveRequest();
  void MasterResponse();
  void SlaveResponse();


  //direct interfaces methods
  virtual bool MputDirect(int, bool, Td_tl2*, Ta_tl2, int);
  virtual bool SputDirect(int, bool, Td_tl1*, Ta_tl1, int);

  //adapter methods
  void reset_request_parameters(int);
  bool same_request_fields(request_parameters&, request_parameters&);
  bool same_burst_field(request_parameters&, request_parameters&);
  bool is_address_increment_right(request_parameters&, request_parameters&);
  void put_TL2_request_fields(request_parameters&);
  AdapterTask define_adapter_task(void);

 private :
  void end_of_elaboration();

  // pointer to the TL1 and TL2 data structures of the channel
  TdataCl_tl1          *m_DataCl_tl1;
  TdataCl_tl2          *m_DataCl_tl2;
  
  //pointer to param class
  ParamCl<TdataCl_tl1> *m_ParamCl_tl1;
  ParamCl<TdataCl_tl2> *m_ParamCl_tl2;

  //pointer to communication class
  CommCl           *m_CommCl_tl1;
  CommCl           *m_CommCl_tl2;

  // parameters given as constructor parameters
  int max_burst_length;        //maximum length of a burst transaction
  int buffer_size;             //size of the internal buffer for data and request fields (MCmd, MAddr, MAddrSpace...)
  int adapter_depth;		  //depth of requests and response storage 

  //internal parameters
  bool new_burst_transaction;  //indicates if the next transaction is the beginning of a new transaction
  bool last_datum_sent;		   // last datum of the burst is sent with the burst
  int m_NumBytesPerWord;	   // size of the data
  
  //indicates a later release of the request or response thread, when the internal buffer for storing request/response is full
  bool release_request_thread; 
  bool release_response_thread; 
  
  //index of the currently processed request/response
  int current_processed_request;
  int current_processed_response;
  
  int beginning_burst_index, current_index;   //index for internal buffers
  
  //buffers for storing beginning index and length of request/response
  int* burst_request_index;		//beginning index of the TL2 request
  int* burst_request_length;	//length of the TL2 request
  int* response_index;	//beginning index of the TL1 response
  int* response_length;	//number of TL1 responses to send

  int m_pending_request;   //indicates if there is a pending request which has arrived, during the processing of the current request
  int m_request_number;	   //ID of the request currently processed
  int m_pending_response;  //indicates if there is a pending response which has arrived, during the processing of the current request
  int m_response_number;   //ID of the response currently processed

  //internal buffers
  request_parameters* m_request_buffer;  //for storing TL1 request fields
  response_parameters* m_response_buffer; //for storing TL2 response fields
  Td_tl1* m_data;							//for storing data
  Td_tl2* m_response_data;					//for storing response
  
  //events
  sc_event send_TL2_request_event;	//event for sending TL2 request, made of TL1 atomic requests
  sc_event send_TL1_response_event; //event for sending TL1 responses, made of TL2 response
  sc_event thread_released_event;   //event indicating the release of the request thread

};


#endif // _OCP_TL1_TL2_MASTER_ADAPTER_H
