// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level 
//      Authors : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//                Anssi Haverinen, Nokia Inc., anssi.haverinen@nokia.com
//                Joe Chou, Sonics Inc., joechou@sonicsinc.com 
//                Alan Kamas, www.kamas.com
//         Date : 06/12/2003
//
//  Description : Transaction Level - Layer-1 data class for
//                the Open Core Protocol (OCP) 1.0
//                The data class can be used in conjunction with
//                OCP transaction channel (tl_channel.h) to transport
//                data between Initiator and Target modules.
//                Contains member variables and access methods
//                to set or read the member variables.
//                The class contains members for transporting all OCP signals
//
//                The members are split in to four groups:
//                - Request-Channel members (set by Master read by Slave)
//                - Data Request-Channel members (set by Master read by Slave)
//                - Response-Channel members (set by Slave, read by Master)
//                - Sideband channel members
//
//                See definition of these groups in OCP 1.0 specification
//
// ============================================================================

// This file contains the definition of the classes:
// template <class Td, class Ta> class OCP_TL1_DataCl
// 
// Template parameters Td: OCP data field type, Ta: OCP address field type 

#ifndef _OCP_TL1_DATA_CL
#define _OCP_TL1_DATA_CL

#ifndef DATAFLOW_ENUM
#define DATAFLOW_ENUM

// -------------------------------------------------
// MCmd encoding (OCP 1.0)
// -------------------------------------------------
enum OCPMCmdType {
  OCP_MCMD_IDLE = 0,
  OCP_MCMD_WR,
  OCP_MCMD_RD,
  OCP_MCMD_RDEX,
  OCP_MCMD_RESERVED4,
  OCP_MCMD_WRNP,
  OCP_MCMD_RESERVED6,
  OCP_MCMD_BCST
};

// -------------------------------------------------
// SResp encoding (OCP 1.0)
// -------------------------------------------------
enum OCPSRespType {
  OCP_SRESP_NULL = 0,
  OCP_SRESP_DVA,
  OCP_SRESP_RESERVED2,
  OCP_SRESP_ERR
};

// -------------------------------------------------
// MBurst encoding (OCP 1.0)
// -------------------------------------------------
enum OCPMBurstType {
  OCP_MBURST_LAST = 0,
  OCP_MBURST_DFLT1,
  OCP_MBURST_TWO,
  OCP_MBURST_DFLT2,
  OCP_MBURST_FOUR,
  OCP_MBURST_STRM,
  OCP_MBURST_EIGHT,
  OCP_MBURST_CONT
};

#endif

template <class Td, class Ta> class OCP_TL1_DataCl : public sc_prim_channel
{
public:

  typedef Td DataType;
  typedef Ta AddrType;

  // Constructor
  OCP_TL1_DataCl()
  {
    // initialize public members
    Reset();
  }

  // Destructor
  virtual ~OCP_TL1_DataCl()
  {
  }

  void Reset()
  {
    // Request Signal Group
    MAddr[0] = MAddr[1]           = 0;
    MAddrSpace[0] = MAddrSpace[1] = 0;
    MBurst[0] = MBurst[1]         = OCP_MBURST_LAST;
    MByteEn[0] = MByteEn[1]       = 0;
    MCmd[0] = MCmd[1]             = OCP_MCMD_IDLE;
    MConnID[0] = MConnID[1]       = 0;
    MData[0] = MData[1]           = 0;
    MThreadID[0] = MThreadID[1]   = 0;

    // Response Signal Group
    SData[0] = SData[1] = 0;
    SResp[0] = SResp[1] = OCP_SRESP_NULL;
    SThreadID[0] = SThreadID[1] = 0;

    // Datahandshake Signal Group
    MDataValid[0] = MDataValid[1] = false;
    MDataThreadID[0] = MDataThreadID[1] = 0;

    // initialize internal members
    m_Synchron = true; // hard coded, use TL_SYNCHRON_DATA_CL to configure

    m_Request  = false;
    m_ReqToggle = 0;

    m_Response = false;
    m_ResToggle = 0;
    
    m_DataRequest   = false;
    m_DataReqToggle = 0;

    m_UgRequest  = false;
    m_UgReqToggle = 0;

    m_UgResponse = false;
    m_UgResToggle = 0;

    // Reset Sideband Signals
    m_MError = false;
    m_nextMError = false;
    m_MFlag = 0;
    m_nextMFlag = 0;
    m_SError = false;
    m_nextSError = false;
    m_SFlag = 0;
    m_nextSFlag = 0;
    m_SInterrupt = false;
    m_nextSInterrupt = false;
    m_Control = 0;
    m_nextControl = 0;
    m_ControlWr = false;
    m_nextControlWr = false;
    m_ControlBusy = false;
    m_nextControlBusy = false;
    m_Status = 0;
    m_nextStatus = 0;
    m_StatusRd = false;
    m_nextStatusRd = false;
    m_StatusBusy = false;
    m_nextStatusBusy = false;
  }

//__________________________________________________________
//
// START of ACCESS METHODS
//__________________________________________________________

// __________________________________________________________
//
// Channel Methods
// These methods are called by the transaction channel, and
// they are mandatory.
// __________________________________________________________

  

  // Test and set access direction (read or write) 
  bool IsWriteRequest()
  {
    return((MCmd[1 - m_ReqToggle] == OCP_MCMD_WR) ? true : false);
  }
  void SetWriteRequest()
  {
    MCmd[m_ReqToggle] = OCP_MCMD_WR;
  }
  void SetReadRequest() 
  {
    MCmd[m_ReqToggle] = OCP_MCMD_RD;
  }


  // Makes new initiator request data visible to the target and
  // implements inertial assignment for request fields.
  // ToggleRequest() must be called for: 
  // SetWriteRequest(), SetReadRequest(), MputMCmd()
  void ToggleRequest()
  {
    m_Request = true;
    if (m_Synchron) request_update();
    else update();
  }

  // Makes new target response data visible to the initiator and
  // implements inertial assignment for response fields.
  // ToggleRequest() must be called for: SputSResp()
  void ToggleResponse()
  {
    m_Response = true;
    if (m_Synchron) request_update();
    else update();
  }

  // Makes new initiator request data (second request channel)
  // visible to the target and
  // implements inertial assignment for data request fields.
  // ToggleDataRequest() must be called for: MDataValid()
  void ToggleDataRequest()
  {
    m_DataRequest = true;
    if (m_Synchron) request_update();
    else update();
  }

// __________________________________________________________
//
// Request Signal Group 
// __________________________________________________________

  // Access methods for MAddr
  void MputMAddr(Ta a)
  {
    MAddr[m_ReqToggle] = a;
  }
  Ta SgetMAddr()
  {
    return MAddr[1 - m_ReqToggle];
  }

  // Access methods for MAddrSpace
  void MputMAddrSpace(int a)
  {
    MAddrSpace[m_ReqToggle] = a;
  }
  int SgetMAddrSpace()
  {
    return MAddrSpace[1 - m_ReqToggle];
  }

  // Access methods for MBurst
  void MputMBurst(OCPMBurstType a)
  {
    MBurst[m_ReqToggle] = a;
  }
  OCPMBurstType SgetMBurst()
  {
    return MBurst[1 - m_ReqToggle];
  }

  // Access methods for MByteEn
  void MputMByteEn(int a)
  {
    MByteEn[m_ReqToggle] = a;
  }
  int SgetMByteEn()
  {
    return MByteEn[1 - m_ReqToggle];
  }

  // Access methods for MCmd
  void MputMCmd(OCPMCmdType a)
  {
    MCmd[m_ReqToggle] = a;
  }
  OCPMCmdType SgetMCmd()
  {
    OCPMCmdType a = MCmd[1 - m_ReqToggle];
    MCmd[1 - m_ReqToggle] = OCP_MCMD_IDLE; // reset MCmd after testing it
    return a;
  }

  // Access methods for MConnID
  void MputMConnID(int a)
  {
    MConnID[m_ReqToggle] = a;
  }
  int SgetMConnID()
  {
    return MConnID[1 - m_ReqToggle];
  }

  // Access methods for MData
  void MputMData(Td d)
  {
    MData[m_ReqToggle] = d;
  }
  void SgetMData(Td &d)
  {
    d = MData[1 - m_ReqToggle];
  }
  Td SgetMData()
  {
    return MData[1 - m_ReqToggle];
  }

  // Access methods for MThreadID
  void MputMThreadID(int a)
  {
    MThreadID[m_ReqToggle] = a;
  }
  int SgetMThreadID()
  {
    return MThreadID[1 - m_ReqToggle];
  }

// __________________________________________________________
//
// Response Signal Group 
// __________________________________________________________

  // Access methods for SData
  void SputSData(Td d)
  {
    SData[m_ResToggle] = d;
  }
  void MgetSData(Td &d)
  {
    d = SData[1 - m_ResToggle];
  }
  Td MgetSData()
  {
    return SData[1 - m_ResToggle];
  }

  // Access methods for SResp
  void SputSResp(OCPSRespType a)
  {
    SResp[m_ResToggle] = a;
  }
  OCPSRespType MgetSResp()
  {
    OCPSRespType a = SResp[1 - m_ResToggle];
    SResp[1 - m_ResToggle] = OCP_SRESP_NULL; // reset SResp after testing it
    return a;
  }

  // Access methods for SThreadID
  void SputSThreadID(int a)
  {
    SThreadID[m_ResToggle] = a;
  }
  int MgetSThreadID()
  {
    return SThreadID[1 - m_ResToggle];
  }

// __________________________________________________________
//
// Datahandshake Signal Group 
// __________________________________________________________

  // Access methods for MData with data handshake
  void MputMDataHS(Td d)
  {
    MData[m_DataReqToggle] = d;
  }
  void SgetMDataHS(Td &d)
  {
    d = MData[1 - m_DataReqToggle];
  }
  Td SgetMDataHS()
  {
    return MData[1 - m_DataReqToggle];
  }

  // Access methods for MDataValid
  void MputMDataValid(bool a)
  {
    MDataValid[m_DataReqToggle] = a;
  }
  bool SgetMDataValid()
  {
    bool a = MDataValid[1 - m_DataReqToggle];
    MDataValid[1 - m_DataReqToggle] = false; // reset
    return a;
  }

  // Access methods for MDataThreadID
  void MputMDataThreadID(int a)
  {
    MDataThreadID[m_DataReqToggle] = a;
  }
  int SgetMDataThreadID()
  {
    return MDataThreadID[1 - m_DataReqToggle];
  }

  
//__________________________________________________________
//
// The following members have not been throughly tested
// for this release, and may change in later releases.
// They are included for completeness.
//__________________________________________________________

// __________________________________________________________
//
// Thread busy signaling. Ungrouped. (Alpha version)
// __________________________________________________________

  // Access methods for MThreadBusy
  void MputMThreadBusy(int a)
  {
    MThreadBusy[m_UgReqToggle] = a;
    m_UgRequest = true;
    if (m_Synchron) request_update();
    else update();
  }
  int SgetMThreadBusy()
  {
    return MThreadBusy[1 - m_UgReqToggle];
  }

  // Access methods for SThreadBusy
  void SputSThreadBusy(int a)
  {
    SThreadBusy[m_UgResToggle] = a;
    m_UgResponse = true;
    if (m_Synchron) request_update();
    else update();
  }
  int MgetSThreadBusy()
  {
    return SThreadBusy[1 - m_UgResToggle];
  }

  // __________________________________________________________
  //
  // Sideband Signal Group. (Second Alpha version)
  // 
  // Each sideband signal generates its own event.
  // NOTE: there is no checking that the proper port is calling
  //       the proper command. ie There is nothing to stop the
  //       the Slave from issuing a MputMError(true) command.
  // __________________________________________________________

  // Update function for sideband signals
  // NOTE: that this simple function could be inlined into the
  //       sideband "put" functions to save the function call
  //       overhead. Instead it is a seperate function because
  //       the synchronization of the OCP channel is still an
  //       open issue. This function provides a single change
  //       point to implement any changes in the synchronization
  //       scheme.
  void requestSidebandUpdate()
  {
    if (m_Synchron) {
      request_update();
    } else {
      update();
    }
  }
  
  // Access methods for MError
  void MputMError(bool nextValue)
  {
    m_nextMError = nextValue;
    requestSidebandUpdate();
  }
  bool SgetMError() const
  {
    return m_MError;
  }
  const sc_event& SidebandMErrorEvent() const { 
    return (m_SidebandMErrorEvent); 
  }

  // Access methods for MFlag
  void MputMFlag(int nextValue)
  {
    m_nextMFlag = nextValue;
    requestSidebandUpdate();
  }
  int SgetMFlag() const
  {
    return m_MFlag;
  }
  const sc_event& SidebandMFlagEvent() const { 
    return (m_SidebandMFlagEvent); 
  }

  // Access methods for SError
  void SputSError(bool nextValue)
  {
    m_nextSError = nextValue;
    requestSidebandUpdate();
  }
  bool MgetSError() const
  {
    return m_SError;
  }
  const sc_event& SidebandSErrorEvent() const { 
    return (m_SidebandSErrorEvent); 
  }

  // Access methods for SFlag
  void SputSFlag(int nextValue)
  {
    m_nextSFlag = nextValue;
    requestSidebandUpdate();
  }
  int MgetSFlag() const
  {
    return m_SFlag;
  }
  const sc_event& SidebandSFlagEvent() const { 
    return (m_SidebandSFlagEvent); 
  }

  // Access methods for SInterrupt
  void SputSInterrupt(bool nextValue)
  {
    m_nextSIterrupt = nextValue;
    requestSidebandUpdate();
  }
  bool MgetSInterrupt() const
  {
    return m_SIterrupt;
  }
  const sc_event& SidebandSInterruptEvent() const { 
    return (m_SidebandSInterruptEvent); 
  }


  //---------------------------------------------------------------------
  // system/core signal group. (Second Alpha version)
  // 
  // NOTE: there is no control over how these functions are accessed.
  //       ie. The System side of the channel may call any of the 
  //       Core side's functions without restriction
  //       
  //---------------------------------------------------------------------

  // Access methods for Control
  bool SysputControl(int nextValue)
  {
    // cannot change control signal if ControlBusy is true.
    if (m_ControlBusy){
      return false;
    }
    m_nextControl = nextValue;
    requestSidebandUpdate();
    return true;
  }
  int CgetControl() const
  {
    return m_Control;
  }
  const sc_event& SidebandControlEvent() const { 
    return (m_SidebandControlEvent); 
  }

  // Access methods for ControlBusy
  void CputControlBusy(bool nextValue)
  {
    m_nextControlBusy = nextValue;
    requestSidebandUpdate();
  }
  bool SysgetControlBusy() const
  {
    return m_ControlBusy;
  }
  const sc_event& SidebandControlBusyEvent() const { 
    return (m_SidebandControlBusyEvent); 
  }

  // Access methods for ControlWr
  void SysputControlWr(bool nextValue)
  {
    m_nextControlWr = nextValue; 
    requestSidebandUpdate();
  }
  bool CgetControlWr() const
  {
    return m_ControlWr;
  }
  const sc_event& SidebandControlWrEvent() const { 
    return (m_SidebandControlWrEvent); 
  }

  // Access methods for Status
  void CputStatus(int nextValue)
  {
    m_nextStatus = nextValue;
    requestSidebandUpdate();
  }
  int SysgetStatus() const
  {
    return m_Status;
  }
  bool readStatus(int& currentValue) const
  {
    // Cannot read the Status value of StatusBusy is true
    if (m_StatusBusy) {
      return false;
    }
    currentValue = m_Status;
    return true;
  }
  const sc_event& SidebandStatusEvent() const { 
    return (m_SidebandStatusEvent); 
  }

  // Access methods for StatusBusy
  void CputStatusBusy(bool nextValue)
  {
    m_nextStatusBusy = nextValue;
    requestSidebandUpdate();
  }
  bool SysgetStatusBusy() const
  {
    return m_StatusBusy;
  }
  const sc_event& SidebandStatusBusyEvent() const { 
    return (m_SidebandStatusBusyEvent); 
  }

  // Access methods for StatusRd
  void SysputStatusRd(bool nextValue)
  {
    m_nextStatusRd = nextValue;
    requestSidebandUpdate();
  }
  bool CgetStatusRd() const
  {
    return m_StatusRd;
  }
  const sc_event& SidebandStatusRdEvent() const { 
    return (m_SidebandStatusRdEvent); 
  }

//__________________________________________________________
//
// END of ACCESS METHODS
//__________________________________________________________


//__________________________________________________________
//
// This method updates the data members, making the new 
// initiator data visible to the target. The method is either
// called synchronously (via request_update()) or asynchronously
// (as direct call), depending on the value of m_Synchron.
//__________________________________________________________

  void update()
  {
    if (m_Request)  m_ReqToggle = 1 - m_ReqToggle;
    if (m_Response) m_ResToggle = 1 - m_ResToggle;
    if (m_DataRequest) m_DataReqToggle = 1 - m_DataReqToggle;
    if (m_UgRequest)  m_UgReqToggle = 1 - m_UgReqToggle;
    if (m_UgResponse) m_UgResToggle = 1 - m_UgResToggle;
    m_Request  = false;
    m_Response = false;
    m_DataRequest = false;
    m_UgRequest  = false;
    m_UgResponse = false;

    // Update sideband signals and set events
    // NOTE: if this is an Asynchronous update then notify should be immediate.
    //       if it is a synchronous update, then it sould be SC_ZERO_TIME.
    // NOTE: Need to build a correct async scheme.
    if (m_MError != m_nextMError) {
      m_MError = m_nextMError;
      m_SidebandMErrorEvent.notify(SC_ZERO_TIME);
    }
    if (m_MFlag != m_nextMFlag) {
      m_MFlag = m_nextMFlag;
      m_SidebandMFlagEvent.notify(SC_ZERO_TIME);
    }
    if (m_SError != m_nextSError) {
      m_SError = m_nextSError;
      m_SidebandSErrorEvent.notify(SC_ZERO_TIME);
    }
    if (m_SFlag != m_nextSFlag) {
      m_SFlag = m_nextSFlag;
      m_SidebandSFlagEvent.notify(SC_ZERO_TIME);
    }
    if (m_SInterrupt != m_nextSInterrupt) {
      m_SInterrupt = m_nextSInterrupt;
      m_SidebandSInterruptEvent.notify(SC_ZERO_TIME);
    }
    if (m_Control != m_nextControl) {
      m_Control = m_nextControl;
      m_SidebandControlEvent.notify(SC_ZERO_TIME);
    }
    if (m_ControlWr != m_nextControlWr) {
      m_ControlWr = m_nextControlWr;
      m_SidebandControlWrEvent.notify(SC_ZERO_TIME);
    }
    if (m_ControlBusy != m_nextControlBusy) {
      m_ControlBusy = m_nextControlBusy;
      m_SidebandControlBusyEvent.notify(SC_ZERO_TIME);
    }
    if (m_Status != m_nextStatus) {
      m_Status = m_nextStatus;
      m_SidebandStatusEvent.notify(SC_ZERO_TIME);
    }
    if (m_StatusRd != m_nextStatusRd) {
      m_StatusRd = m_nextStatusRd;
      m_SidebandStatusRdEvent.notify(SC_ZERO_TIME);
    }
    if (m_StatusBusy != m_nextStatusBusy) {
      m_StatusBusy = m_nextStatusBusy;
      m_SidebandStatusBusyEvent.notify(SC_ZERO_TIME);
    }
  }


//__________________________________________________________
//
// OCP Data Members
//__________________________________________________________

// --------------------
// Request Signal Group
// --------------------
  Ta MAddr[2];
  int MAddrSpace[2];
  OCPMBurstType MBurst[2];
  int MByteEn[2];
  OCPMCmdType MCmd[2];
  int MConnID[2];
  Td MData[2];
  int MThreadID[2];

// ---------------------
// Response Signal Group
// ---------------------
  Td SData[2];
  OCPSRespType SResp[2];
  int SThreadID[2];

// --------------------------
// Datahandshake Signal Group
// -------------------------
  bool MDataValid[2];
  int MDataThreadID[2];
//  sc_event MDataValidEvent;
//  sc_event SDataAcceptEvent;

// -----------------
// Ungrouped Signals
// -----------------
  bool MThreadBusy[2];
  bool SThreadBusy[2];

// ---------------------
// Sideband Signal Group
// ---------------------
  bool m_MError;
  bool m_nextMError;
  sc_event m_SidebandMErrorEvent;
  int m_MFlag;
  int m_nextMFlag;
  sc_event m_SidebandMFlagEvent;
  bool m_SError;
  bool m_nextSError;
  sc_event m_SidebandSErrorEvent;
  int m_SFlag;
  int m_nextSFlag;
  sc_event m_SidebandSFlagEvent;
  bool m_SInterrupt;
  bool m_nextSInterrupt;
  sc_event m_SidebandSInterruptEvent;
  int m_Control;
  int m_nextControl;
  sc_event m_SidebandControlEvent;
  bool m_ControlWr;
  bool m_nextControlWr;
  sc_event m_SidebandControlWrEvent;
  bool m_ControlBusy;
  bool m_nextControlBusy;
  sc_event m_SidebandControlBusyEvent;
  int m_Status;
  int m_nextStatus;
  sc_event m_SidebandStatusEvent;
  bool m_StatusRd;
  bool m_nextStatusRd;
  sc_event m_SidebandStatusRdEvent;
  bool m_StatusBusy;
  bool m_nextStatusBusy;
  sc_event m_SidebandStatusBusyEvent;

//__________________________________________________________
//
// End of OCP Data Members
//__________________________________________________________


// private: 
// We make them public for read access.

  // synchronous/asynchronous update of the data members
  bool m_Synchron;

  // request channel update
  bool m_Request;
  int m_ReqToggle;

  // response channel update
  bool m_Response;
  int m_ResToggle;

  // second request channel (for data handshake) update
  bool m_DataRequest;
  int m_DataReqToggle;

  // ungrouped signals update
  bool m_UgRequest;
  int m_UgReqToggle;
  bool m_UgResponse;
  int m_UgResToggle;

};

#endif // _OCP_TL1_DATA_CL
