// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level Layer-1
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//                Anssi Haverinen, Nokia Inc., anssi.haverinen@nokia.com
//         Date : 02/15/2003
//
//  Description : Transaction Level - Layer-1 example Master
//    Features:
//    - Asynchronous master using OCP data class
//    - Using data handshake
//    - Non-blocking methods are used
//    - Pipelined Request and Response threads
//                
// Parameter    :
//   Template arguments.
//   - TdataCl: Data class containing data members and
//              access methods for the data exchanged between
//              Masters and Slaves
//   - Td: Data type
//   - Ta: Address type
//                
//   Constructor arguments.
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//     - Priority: Positive number specifying the priority relative to the
//                 other Masters for Bus access. Higher numbers means higher
//                 priority. Masters can have the same priority.
//     - DelayPct: Non-negative number specifying the delay in percent
//                 of a clock cycle, the data are transmitted.
// ============================================================================

#include "ocp_tl1_master_async_hs.h"
#include "ocp_tl1_data_cl.h"

// ----------------------------------------------------------------------------
// Process : OCP_TL1_Master_Async_HS::OCP_TL1_Master_Async_HS
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL1_Master_Async_HS<TdataCl>::OCP_TL1_Master_Async_HS
      (
        sc_module_name name_
      , int ID
      , int Priority
      , double DelayPct
      )
        : sc_module         (name_)
        , MasterP           ("MasterPort")
        , m_ID              (ID)
        , m_Priority        (Priority)
        , m_DelayPct        (DelayPct)
{
  // InitParameters();

  SC_THREAD(Request);
  sensitive_pos(clk); 
  SC_THREAD(Handshake);
  sensitive_pos(clk); 
  SC_THREAD(Response);
  sensitive << MasterP; 
}

// ----------------------------------------------------------------------------
// Process : OCP_TL1_Master_Async_HS::~OCP_TL1_Master_Async_HS
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL1_Master_Async_HS<TdataCl>::~OCP_TL1_Master_Async_HS()
{
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Async_HS::Request()
//
//  Top level request method of the Master
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Master_Async_HS<TdataCl>::Request()
{
  int i = 0;
  int Pause[] = {1, 0, 6}; // # of cycles between requests;
  bool tmp = 0;
  Ta Addr = 0;
  int cnt = 1;

  while(true)
  {
    wait(); // wait for rising edge of clock

    if (cnt == 0)
    {
      // check if channel is free
      tmp = !MasterP->MgetSbusy();
      if (tmp)
      {
        // get new Pause value
        cnt = Pause[i%3]; i++;

        // Delay data signal
	// This must work with or without the next line
        // The delay must be less than 1 clk cycle
 	if (m_DelayPct > 0)
          wait(TL_CLK_PERIOD * (m_DelayPct/100.0), TL_TIME_UNIT);

        // Put data into the channel
	m_DataCl->MputMAddr(Addr);
        MasterP->MputWriteRequest(); // Toggles the request data members

#ifdef DEBUG_G1
      cout << "Master sent address "
           << " delta " << simcontext()->delta_count()
           << " time " << sc_time_stamp().to_seconds()
           << " Addr " << Addr << endl;
#endif

        Addr += 4;
      }
    }
    else
    {
      cnt--;
    }
  } // end of while loop
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Async_HS::Handshake()
//
//  Top level request method of the Master
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Master_Async_HS<TdataCl>::Handshake()
{
  int Pause[] = {1, 0, 5}; // # of cycles between requests;
  int cnt = 2;
  bool tmp = false;
  int i = 0;
  Td Data = 0;

  while (true)
  {
    wait();  // wait for rising edge of clock

    if (cnt == 0)
    {
      tmp = !MasterP->MgetSbusyData();
      if (tmp)
      {
        cnt = Pause[i%3]; i++;

        m_DataCl->MputMDataHS(Data);
        m_DataCl->MputMDataValid(true); // Not mandatory
        MasterP->MputDataRequest(); // Toggles the handshake data members

#ifdef DEBUG_G1
        cout << "Master sent data "
             << " delta " << simcontext()->delta_count()
            << " time " << sc_time_stamp().to_seconds()
             << " data " << Data << endl;
#endif
        Data++;
      }
    }
    else
      cnt--;
  }
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Async_HS::Response()
//
//  Top level reponse method of the Master
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Master_Async_HS<TdataCl>::Response()
{
  Td Rdata;

  while(true)
  {
    wait(); // for default event

    if (MasterP->MgetResponse(true))
    {
      m_DataCl->MgetSData(Rdata);

#ifdef DEBUG_G1
      cout << "Master got response "
           << " delta " << simcontext()->delta_count()
           << " time " << sc_time_stamp().to_seconds()
           << " data " << Rdata << endl;
#endif
    }
  } // end of while(true) loop
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Async_HS::SputDirect()
//
//  Debug interface method.
//  Read/Write data, Slave to Master direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl> bool OCP_TL1_Master_Async_HS<TdataCl>::SputDirect(
         int SlaveID, bool IsWrite, Td *Data, Ta Address, int NumWords)

{
  // not implemented
  return(false);
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Async_HS::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Master_Async_HS<TdataCl>::end_of_elaboration()
{
  sc_module::end_of_elaboration();

  // Initialize debug interface
  MasterP->MregisterDirectIF(this);

  // Get data structure
  m_DataCl = MasterP->GetDataCl();

  // Get communication structure
  m_CommCl = MasterP->GetCommCl();

  // Get system parameter structure
  m_ParamCl = MasterP->GetParamCl();

  // Put parameter into Channel
  // No parameters needed
}


// ----------------------------------------------------------------------------
//
//  Instantiation of the Master
//
// ----------------------------------------------------------------------------
template class OCP_TL1_Master_Async_HS<TL1_TEMPL_DATA_CL >; // see ocp_tl1_globals.h
