// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level Layer-1
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//                Anssi Haverinen, Nokia Inc., anssi.haverinen@nokia.com
//                Joe Chou, Sonics Inc., joechou@sonicsinc.com 
//         Date: 06/13/2003
//
//  Description : OCP Transaction Level Channel
//    The Channel can be used at all three Transaction Levels.
//    It is based on the generic TL channel released to OSCI
//    (www.systemc.org).
//
//    The Channel is quite generic still, and can be used to implement
//    other interface protocols with different data classes.
//
//    The TL Channel implements the communication between user-written
//    modules. In particular the Channel implements a 
//    Master and a Slave interface. The main features are:
//    - Two way communication: The Channel consists of a
//      . Request Channel, modeling the Master to Slave transfer,
//      . Data Request Channel, modeling the Master to Slave data transfer and a
//      . Response Channel, modeling the Slave to Master transfer
//    - Blocking and non-blocking methods: Read and write methods
//      implemented in the Channel can be blocking or non-blocking.
//    - Direct access methods: Read and write methods to transport
//      data without using any resources and without affecting
//      the timing. This is useful for initialization, debugging or
//      fast access to pure functional Slave.
//    - User configurable data class: The Channel gives Master
//      and Slave access to the user configurable data class.
//      This data class can be used to pass on data and
//      control information between Master and Slave.
//      The data class is a template argument allowing the usage
//      of different data classes in a system.
//    - User configurable parameter class: The parameter class
//      is intented to pass on parameters or setup values like
//      Slave addresses or Master priorities to other modules e.g. the Bus.
//    - Synchronization states and events used in the Channel
//      can be exported and hence made visible to the attached
//      modules by means of the provided communication class.
//
//   The flow of events in the Channel:
//   a) Request Channel
//   - Master submits request for Slave to the Channel.
//     . Channel is blocked until Slave frees the Channel.
//     . If Master used non-blocking method the Master can probe
//       if Channel has been freed by the Slave.
//     . If Master used blocking method, the Master is blocked until the
//       Slave frees the Channel.
//   - Channel triggers Slave that there is a pending request.
//     . Slave can be made sensitive to a Channel event.
//     . Slave can probe the Channel for a pending request.
//   - Slave receives the pending request and frees the Channel
//   - After the Request Channel has been freed by the Slave the Master
//     is allowed to submit another request.
//   b) Data Request Channel
//   The Data Request Channel works the same as the Request Channel;
//   it is meant for implementing interfaces which use separate data handshake
//   c) Response-Channel
//   The Slave-Response Channel works the same as the Request Channel,
//   except that the roles of Master and Slaves are inverted. That is, the
//   Slave submits the response to the Master and the Master must free the
//   Response Channel for the next Slave response.
//     
// Parameters    :
//   Template arguments.
//     - TdataCl: Data class containing data members and access methods for
//                the data exchanged between Masters and Slaves
//     - Td: Data type
//     - Ta: Address type
//   Constructor arguments.
//     - sc_module_name: Name of the module (channel) instance
//     - Synchron: Specifies if the Channel should do a synchronous
//         update (=true) or and asynchronous update (=false)
//         of the Channel events and states.
//         Use the default value unless you know exactly
//         what you are doing.
//     - SyncEvents: Specifies if the Channel should use synchronization
//         events for the communication scheme. The Channel is faster if
//         no synchronization events are used. This could be used e.g.
//         in a Layer-1 application where only non-blocking methods
//         are employed. If blocking methods are used SyncEvents must
//         be true.
//         Use the default value (= true) unless you know exactly
//         what you are doing.
//     - DefaultEvents: Specifies if the Channel should trigger the
//         default event. The Channel is faster if no default event is
//         triggered. DefaultEvent should be set to false if the
//         attached modules are not sensitve to their ports, with other words,
//         if at least one attached module is sensitive to the port,
//         DefaultEvent should be set to true.
//         Use the default value (= true) unless you know exactly
//         what you are doing.
//                 
// =========================================================================

#ifndef _TL_CHANNEL_H
#define _TL_CHANNEL_H

#include <math.h>
#include <stdio.h>
#include <fstream.h>

#include "systemc.h"

#include "tl_master_if.h"
#include "tl_slave_if.h"


template<class TdataCl> class TL_Channel
  : public sc_module
  , public TLmasterIF<TdataCl>
  , public TLslaveIF<TdataCl>
{

//---------------------------------------------------------------
//
// Synchronizer class. This class is used to perform a synchronous update
// of the states and events which control the communication between
// Master and Slave. In the asynchronouse case the update() method
// is executed directly from the module TL_Channel, while in the synchronous
// case this sc_prim_channel is used to call the request_update() method.
//
//---------------------------------------------------------------
class Synchronizer : public sc_prim_channel
{
public:
  Synchronizer(TL_Channel* Channel) : m_Channel (Channel){}
  void Update() {request_update();}
private:
  virtual void update() {m_Channel->update();}
  TL_Channel* m_Channel;
};


public:

  typedef typename TdataCl::DataType Td;
  typedef typename TdataCl::AddrType Ta;

  SC_HAS_PROCESS(TL_Channel);

//---------------------------------------------------------------
//
// Constructor
//
//---------------------------------------------------------------
  TL_Channel(sc_module_name name
     , bool Synchron     = true
     , bool SyncEvent    = true
     , bool DefaultEvent = true
            )
     :sc_module        (name)
     , m_Master         (0)
     , m_Slave          (0)
     , m_Synchron       (Synchron)
     , m_SyncEvent      (SyncEvent)
     , m_DefaultEvent   (DefaultEvent)
     , m_EventSelectFw  (0)
     , m_EventSelectFwD (0)
     , m_EventSelectBw  (0)
     , m_MdirectIF      (0)
     , m_SdirectIF      (0)
     , m_Synchronizer   (this)
  {
#ifdef TRACE_C
    outFile = new ofstream(name);
#endif

    if (!m_Synchron) {
      cout << "WARNING, OCP channel model currently does not support Asynchronuos mode" << endl;
    }
    m_DataCl  = new TdataCl();
    m_CommCl  = new CommCl();
    m_ParamCl = new ParamCl<TdataCl>();

    SC_METHOD(ReqEnd);
    sensitive << m_CommCl->RequestEndEventN;
    dont_initialize();

    SC_METHOD(ResEnd);
    sensitive << m_CommCl->ResponseEndEventN;
    dont_initialize();

    SC_METHOD(DataReqEnd);
    sensitive << m_CommCl->DataRequestEndEventN;
    dont_initialize();
  }

//---------------------------------------------------------------
//
// Destructor
//
//---------------------------------------------------------------
  virtual ~TL_Channel()
  {
#ifdef TRACE_C
    delete outFile;
#endif

    delete m_DataCl;
    delete m_CommCl;
    delete m_ParamCl;
  }


//---------------------------------------------------------------
//
// SC_METHOD/SC_THREAD methods.
// Processes implementing a delayed Channel release.
//
//---------------------------------------------------------------
void ReqEnd()   { RequestUpdateFw(2); }
void ResEnd()   { RequestUpdateBw(2); }
void DataReqEnd()   { RequestUpdateFwD(2); }

//---------------------------------------------------------------
//
// Register ports and perform static check.
//
//---------------------------------------------------------------
  virtual void register_port(sc_port_base& port, const char* if_typename)
  {
    sc_string nm(if_typename);
    if (nm == typeid(TLmasterIF<TdataCl> ).name())
    {
      // only one master can be connected
      assert(m_Master == 0);
      m_Master = &port;
    }
    else if (nm == typeid(TLslaveIF<TdataCl> ).name())
    {
      // only one slave can be connected
      assert(m_Slave == 0);
      m_Slave = &port;
    }
  }

//---------------------------------------------------------------
//
// Externally visible events.
// Event is triggered at Sput and Mput.
// It allows making Master/Slave sensitive to the Master/Slave port(s).
//
//---------------------------------------------------------------
  const sc_event& default_event() const { 
    return (m_CommCl->StartEvent); 
  }

//---------------------------------------------------------------
//
// Data interface access methods.
// These methods return a pointer to the corresponding class.
// This gives Master/Slaves access to the data structures.
// The data structures must be provided by the user.
//
//---------------------------------------------------------------
  TdataCl          *GetDataCl()  { return m_DataCl; }
  ParamCl<TdataCl> *GetParamCl() { return m_ParamCl; }

//---------------------------------------------------------------
// Returns pointer to communication class, which
// contains state variables and channel events.
// These maybe used to create richer transactions by the user
//---------------------------------------------------------------
  CommCl           *GetCommCl()  { return m_CommCl; }


//---------------------------------------------------------------
//
// Explicit Channel release methods.
// These methods unlock the Channel.
// Srelease() unlocks the first request channel.
// Mrelease() unlocks the first response channel.
// SreleaseData() unlocks the second request channel (for data handshake)
//
//---------------------------------------------------------------
  void Mrelease(sc_time time) {m_CommCl->ResponseEndEventN.notify(time);}
  void Srelease(sc_time time) {m_CommCl->RequestEndEventN.notify(time);}
  void Mrelease(void) { RequestUpdateBw(2); }
  void Srelease(void) { RequestUpdateFw(2); }
  void SreleaseData(sc_time time) {m_CommCl->DataRequestEndEventN.notify(time);}
  void SreleaseData(void) { RequestUpdateFwD(2); }

//---------------------------------------------------------------
//
// Preemptive Channel release methods.
// These methods unlock the channel immediately after the
// request/response has been posted
// SreleasePE() unlocks the first request channel.
// MreleasePE() unlocks the first response channel.
// SreleaseDataPE() unlocks the second request channel.
// 
// These are meant to be used in TL1 with fully synchronous
// masters and slaves, which want to call release before they
// receive request.
//---------------------------------------------------------------
  void MreleasePE(void) {wait(SC_ZERO_TIME); MgetResponseBlocking(true);}
  void SreleasePE(void) {wait(SC_ZERO_TIME); SgetRequestBlocking(true);} 
  void SreleaseDataPE(void) {wait(SC_ZERO_TIME); SgetDataRequestBlocking(true);}
  
//---------------------------------------------------------------
//
// Methods for the master to check if the request channels are busy.
//
//---------------------------------------------------------------
  bool MgetSbusy() {
    return (m_CommCl->RequestPending || m_CommCl->BlockingRequestPending);}
  bool MgetSbusyData() {
    return (m_CommCl->DataRequestPending || m_CommCl->BlockingDataRequestPending);}

//---------------------------------------------------------------
//
// Method for the slave to check if the response channel is busy
//
//---------------------------------------------------------------
  bool SgetMbusy() {
    return (m_CommCl->ResponsePending || m_CommCl->BlockingResponsePending); }

//---------------------------------------------------------------
//
// Method for Slave to check if the request is a read or write
// request. Note that the user must implement the underlying 
// m_DataCl->IsWriteRequest() method in the data interface class.
//
//---------------------------------------------------------------
  bool IsWrite() { return m_DataCl->IsWriteRequest(); }


//---------------------------------------------------------------
//
// Direct interface methods (for debugging).
// Note that the MputDirect method must be implemented in the
// Slave while the SputDirect method (if used) must be implemented
// in the Master.
//
//---------------------------------------------------------------
  virtual bool MputDirect(
			  int MasterID, bool IsWrite, Td *Data, Ta Address, int NumWords)
  {
    if (m_SdirectIF != 0)
      return(m_SdirectIF->MputDirect(MasterID, IsWrite, Data, Address, NumWords));
    else
      return(false);
  }

  virtual bool SputDirect(
			  int SlaveID, bool IsWrite, Td *Data, Ta Address, int NumWords)
  {
    if (m_MdirectIF != 0)
      return(m_MdirectIF->SputDirect(SlaveID, IsWrite, Data, Address, NumWords));
    else
      return(false);
  }

  void SregisterDirectIF(SdirectIF<TdataCl> *A) { m_SdirectIF = A; }
  void MregisterDirectIF(MdirectIF<TdataCl> *A) { m_MdirectIF = A; }


//---------------------------------------------------------------
//
// Methods for the Master to initiate a write or read transfer.
// There exists non-blocking and blocking versions.
//
//---------------------------------------------------------------
  bool MputWriteRequest() 
  {
    bool tmp;
    if ((tmp = !MgetSbusy())) {
      m_DataCl->SetWriteRequest();
      tmp = MputRequest();
    }
    return tmp;
  }
  bool MputReadRequest() 
  {
    bool tmp;
    if ((tmp = !MgetSbusy())) {
      m_DataCl->SetReadRequest();
      tmp = MputRequest();
    }
    return tmp;
  }
  bool MputWriteRequestBlocking() 
  {
    m_DataCl->SetWriteRequest(); return MputRequestBlocking();
  }

  bool MputReadRequestBlocking() 
  {
    m_DataCl->SetReadRequest(); return MputRequestBlocking();
  } 


//---------------------------------------------------------------
//
// The following 4 methods implement the complete communication scheme.
// For each of the 4 methods there exist a blocking and non-blocking version.
//
// The above Mput methods are wrappers around the MputRequest() methods.
//
// It may be useful to call the MputRequest() method directly in order
// to avoid the calling of SetWriteRequest() or SetReadRequest().
//
// MputRequest(): Master method to initiate a request (read or write)
// SgetRequest(): Slave method to accept a request
// SputRequest(): Slave method to initiate a reponse
// MgetRequest(): Master method to accept the response
//
//---------------------------------------------------------------
  // Master method to initiate a request (read or write)
  bool MputRequest()
  {
    if ((!m_CommCl->RequestPending) && (!m_CommCl->BlockingRequestPending))
    {
      // update data
      m_DataCl->ToggleRequest();

      // update synchronization
      m_CommCl->RequestPending = true;
      m_CommCl->RequestEnd     = false;
      m_CommCl->RequestPendingPE[m_CommCl->RequestTogglePE] = true; // For TL1 pre-emptive release
      RequestUpdateFw(1); // _RequestStartEvent
      if (!m_ParamCl->CmdAccept) SreleasePE();
      return(true);
    }
    else return(false);
  }
  bool MputRequestBlocking()
  {
    // wait for the channel to be released
    if (!(m_CommCl->BlockingRequestPending))
    {
      m_CommCl->BlockingRequestPending = true;
      if (m_CommCl->RequestPending)
      {
#ifdef DEBUG_C
        cout << name() << ": Master waits put request" << endl;
#endif
        wait(m_CommCl->RequestEndEvent);
      }
      // update data
      m_DataCl->ToggleRequest();

      // update synchronization
      m_CommCl->RequestPending = true;
      m_CommCl->RequestPendingPE[m_CommCl->RequestTogglePE] = true; // For TL1 pre-emptive release
      RequestUpdateFw(1); // _RequestStartEvent

      if (!(m_CommCl->RequestEnd))
      {
#ifdef DEBUG_C
      cout << name() << ": Master waits put request" << endl;
#endif
        wait(m_CommCl->RequestEndEvent);
      }
#ifdef DEBUG_C
      cout << name() << ": Master finished waiting put request" << endl;
#endif
      m_CommCl->RequestEnd             = false;
      m_CommCl->BlockingRequestPending = false;
      return(true);
    }
    else return(false);
  }


//---------------------------------------------------------------
//
// Methods for the Slave to get a Master request.
// There exist non-blocking and blocking versions.
//
//---------------------------------------------------------------
  bool SgetRequest(bool Release)
  {
    if ((!m_CommCl->GetRequestPending) &&
        (!m_CommCl->BlockingGetRequestPending))
    {
      // update synchronization
      m_CommCl->GetRequestPending = true;
      m_CommCl->RequestStart = false;
      if (Release) RequestUpdateFw(2); // _RequestEndEvent
      return(true);
    }
    else return(false);
  }
  bool SgetRequestBlocking(bool Release)
  {
    if (!(m_CommCl->BlockingGetRequestPending))
    {
      m_CommCl->GetRequestPending = true;
      m_CommCl->BlockingGetRequestPending = true;
      if (!(m_CommCl->RequestStart))
      {
#ifdef DEBUG_C
      cout << name() << ": Slave waits get request" << endl;
#endif
        wait(m_CommCl->RequestStartEvent);
      }
#ifdef DEBUG_C
      cout << name() << ": Slave finished waiting get request" << endl;
#endif

      // update synchronization
      m_CommCl->RequestStart              = false;
      m_CommCl->BlockingGetRequestPending = false;
      if (Release) RequestUpdateFw(2); // _RequestEndEvent
      return(true);
    }
    else return(false);
  }
  bool SgetRequestPE() // For use with SreleasePE(). Do not mix with SgetRequest()
  {
    if (m_CommCl->RequestPendingPE[1-m_CommCl->RequestTogglePE])
    {
      // update synchronization
      m_CommCl->RequestPendingPE[1-m_CommCl->RequestTogglePE] = false;
      return(true);
    }
    else return(false);
  }

//---------------------------------------------------------------
//
// Methods for the Slave to answer to a Master request.
// There exist non-blocking and blocking versions.
//
//---------------------------------------------------------------
  bool SputResponse()
  {
    if ((!m_CommCl->ResponsePending) && (!m_CommCl->BlockingResponsePending))
    {
      // update data
      m_DataCl->ToggleResponse();

      // update synchronization
      m_CommCl->ResponsePending = true;
      m_CommCl->ResponsePendingPE[m_CommCl->ResponseTogglePE] = true; // For TL1 pre-emptive release
      m_CommCl->ResponseEnd     = false;
      RequestUpdateBw(1); // _ResponseStartEvent
      if (!m_ParamCl->RespAccept) MreleasePE();
      return(true);
    }
    else return(false);
  }
  bool SputResponseBlocking()
  {
    // wait for the channel to be released
    if (!(m_CommCl->BlockingResponsePending))
    {
      m_CommCl->BlockingResponsePending = true;

      if (m_CommCl->ResponsePending)
      {
#ifdef DEBUG_C
        cout << name() << ": Slave waits put request" << endl; 
#endif
        wait(m_CommCl->ResponseEndEvent);
      }
      // update data
      m_DataCl->ToggleResponse();

      // update synchronization
      m_CommCl->ResponsePending = true;
      m_CommCl->ResponsePendingPE[m_CommCl->ResponseTogglePE] = true; // For TL1 pre-emptive release
      RequestUpdateBw(1); // _ResponseStartEvent

      if (!(m_CommCl->ResponseEnd))
      {
#ifdef DEBUG_C
      cout << name() << ": Slave waits put request" << endl; 
#endif
        wait(m_CommCl->ResponseEndEvent);
      }
#ifdef DEBUG_C
      cout << name() << ": Slave finished waiting put request" << endl;
#endif
      m_CommCl->ResponseEnd             = false;
      m_CommCl->BlockingResponsePending = false;
      return(true);
    }
    else return(false);
  }


//---------------------------------------------------------------
//
// Methods for the Master to request the response from the Slave.
// There exist non-blocking and blocking versions.
//
//---------------------------------------------------------------
  bool MgetResponse(bool Release)
  {
    if ((!m_CommCl->GetResponsePending) &&
        (!m_CommCl->BlockingGetResponsePending))
    {
      // update synchronization
      m_CommCl->GetResponsePending = true;
      m_CommCl->ResponseStart = false;
      if (Release) RequestUpdateBw(2); // _ResponseEndEvent
      return(true);
    }
    else return(false);
  }
  bool MgetResponseBlocking(bool Release)
  {
    if (!(m_CommCl->BlockingGetResponsePending))
    {
      m_CommCl->GetResponsePending = true;
      m_CommCl->BlockingGetResponsePending = true;

      if (!(m_CommCl->ResponseStart))
      {
        wait(m_CommCl->ResponseStartEvent);
      }
      // update synchronization
      m_CommCl->ResponseStart              = false;
      m_CommCl->BlockingGetResponsePending = false;
      if (Release) RequestUpdateBw(2); // _ResponseEndEvent
      return(true);
    }
    else return(false);
  }
  bool MgetResponsePE()// For use with MreleasePE(). Do not mix with MgetResponse()
  {
    if (m_CommCl->ResponsePendingPE[1-m_CommCl->ResponseTogglePE])
    {
      // update synchronization
      m_CommCl->ResponsePendingPE[1-m_CommCl->ResponseTogglePE] = false;
      return(true);
    }
    else return(false);
  }

//---------------------------------------------------------------
//
// Methods for the master to post a request on
// second channel, used for data handshake
// There exist non-blocking and blocking versions.
//
//---------------------------------------------------------------
  bool MputDataRequest()
  {
    if ((!m_CommCl->DataRequestPending) && (!m_CommCl->BlockingDataRequestPending))
    {
      // update data
      m_DataCl->ToggleDataRequest();

      // update synchronization
      m_CommCl->DataRequestPending = true;
      m_CommCl->DataRequestPendingPE[m_CommCl->DataRequestTogglePE] = true; // For TL1 pre-emptive release
      m_CommCl->DataRequestEnd     = false;
      RequestUpdateFwD(1); // _RequestStartEvent
      if (!m_ParamCl->DataAccept) SreleaseDataPE();
      return(true);
    }
    else return(false);
  }
  bool MputDataRequestBlocking()
  {
    // wait for the channel to be released
    if (!(m_CommCl->BlockingDataRequestPending))
    {
      m_CommCl->BlockingDataRequestPending = true;
      if (m_CommCl->DataRequestPending)
      {
#ifdef DEBUG_C
        cout << name() << ": Master waits put data request" << endl;
#endif
        wait(m_CommCl->DataRequestEndEvent);
      }
      // update data
      m_DataCl->ToggleDataRequest();

      // update synchronization
      m_CommCl->DataRequestPending = true;
      m_CommCl->DataRequestPendingPE[m_CommCl->DataRequestTogglePE] = true; // For TL1 pre-emptive release
      RequestUpdateFwD(1); // _RequestStartEvent

      if (!(m_CommCl->DataRequestEnd))
      {
#ifdef DEBUG_C
      cout << name() << ": Master waits put data request" << endl;
#endif
        wait(m_CommCl->DataRequestEndEvent);
      }
#ifdef DEBUG_C
      cout << name() << ": Master finished waiting put data request" << endl;
#endif
      m_CommCl->DataRequestEnd             = false;
      m_CommCl->BlockingDataRequestPending = false;
      return(true);
    }
    else return(false);
  }

//---------------------------------------------------------------
//
// Methods for the slave to get a master request,
// second channel, used for data handshake
// There exist non-blocking and blocking versions.
//
//---------------------------------------------------------------
  bool SgetDataRequest(bool Release)
  {
    if ((!m_CommCl->GetDataRequestPending) &&
        (!m_CommCl->BlockingGetDataRequestPending))
    {
      // update synchronization
      m_CommCl->GetDataRequestPending = true;
      m_CommCl->DataRequestStart = false;
      if (Release) RequestUpdateFwD(2); // _RequestEndEvent
      return(true);
    }
    else return(false);
  }
  bool SgetDataRequestBlocking(bool Release)
  {
    if (!(m_CommCl->BlockingGetDataRequestPending))
    {
      m_CommCl->GetDataRequestPending = true;
      m_CommCl->BlockingGetDataRequestPending = true;
      if (!(m_CommCl->DataRequestStart))
      {
#ifdef DEBUG_C
      cout << name() << ": Slave waits get data request" << endl;
#endif
        wait(m_CommCl->DataRequestStartEvent);
      }
#ifdef DEBUG_C
      cout << name() << ": Slave finished waiting get data request" << endl;
#endif

      // update synchronization
      m_CommCl->DataRequestStart              = false;
      m_CommCl->BlockingGetDataRequestPending = false;
      if (Release) RequestUpdateFwD(2); // _RequestEndEvent
      return(true);
    }
    else return(false);
  }
  bool SgetDataRequestPE() // For use with SreleaseDataPE(). Do not mix with SgetDataRequest()
  {
    if (m_CommCl->DataRequestPendingPE[1-m_CommCl->DataRequestTogglePE])
    {
      // update synchronization
      m_CommCl->DataRequestPendingPE[1-m_CommCl->DataRequestTogglePE] = false;
      return(true);
    }
    else return(false);
  }

//---------------------------------------------------------------
// Update method. This method is used to update the states and events
// which control the communication between Master and Slave.
// This method is not part of the API. However it is public because
// it is used by the Synchronizer class.
// AsyncUpdate() use immediate notify 
// update() use delayed notify 
//---------------------------------------------------------------

  void AsyncUpdate()
  {
#ifdef TRACE_C
    double ttime;
#endif

    switch(m_EventSelectFw)
    {
      case 1: 
        if (m_SyncEvent) m_CommCl->RequestStartEvent.notify();
        m_CommCl->RequestStart = true;
        m_CommCl->GetRequestPending = false;
	m_CommCl->RequestTogglePE = 1 - m_CommCl->RequestTogglePE;
        if (m_DefaultEvent) {
          m_CommCl->StartEvent.notify();
        }
#ifdef TRACE_C
        ttime = sc_time_stamp().to_seconds();
        *outFile<<" "<<m_DataCl->SgetMAddr()<<" Req start at "<<ttime<<endl;
#endif
        break;
      case 2:
        if (m_SyncEvent) m_CommCl->RequestEndEvent.notify();
        if (m_CommCl->RequestPending == false)
          m_CommCl->RequestEnd = true;
        else
          m_CommCl->RequestPending = false;
#ifdef TRACE_C
        ttime = sc_time_stamp().to_seconds();
        *outFile<<" "<<m_DataCl->SgetMAddr()<<" Req end at "<<ttime<<endl;
#endif
        break;
      default: break;
    }
    switch(m_EventSelectFwD)
    {
      case 1: 
        if (m_SyncEvent) m_CommCl->DataRequestStartEvent.notify();
        m_CommCl->DataRequestStart = true;
        m_CommCl->GetDataRequestPending = false;
	m_CommCl->DataRequestTogglePE = 1 - m_CommCl->DataRequestTogglePE;
        if (m_DefaultEvent) m_CommCl->StartEvent.notify();
#ifdef TRACE_C
        ttime = sc_time_stamp().to_seconds();
        *outFile<<"Data Req start at "<<ttime<<endl;
#endif
        break;
      case 2:
        if (m_SyncEvent) m_CommCl->DataRequestEndEvent.notify();
        if (m_CommCl->DataRequestPending == false)
          m_CommCl->DataRequestEnd = true;
        else
          m_CommCl->DataRequestPending = false;
#ifdef TRACE_C
        ttime = sc_time_stamp().to_seconds();
        *outFile<<"Data Req end at "<<ttime<<endl;
#endif
        break;
      default: break;
    }
    switch(m_EventSelectBw)
    {
      case 1:
        if (m_SyncEvent) {
          m_CommCl->ResponseStartEvent.notify();
        }
        m_CommCl->ResponseStart = true;
        m_CommCl->GetResponsePending = false;
	m_CommCl->ResponseTogglePE = 1 - m_CommCl->ResponseTogglePE;
        if (m_DefaultEvent) m_CommCl->StartEvent.notify();
#ifdef TRACE_C
        ttime = sc_time_stamp().to_seconds();
        *outFile<<"                                         "
			<<" Resp start at "<<ttime<<endl;
#endif
        break;
      case 2:
        if (m_SyncEvent) m_CommCl->ResponseEndEvent.notify();
        if (m_CommCl->ResponsePending == false)
          m_CommCl->ResponseEnd = true;
        else
          m_CommCl->ResponsePending = false;
#ifdef TRACE_C
       ttime = sc_time_stamp().to_seconds();
       *outFile<<"                                         "
			<<" Resp end at "<<ttime<<endl;
#endif
              break;
      default: break;
    }
    m_EventSelectFwD = 0;
    m_EventSelectFw = 0;
    m_EventSelectBw = 0;
  }

  void update()
  {
#ifdef TRACE_C
    double ttime;
#endif

    switch(m_EventSelectFw)
    {
      case 1: 
        if (m_SyncEvent) m_CommCl->RequestStartEvent.notify(SC_ZERO_TIME);
        m_CommCl->RequestStart = true;
        m_CommCl->GetRequestPending = false;
	m_CommCl->RequestTogglePE = 1 - m_CommCl->RequestTogglePE;
        if (m_DefaultEvent) m_CommCl->StartEvent.notify(SC_ZERO_TIME);
#ifdef TRACE_C
        ttime = sc_time_stamp().to_seconds();
        *outFile<<" "<<m_DataCl->SgetMAddr()<<" Req start at "<<ttime<<endl;
#endif
        break;
      case 2:
        if (m_SyncEvent) m_CommCl->RequestEndEvent.notify(SC_ZERO_TIME);
        if (m_CommCl->RequestPending == false)
          m_CommCl->RequestEnd = true;
        else
          m_CommCl->RequestPending = false;
#ifdef TRACE_C
        ttime = sc_time_stamp().to_seconds();
        *outFile<<" "<<m_DataCl->SgetMAddr()<<" Req end at "<<ttime<<endl;
#endif
        break;
      default: break;
    }
    switch(m_EventSelectFwD)
    {
      case 1: 
        if (m_SyncEvent) m_CommCl->DataRequestStartEvent.notify(SC_ZERO_TIME);
        m_CommCl->DataRequestStart = true;
        m_CommCl->GetDataRequestPending = false;
	m_CommCl->DataRequestTogglePE = 1 - m_CommCl->DataRequestTogglePE;
        if (m_DefaultEvent) m_CommCl->StartEvent.notify(SC_ZERO_TIME);
#ifdef TRACE_C
        ttime = sc_time_stamp().to_seconds();
        *outFile<<"Data Req start at "<<ttime<<endl;
#endif
        break;
      case 2:
        if (m_SyncEvent) m_CommCl->DataRequestEndEvent.notify(SC_ZERO_TIME);
        if (m_CommCl->DataRequestPending == false)
          m_CommCl->DataRequestEnd = true;
        else
          m_CommCl->DataRequestPending = false;
#ifdef TRACE_C
        ttime = sc_time_stamp().to_seconds();
        *outFile<<"Data Req end at "<<ttime<<endl;
#endif
        break;
      default: break;
    }
    switch(m_EventSelectBw)
    {
      case 1:
        if (m_SyncEvent) {
          m_CommCl->ResponseStartEvent.notify(SC_ZERO_TIME);
        }
        m_CommCl->ResponseStart = true;
        m_CommCl->GetResponsePending = false;
	m_CommCl->ResponseTogglePE = 1 - m_CommCl->ResponseTogglePE;
        if (m_DefaultEvent) m_CommCl->StartEvent.notify(SC_ZERO_TIME);
#ifdef TRACE_C
        ttime = sc_time_stamp().to_seconds();
        *outFile<<"                                         "
			<<" Resp start at "<<ttime<<endl;
#endif
        break;
      case 2:
        if (m_SyncEvent) m_CommCl->ResponseEndEvent.notify(SC_ZERO_TIME);
        if (m_CommCl->ResponsePending == false)
          m_CommCl->ResponseEnd = true;
        else
          m_CommCl->ResponsePending = false;
#ifdef TRACE_C
       ttime = sc_time_stamp().to_seconds();
       *outFile<<"                                         "
			<<" Resp end at "<<ttime<<endl;
#endif
              break;
      default: break;
    }
    // Reset dataflow update state
    m_EventSelectFwD = 0;
    m_EventSelectFw = 0;
    m_EventSelectBw = 0;

  }

//---------------------------------------------------------------
//
// Access to the events of the Communication class (Added by Prosilog) 
//
//---------------------------------------------------------------

	virtual const sc_event& RequestStartEvent() const
	{ return m_CommCl->RequestStartEvent; }
	virtual const sc_event& RequestEndEvent() const
	{ return m_CommCl->RequestEndEvent; }
	virtual const sc_event& DataRequestStartEvent() const
	{ return m_CommCl->DataRequestStartEvent; }
	virtual const sc_event& DataRequestEndEvent() const
	{ return m_CommCl->DataRequestEndEvent; }
	virtual const sc_event& ResponseStartEvent() const
	{ return m_CommCl->ResponseStartEvent; }
	virtual const sc_event& ResponseEndEvent() const
	{ return m_CommCl->ResponseEndEvent; }


//------------------ End of public methods ----------------------


private:

  void RequestUpdateFw(int i)
  {
    m_EventSelectFw = i;
    if (m_Synchron) m_Synchronizer.Update();
    else AsyncUpdate();
  }
  
  void RequestUpdateFwD(int i)
    {
    m_EventSelectFwD = i;
    if (m_Synchron) m_Synchronizer.Update();
    else AsyncUpdate();
  }

  void RequestUpdateBw(int i)
  {
    m_EventSelectBw = i;
    if (m_Synchron) m_Synchronizer.Update();
    else AsyncUpdate();
  }

//---------------- End of Methods --------------------------------------


//---------------- Private Data ----------------------------------------

  // master and slave ports to be bounded
  sc_port_base* m_Master;
  sc_port_base* m_Slave;

  // contructor parameter
  bool m_Synchron;
  bool m_UseEvents;
  bool m_SyncEvent;
  bool m_DefaultEvent;

  // Switches for controlling the update() method
  int m_EventSelectFw;
  int m_EventSelectFwD;
  int m_EventSelectBw;

  // Direct interface pointer
  MdirectIF<TdataCl> *m_MdirectIF;
  SdirectIF<TdataCl> *m_SdirectIF;

  // Synchronization classes
  Synchronizer m_Synchronizer;

  // Pointers to the data structure classes provided by the user
  TdataCl          *m_DataCl;
  ParamCl<TdataCl> *m_ParamCl;
  
  // Pointer to the communication class
  CommCl           *m_CommCl;

#ifdef TRACE_C
  // Output file to trace the communication
  ofstream *outFile;
#endif
}; // end module TL_Channel

#endif  // _TL_CHANNEL_H



