/*
 *  Copyright
 * ========================================================================================
 * Project:		Accurate DRAM Model
 * Author:		Yi Wang, KTH
 * ID:			adm_dram_core.h, 1.0, 2010/11/23
 *
 * Description:	A DRAM core that has the following 2 major functions
 * 				1. Perform the command from the command decoder and add inner delay
 * 				2. Inspect the time queue to provide the controller with appropriate responses
 *
 * ========================================================================================
 * Version History
 * ========================================================================================
 * Version 1.0: First Executable version.
 * Version 0.1: Draft version
 * ========================================================================================
 */

#ifndef ADM_DRAM_CORE_H_
#define ADM_DRAM_CORE_H_

#include "systemc.h"
#include "adm_dram_ctrl_ifs.h"
#include "adm_configuration.h"
#include <deque>
#include <map>

class adm_dram_core: public sc_module, public adm_core_if {
public:
	/* IO ports */
	sc_in<bool> clk;
	sc_export<adm_core_if> core_target_port;
	sc_port<adm_ctrl_if> core_initiator_port;

	/* Constructors */
	SC_HAS_PROCESS(adm_dram_core);
	adm_dram_core(sc_module_name nm, adm_configuration admConfig)
			: sc_module(nm)
			, m_admConfig(admConfig)
			, core_initiator_port("Core_Initiator_Port")
			, core_target_port("Core_Target_Port")
	{
		core_target_port(*this);
		SC_THREAD(responseRead);
		sensitive<<clk.pos();
		dont_initialize();
	}

	void responseRead()
	{
		while (true){
			wait();
			wait(SC_ZERO_TIME);
			wait(SC_ZERO_TIME);
			if(m_timeQueue.size() != 0){
				adm_data resultData = m_timeQueue.back();
				m_timeQueue.pop_back();
				if (resultData.getLength() != 0) {
					core_initiator_port->putResponse(resultData);
				}
			}

		}
	}

	void coreHopRow()
	{
		virtualDelay(m_admConfig.m_tHopRow);
	}

	void coreReactivate()
	{
		virtualDelay(m_admConfig.m_tRCD);
	}

	void coreWrite(uint64 address, const adm_data & data, const int tPreparation, const int burstCycle)
	{
		for (int i = 0; i < data.getLength(); ++i) {
			m_storage.insert(std::pair<const uint64, unsigned char>(address+i , data[i]));
		}
		if (tPreparation != 0) {
			virtualDelay(m_admConfig.m_tDQSS);
		}
		else {
			virtualDelay(m_admConfig.m_tDQSS - m_timeQueue.size());
		}
		virtualDelay(burstCycle);
	}

	void coreRead(uint64 address, int length, int tPreparation, int burstCycle)
	{
		adm_data responseData(length);
		adm_data dummyTime;
		if (tPreparation != 0) {
			virtualDelay(m_admConfig.m_tCAS);
		}
		else {
			virtualDelay(m_admConfig.m_tCAS - m_timeQueue.size());
		}
		for (int i = 0; i < length; ++i) {
			responseData[i] = m_storage[address + i];
		}
		m_timeQueue.push_front(responseData);
		virtualDelay(burstCycle - 1);
	}

	void virtualDelay(int delay)
	{
		adm_data dummyTime = adm_data();
		for (int i = 0; i < delay; ++i) {
			m_timeQueue.push_front(dummyTime);
		}
	}

	int getRemainCycles()
	{
		return m_timeQueue.size();
	}

	~adm_dram_core()
	{
	}
private:
	adm_configuration m_admConfig;
	std::deque<adm_data> m_timeQueue;
	std::map<uint64, unsigned char> m_storage;
};

#endif /* ADM_DRAM_CORE_H_ */
