/* Copyright
* ========================================================================================
* Project:       Reference DRAM model
* Author:        Lei Liang, KTH
* ID:            sampdram.cpp
* ========================================================================================
*/

#include "sampdram.h"

using std::endl;
using std::cout;


template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> sampdram <BUSWIDTH, ADDRWIDTH>::sampdram (sc_core::sc_module_name name_, int delay)
: sc_core::sc_module (name_), ocpInitPort("tpPort"), req(NULL),reqTemp(NULL),response(NULL),m_delayCycles(delay > 0 ? delay - 1 : 0) {

	SC_METHOD(proc);
	sensitive<<clk.pos();
	dont_initialize();
	/*bind the transaction */
	ocpInitPort.register_nb_transport_fw(this, &sampdram::nb_transport);

	m_burstLength=0;
    m_respOngoing=false;
    m_srmdOn=false;
	m_queueStopped=false;
	m_posted=false;
	m_burstOn=false;
	m_needSrmdpost=false;
	m_needMrmdpost=false;
	m_needendsrmd=false;


	/*put empty pointers into the queue */

	for(int i=0; i<m_delayCycles;i++){
		queue.push_back(queuedata());
	}
	/*OCP-IP synchronization, the proc can run before the nb_transport function */
	ocpInitPort.activate_synchronization_protection();
}


/* Destructor */
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> sampdram<BUSWIDTH, ADDRWIDTH>::~sampdram(){}

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> void sampdram<BUSWIDTH, ADDRWIDTH>::proc(){

	bool readFlag=false;
	tlm::tlm_command request;										//to store the request command


	/* When the master do not accepted the data sent by this sample DRAM, the DRAM will not send END_REQ
	 * when it got the REQ operation. Thus when the data is accepted, an END_REQ need to be sent here
	 * When END_REQ is sent, no read or write operation can be operated in this cycle since only one
	 * response can be sent in one cycle
	 *
	 * m_respOngoing will be false if the master accepted the data. when the END_RESP is not sent in last
	 * cycle, m_respOngoing is true and m_queueStopped is trued in this cycle. So when m_respOngoing is
	 * false and m_queueStopped is true means the END_RESP is sent and the transaction has been unblocked
	 * in the last cycle. So extra END_REQ need to be sent at this time
	 */
	if((!m_respOngoing) && (m_queueStopped)){
		extraendreq();
		m_queueStopped=false;
	   }

	/*If two burst write or read operation REQ is too close to let the burst operation finished
	 * The second REQ will not accepted until the first burst operation finished. When it is
	 * finished, an END_REQ need to be sent at this time. And no other operation can be done in
	 * this cycle
	 * */
	else if((!m_srmdOn) && (m_needendsrmd)){
			m_srmdOn=true;
			m_needendsrmd=false;
			req=reqTempburst;
			time=sc_core::SC_ZERO_TIME;
			phase=tlm::END_REQ;
			ocpInitPort->nb_transport_bw(*req,phase, time);

			}
	/* The following code contains three parts:
	 * A: Write data into the memory
	 * B: Push pointer into the queue
	 * C: Pop pointer out of the queue
	 * */
	else{
		ocpip::burst_length* b_len;

		/* Go in if transaction is not blocked and there is a REQ operation
		 * Write the data into the memory or true the flag for read operation
		 * */
		if((req) && (!m_respOngoing))
		{
			request = req->get_command();

			/* Capture the burst_length. If there is no burst_length, calculate it.*/
			ocpInitPort.ocpip::extension_api::get_extension<ocpip::burst_length>(b_len, *req);
			if (b_len){m_burstLength=b_len->value;}
			else{
				SC_REPORT_WARNING(SC_ID_WITHOUT_MESSAGE_, "burst_length extension is not used. Calculating burst length from data length.");
			    m_burstLength=(req->get_data_length() / sizeof(Td));

			}

			/*Capture the parameter to see if srmd or mrmd is needed*/
			bool has_srm =ocpip::extension_api::get_extension<ocpip::srmd>(*req);
			if (has_srm) m_srmdOn=true;								//m_srmdOn will be set false when Srmd write finished
			else m_srmdOn=false;

			/*Judge Posted write or None posted is needed*/
			m_posted=ocpip::extension_api::get_extension<ocpip::posted>(*req);

			if(request==tlm::TLM_WRITE_COMMAND){
				m_burstOn=true;												//set false when burst finished
				burstwrite();

			}
			else{readFlag=true;}
		}

	/* Push pointer into the queue.
	 * A: Read operation: push the data into the transaction and put pointer into the queue
	 * B: Write operation: push the wrtie ACK into the queue if needed
	 * C: Empty operation: push a empty into the queue if there is no REQ
	 * */
		if(!m_respOngoing){													//not blocked
			if(readFlag==true) {
				m_burstOn=true;													//set false when read finished
				push();															//A operation
			}
			else if((m_needMrmdpost)||(m_needSrmdpost)){						//B operation
					queue.push_back(queuedata(req));
					req=NULL;
					if(m_needMrmdpost==true){									//Mrmd ACK pushed,false the flag
						m_needMrmdpost=false;
//						cout<<"[From sampdram @ "<<sc_time_stamp()<<"]"<<" [Mrmd_posted_write] "
//											   	 <<"[pointer pushed into queue is ] "<<req<<endl;
					}
					else {														//Srmd ACK pushed,false the flag
						m_needSrmdpost=false;
//						cout<<"[From sampdram @ "<<sc_time_stamp()<<"]"<<" [Srmd_posted_write] "
//																	   	 <<"[data pushed into queue is ] "<<req<<endl;
					}
			}
			else{																//C operation
					queue.push_back(queuedata());
//					cout<<"[From sampdram @ "<<sc_time_stamp()<<"]"<<" [push] "
//					   	 <<"[data pushed into queue is ] "<<"empty"<<endl;

			}
			queuePop();
			m_queueStopped=false;
		}

		else{																	//transaction blocked
			m_queueStopped=true;
			reqTemp=req;
//			cout<<"[From sampdram @ "<<sc_time_stamp()<<"]"<<"[Transaction is blocked"
//					<<"since RESP is not Ended"<<endl;
			}
	}

}


template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
tlm::tlm_sync_enum sampdram<BUSWIDTH, ADDRWIDTH>::nb_transport(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim){
    if(ph==ocpip::BEGIN_DATA){
    	req=&txn; ph=ocpip::END_DATA; return tlm::TLM_UPDATED;
    }
    else{
    	switch(ph){
    		/*
    		 * In 2 situation the END_REQ will not be sent
    		 * A: Two write or read is requested to close to let the first finished
    		 * B: END_RESP is not sent
    		 */
    		case tlm::BEGIN_REQ:
    				if(m_burstOn==false){
    					if(m_queueStopped==0){
    						req=&txn; ph=tlm::END_REQ; return tlm::TLM_UPDATED; break;}
    					else{
    						req=&txn; return tlm::TLM_ACCEPTED;break;}
    				}
    				else{
    					reqTempburst=&txn; m_needendsrmd=true;return tlm::TLM_ACCEPTED;}

    		case tlm::END_RESP:  m_respOngoing=0;
//    				cout<<"END_RESP is sent out at time "<<sc_time_stamp()<<endl;
    				return tlm::TLM_ACCEPTED; break;
    		default:
    			std::cerr<<"Error in "<<name()<<" got unexpected phase "<<phase<<" in nb_transport_fw"<<std::endl;
    			exit(1);
    	}
    }
return tlm::TLM_ACCEPTED;
}

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> void sampdram<BUSWIDTH, ADDRWIDTH>::queuePop(){
	if (queue.front().empty)  {								//not pointer stored in this cycle
		queue.pop_front();
//		cout<<"[From sampdram @ "<<sc_time_stamp()<<"]"<<" [pop] "
//					    	 <<"[data popped out from queue is empty] "<<endl;
	}
	else {
		m_respOngoing=true;									//mark the flag to the status of response,this will be false when END_RESP sent
		time=sc_core::SC_ZERO_TIME;
		phase=tlm::BEGIN_RESP;
		response=queue.front().rsp;
		tlm::tlm_sync_enum retVal=ocpInitPort->nb_transport_bw(*response,phase, time);
		switch(retVal){
				     case tlm::TLM_ACCEPTED:
//				    	 cout<<"[From sampdram @ "<<sc_time_stamp()<<"]"<<" [pop] "
//				    	 <<"[data popped out from queue is accepted] "<<endl;
				    	 break;
				     /*	when activate_synchronization_protection() is used, TLM_UPDATED will not accepted
				      * But m_respOngoing will also be false when END_REQ is sent 2p after.
				      * */
				     case tlm::TLM_UPDATED: //For this programme, this case is never used
//				    	  cout<<"Printed out from sampdram transport_bw: Status- BEGIN_RESP is updated"<<endl;
				    	  m_respOngoing=false;
				          break;
				     case tlm::TLM_COMPLETED:
				    	  std::cout<<"Error"<<" Printed out from sampdram transport_bw: Error happened "<<endl;;
				    	  break;

				      }
		queue.pop_front();

	}

}

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> void sampdram<BUSWIDTH, ADDRWIDTH>::burstwrite(){
	static int bc=0;
	if(m_srmdOn){
		if(bc==0) bc++;
		else{
				m_Memory.write((req->get_address()+(bc-1)*4),*(((Td*)(req->get_data_ptr()))+bc-1));
//				cout<<"[From sampdram @ "<<sc_time_stamp()<<"]"<<" [burstwrite] "<<"[data written into the memory] "
//									<<*(((Td*)(req->get_data_ptr()))+bc-1)<<endl;
				if (bc==m_burstLength){									//write finished
						m_srmdOn=false;
						m_burstOn=false;
						if (m_posted==false)m_needSrmdpost=true;
						else req=NULL;
						bc=0;

				}
				else bc++;
		}
	}
	else{
			m_Memory.write((req->get_address()+bc*4),*(((Td*)(req->get_data_ptr()))+bc));
			//	cout<<"[From sampdram @ "<<sc_time_stamp()<<"]"<<" [burstwrite] "<<"[data written into the memory] "
			//			<<*(((Td*)(req->get_data_ptr()))+bc)<<endl;
			bc++;
			if (!m_posted)m_needMrmdpost=true;
			else req=NULL;
			if (bc==m_burstLength) {
					m_burstOn=false;
					bc=0;


			}
	}

}


template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> void sampdram<BUSWIDTH, ADDRWIDTH>::push(){
	static int brbc=0;

	*(((Td*)(req->get_data_ptr()))+brbc) = m_Memory.read(req->get_address()+brbc*4);
	queue.push_back(req);

//	cout<<"[From sampdram @ "<<sc_time_stamp()<<"]"<<" [push] "<<"[data pushed into the queue is] "
//			<<m_Memory.read(req->get_address()+brbc*4)<<req<<endl;

	brbc++;

	if (brbc==m_burstLength){
		brbc=0;
		m_srmdOn=false;
		m_burstOn=false;
		req=NULL;

	}
}


template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> void sampdram<BUSWIDTH, ADDRWIDTH>::extraendreq(){
	req=reqTemp;
	time=sc_core::SC_ZERO_TIME;
	phase=tlm::END_REQ;
	ocpInitPort->nb_transport_bw(*req,phase, time);
//	cout<<"[From sampdram @ "<<sc_time_stamp()<<"]"<<" [Extra_END_REQ] "<<"[An extra END_REQ has been sent] "<<endl;

}

/*Instantiation of the sampdram*/
template class sampdram< 32,32 >;
