/* Copyright
 * ========================================================================================
 * Project:		Accurate DRAM Model
 * Author:		Nan Li, KTH
 * ID:			test_gen.h, v1.2, 2010/12/04
 *
 * Description:	Test generator for accurate DRAM model
 *
 * ========================================================================================
 * Version History
 * ========================================================================================
 * Version 1.2: Added support for thread_id extension
 * Version 1.1: Generates CSV file instead of simple text file.
 * Version 1.0: implemented capability to read in requests from file and to generate a
 *              simple log file.
 * Version 0.5: burstWrite() and burstRead() implemented. Tested.
 * Version 0.1: Draft version
 * ========================================================================================
 */

#ifndef TEST_GEN_H_
#define TEST_GEN_H_

#include "systemc.h"
#include "ocpip.h"
#include <vector>

#define BUSWIDTH 32

class adm_test_gen : public sc_module {
public:
	typedef ocpip::ocp_data_class_unsigned<BUSWIDTH, 32>::DataType Td;
	struct txn_info {
		unsigned int id;
		bool write;
		uint64 address;
		unsigned int length;	/* number of words */
		uint64 start;
		uint64 end;
		unsigned int thread_id;
	};

	ocpip::ocp_master_socket_tl1<sizeof(Td)*8> ocpInitPort;
	sc_in_clk clk;

	SC_HAS_PROCESS(adm_test_gen);
	adm_test_gen(const sc_module_name &nm, const char *testFile, const char *logFile)
			: sc_module(nm),
			  ocpInitPort("init_port", ocpip::ocp_master_socket_tl1<sizeof(Td)*8>::mm_txn_with_data()),
			  req(0),
			  rsp(0),
			  finTest(testFile),
			  foutLog(logFile),
			  cycle(0),
			  txnId(0),
			  nextRspId(0)
	{
        if (!finTest) {
            cerr << "Cannot open file: " << testFile << endl;
            exit(1);
        }
        if (!foutLog) {
            cerr << "Cannot open file for write: " << logFile << endl;
            exit(1);
        }

		SC_THREAD(test);
		sensitive << clk.pos();
		dont_initialize();

		SC_METHOD(handleResponse);
		sensitive << clk.pos();
		dont_initialize();

		SC_THREAD(countCycle);
		sensitive << clk.pos();
		dont_initialize();

		ocpInitPort.register_nb_transport_bw(this, &adm_test_gen::nb_transport_bw);
		ocpInitPort.activate_synchronization_protection();
	}
	virtual ~adm_test_gen() {}

private:
	/* processes */
	virtual void test();
	void handleResponse();
	void countCycle();

	tlm::tlm_sync_enum nb_transport_bw(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim);
	void burstWrite(uint64 address, unsigned int threadId, unsigned int length, bool posted = false, bool srmd = false);
	void burstRead(uint64 address, unsigned int threadId, unsigned int length, bool srmd = false);

	unsigned int calculateBurstLength(tlm::tlm_generic_payload &txn);

	tlm::tlm_generic_payload *req, *rsp;
	std::ifstream finTest;
	std::ofstream foutLog;

	uint64 cycle;
	unsigned int txnId;
	std::map<tlm::tlm_generic_payload *, unsigned int> txnIdMap;

	std::vector<txn_info> txnSummary;
	unsigned int nextRspId;
};


#endif /* TEST_GEN_H_ */
