/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE(delay)
{
  sca_tdf::sca_in<double> inp;
  sca_tdf::sca_out<double> outp;

  void set_attributes();
  void initialize();
  void processing();

  SCA_CTOR(delay) : inp("inp"), outp("outp") {}
};

void delay::set_attributes()
{
  inp.set_delay(1);
}

void delay::initialize()
{
  inp.initialize(0.0);
}

void delay::processing()
{
  outp = inp;
}

/////////////////////////////

SCA_TDF_MODULE(adder)
{
  sca_tdf::sca_in<double> inp1;
  sca_tdf::sca_in<double> inp2;
  sca_tdf::sca_out<double> outp;

  void processing();

  SCA_CTOR(adder) 
  : inp1("inp1"), inp2("inp2"), outp("outp") {}
};


void adder::processing()
{
  outp.write(inp1.read() + inp2.read());
}

///////////////////////////////////

SCA_TDF_MODULE(const_src)
{
  sca_tdf::sca_out<double> outp;

  void set_attributes();
  void processing();

  SCA_CTOR(const_src) : outp("outp") {}
};

void const_src::set_attributes()
{
  outp.set_timestep(1.0, sc_core::SC_MS);
}

void const_src::processing()
{
  outp = 1.0;
}

SCA_TDF_MODULE(sink)
{
  sca_tdf::sca_in<double> inp;

  void processing();

  SCA_CTOR(sink) : inp("inp") {}
};

void sink::processing()
{
  double tmp = inp.read();

  std::ostringstream str;
  str << get_time() << "  " << name()
      << " current value: " << tmp;

  SC_REPORT_INFO("Simple_TDF", str.str().c_str());
}

////////////////////////////////

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1("sig1");
  sca_tdf::sca_signal<double> sig2;
  sca_tdf::sca_signal<double> sig3("sig3");

  adder add1("add1");
  add1.inp1(sig1);
  add1.inp2(sig2);
  add1.outp(sig3);

  delay del1("del1");
  del1(sig3, sig2); // positional binding

  const_src src1("src1");
  src1.outp(sig1);

  sink sink1("sink1");
  sink1.inp(sig3);

  sc_core::sc_start(10.0, sc_core::SC_MS);

  TEST_LABEL_END;

  return 0;
}
