/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/


#include <systemc-ams>
#include "test_utilities.h"
#include <string>

bool check_val(double val, double e_val, double rel_err, double abs_err)
{
  if ( (std::fabs(val - e_val) > abs_err) && (std::fabs(val - e_val) / e_val > rel_err))
  {
    std::cout << " Expect value: " << e_val << " read value: " << val << "  diff: " << val - e_val << std::endl;
    return true;
  }

  return false;
}

int check_values(double time, double delay, double rampv, double rel_err = 1e-10, double abs_err = 1e-10)
{
  double ctime = time-delay;
  double e_rampv = 0.0;


  if (ctime >= 0.0)
  {
    e_rampv = ctime;
  }
  else
  {
    return 0;
  }

  if (check_val(rampv, e_rampv, rel_err, abs_err))
  {
    return 2;
  }

  return 0;
}


bool check_file(std::string fname)
{
  std::ifstream fin;
  fin.open(fname.c_str());

  if (fin.fail())
  {
    std::ostringstream str;
    str << "Can't open file: " << fname;
    SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
  }

  std::string line_in;
  unsigned long line_cnt = 0;
  if (!std::getline(fin, line_in))
  {
    std::ostringstream str;
    str << "Can't get header line from file: " << fname;
    SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
  }
  line_cnt++;

  std::vector<std::string> names;
  std::istringstream nistr(line_in);

  std::string name;
  if (!(nistr >> name) && (name != "%time"))
  {
    std::ostringstream str;
    str << "Wrong header line in file: " << fname;
    SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
  }

  while (nistr >> name)
  {
    names.push_back(name);
  }

  while (std::getline(fin, line_in))
  {
    std::istringstream istr(line_in);
    line_cnt++;

    double ctime;
    istr >> ctime;

    if (istr.fail())
    {
      std::ostringstream str;
      str << "Can't get time in file: " << fname << " line: " << line_cnt;
      SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
    }

    for (unsigned long i = 0; i < names.size(); i++)
    {
      double rval;
      istr >> rval;

      if (istr.fail())
      {
        std::ostringstream str;
        str << "Failed to read values from file: " << fname << " line: " << line_cnt;
        str << " for: " << names[0] << " ";
        str << " at time: " << ctime;
        SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
      }

      int err = check_values(ctime, sca_core::sca_time(0.8, sc_core::SC_US).to_seconds(), rval);
      if (err != 0)
      {
        std::ostringstream str;
        str << "Wrong value in file: " << fname << " line: " << line_cnt;
        str << " for signal: " << names[0] << " at time " << ctime;
        str << " delay: " << sca_core::sca_time(1.6, sc_core::SC_US) << " d index: " << i;
        SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
      }
    }
  }

  return false;
}

///////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(ltf_nd_src)
{
  sca_tdf::sca_out<double> outp_ramp_r;

  struct params
  {
    int rate;
    sca_core::sca_time sample_time;
    double f_sin;

    params()
    {
      rate = 5;
      sample_time = sca_core::sca_time(1.0, sc_core::SC_US);
      f_sin = 13.3e3;
    }
  };

  void set_attributes()
  {
    set_timestep(p.sample_time);
    outp_ramp_r.set_rate(p.rate);
    does_attribute_changes();
  }

  void initialize();
  void processing();

  void change_attributes()
  {
    switch(activation)
    {
      case 0: // mstep=1ms
        break;

      case 5:
        break;

      case 6:
        break;

      case 7:
        break;

      case 8: // change timestep to 2sec and calculate to end
        set_timestep(2.0, sc_core::SC_US);
        break;

      case 9:
        break;

      case 10:
        break;

      case 11:
        break;

      case 12:
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 13:
        break;

      case 14: // 3 zero timestep module iterations, calculation to end
      case 15:
      case 16:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 17: // 4 zero timestep
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 18: // module timestep 1sec 4 zero tstep iterations
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 19: // calculate to end
        break;

      case 20: // 2 zero module timestep iterations
      case 21:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 22:
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 23:
      case 24:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 25:
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 26:
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 27:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 28:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 29:
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 30:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 31: // reset to begin of case 28
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 32:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 33: // normal full timestep
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 34: // reset to tstep before last timestep (before last iteration of 32)
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 35: // repeat as normal full timestep
        request_next_activation(0.0, sc_core::SC_SEC);
        break;

      case 36:
        sc_core::sc_stop();
        break;

      default: break;
    }
  }

  ltf_nd_src(sc_core::sc_module_name nm, params pa = params()) :
    p(pa)
  {
    activation = 0;
  }
  params p;

private:
  sca_tdf::sca_ltf_nd nd41;
  sca_tdf::sca_ltf_nd ndelay;

  sca_util::sca_vector<double> N1, D1;
  sca_util::sca_vector<double> N2, D2;

  int activation;
};

void ltf_nd_src::initialize()
{
  N1(0) = 1.0;
  D1(0) = 1.0;  // y=x - x=const=1.0

  N2(0) = 1.0;
  D2(1) = 1.0;  // y=t*x - x=const=1.0

  ndelay.set_max_delay(sc_core::sc_time(1600, sc_core::SC_NS));
}

// time domain implementation
void ltf_nd_src::processing()
{
  sca_util::sca_vector<double> out_vec1(p.rate);

  out_vec1 = nd41(N2, D2, 1.0);
  outp_ramp_r = ndelay(N1, D1, sc_core::sc_time(1600, sc_core::SC_NS), out_vec1);
  activation++;
}

int sc_main(int argn, char* argc[])
{
  sc_core::sc_set_time_resolution(1.0, sc_core::SC_FS);

  sca_util::sca_information_off();

  TEST_LABEL_START;

  // define non-conservative signals
  sca_tdf::sca_signal<double> s_out_rampr;

  // instantiate and connect components

  ltf_nd_src* i_src = new ltf_nd_src("i_src");
  i_src->outp_ramp_r(s_out_rampr);

  /////// Tracing -> for each rate a different file

  sca_util::sca_trace_file* tfr = sca_util::sca_create_tabular_trace_file("tdf_nd_ramp_rate.dat");

  sca_util::sca_trace(tfr, s_out_rampr, "s_out_rampr");

  ////////////////////////////////////////////////

  sc_core::sc_start();

  sca_util::sca_close_tabular_trace_file(tfr);

  check_file("tdf_nd_ramp_rate.dat");

  TEST_LABEL_END;

  return 0;
}

