/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

//  circuit2.cpp  - eln regression test

#include <systemc-ams>
#include "test_utilities.h"

SC_MODULE(circuit2)
{  
  // port declaration
  sca_tdf::sca_in<double> inp1;
  sc_core::sc_in<bool>    inp2;

  // node declaration
  sca_eln::sca_node n1;
  sca_eln::sca_node_ref n2; // reference node
  sca_eln::sca_node n3;
  sca_eln::sca_node n4;
  sca_eln::sca_node n5;

  // component declaration
  sca_eln::sca_tdf_vsource *vin;
  sca_eln::sca_de_rswitch  *rswitch;
  sca_eln::sca_r           *r1, *r2, *r3, *r4;

  struct params // parameter
  {
    double r1_val;
    double r2_val;
    double r3_val;
    double r4_val;

    params() // default for parameter
    {
      r1_val = 1.0e3;
      r2_val = 4.0e3;
      r3_val = 2.0e3;
      r4_val = 3.0e3;
    }
  };

  circuit2( sc_core::sc_module_name nm, params pa = params() ) 
  : inp1("inp1"), inp2("inp2"), n1("n1"), n2("n2"), n3("n3"), n4("n4"), n5("n5"), p(pa)
  {
    architecture(); // generate netlist
  }

  ~circuit2();

 private:
  params p;
  void architecture(); // method for netlist
};

// netlist implementation

void circuit2::architecture()
{
  r1 = new sca_eln::sca_r("r1");
  r1->value = p.r1_val;
  r1->n(n1);
  r1->p(n5);

  r2 = new sca_eln::sca_r("r2");
  r2->value = p.r2_val;
  r2->n(n2);
  r2->p(n1);

  r3 = new sca_eln::sca_r("r3");
  r3->value = p.r3_val;
  r3->n(n3);
  r3->p(n2);

  r4 = new sca_eln::sca_r("r4");
  r4->value = p.r4_val;
  r4->n(n4);
  r4->p(n3);

  rswitch = new sca_eln::sca_de_rswitch("rswitch");
  rswitch->n(n3);
  rswitch->p(n1);
  rswitch->ctrl(inp2);
  rswitch->ron = 1.0e-12;
  rswitch->roff = 1.0e12;
  rswitch->off_state = false;

  vin = new sca_eln::sca_tdf_vsource("vin");
  vin->inp(inp1);
  vin->p(n5);
  vin->n(n4);
}

circuit2::~circuit2()
{
  delete r1, r2, r3, r4, rswitch, vin;
}

void check_td_results(std::string fname)
{
  bool test_passed_with_success = true;
  std::ifstream tdf(fname.c_str());
  std::string line;
  long test_cnt = 0;

  std::cerr.precision(15);

  getline(tdf, line);
  std::cout << "1..32" << std::endl;
  while (getline(tdf, line))
  {
    std::istringstream str(line);

    double time;
    str >> time;

    double node1, node2, node3, i_vin, i1, i2, i3, i4, i5;
    str >> node1 >> node2 >> node3 >> i_vin >> i1 >> i2 >> i3 >> i4 >> i5;

    if ( !((test_util::check_value(i5, 6e-12, 1e-6) && (time <= 0.04)) ||
        (test_util::check_value(i5, 0.0025) && (time > 0.04))) )
    {
      std::cerr.precision(15);
      std::cerr << "Error TD-check failed for i5 at: " << time
                << " i5: " << " expect: " << 6e-12 << "  " << i5 << std::endl;
      std::cout << "not ";
      test_passed_with_success = false;
    }
    std::cout << "ok " << ++test_cnt << std::endl;

    if ( !( (test_util::check_value(i1, i2+i5) && test_util::check_value(i2, i3) && (time <= 0.04) ) ||
        (test_util::check_value(i4, i5+i3) && test_util::check_value(i2, i3, 1.4e-18) &&
         test_util::check_value(i1, i2+i5) && (time > 0.03)) ) )
    {
      std::cerr << "Error TD-check failed for verification of currents at: " << time << std::endl;
      std::cout << "not ";
      test_passed_with_success = false;
    }
    std::cout << "ok " << ++test_cnt << std::endl;

    if ( !((test_util::check_value(node1, 3.9999999904) && (time <= 0.04)) ||
        (test_util::check_value(node1, 1.6e-15, 2e-15) && (time > 0.04))) )
    {
      std::cerr << "Error TD-check failed for node1 at: " << time << std::endl;
      std::cout << "not ";
      test_passed_with_success = false;
    }
    std::cout << "ok " << ++test_cnt << std::endl;

    if ( !((test_util::check_value(node3, -1.9999999952) && (time <= 0.04)) ||
        (test_util::check_value(node3, -0.8e-15, 2e-15) && (time > 0.04))) )
    {
      std::cerr << "Error TD-check failed for node3 at: " << time << std::endl;
      std::cout << "not ";
      test_passed_with_success = false;
    }
    std::cout << "ok " << ++test_cnt << std::endl;
  };

  if (test_passed_with_success) std::cout << "results ok." << std::endl;
}

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  // define non-conservative signals
  sca_tdf::sca_signal<double> signal1;
  sc_core::sc_signal<bool> signal2;

  test_util::const_src::params src_params;
  src_params.ampl = 10;

  test_util::const_src* src = new test_util::const_src("source", src_params);
  src->outp(signal1);

  circuit2* cir = new circuit2("circuit");
  cir->inp1(signal1);
  cir->inp2(signal2);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("output2.dat");
  sca_util::sca_trace(tf, cir->n1, "node1");
  sca_util::sca_trace(tf, cir->n2, "node2");
  sca_util::sca_trace(tf, cir->n3, "node3");
  sca_util::sca_trace(tf, cir->vin, "i_through_vin");
  sca_util::sca_trace(tf, cir->r1, "i1");
  sca_util::sca_trace(tf, cir->r2, "i2");
  sca_util::sca_trace(tf, cir->r3, "i3");
  sca_util::sca_trace(tf, cir->r4, "i4");
  sca_util::sca_trace(tf, cir->rswitch, "i5");

  signal2 = false;

  sc_core::sc_start(30, sc_core::SC_MS);

  signal2 = true;

  sc_core::sc_start(50, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(tf);

  check_td_results("output2.dat");

  TEST_LABEL_END;

  delete src, cir;
  
  return 0;
}
