/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

#include <systemc-ams>
#include "test_utilities.h"

SC_MODULE(test01)
{
  sca_tdf::sca_in<double> inp1;                          // port declaration

  sca_eln::sca_node     n1;                              // node declaration
  sca_eln::sca_node_ref n2;
  sca_eln::sca_node     n_gnd;                           // reference is between the capacitor and the resistance, so the capacitor voltage will be negative

  sca_eln::sca_vsource    *src1;                         // component declaration
  sca_eln::sca_tdf::sca_l *l1;
  sca_eln::sca_r          *r1;

  struct params                                          // parameter
  {
    params()                                             // default for parameter
    {
    }
  };

  test01( sc_core::sc_module_name nem, params pa = params() )
  : n1("n1"), n2("n2"), n_gnd("n_gnd")
  {
    architecture();                                      // generate netlist
  }

  ~test01();

 private:
  params p;
  void architecture();                                   // method for netlist
};

void test01::architecture()                              // netlist implementation
{
  src1 = new sca_eln::sca_vsource("src1");
  src1->amplitude = 5.0;
  src1->frequency = 1.0e4;
  src1->offset = 0;
  src1->p(n1);
  src1->n(n_gnd);

  r1 = new sca_eln::sca_r("r1");
  r1->p(n1);
  r1->n(n2);
  r1->value = 1e3;

  l1 = new  sca_eln::sca_tdf::sca_l("l1");
  l1->p(n2);
  l1->n(n_gnd);
  l1->inp(inp1);
  l1->scale = 1.0e-3;
}

test01::~test01()
{
  delete src1, r1, l1;
}

bool check_value(double val1, double val2, double abs_err = 1e-3, double rel_err = 1e-1)
{
    if ( (std::fabs(val1) < abs_err) || (std::fabs(val2) < abs_err) )
    {
        if (std::fabs(val1 - val2) > abs_err) return false;
    }
    else
    {
        if (std::fabs((val1 - val2) / (val1 + val2) / 2.0) > rel_err)
        {
            return false;
        }
    }
    return true;
}

void check_td_results(std::string fname)
{
  bool test_passed_with_success = true;
  std::ifstream tdf(fname.c_str());
  std::string line;

  getline(tdf, line);
  while (getline(tdf, line))
  {
    std::istringstream str(line);

    double time;
    str >> time;

    double i_res;
    str >> i_res;

    double n_gnd;
    str >> n_gnd;

    double inp1;
    str >> inp1;

    double theory;
    double wl = 2.0 * M_PI * 10000 * (1e-3 * inp1);
    double z = sqrt(pow(1000.0, 2.0) + pow((wl), 2.0));
    double phi = atan(wl / (1000));

    theory = ((-5) / (z) * exp((-time) / ((0.001 * inp1)/(1000))) * std::sin(-phi)) + 
              ((5) / (z)) * std::sin( (2 * M_PI * 10000 * time) - phi);

    if (time >= 1e-6)
    {
      if (!check_value(i_res, theory))
      {
        std::cout << "simul " << i_res << " teor= " << theory << std::endl;
        test_passed_with_success = false;
      }
    }
  };

  if (test_passed_with_success) std::cout << "Trace comparison check ok" << std::endl;
}

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> signal1; // non-conservative signals

  test_util::sin_src::params src_params2;
  src_params2.ampl = 5;
  src_params2.freq = 500;
  src_params2.offset = 6;
  src_params2.sample_time = sc_core::sc_time(1, sc_core::SC_US);

  test_util::sin_src* src2 = new test_util::sin_src("source2", src_params2);
  src2->outp(signal1);

  test01* cir = new test01("circuit");
  cir->inp1(signal1);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("test01");
  sca_util::sca_trace(tf, cir->r1, "i_r1");
  sca_util::sca_trace(tf, cir->n_gnd, "n_gnd");
  sca_util::sca_trace(tf, cir->inp1, "inp1");

  sc_core::sc_start(8, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(tf);

  check_td_results("test01.dat");

  TEST_LABEL_END;

  delete src2, cir;
  
  return 0;
}
